# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

"""
Scan the resource providers, add some metadata and store it in the database
"""

from elisa.core import common
from elisa.core.media_uri import MediaUri

from twisted.internet import task, defer, reactor
from twisted.python.failure import Failure

import time

from elisa.core.components.resource_provider import ResourceProvider

from elisa.plugins.database.scanner_models import Statistic, ScanResource
from elisa.plugins.database.database_parser import DatabaseParser

from elisa.plugins.base.local_resource import LocalResource

from elisa.plugins.base.models.file import DirectoryModel


# FIXME: we put here also schema stuff not strictly related with the media
# scanner (the "favorites" table), because we don't have a general
# mechanism to do database stuff yet
SCHEMA = [
        "CREATE TABLE IF NOT EXISTS music_albums" \
                " (name VARCHAR PRIMARY KEY, cover_uri VARCHAR," \
                " release_date VARCHAR);",
        "CREATE TABLE IF NOT EXISTS artists"
                " (name VARCHAR PRIMARY KEY, image_uri VARCHAR);",
        "CREATE TABLE IF NOT EXISTS track_artists" \
                " (artist_name VARCHAR, track_path VARCHAR," \
                " PRIMARY KEY (artist_name, track_path));",
        "CREATE TABLE IF NOT EXISTS music_tracks" \
                " (file_path VARCHAR PRIMARY KEY, title VARCHAR,"\
                " track_number INTEGER, album_name VARCHAR, "\
                " duration INTEGER, genre VARCHAR);",
        "CREATE TABLE IF NOT EXISTS tags"\
                " (name VARCHAR PRIMARY KEY);",
        "CREATE TABLE IF NOT EXISTS files" \
                " (path VARCHAR PRIMARY KEY, source VARCHAR," \
                " modification_time INTEGER DEFAULT 0, mime_type VARCHAR," \
                " deleted INTEGER, playcount INTEGER DEFAULT 0, last_played" \
                " INTEGER DEFAULT 0);",
        "CREATE TABLE IF NOT EXISTS file_tags" \
                " (file_path VARCHAR, tag_name VARCHAR,"\
                " PRIMARY KEY(file_path, tag_name));",
        "CREATE TABLE IF NOT EXISTS images"\
                " (file_path VARCHAR PRIMARY KEY, size VARCHAR, " \
                " shot_time INTEGER, with_flash INTEGER DEFAULT 0," \
                " orientation INTEGER, gps_altitude INTEGER, " \
                " gps_latitude VARCHAR, gps_longitude VARCHAR," \
                " album_name VARCHAR);",
        "CREATE TABLE IF NOT EXISTS photo_albums"\
                " (name VARCHAR PRIMARY KEY, preview_uri VARCHAR);",
        "CREATE TABLE IF NOT EXISTS videos"\
                " (file_path VARCHAR PRIMARY KEY, size VARCHAR,"\
                " duration VARCHAR, codec VARCHAR, thumbnail_uri VARCHAR);",

        "CREATE TABLE IF NOT EXISTS favorites"\
                " (id INTEGER PRIMARY KEY, section TEXT,"\
                " added_time TEXT, title TEXT, subtitle,"\
                " foreign_id TEXT, foreign_class TEXT, uri TEXT);",
        ]

class MediaScanner(ResourceProvider):
    """
    Asynchronous working media scanner.
    
    @ivar store: the store for the data
    @type store: L{storm.twisted.storm.DeferredStore}
    """

    default_config = {"delay" : 0.1}
    config_doc = {
        "delay": "the delay (in seconds) between processing two files"}

    supported_uri = '^media_scanner://localhost/'

    def __init__(self):
        super(MediaScanner, self).__init__()
        self.store = None
        self._current_delay = None
        self._start_delay = None

        self.scan_stat = Statistic()
        self.scan_stat.uri = MediaUri('media_scanner://localhost/statistic')

        self.running = False

    def _scan_directories(self):
        # ugly hack to start the scanning process for the directories the user
        # has set up in the configuration file
        directories_section = common.application.config.get_section('directories')
        if not directories_section:
            self.debug("no default directories")
            return
        for place in ('music', 'pictures'):
            paths = directories_section.get(place, [])
            for path in paths:
                if path != '*default*': # skip the default configuration thingy
                    uri = MediaUri('file://%s' % path)
                    self.debug("adding %s" % uri)
                    self.put(uri, None)

    def initialize(self):

        def set_parser(parser):
            self.parser = parser
            self.parser.store = self.store
            # start the scanner with a delay of 5 seconds
            self._start_delay = reactor.callLater(5, self._scan_directories)
            return self
 
        def set_local_resource(local_resource):
            self.local_resource = local_resource
            return self

        dfr = super(MediaScanner, self).initialize()
        store = common.application.store
        self.store = store

        dfr.addCallback(lambda x: self.create_schema())
        # FIXME: add configuration system
        dfr.addCallback(lambda x: DatabaseParser.create({}))
        dfr.addCallback(set_parser)
        dfr.addCallback(lambda x: LocalResource.create({}))
        dfr.addCallback(set_local_resource)
        return dfr

    def clean(self):
        if self._current_delay and not self._current_delay.called:
            self._current_delay.cancel()

        if self._start_delay and not self._start_delay.called:
            self._start_delay.cancel()

        return super(MediaScanner, self).clean()
        
    def _update_layout(self, result):
        def get_columns(result):
            return [ x[1] for x in result ]

        def get_all(result):
            return result.get_all()

        def sqlexec(string):
            return self.store.execute(string)
            
        def get_table_info(table_name):
            dfr = sqlexec("PRAGMA table_info('%s')" % table_name)
            dfr.addCallback(get_all)
            dfr.addCallback(get_columns)
            return dfr

        def got_files(result):
            commit = False
            if not u'playcount' in result:
                sqlexec("ALTER TABLE files ADD playcount INTEGER DEFAULT 0;")
                commit = True
            if not u'last_played' in result:
                sqlexec("ALTER TABLE files ADD last_played INTEGER DEFAULT 0;")
                commit = True

            if commit:
                return self.store.commit()

        def got_images(result):
            if not 'shot_time' in result:
                # shot_time is part of the first update we did for the image
                # table. as it is not there, we have to update the table
                # FIXME: this way to identify an Database Update is not nice
                sqlexec("ALTER TABLE images ADD shot_time INTEGER;")
                sqlexec("ALTER TABLE images ADD with_flash INTEGER DEFAULT 0;")
                sqlexec("ALTER TABLE images ADD orientation INTEGER;")
                sqlexec("ALTER TABLE images ADD gps_altitude INTEGER;")
                sqlexec("ALTER TABLE images ADD gps_latitude VARCHAR;")
                sqlexec("ALTER TABLE images ADD gps_longitude VARCHAR;")
                sqlexec("ALTER TABLE images ADD album_name VARCHAR;")
                return self.store.commit()

        def got_albums(result):
            if not u'release_date' in result:
                sqlexec("ALTER TABLE music_albums ADD release_date VARCHAR;")
                return self.store.commit()

        def do_files_update():
            dfr = get_table_info('files')
            dfr.addCallback(got_files)
            return dfr

        def do_albums_update(old_res=None):
            dfr = get_table_info('music_albums')
            dfr.addCallback(got_albums)
            return dfr

        def do_images_update(old_res=None):
            dfr = get_table_info("images")
            dfr.addCallback(got_images)
            return dfr

        dfr = do_files_update()
        dfr.addCallback(do_albums_update)
        dfr.addCallback(do_images_update)
        return dfr

    def create_schema(self):
        def go_through(queries):
            for query in queries:
                dfr = self.store.execute(query)
                yield dfr

        dfr = task.coiterate(go_through(SCHEMA))
        dfr.addCallback(lambda result: self.store.commit())
        dfr.addCallback(self._update_layout)
        return dfr

    def running_get(self):
        return self._running

    def running_set(self, value):
        self._running = value
        self.scan_stat.running = value

    running = property(fget=running_get, fset=running_set)

    # general Resource Provider API
    def get(self, uri, context_model=None):
        """
        If the filename is C{statistic} you receive the
        L{elisa.plugins.database.scanner_models.Statistic} for this scanner
        """
        if uri.filename == 'statistic':
            return (self.scan_stat, defer.succeed(self.scan_stat))

        return (None, defer.fail(NotImplementedError()))

    def put(self, source_uri, container_uri, context_model=None):
        """
        put another uri into the queue and start the scanning process if it is
        not yet running
        """
        # FIXME check the container uri
        # FIXME no check if this uri might already in the queue...

        if source_uri.filename != '':
            # we want to have the content, so we have to add a trailing slash
            source_uri = source_uri.join('')

        def start_scanning(result, source_uri):
            s = ScanResource(source_uri)
            self.scan_stat.queue.put(s)

            if not self.running:
                self._scan()

            return s.defer

        return self.parser.mark_deleted(source_uri.path).addCallback(start_scanning,
                source_uri)
       
    # internally used
    def _update_stat(self, result, model, stat):
        stat.files_scanned += 1

        if isinstance(result, Failure):
            stat.files_failed.append( (model.uri, result) )

    def _file_found(self, model, stat):
        stat.files_total += 1

        # make a pause to take a breath
        delay_dfr = defer.Deferred()

        def delay_callback(result, delay_dfr):
            self._current_delay = reactor.callLater(
                float(self.config.get('delay')), delay_dfr.callback, result)
            return result

        dfr = self.parser.query_model(model, stat)
        dfr.addBoth(self._update_stat, model, stat)
        dfr.addBoth(delay_callback, delay_dfr)

        return delay_dfr

    def _scan(self):

        def run_next(result, scan_resource):
            if scan_resource.pending_dfrs:
                # some metadata pending
                scan_resource.state = ScanResource.SCANNING_METADATA
            else:
                # nothing pending, let's go on
                scan_resource.state = ScanResource.SCANNING_DONE
                scan_resource.defer.callback(ScanResource.SCANNING_DONE)

            self.scan_stat.currently_scanning = None
            self.scan_stat.scanned.append(scan_resource)

            if self.scan_stat.queue.empty():
                # stop processing
                self.running = False
            else:
                return self._scan()

        self.running = True

        scan_resource = self.scan_stat.queue.get_nowait()
        self.scan_stat.currently_scanning = scan_resource

        # clear stat
        scan_resource.last_scan = time.time()
        scan_resource.files = 0
        scan_resource.dirs = 0
        scan_resource.state = ScanResource.SCANNING_FS

        return self._scan_recursive(scan_resource).addCallback(run_next, scan_resource)


    def _scan_recursive(self, scan_resource):
        #resources = common.application.resource_manager
        resources = self.local_resource
        uri = scan_resource.root_uri

        def stopper(data, file):
            self.warning("%s failed: %s" % (file, data))
            return None

        def scan_children(model, dirs):
            def iterator(model, dirs):
                while len(model.files):
                    item = model.files.pop(0)
                    if isinstance(item, DirectoryModel):
                        # delay searching the subfolders for later
                        dirs.append(item.uri)
                    else:
                        dfr = self._file_found(item, scan_resource)
                        yield dfr

            return task.coiterate(iterator(model, dirs))

        def scan_dirs(dirs):
            def iterator(dirs):
                while len(dirs):
                    item_uri = dirs.pop(0)
                    rewrite = '%s'
                    if item_uri.filename != '':
                        # we have a filename append the slash to get the
                        # content
                        rewrite += '/'
                    uri = MediaUri(rewrite % item_uri)
                    model, dfr = resources.get(uri)
                    dfr.addCallback(scan_children, dirs)
                    dfr.addErrback(stopper, uri.path)
                    yield dfr

            return task.coiterate(iterator(dirs))

        return scan_dirs([uri])
