# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core.utils.i18n import install_translation
_ = install_translation('database')

from twisted.internet import defer

from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.link import Link
from elisa.plugins.base.models.image import ImageModel

from elisa.plugins.poblesec.base.preview_list import \
    MenuItemPreviewListController, DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.poblesec.base.list import  GenericListViewMode

from elisa.plugins.database.models import Image, PhotoAlbum
from elisa.core import common
from elisa.core.media_uri import MediaUri

import os

def photo_lib_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/photo_library'
    link.label = _('Photo Library')
    link.icon = 'elisa.plugins.poblesec.photo_library'
    controller.model.append(link)

    return defer.succeed(None)

def photo_lib_albums_decorator(controller):
    albums = Link()
    albums.controller_path = '/poblesec/database/photo/albums'
    albums.label = _('Browse by Albums')
    albums.icon = 'elisa.plugins.poblesec.photo_album'
    controller.model.append(albums)
    return defer.succeed(None)


def photo_lib_images_decorator(controller):
    tracks = Link()
    tracks.controller_path = '/poblesec/database/photo/list'
    tracks.label = _('Browse all Photos')
    tracks.icon = 'elisa.plugins.poblesec.photo'
    controller.model.append(tracks)
    return defer.succeed(None)


def photo_lib_times_decorator(controller):
    times = Link()
    times.controller_path = '/poblesec/database/photo/by_time'
    times.label = _('By time')
    times.icon = 'elisa.plugins.poblesec.by_genre' # FIXME
    controller.model.append(times)
    return defer.succeed(None)

class ThumbnailMixin(object):

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = common.application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        self.gst_metadata = controllers[0].gst_metadata

    def _updated_thumbnail(self, thumbnail, item):
        item.thumbnail = thumbnail
        return thumbnail

    def _request_thumbnail(self, path):
        def got_metadata(metadata):
            return metadata['thumbnail']

        metadata = {'uri': MediaUri("file://%s" % path), 'thumbnail': None,}

        dfr = self.gst_metadata.get_metadata(metadata)
        dfr.addCallback(got_metadata)
        return dfr

class PhotoViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        return defer.succeed(os.path.basename(item.file_path))

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.photo'
        return resource

    def get_image(self, item, theme):
        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._request_thumbnail(item.file_path)
        dfr.addCallback(self._updated_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class PhotoController(HierarchyController):

    def initialize(self, album=None):
        dfr = super(PhotoController, self).initialize()
        dfr.addCallback(self._start, album)
        return dfr

    def _start(self, result, album):
        self.album = album

        def got_images(images):
            self.model.extend(images)
            return self

        def sort_images(result_set):
            result_set.order_by(Image.shot_time, Image.file_path)
            return result_set.all()

        store = common.application.store

        if album:
            dfr = store.find(Image, Image.album_name == album.name)
        else:
            dfr = store.find(Image)
        
        dfr.addCallback(sort_images)
        dfr.addCallback(got_images)
        return dfr

    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
            slideshow_controller = controllers[0]
            slideshow_controller.player.clear_playlist()

            for my_model in self.model:
                if my_model == model:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                image = ImageModel()
                if hasattr(my_model, 'thumbnail') and my_model.thumbnail != None:
                    thumbnail_uri = MediaUri("file://"+my_model.thumbnail)
                    image.references.append(thumbnail_uri)
                image.references.append(MediaUri("file://"+my_model.file_path))
                image.title = os.path.basename(my_model.file_path)
                slideshow_controller.player.enqueue_to_playlist(image)

            slideshow_controller.player.jump_to_index(index)

            controllers = self.frontend.retrieve_controllers('/poblesec')
            main = controllers[0]
            main.show_slideshow_player()
            self.stop_loading_animation()


class AlbumViewMode(GenericListViewMode, ThumbnailMixin):

    def get_label(self, item):
        return defer.succeed(item.name)

    def get_default_image(self, item):
        resource = 'elisa.plugins.poblesec.photo_album'
        return resource

    def _retrieve_file_path(self, album):
        def got_one(one):
            return one.file_path

        dfr = album.photos.any()
        dfr.addCallback(got_one)
        return dfr

    def _got_thumbnail(self, thumbnail, item):
        item.thumbnail = thumbnail
        return None

    def get_image(self, item, theme):
        if hasattr(item, 'thumbnail') and item.thumbnail != None:
            return defer.succeed(item.thumbnail)

        dfr = self._retrieve_file_path(item)
        dfr.addCallback(self._request_thumbnail)
        dfr.addCallback(self._got_thumbnail, item)
        return dfr

    def get_preview_image(self, item, theme):
        if hasattr(item, 'thumbnail'):
            return item.thumbnail


class AlbumsController(HierarchyController):

    def initialize(self):
        dfr = super(AlbumsController, self).initialize()
        dfr.addCallback(self._start)
        return dfr

    def _start(self, result):

        def got_albums(albums):
            self.model.extend(albums)
            return self

        def sort_albums(result_set):
            result_set.order_by(PhotoAlbum.name)
            return result_set.all()

        store = common.application.store

        dfr = store.find(PhotoAlbum)

        dfr.addCallback(sort_albums)
        dfr.addCallback(got_albums)
        return dfr

    # copy-paste-code again :(
    def node_clicked(self, widget, model):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(model))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            history = controllers[0].history
            dfr = history.append_controller('/poblesec/database/photo/list',
                                                    model.name, album=model)

# Photo
class PhotoVerticalWithPreview(PhotoController, MenuItemPreviewListController):
    view_mode = PhotoViewMode

class PhotoCoverflow(PhotoController, ImageWithReflectionCoverflowController):
    view_mode = PhotoViewMode

class PhotoGrid(PhotoController, GridItemGridController):
    view_mode = PhotoViewMode

class PhotoListSwitcherController(ListSwitcherController):
    modes = [PhotoVerticalWithPreview,
             PhotoCoverflow,
             PhotoGrid]
    default_mode = PhotoGrid

# Albums
class AlbumsVerticalWithPreview(AlbumsController, MenuItemPreviewListController):
    view_mode = AlbumViewMode

class AlbumsCoverflow(AlbumsController, ImageWithReflectionCoverflowController):
    view_mode = AlbumViewMode

class AlbumsGrid(AlbumsController, GridItemGridController):
    view_mode = AlbumViewMode

class AlbumsListSwitcherController(ListSwitcherController):

    modes = [AlbumsVerticalWithPreview,
             AlbumsCoverflow,
             AlbumsGrid]
    default_mode = AlbumsVerticalWithPreview
