# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Benjamin Kampmann <benjamin@fluendo.com>

from elisa.core import common

from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController

from elisa.plugins.database.photo_controller import PhotoVerticalWithPreview, \
    PhotoCoverflow, PhotoGrid

from elisa.plugins.poblesec.base.preview_list import \
    DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
    ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController



from elisa.plugins.database.music_controller import GenericTracksDbController, \
                                              DBTracksViewMode

from elisa.plugins.database.photo_controller import PhotoController, \
                                              PhotoViewMode
from elisa.plugins.poblesec.link import Link

from elisa.plugins.database.models import MusicTrack, File, Image
from storm.expr import Desc

from twisted.internet import defer

# i18n
from elisa.core.utils.i18n import install_translation
_ = install_translation('database')


def most_played_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music/tracks_played'
    link.label = _('Most Played Tracks')
    link.controller_args = {'limit': 20,}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)


def last_played_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music/tracks_played'
    link.label = _('Recently Played Tracks')
    link.controller_args = {'limit':20,  'order_by' : 'last_played'}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)


def recently_added_tracks_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music/tracks_added'
    link.label = _('Recently Added Tracks')
    link.controller_args = {'limit' : 20}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)

def music_playlists_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/music/playlists'
    link.label = _('Playlists')
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)

def recently_added_photos_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/photo/photos_added'
    link.label = _('Recently added Photos')
    link.controller_args = {'limit' : 20}
    link.icon = 'elisa.plugins.poblesec.recently_added_photos'
    controller.model.append(link)

    return defer.succeed(None)

def last_played_photos_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/photo/photos_played'
    link.label = _('Recently viewed Photos')
    link.controller_args = {'limit':20,  'order_by' : 'last_played'}
    link.icon = 'elisa.plugins.poblesec.recently_viewed_photos'
    controller.model.append(link)

    return defer.succeed(None)

def most_played_photos_decorator(controller):
    # disabled for now
    return defer.succeed(None)
    link = Link()
    link.controller_path = '/poblesec/database/photo/photos_played'
    link.label = _('Most watched Photos')
    link.controller_args = {'limit': 20,}
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)

def pictures_playlists_decorator(controller):
    link = Link()
    link.controller_path = '/poblesec/database/pictures/playlists'
    link.label = _('Playlists')
    link.icon = 'elisa.plugins.poblesec.playlist'
    controller.model.append(link)

    return defer.succeed(None)

class PlayedMixin(object):
    def _load(self, result, filter, order_by, limit):
        self.filter = filter

        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        def set_limit(result_set, limit):
            result_set.config(limit=limit)
            return result_set

        def sort_by_count(result_set):
            result_set.order_by(Desc(File.playcount))
            return result_set

        def sort_by_date(result_set):
            result_set.order_by(Desc(File.last_played))
            return result_set

        store = common.application.store

        dfr = store.find(filter, filter.file_path == File.path, \
                File.playcount != 0)

        if order_by == "last_played":
            dfr.addCallback(sort_by_date)
        else:
            dfr.addCallback(sort_by_count)

        dfr.addCallback(set_limit, limit)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr

class AddedMixin(object):
    def _load(self, result, filter, limit):
        self.filter = filter

        def got_items(models):
            self.model.extend(models)
            return self

        def get_items(result_set):
            return result_set.all()

        def set_limit(result_set, limit):
            result_set.config(limit=limit)
            return result_set

        def sort_by_added(result_set):
            # this is not exactly the time it was added but last changed
            result_set.order_by(Desc(File.modification_time))
            return result_set

        store = common.application.store

        dfr = store.find(filter, filter.file_path == File.path)

        dfr.addCallback(sort_by_added)

        dfr.addCallback(set_limit, limit)
        dfr.addCallback(get_items)
        dfr.addCallback(got_items)
        return dfr


class PlayedTracksController(GenericTracksDbController, PlayedMixin):
    def initialize(self, order_by=None, limit=10):
        dfr = super(PlayedTracksController, self).initialize()

        dfr.addCallback(self._load, MusicTrack, order_by, limit)
        return dfr

class AddedTracksController(GenericTracksDbController, AddedMixin):

    def initialize(self, limit=10):
        dfr = super(AddedTracksController, self).initialize()

        dfr.addCallback(self._load, MusicTrack, limit)
        return dfr

class PlayedPhotosController(PhotoController, PlayedMixin):
    def initialize(self, order_by=None, limit=10):
        dfr = super(PlayedPhotosController, self).initialize()

        dfr.addCallback(self._load, Image, order_by, limit)
        return dfr

class AddedPhotosController(PhotoController, AddedMixin):

    def initialize(self, limit=10):
        dfr = super(AddedPhotosController, self).initialize()

        dfr.addCallback(self._load, Image, limit)
        return dfr

# Added Tracks 
class AddedTracksVerticalWithPreview(AddedTracksController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode

class AddedTracksCoverflow(AddedTracksController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode

class AddedTracksGrid(AddedTracksController, GridItemGridController):
    view_mode = DBTracksViewMode

class AddedTracksListSwitcherController(ListSwitcherController):
    modes = [AddedTracksVerticalWithPreview,
             AddedTracksCoverflow,
             AddedTracksGrid]
    default_mode = AddedTracksVerticalWithPreview


# Played Tracks
class PlayedTracksVerticalWithPreview(PlayedTracksController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksViewMode

class PlayedTracksCoverflow(PlayedTracksController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksViewMode

class PlayedTracksGrid(PlayedTracksController, GridItemGridController):
    view_mode = DBTracksViewMode

class PlayedTracksListSwitcherController(ListSwitcherController):
    modes = [PlayedTracksVerticalWithPreview,
             PlayedTracksCoverflow,
             PlayedTracksGrid]
    default_mode = PlayedTracksVerticalWithPreview

# Added Photos 
class AddedPhotosVerticalWithPreview(AddedPhotosController, PhotoVerticalWithPreview):
    view_mode = PhotoViewMode

class AddedPhotosCoverflow(AddedPhotosController, PhotoCoverflow):
    view_mode = PhotoViewMode

class AddedPhotosGrid(AddedPhotosController, PhotoGrid):
    view_mode = PhotoViewMode

class AddedPhotosListSwitcherController(ListSwitcherController):
    modes = [AddedPhotosVerticalWithPreview,
             AddedPhotosCoverflow,
             AddedPhotosGrid]
    default_mode = AddedPhotosGrid

# Played Photos 
class PlayedPhotosVerticalWithPreview(PlayedPhotosController, PhotoVerticalWithPreview):
    view_mode = PhotoViewMode

class PlayedPhotosCoverflow(PlayedPhotosController, PhotoCoverflow):
    view_mode = PhotoViewMode

class PlayedPhotosGrid(PlayedPhotosController, PhotoGrid):
    view_mode = PhotoViewMode

class PlayedPhotosListSwitcherController(ListSwitcherController):
    modes = [PlayedPhotosVerticalWithPreview,
             PlayedPhotosCoverflow,
             PlayedPhotosGrid]
    default_mode = PlayedPhotosGrid
