# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Guido Amoruso <guidonte@fluendo.com>


from elisa.core import common

from elisa.core.media_uri import MediaUri
from elisa.plugins.base.models.media import PlayableModel

from twisted.internet import defer

from random import shuffle
import datetime


class Action(object):
    """Contextul action for a Controller object.

    The intended use is for controllers to subclass this and override the run()
    method, that can do everything you want, assuming to deal with a specific
    controller class.

    @ivar controller: the controller on which to run the action
    @itype controller: L{elisa.plugins.pigment.pigment_controller.PigmentController}
    """

    icon = 'elisa.plugins.poblesec.file'
    title = ''
    subtitle = ''

    def __init__(self, controller):
        self.controller = controller

    def run(self):
        raise NotImplementedError()


class PlayAllAction(Action):

    icon = 'elisa.plugins.poblesec.play_all'

    def playable_models_from_items(self, items):
        models = []
        for item in items:
            playable_model = PlayableModel()
            playable_model.uri = MediaUri('file://' + item.file_path)
            models.append(playable_model)

        return models

    def get_items(self, item):
        raise NotImplementedError()

    def run(self):
        items = [item for item in self.controller.model if not isinstance(item, Action)]
        tracks = []
        dfrs = []
        for item in items:
            dfr = self.get_items(item)
            dfr.addCallback(self.playable_models_from_items)
            dfr.addCallback(lambda playable_items: tracks.extend(playable_items))
            dfrs.append(dfr)

        if len(dfrs) == 0:
            self.controller.stop_loading_animation()
            return defer.succeed(None)

        dfr_list = defer.DeferredList(dfrs)
        dfr_list.addCallback(lambda result: self.order_tracks(tracks))
        dfr_list.addCallback(self.play)

        return dfr_list

    def order_tracks(self, tracks):
        return tracks

    def play(self, items):
        # FIXME: this is not dynamic enough, one needs to first build the list
        # of PlayableModel's before actually starting to play the first item.
        # The playback of the first item should start first and then the list
        # of following items to enqueue in the playlist be built.
        if items is not None and len(items) > 0:
            poblesec = self.controller.frontend.retrieve_controllers('/poblesec')[0]
            player = poblesec.music_player.player
            poblesec.show_music_player()
            self.controller.stop_loading_animation()

            player.play_model(items[0])
            for item in items[1:]:
                player.enqueue_to_playlist(item)


class ShuffleAction(PlayAllAction):

    icon = 'elisa.plugins.poblesec.shuffle'

    def order_tracks(self, tracks):
        """Actually shuffle tracks."""
        shuffle(tracks)
        return tracks


class FavoriteNotFoundError(Exception):
    pass


class AddToFavoritesAction(Action):

    toggled = False

    toggled_icon = 'elisa.plugins.poblesec.remove_from_favorites'
    untoggled_icon = 'elisa.plugins.poblesec.add_to_favorites'
    toggled_title = ''
    untoggled_title = ''
    toggled_subtitle = ''
    untoggled_subtitle = ''

    def setup(self, favorites):
        if not favorites:
            self.toggle(False)
        else:
            self.toggle(True)

        return favorites

    def store(self, favorite):
        if favorite is None:
            self.controller.warning("Not storing None favorite")
            return

        self.controller.debug("Storing %s" % favorite)

        common.application.store.add(favorite)
        common.application.store.commit()

        self.controller.stop_loading_animation()
        self.toggle(not self.toggled)

        return favorite

    def remove(self, favorite):
        if favorite is None:
            self.controller.warning("Favorite not found: not removing")
            return

        self.controller.debug("Removing %s" % favorite)

        common.application.store.remove(favorite)
        common.application.store.commit()

        self.controller.stop_loading_animation()
        self.toggle(not self.toggled)

        return favorite

    def toggle(self, status):
        self.controller.debug("Toggling %s" % status)

        self.toggled = status
        self.refresh()

    def refresh(self):
        if self.toggled:
            self.icon = self.toggled_icon
            self.title = self.toggled_title
            self.subtitle = self.toggled_subtitle
        else:
            self.icon = self.untoggled_icon
            self.title = self.untoggled_title
            self.subtitle = self.untoggled_subtitle

        # actually refresh the list item
        index = self.controller.model.index(self)
        self.controller.model[index] = self


class AddToPlaylistAction(Action):

    icon = 'elisa.plugins.poblesec.add_to_playlist'
    toggled = False


class ActionPathNotSetError(Exception):
    pass


class LinkAction(Action):

    path = None
    args = {}

    def run(self):
        if self.path is None:
            raise ActionPathNotSetError()

        browser = self.controller.frontend.retrieve_controllers('/poblesec/browser')[0]
        dfr = browser.history.append_controller(self.path, self.title, **self.args)

        return dfr


