# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import os.path

from elisa.core.common import application
from elisa.core.media_uri import MediaUri
from elisa.core.utils import typefinding
from elisa.core.utils.i18n import install_translation

from elisa.plugins.base.models.file import FileModel, DirectoryModel
from elisa.plugins.base.models.media import PlayableModel
from elisa.plugins.base.models.image import ImageModel
from elisa.plugins.base.messages.device import DeviceRemoved

from elisa.plugins.poblesec.notifying_dict import NotifyingDictionary
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list import GenericListViewMode
from elisa.plugins.poblesec.base.preview_list import MenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController

from twisted.internet import defer


_ = install_translation('poblesec')


class FilesystemController(HierarchyController):

    default_uri = MediaUri('file://' + os.path.expanduser('~'))
    hide_files = False
    media_type = None

    def initialize(self, uri=None, media_type=None):
        dfr = super(FilesystemController, self).initialize()
        bus = application.bus
        bus.register(self._device_remove_cb, DeviceRemoved)

        self._load_error = None
        self.uri = uri
        self.media_type = media_type
        self.gone_back = False

        def resource_loaded(resource):
            def is_not_hidden(file_model):
                return not file_model.name.startswith('.')

            non_hidden_files = filter(is_not_hidden, resource.files)
            non_hidden_files.sort(cmp=lambda x, y: cmp(x.name.lower(), y.name.lower()))

            if self.hide_files == True:
                def is_directory(file_model):
                    return isinstance(file_model, DirectoryModel)

                non_hidden_files = filter(is_directory, non_hidden_files)

            for my_file_model in non_hidden_files:
                try:
                    media_types = typefinding.typefind(MediaUri(my_file_model.uri))
                except typefinding.UnknownFileExtension, e:
                    media_types = []

                if self.media_type in media_types:
                    my_file_model.media_type = self.media_type
                else:
                    continue

            if self.media_type:
                def is_controller_mediatype(file_model):
                    if isinstance(file_model, DirectoryModel):
                        return True
                    return file_model.media_type == self.media_type

                non_hidden_files = filter(is_controller_mediatype, non_hidden_files)

            self.model.extend(non_hidden_files)
            return self

        def got_error(failure, resource):
            self._load_error = failure
            return resource

        def load_resource(controller):
            # we assume that self.uri is a directory
            # adds a slash in order to get the content of the directory
            self.uri = self.uri.join("")
            resource, dfr_resource = application.resource_manager.get(self.uri)
            dfr_resource.addErrback(got_error, resource)
            return dfr_resource

        if self.uri != None:
            dfr.addCallback(load_resource)
            dfr.addCallback(resource_loaded)

        return dfr

    def clean(self):
        dfr = super(FilesystemController, self).clean()
        bus = application.bus
        bus.unregister(self._device_remove_cb)
        return dfr

    def _device_remove_cb(self, msg, sender):
        if None not in (self.uri, msg.udi) and msg.udi == self.uri:
            # get history and go back to Music menu
            controllers = self.frontend.retrieve_controllers('/poblesec/browser')
            browser = controllers[0]
            if not self.gone_back:
                self.gone_back = True
                browser.history.go_back()

    def nodes_setup(self):
        super(FilesystemController, self).nodes_setup()
        media_type = self.media_type
        # FIXME: please clean the media_type values mess.
        if self._load_error:
            msg = _("There was an error loading the folder.")
        if media_type == 'video':
            msg = _("This folder does not contain any video files.")
        elif media_type in ('music', 'audio'):
            msg = _("This folder does not contain any audio files.")
        elif media_type in ('pictures', 'image'):
            msg = _("This folder does not contain any image files.")
        else:
            msg = _("This folder does not contain any multimedia files")
        self.nothing_to_display_widget.text.label = msg
        if not len(self.model) or self._load_error:
            self.nothing_to_display_widget.visible = True

    def set_frontend(self, frontend):
        # retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        self.gst_metadata = controllers[0].gst_metadata

        super(FilesystemController, self).set_frontend(frontend)

        self.nodes.focus = True

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)

            self._previous_clicked = selected_item

            if isinstance(item, DirectoryModel):
                file_uri = item.uri

                # we assume there was only one controller created with that path
                controllers = self.frontend.retrieve_controllers('/poblesec/browser')
                browser = controllers[0]
                dfr = browser.history.append_controller(self.path, item.name,
                                                        uri=file_uri, media_type=self.media_type)

            elif isinstance(item, FileModel):
                if item.media_type == "image":
                    self.play_image(item)
                elif item.media_type == "audio":
                    self.play_music_or_video(item, 'music')
                elif item.media_type == "video":
                    self.play_music_or_video(item, 'video')

                dfr = defer.succeed(True)

    def play_image(self, item):
        controllers = self.frontend.retrieve_controllers('/poblesec/slideshow_player')
        slideshow_controller = controllers[0]
        slideshow_controller.player.clear_playlist()

        for file in self.model:
            if isinstance(file, FileModel) and file.media_type == "image":
                if file == item:
                    # save the index of the image to be displayed first
                    index = len(slideshow_controller.player.playlist)
                image = ImageModel()
                if file.thumbnail != None:
                    thumbnail_uri = MediaUri("file://"+file.thumbnail)
                    image.references.append(thumbnail_uri)
                image.references.append(file.uri)
                image.title = file.name
                slideshow_controller.player.enqueue_to_playlist(image)

        slideshow_controller.player.jump_to_index(index)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        main.show_slideshow_player()
        self.stop_loading_animation()

    def play_music_or_video(self, item, player):
        # player should be one of ('music', 'video')
        controllers = self.frontend.retrieve_controllers('/poblesec/%s_player' % player)
        player_controller = controllers[0]

        # enqueue and play the clicked file
        playable_model = PlayableModel()
        playable_model.uri = item.uri
        player_controller.player.play_model(playable_model)

        controllers = self.frontend.retrieve_controllers('/poblesec')
        main = controllers[0]
        eval('main.show_%s_player()' % player)
        self.stop_loading_animation()

        # enqueue all the following files of the directory
        index = self.model.index(item)
        for file in self.model[index+1:]:
            if isinstance(file, DirectoryModel):
                continue
            playable_model = PlayableModel()
            playable_model.uri = file.uri
            player_controller.player.enqueue_to_playlist(playable_model)


class FilesystemViewMode(GenericListViewMode):

    """
    Implementation of the common view modes API.
    """

    filetype_to_resource = {
        'video': 'elisa.plugins.poblesec.file_video',
        'audio': 'elisa.plugins.poblesec.file_music',
        'image': 'elisa.plugins.poblesec.file_picture',
        'directory': 'elisa.plugins.poblesec.folder',
    }

    def __init__(self):
        # FIXME: we need the frontend to get a reference to the gst_metadata
        # instance. This a cheap - UGLY - way to get the frontend without
        # changing a lot of client code. It is really ugly as we assume there
        # is only one frontend, which might not be the case in the future...
        frontend = application.interface_controller.frontends.values()[0]
        # Retrieve and store a reference to gst_metadata
        controllers = frontend.retrieve_controllers('/poblesec')
        self.gst_metadata = controllers[0].gst_metadata

    def get_label(self, item):
        return defer.succeed(item.name)

    def get_default_image(self, item):
        fallback = 'elisa.plugins.poblesec.file'
        resource = self.filetype_to_resource.get(item.media_type, fallback)
        return resource

    def _request_thumbnail(self, item):
        return self._request_type_and_thumbnail(item)

    def _request_type_and_thumbnail(self, item):
        def on_metadata_set(notifier, metadata, key, value):
            if key == 'file_type':
                if item.media_type is None:
                    item.media_type = value
            elif key == 'thumbnail':
                item.thumbnail = value

        metadata = {'uri': item.uri,
                    'thumbnail': None,
                    'file_type': item.media_type,
                    'mime_type': None,
                   }
        metadata = NotifyingDictionary(metadata)
        handler_id = metadata.notifier.connect('item-changed', on_metadata_set)

        if self.gst_metadata.able_to_handle(metadata):
            deferred = self.gst_metadata.get_metadata(metadata)
            return deferred.addBoth(lambda result: self._disconnect_metadata(metadata, handler_id, item))

    def _disconnect_metadata(self, metadata, handler_id, item):
        metadata.notifier.disconnect(handler_id)
        return item.thumbnail

    def get_image(self, item, theme):
        if item.media_type is not None:
            if item.thumbnail is not None:
                return defer.succeed(item.thumbnail)
            else:
                return self._request_thumbnail(item)
        else:
            return self._request_type_and_thumbnail(item)

    def get_preview_image(self, item, theme):
        return item.thumbnail


class FilesystemPreviewListController(FilesystemController, MenuItemPreviewListController):
    view_mode = FilesystemViewMode
    fastscroller = True

    def item_to_label(self, item):
        return item.name


class FilesystemCoverflowController(FilesystemController, ImageWithReflectionCoverflowController):
    view_mode = FilesystemViewMode


class FilesystemGridController(FilesystemController, GridItemGridController):
    view_mode = FilesystemViewMode


class FilesystemListSwitcherController(ListSwitcherController):
    modes = [FilesystemPreviewListController,
             FilesystemCoverflowController,
             FilesystemGridController]
    default_mode = FilesystemPreviewListController


class FilesystemListSwitcherGridController(FilesystemListSwitcherController):
    default_mode = FilesystemGridController
