# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Gernot Klimscha <gernot@fluendo.com>

"""
WMDResourceProvider
"""

from elisa.core.utils import defer
from elisa.core.media_uri import MediaUri

from elisa.core.components.resource_provider import ResourceProvider
from elisa.core import common

from elisa.plugins.base.messages.device import NewDeviceDetected, NewUnknownDevice,\
        DeviceRemoved

from elisa.plugins.base.models.volume import VolumesModel, VolumeModel
from elisa.plugins.pigment.messages import ViewportWindowCreated

from twisted.internet import reactor
import os, time, thread

# win32api
import win32api, win32con, win32gui
import win32com.client
from ctypes import *

# duration of shift-key simulation for blocking autoplay
WMD_TIMEOUT = 5
# name of iTunes application
WMD_ITUNES_CAPTION = "iTunes"
# classname
#WMD_CLASSNAME = "Windows Media Detection Listener"

# drivetypes we are interested in
DRIVETYPE_REMOVEABLE = 2
DRIVETYPE_LOCALDISK = 3
DRIVETYPE_CDROM = 5

# device notifications for WM_DEVICECHANGE wParam
DBT_DEVICEARRIVAL = 0x8000
DBT_DEVICEREMOVECOMPLETE = 0x8004

# device type in DEV_BROADCAST_HDR
DBT_DEVTYP_VOLUME = 0x00000002

WORD = c_ushort
DWORD = c_ulong

class DEV_BROADCAST_HDR(Structure):
  _fields_ = [
    ("dbch_size", DWORD),
    ("dbch_devicetype", DWORD),
    ("dbch_reserved", DWORD)
  ]

class DEV_BROADCAST_VOLUME(Structure):
  _fields_ = [
    ("dbcv_size", DWORD),
    ("dbcv_devicetype", DWORD),
    ("dbcv_reserved", DWORD),
    ("dbcv_unitmask", DWORD),
    ("dbcv_flags", WORD)
  ]

class WMDResource(ResourceProvider):

    supported_uri='^volumes://'

    def __init__(self):
        super(WMDResource, self).__init__()
        self._udi_mappings = {}

    def initialize(self):
        bus = common.application.bus
        bus.register(self._viewport_created, ViewportWindowCreated)

        return super(WMDResource, self).initialize()

    def _viewport_created(self, msg, sender):
        viewport = msg.viewport
        if viewport.message_filter:
            viewport.message_filter+=(win32con.WM_DEVICECHANGE,)
        else:
            viewport.message_filter=(win32con.WM_DEVICECHANGE,)
        viewport.connect('win32-message-event', self._got_win32_message)

    def _got_win32_message(self, frontend, event):
        if event.message == win32con.WM_DEVICECHANGE:
            self._on_device_change(event.message, event.wparam, event.lparam)
    
    def get(self, uri, model):
        """
        Simple method to retrieve volumes. You can access it with C{volumes://}
        and apply a filter parameter. This is an and filter. If you specify it
        only these kind of volumes show up. Example:
        C{volumes://localhost/?filter=dvd,cdda} would only give you dvds and
        cddas. The three knows filters are: dvd, removable and cdda.

        The default is that all filters are applied (like
        filter=dvd,cdda,removable).

        In return you get a *new* L{elisa.plugins.wmd.models.VolumesModel}
        """

        filter = uri.get_param('filter', 'dvd,cdda,removable')

        filtered = filter.split(',')

        model = self._get_attached_volumes(filter=filtered)
        model.filter = filter

        return model, defer.succeed(model)

    def _device_added(self, udi):
        model = self._get_attached_volumes(drive_letter=udi)

        if not model:
            return self._send_unknown_device(udi)

        self._udi_mappings[udi] = model.mount_point
        msg = NewDeviceDetected()
        msg.udi = udi
        msg.model = model

        common.application.bus.send_message(msg, self)
        return self._send_unknown_device(udi)

    def _device_removed(self, udi):
        try:
            uri = self._udi_mappings[udi]
        except KeyError:
            uri = udi
        else:
            del self._udi_mappings[udi]
        common.application.bus.send_message(DeviceRemoved(uri), self)

    def _send_unknown_device(self, udi):
        common.application.bus.send_message(NewUnknownDevice(udi), self)

    def _get_attached_volumes(self, drive_letter='', filter=''):
        # get WMI COM object
        _wmi = win32com.client.GetObject(r"winmgmts:\\.\root\cimv2")

        model = None
        query = "SELECT * FROM Win32_LogicalDisk WHERE (DriveType = %d OR DriveType = %d " \
                "OR DriveType = %d) AND FileSystem IS NOT NULL AND SupportsDiskQuotas = False" % \
                (DRIVETYPE_LOCALDISK, DRIVETYPE_REMOVEABLE, DRIVETYPE_CDROM)

        # if called with drive letter just search for the device and return a VolumeModel instead of a VolumesModel
        if drive_letter:
            query += " AND DeviceID = '%s:'" % drive_letter
        else:
            # create a VolumesModel to append all found volumes to it
            model = VolumesModel()

        if filter and 'removable' in filter:
            # add file and ipod for removable to search in protocols at the end
            filter.append('file')
            filter.append('ipod')

        wmiLogicalDisks = _wmi.ExecQuery(query)
        for wmiLogicalDisk in wmiLogicalDisks:
            mount_point = wmiLogicalDisk.DeviceID
            if wmiLogicalDisk.DriveType == DRIVETYPE_REMOVEABLE:
                # removable media found - get more information
                query = "ASSOCIATORS OF {Win32_LogicalDisk.DeviceID='%s'} WHERE " \
                        "AssocClass = Win32_LogicalDiskToPartition" % mount_point
                wmiDiskPartitions = _wmi.ExecQuery(query)
                for wmiDiskPartition in wmiDiskPartitions:
                    query = "ASSOCIATORS OF {Win32_DiskPartition.DeviceID='%s'} WHERE " \
                       "AssocClass = Win32_DiskDriveToDiskPartition" % wmiDiskPartition.DeviceID
                    wmiDiskDrives = _wmi.ExecQuery(query)
                    for wmiDiskDrive in wmiDiskDrives:
                        name = wmiDiskDrive.Caption
                        self.debug("Found removeable media '%s' in '%s'" % (name, mount_point))
                        if "ipod" in name.lower():
                            # it's an ipod!
                            protocol = 'ipod'
                            # start iTunes killing thread
                            thread.start_new_thread(self._itunes_thread, (WMD_TIMEOUT, ))
                        else:
                            # just a normal file device
                            protocol = 'file'
            elif wmiLogicalDisk.DriveType == DRIVETYPE_CDROM:
                # CD/DVD found
                name = win32api.GetVolumeInformation(mount_point)[0]
                protocol = self._get_cd_media_type(mount_point)
                self.debug("Found CD/DVD '%s' in '%s' which supports '%s'" % \
                           (name, mount_point, protocol))
            else:
                # should be a usb harddisk ...
                name = wmiLogicalDisk.VolumeName
                protocol = 'file'

            # if something has been found add the new volume but only if no filter is used or it is contained in the filter
            if name and not filter or (filter and protocol in filter):
                # create a VolumeModel for the found volume
                volume = VolumeModel()
                volume.protocol = protocol
                volume.label = name or mount_point
                volume.udi = mount_point[0]
                volume.mount_point = MediaUri("%s:///%s:/" % (protocol, mount_point[0]))

                # if called for a specific volume just return the VolumeModel
                if drive_letter:
                    return volume
                else:
                    # else append it to the VolumesModel
                    model.volumes.append(volume)

        return model

    def _on_device_change(self, msg, wparam, lparam):
        dev_broadcast_hdr = DEV_BROADCAST_HDR.from_address(lparam)
        if wparam == DBT_DEVICEARRIVAL or wparam == DBT_DEVICEREMOVECOMPLETE:
            if dev_broadcast_hdr.dbch_devicetype == DBT_DEVTYP_VOLUME:
                dev_broadcast_volume = DEV_BROADCAST_VOLUME.from_address(lparam)
                drive_letter = self._drive_from_mask(dev_broadcast_volume.dbcv_unitmask)
                drive_letter = chr(ord("A") + drive_letter)

                if wparam == DBT_DEVICEARRIVAL:
                    self.debug("new volume arrived in '%s:'" % drive_letter)
                    win32api.keybd_event(win32con.VK_SHIFT, 0)
                    # thread needed because win32gui does not expose SetTimer API
                    thread.start_new_thread(self._timeout_thread, (WMD_TIMEOUT, ))
                    # call later because else there is a problem with the windows COM functions wich cannot be called from an event
                    reactor.callLater(0.10, self._device_added, drive_letter)
                else:
                    self.debug("volume has been removed from '%s:'" % drive_letter)
                    self._device_removed(drive_letter)
        return (False, True)

    def _timeout_thread(self, timeout):
        time.sleep(timeout)
        win32api.keybd_event(win32con.VK_SHIFT, 0, win32con.KEYEVENTF_KEYUP)

    def _itunes_thread(self, timeout):
        w = 0; dec = 1
        self.debug("searching for iTunes window")
        while (w == 0 and timeout != 0):
            w = win32gui.FindWindow(None, WMD_ITUNES_CAPTION)
            if w == 0:
                # window not yet there - try again in 1 second
                time.sleep(dec)
                timeout -= dec
            else:
                # itunes window has been found - kill it!
                time.sleep(1)
                self.debug("iTunes found!")
                win32gui.PostMessage(w, win32con.WM_CLOSE)

    def _drive_from_mask(self, mask):
        n_drive = 0
        while 1:
            if (mask & (2 ** n_drive)):
                return n_drive
            else:
                n_drive += 1

    def _get_cd_media_type(self, drive):
        """
        check if CD is an audio cd or movie dvd
        """

        # a directory every movie dvd should have
        if os.path.isdir(drive + '\\VIDEO_TS'):
            return 'dvd'

        l = os.listdir(drive)
        for f in l:
            if f.endswith('.cda'):
                return 'cdda'

        return 'file'
