# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Authors: Philippe Normand <philippe@fluendo.com>
# ported to elisa 0.5 by: Benjamin Kampmann <benjamin@fluendo.com>

"""
HALResourceProvider
"""

from elisa.core.utils import defer

from elisa.core.components.resource_provider import ResourceProvider
from elisa.core import common
from elisa.core.application import ComponentsLoadedMessage
from elisa.core.media_uri import MediaUri

from elisa.plugins.base.messages.device import NewDeviceDetected, NewUnknownDevice,\
        DeviceRemoved

from elisa.plugins.base.models.volume import VolumesModel, DeviceModel, VolumeModel

import dbus
if getattr(dbus, 'version', (0,0,0)) >= (0,41,0):
    import dbus.glib
from dbus.exceptions import DBusException


try:
    import gconf
except ImportError:
    gconf = None


import platform

from twisted.internet import task

def get_dbus_error_message(exception):
    """
    Retrieve the message of the DBus error. Usually something like
    'org.freedesktop....'

    @param exception: the DBus exception to analyze
    @type exception:  L{dbus.DBusException}
    @rtype:           string
    """
    if dbus.version < (0, 82, 0):
        msg = exception.message
    else:
        msg = "%s: %s" % (exception._dbus_error_name, exception.message)
    return msg


class NotMountable(Exception):
    pass

class HALResource(ResourceProvider):
    log_category = "hal_resource"
    supported_uri='^volumes://'

    def initialize(self):
        bus = common.application.bus
        bus.register(self._components_loaded, ComponentsLoadedMessage)

        self.bus = dbus.SystemBus()
        self.ses = dbus.SessionBus()
        self.hal_manager = self.bus.get_object('org.freedesktop.Hal',
                                               '/org/freedesktop/Hal/Manager')

        self.hal_iface = dbus.Interface(self.hal_manager,
                                        'org.freedesktop.Hal.Manager')

        self._hotplugged_devices = {}
        self.stop_volume_manager_monitoring()
        return super(HALResource, self).initialize()


    def clean(self):
        dfr = super(HALResource, self).clean()
        self.restore_volume_manager_monitoring()
        return dfr

    def get(self, uri, model):
        """
        Simple method to retrieve volumes. You can access it with
        C{volumes://} and apply a filter parameter. If you specify it
        only these kind of volumes show up. Example::

        ::

          C{volumes://localhost/?filter=dvd,cdda}

        would only give you dvds and cddas. The three knows filters
        are: dvd, removable and cdda.

        The default is that all filters are applied (like
        filter=dvd,cdda,removable).

        In return you get a L{elisa.plugins.hal.models.VolumesModel}
        """

        filter = uri.get_param('filter', 'dvd,cdda,removable')

        filtered = filter.split(',')

        model = VolumesModel()
        model.filter = filter

        # we only support retrieval of volume devices atm
        devices = self.hal_iface.FindDeviceByCapability('volume')

        def iterate_devices():
            for udi in devices:
                device = self._get_device_with_udi(udi)
                if device is None:
                    yield udi
                    continue

                props = device.GetAllProperties()

                if props.get('volume.disc.has_audio') and 'cdda' in filtered:
                    volume = self._create_new_volume_model(udi, 'cdda', device, props)
                    if volume:
                        model.volumes.append(volume)
                        yield udi
                        continue

                if props.get('volume.disc.is_videodvd') and 'dvd' in filtered:
                    volume = self._create_new_volume_model(udi, 'dvd', device, props)
                    if volume:
                        model.volumes.append(volume)
                        yield udi
                        continue

                if 'removable' in filtered:
                    # do the check only if the removable is asked
                    parent_udi = device.GetProperty(u'info.parent')
                    parent = self._get_device_with_udi(parent_udi)
                    if parent is None:
                        yield udi
                        continue
                    parent_props = parent.GetAllProperties()
                    if parent_props.get('storage.removable', False) or \
                        parent_props.get('storage.hotpluggable', False):
                        volume = self._create_model_for_volume(udi, device)
                        if volume and isinstance(volume, VolumeModel):
                            model.volumes.append(volume)
                yield udi

        dfr = task.coiterate(iterate_devices())
        dfr.addCallback(lambda gen: model)
        return model, dfr

    def stop_volume_manager_monitoring(self):
        """
        Neutralize some of the volume_manager monitoring settings so
        that the user won't see rhythmbox pop up when an iPod is
        inserted (for example).
        """
        if gconf:
            client = gconf.client_get_default()
            path = '/desktop/gnome/volume_manager'
            autoplay_cda = client.get_bool('%s/autoplay_cda' % path)
            autoplay_dvd = client.get_bool('%s/autoplay_dvd' % path)
            autobrowse = client.get_bool('%s/autobrowse' % path)
            self.volume_manager_config = {'autoplay_cda': autoplay_cda,
                                          'autoplay_dvd': autoplay_dvd,
                                          'autobrowse': autobrowse}
            for prop in self.volume_manager_config.keys():
                client.set_bool('%s/%s' % (path, prop), False)

    def restore_volume_manager_monitoring(self):
        """
        Restore the volume_manager gconf settings
        """
        if gconf:
            client = gconf.client_get_default()
            path = '/desktop/gnome/volume_manager'
            for prop, value in self.volume_manager_config.iteritems():
                client.set_bool('%s/%s' % (path, prop), value)


    def _components_loaded(self, msg, sender):
        self.hal_iface.connect_to_signal('DeviceAdded',
                                         self._device_added_callback)
        self.hal_iface.connect_to_signal('DeviceRemoved',
                                         self._device_removed_callback)

    # send messages
    # new device by hal
    def _device_added_callback(self, udi):
        self.debug("device added: %r", udi)
        # check if this any kind of device we care about.
        device = self._get_device_with_udi(udi)
        self.debug("device object: %r", device)
        if device is not None:
            model = self._create_model_for_device(udi, device)

            if model:
                msg = NewDeviceDetected()
                msg.udi = udi
                msg.model = model
                if isinstance(model, VolumeModel) and not model.mount_point:
                    return
                common.application.bus.send_message(msg, self)
        else:
            return self._send_unknown_device(udi)

    def _device_removed_callback(self, udi):
        msg = None
        if udi in self._hotplugged_devices:
            mount_point = self._hotplugged_devices[udi]
            del self._hotplugged_devices[udi]
            msg = DeviceRemoved(mount_point)
        else:
            for udi, mount_point in self._hotplugged_devices.iteritems():
                if self._get_device_with_udi(udi) is None:
                    del self._hotplugged_devices[udi]
                    msg = DeviceRemoved(mount_point)
                    break
        if msg:
            common.application.bus.send_message(msg, self)

    def _send_unknown_device(self, udi):
        common.application.bus.send_message(NewUnknownDevice(udi), self)

    # models
    def _create_model_for_device(self, udi, device):
        model = None
        if udi not in self._hotplugged_devices:

            # decide on the capabaility which model creation to call
            if device.QueryCapability('volume'):
                model = self._create_model_for_volume(udi, device)
                if isinstance(model, VolumeModel):
                    self._hotplugged_devices[udi] = model.mount_point


            # FIXME: add video/dvb device
            # FIXME: add digital cameras
            # FIXME: add bluetooth-headsets and speakers?

        return model

    def _create_model_for_volume(self, udi, device):
        parent_udi = device.GetProperty(u'info.parent')
        parent = self._get_device_with_udi(parent_udi)
        if parent is None:
            return None

        properties = device.GetAllProperties()

        if parent.PropertyExists('portable_audio_player.type'):
            protocols = parent.GetProperty('portable_audio_player.access_method.protocols')

            # FIXME: add more for other kind players asap
            if 'ipod' in protocols:
                # is ipod
                return self._create_new_volume_model(udi, 'ipod', device, properties)

            # fallback: we don't know this portable player
            return self._create_new_volume_model(udi, 'portable_player', device, properties)

        elif properties.get('volume.disc.is_videodvd', False):
            # is dvd
            return self._create_new_volume_model(udi, 'dvd', device, properties)

        elif properties.get('volume.disc.has_audio', False):
            # is audio cd
            return self._create_new_volume_model(udi, 'cdda', device, properties)

        # this are all type of
        elif properties.get('volume.disc.is_partition', False) or \
                properties.get('volume.is_partition', False) or \
                properties.get('volume.disc.has_data', False) or \
                (parent.PropertyExists('storage.removable') and \
                        parent.GetProperty('storage.removable') == True) or \
                (parent.PropertyExists('storage.hotpluggable') and \
                        parent.GetProperty('storage.hotpluggable') == True):
                # other file based system
                return self._create_new_volume_model(udi, 'file',
                        device, properties)

    def _create_new_volume_model(self, udi, protocol, device, properties=None):
        if not properties:
            properties = device.GetAllProperties()

        if protocol in ('file', 'portable_player', 'ipod'):
            if properties.get('volume.ignore', False):
                self.debug("%s at %s wants to be ignored" % (udi, device))
                return

            model = VolumeModel()

            label = unicode(properties.get('volume.label', ''))
            if not label:
                label = unicode(properties.get('info.product', ''))


            model.label = label

            if not properties.get('volume.is_mounted'):
                try:
                    mount_point = self.mount_device(device)
                except NotMountable, exception:
                    mount_point = None
            else:
                mount_point = unicode(properties.get('volume.mount_point',''))

            if mount_point:
                if protocol == 'ipod':
                    mount_point = MediaUri("ipod://%s/" % mount_point)
                else:
                    mount_point = MediaUri("file://%s/" % mount_point)

            model.mount_point = mount_point
        else:
            model = DeviceModel()

        model.udi = udi
        model.device = unicode(properties.get('block.device', None))

        # protocol is elisa internal
        model.protocol = protocol

        return model

    # helper method
    def _get_device_with_udi(self, udi):
        obj = self.bus.get_object('org.freedesktop.Hal', udi)
        device = dbus.Interface(obj, 'org.freedesktop.Hal.Device')
        try:
            device.GetAllProperties()
        except DBusException, exc:
            unknown_device = 'org.freedesktop.Hal.NoSuchDevice'

            msg = get_dbus_error_message(exc)

            if msg.startswith(unknown_device):
                device = None
            else:
                raise
        return device

    def _get_property(self, device, prop_name, default):
        value = default
        if device.PropertyExists(prop_name):
            value = device.GetProperty(prop_name)
        return value

    def mount_device(self, device):
        prop_name = u'org.freedesktop.Hal.Device.Volume.method_names'
        method_names = self._get_property(device, prop_name, [])
        
        if 'Mount' not in method_names:
            raise NotMountable()

        interface = 'org.freedesktop.Hal.Device.Volume'
        method = device.get_dbus_method('Mount', dbus_interface=interface)

        name = device.GetProperty(u'info.product')
        self.debug("mounting: %r ", name)
        
        try:
            # Let's HAL decide, what the mount point should be
            method('', device.GetProperty(u'volume.fstype'),[])
        except DBusException, exc:
            already_mounted = 'org.freedesktop.Hal.Device.Volume.AlreadyMounted'
            unavailable = 'org.freedesktop.Hal.Device.Volume.MountPointNotAvailable'
            permission_denied = 'org.freedesktop.Hal.Device.Volume.PermissionDenied'
            unknown_failure = 'org.freedesktop.Hal.Device.Volume.UnknownFailure'

            msg = get_dbus_error_message(exc)

            if msg.startswith(already_mounted):
                self.info("Already mounted")
            elif msg.startswith(permission_denied):
                idx = msg.index(permission_denied) + len(permission_denied) + 2
                device_error = msg[idx:]
                self.info("Permission denied: %s", device_error)
            elif msg.startswith(unavailable):
                return None
            elif msg.startswith(unknown_failure):
                return None
            else:
                raise

        mount_point = self._get_mount_point(device)
        return mount_point

    def _get_mount_point(self, device):
        mount_point = device.GetProperty(u'volume.mount_point')
        name = device.GetProperty(u'info.product')
        if not mount_point:
            if platform.system() == 'Linux':
                # this property is not supported by upstream HAL
                # and seems specific to Ubuntu distro as stated there:
                # http://lists.freedesktop.org/archives/hal/2007-April/008062.html
                # FIXME: this needs further investigation. linux.* properties
                #        should not be used.
                mount_point = self._get_property(device,
                                                 'linux.fstab.mountpoint','')
                
        self.debug("mount point of %r: %r", name, mount_point)
        return mount_point
