# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.components.controller import Controller
from elisa.core.components.model import Model
from elisa.extern.log.log import getFailureMessage
from elisa.core import common
from elisa.core.media_uri import MediaUri
from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.weather.report_widget import WeatherWidget
from elisa.plugins.weather.report_provider import WeatherReportModel
from twisted.internet import defer, task

from pgm.timing.implicit import *

class WeatherController(PigmentController):
    default_config = {'interval': 60, 'station': 'LEBL'}
    config_doc = {'interval': 'update interval',
            'station': 'station id (defaults to Barcelona, Spain. Guess why.)'}

    def initialize(self):
        # create an empty model
        self.model = WeatherReportModel()
        # use a looping call to fill and update self.model at regular intervals
        self.looping_call = task.LoopingCall(self._get_report)

        return defer.succeed(self)

    def clean(self):
        if self.looping_call.running:
            self.looping_call.stop()

    def set_frontend(self, frontend):
        # build the UI
        widget = self.build_widget(self.model)
        widget.visible = True
        self.widget.add(widget)
        self.animated = AnimatedObject(widget)
        self.animated.setup_next_animations(repeat_count=INFINITE,
                                            repeat_behavior=REVERSE,
                                            duration=2000)
        self.animated.y -= 0.1

        # get the station id from the configuration file
        station = self.config['station']
        if station is not None:
            interval = int(self.config['interval'])
            self.debug('starting update at interval %s' % interval)
            self.looping_call.start(interval)

    def build_widget(self, model):
        widget = WeatherWidget()
        model.bind('city', widget, 'city')
        model.bind('sky', widget, 'sky')
        model.bind('temperature', widget, 'temperature')
        model.bind('image', widget, 'image')

        return widget

    def _get_report(self):
        # get a report via the ResourceManager
        resource_manager = common.application.resource_manager
        uri = MediaUri('metar://' + self.config['station'])
        model, dfr = resource_manager.get(uri)
        dfr.addCallback(self._get_report_callback)
        dfr.addErrback(self._get_report_errback)

        return dfr
    
    def _get_report_callback(self, model):
        # update self.model so that the changes will be reflected in the UI
        for attribute in ('city', 'sky', 'temperature', 'image'): 
            setattr(self.model, attribute, str(getattr(model, attribute)))

        return model

    def _get_report_errback(self, failure):
        self.warning('error updating weather status: %s' % 
                getFailureMessage(failure))

        # ignore so that the looping call won't stop
        return None
