# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

import inspect
import os
import sys

import pkg_resources

from twisted.internet import defer
from twisted.python import reflect
from twisted.python.rebuild import rebuild

from elisa.core import common
from elisa.core.component import ComponentError
from elisa.core.components.message import Message
from elisa.core.log import Loggable
from elisa.extern import enum

default_plugin_dirs = []
default_plugin_dirs.append(os.path.join(os.path.expanduser('~'), '.elisa-0.5', 'plugins'))

class InvalidComponentPath(ComponentError):
    def __init__(self, component_name):
        super(InvalidComponentPath, self).__init__(component_name)

    def __str__(self):
        return "Invalid component path %s" % self.component_name

class ComponentNotFound(ComponentError):
    def __init__(self, component_name):
        super(ComponentNotFound, self).__init__(component_name)

    def __str__(self):
        return "Component %r not found" % self.component_name

class PluginNotFound(Exception):
    pass

class PluginAlreadyEnabled(Exception):
    pass

class PluginAlreadyDisabled(Exception):
    pass

class PluginStatusMessage(Message):
    """
    A plugin has been enabled or disabled.

    @ivar action: C{ActionType.ENABLED} or C{ActionType.DISABLED}
    @type action: C{ActionType}
    @ivar plugin_name: name of the plugin
    @type plugin_name: C{str}
    """
    ActionType = enum.Enum('DISABLED', 'ENABLED')

    def __init__(self, plugin_name, action):
        self.plugin_name = plugin_name
        self.action = action

    def __str__(self):
        '<PluginStatusMessage %s %s>' % (self.plugin_name, self.action)

class PluginRegistry(Loggable):
    def __init__(self, plugin_dirs=None):
        super(PluginRegistry, self).__init__()

        if plugin_dirs is None:
            plugin_dirs = default_plugin_dirs[:]

        self.plugin_dirs = plugin_dirs
        self._plugin_status = {}

    # here be the dragons
    def _deactivate_dist(self, working_set, name='elisa'):
        # the list of deactivated distributions
        dists = []
        # the paths containing the deactivated distributions
        paths = []

        for dist in list(working_set):
            if not dist.key.startswith(name):
                continue

            dists.append(dist)

            self.debug('removing dist %s' % dist)
            # remove dist from the working set
            working_set.entry_keys[dist.location].remove(dist.key)
            del working_set.by_key[dist.key]

            if dist.location not in paths:
                # this is the first distribution installed in dist.location that
                # we deactivate
                paths.append(dist.location)

                # dist_paths is a list of sys.path-like entries that we need to
                # remove dist.location from
                dist_paths = [working_set.entries]

                if working_set.entries is not sys.path:
                    # make sure that we remove the location from sys.path
                    dist_paths.append(sys.path)

                # for uninstalled distributions, we add the location to
                # elisa.plugins.__path__ (or elisa.__path__ for core) so we need
                # to remove it here
                if self._is_uninstalled_plugin(dist):
                    self._fix_uninstalled_plugin(dist, 'remove')

                for entries in dist_paths:
                    # remove dist.location from the working set, it will be added
                    # back later when the first distribution in dist.location is
                    # activated.

                    # FIXME: if in the next load_plugins() call dist.location
                    # doesn't contain any distribution we need to manually add
                    # it back

                    try:
                        entries.remove(dist.location)
                    except ValueError:
                        # dist.location is always normalized but "entries"
                        # contains unnormalized paths. This usually happens on
                        # windows.
                        normalized_entries = [pkg_resources.normalize_path(entry)
                                for entry in entries]
                        try:
                            index = normalized_entries.index(dist.location)
                            del entries[index]
                        except ValueError:
                            pass

        return dists, paths

    def _is_uninstalled_plugin(self, dist):
        return os.path.isdir(dist.location) and \
                not os.path.isdir(os.path.join(dist.location, 'elisa'))

    def _fix_uninstalled_plugin(self, dist, action):
        assert action in ('add', 'remove')

        # dist is an uncompressed egg that contains the modules
        # directly under dist.location and not under
        # dist.location/elisa/plugins. With our current layout this
        # means all the plugins when running elisa uninstalled.
        self.info('%s is an uninstalled plugin' % dist)

        normalized_location = pkg_resources.normalize_path(dist.location)
        parent = os.path.dirname(normalized_location)
        path = None
        if dist.key == 'elisa':
            import elisa
            path = elisa.__path__
        else:
            import elisa.plugins
            path = elisa.plugins.__path__

        if action == 'add':
            if parent not in path:
                path.append(parent)
        elif action == 'remove':
            try:
                path.remove(parent)
            except ValueError:
                pass

    def load_plugins(self, plugin_names=None):
        """
        Load plugins from self.plugin_dirs.

        This function should be called as early as possible at startup, _before_
        using any plugin.
        Note that this function runs without returning to the reactor for as
        long as it takes. There's no point in making it return before it's
        done as the plugin environment needs to be setup before any other part
        of elisa can run correctly.

        You can optionally pass a list of plugin names to enable. If you don't
        or pass None, all the available plugins are enabled.

        @param plugin_names: the names of the plugins to enable. If None, all
                             the available plugins are enabled.
        @type plugin_names: sequence of strings or None
        """
        self.info('loading plugins from %s' % self.plugin_dirs)

        # the plugin registry is in elisa.core so the core must always be loaded
        # at this point
        core = pkg_resources.working_set.by_key['elisa']
        self.info('currently loaded core %s' % core)

        # deactivate the plugins in the current working set so we can activate
        # new versions in self.plugin_dirs
        old_dists, paths = self._deactivate_dist(pkg_resources.working_set, 'elisa')

        # paths contains a list with the paths of the distributions that we
        # deactivated in deactivated_dist(). When load_plugins() is called for
        # the first time it contains the path of the plugins installed system
        # wide (if there's one).
        plugin_dirs = self.plugin_dirs + paths

        env = pkg_resources.Environment(plugin_dirs)
        distributions, errors = pkg_resources.working_set.find_plugins(env)

        for error in errors:
            self.warning('plugin conflict %s' % error)

        # build a map {distribution_name: [dist_object, False]} with new
        # distributions plus the distributions that we deactivated in
        # _deactivate_dist(). We use this map to keep track of the plugins that
        # are currently loaded and for which there's a new version available.
        reload_map = dict([(dist.key, [dist, False])
                for dist in distributions + old_dists])

        # add the plugins to the active working set
        for dist in distributions:
            if not dist.key.startswith('elisa'):
                continue

            self.log('loading plugin %s' % dist)

            # do some extra work if the plugin is an uncompressed, uninstalled
            # plugin
            if self._is_uninstalled_plugin(dist):
                self._fix_uninstalled_plugin(dist, 'add')

            # check if the plugin was already called before calling
            # load_plugins().
            old_dist = reload_map[dist.key][0]
            if old_dist.version < dist.version:
                # it was loaded and we have a new version, mark it as to be
                # reloaded
                reload_map[dist.key][1] = True

            # add the distribution to the working set
            pkg_resources.working_set.add(dist)

        if plugin_names:
            plugin_names = set(plugin_names)

        for key, (dist, to_reload) in reload_map.iteritems():
            if to_reload:
                # FIXME: this code needs to go as soon as the bootstrap code
                # becomes stable.

                if dist.key == 'elisa':
                    # core
                    package = 'elisa'
                else:
                    entry_point = dist.get_entry_map('elisa.plugins').values()[0]
                    package = entry_point.module_name

                self.info('reloading plugin %s, package %s' % (key, package))
                for module_name, module in sys.modules.items():
                    if module is None or not module_name.startswith(package):
                        continue

                    self.info('reloading module %s' % module)
                    rebuild(module)
                    self.info('reloaded module %s' % module)
 
            if plugin_names is None or key in plugin_names:
                self._plugin_status[key] = True
            else:
                self._plugin_status[key] = False 

        self.info('loaded %d plugins' % len(distributions))

    def enable_plugin(self, plugin_name):
        """
        Enable a plugin.

        @param plugin_name: the name of the plugin to enable
        @type plugin_name: C{str}
        """
        try:
            if self._plugin_status[plugin_name]:
                raise PluginAlreadyEnabled(plugin_name)
        except KeyError:
            raise PluginNotFound(plugin_name)

        self._plugin_status[plugin_name] = True
        message = PluginStatusMessage(plugin_name,
                PluginStatusMessage.ActionType.ENABLED)
        common.application.bus.send_message(message)

    def disable_plugin(self, plugin_name):
        """
        Disable a plugin.

        @param plugin_name: the name of the plugin to disable
        @type plugin_name: C{str}
        """

        try:
            if not self._plugin_status[plugin_name]:
                raise PluginAlreadyDisabled(plugin_name)
        except KeyError:
            raise PluginNotFound(plugin_name)
 
        self._plugin_status[plugin_name] = False
        message = PluginStatusMessage(plugin_name,
                PluginStatusMessage.ActionType.DISABLED)
        common.application.bus.send_message(message)
    
    def get_plugins(self):
        """
        Get the list of available plugins.

        This call returns (plugin_name, status) tuples, where status is True if
        the plugin is enabled, False otherwise.

        @return generator yielding (plugin_name, status) tuples
        @rtype: C{generator}
        """
        return self._plugin_status.iteritems()

    def get_enabled_plugins(self):
        """
        Get the list of enabled plugins.

        @return: generator yielding plugin names
        @rtype: C{generator}
        """
        for name, status in self._plugin_status.iteritems():
            if status == False:
                continue

            yield name

    def get_plugin_names(self, path='elisa.plugins'):
        """
        Get the names of the installed plugins.

        @param path: plugin path, defaults to 'elisa.plugins'
        @type path: C{str}
        @return: a C{generator} object yielding plugin names
        @rtype: C{generator}
        """
        return self._plugin_status.iterkeys()

    def create_component(self, path, config=None, **kwargs):
        """
        Create a component given its path.

        The path is in module:Component syntax, eg
        elisa.plugins.my_plugin:MyComponent.

        @param path: the component path
        @type path: C{str}
        @param config: the configuration to set for the component
        @type config: L{elisa.core.config.Config}
        @return: an instance of the component identified by C{path}
        @rtype: L{elisa.core.component.Component} or a subclass
        """

        if not path.startswith('elisa.'):
            # a shortcut so we don't have to specify the whole path in the conf
            # file
            path = 'elisa.plugins.' + path

        self.debug('creating component %s' % path)

        try:
            module, klass = path.split(':', 1)
        except ValueError:
            return defer.fail(InvalidComponentPath(path))

        try:
            component_class = reflect.namedAny('%s.%s' % (module, klass))
        except:
            # it's ok to catch everything here and errback
            return defer.fail()

        self.debug('got class %s, calling create()'
                % reflect.qual(component_class))
        return component_class.create(config, **kwargs)
