/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007, 2008 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_BACKEND_H__
#define __PGM_BACKEND_H__

/* pgmbackend.h and pgmcontext.h include eachother */
typedef struct _PgmBackend      PgmBackend;
typedef struct _PgmBackendClass PgmBackendClass;

#include <pgm/pgm.h>
#include "pgmcontext.h"

G_BEGIN_DECLS

#define PGM_TYPE_BACKEND (pgm_backend_get_type ())
#define PGM_BACKEND(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj), PGM_TYPE_BACKEND, PgmBackend))
#define PGM_BACKEND_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass), PGM_TYPE_BACKEND, PgmBackendClass))
#define PGM_IS_BACKEND(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj), PGM_TYPE_BACKEND))
#define PGM_IS_BACKEND_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass), PGM_TYPE_BACKEND))
#define PGM_BACKEND_GET_CLASS(obj) \
  (G_TYPE_INSTANCE_GET_CLASS((obj), PGM_TYPE_BACKEND, PgmBackendClass))

struct _PgmBackend {
  GstObject parent;

  /* Reference to the PgmContext */
  PgmContext *context;
};

struct _PgmBackendClass {
  GstObjectClass parent_class;

  /* virtual methods for subclasses */

  /* Create the window */
  gboolean (*create_window)           (PgmBackend *backend);

  /* Destroy the window */
  gboolean (*destroy_window)          (PgmBackend *backend);

  /* Define the title of the window */
  void     (*set_title)               (PgmBackend *backend,
                                       const gchar *title);

  /* Show or hide the window decorations (titlebar, resize buttons, ...) */
  gboolean (*set_decorated)           (PgmBackend *backend,
                                       gboolean decoration);

  /* Swap the buffers */
  void     (*swap_buffers)            (PgmBackend *backend);

  /* Get the address of the OpenGL extension function named proc_name */
  gpointer (*get_proc_address)        (PgmBackend *backend,
                                       const gchar *proc_name);

  /* Set the window size */
  gboolean (*set_size)                (PgmBackend *backend,
                                       gint width,
                                       gint height);

  /* Set whether the window should be displayed fullscreen */
  gboolean (*set_fullscreen)          (PgmBackend *backend,
                                       gboolean fullscreen);

  /* Show/Hide (map/unmap) the window */
  gboolean (*set_visibility)          (PgmBackend *backend,
                                       gboolean visible);

  /* Retrieve the size of the physical screen in millimeters */
  void     (*get_screen_size_mm)      (PgmBackend *backend,
                                       gint *width,
                                       gint *height);

  /* Change the resolution of the desktop */
  gboolean (*set_screen_resolution)   (PgmBackend *backend,
                                       gint width,
                                       gint height);

  /* Retrieve the desktop resolution */
  void     (*get_screen_resolution)   (PgmBackend *backend,
                                       gint *width,
                                       gint *height);

  /* Build rasterized text display lists (used for debugging purposes) */
  gboolean (*build_text_lists)        (PgmBackend *backend);

  /* Destroy rasterized text display lists (used for debugging purposes) */
  gboolean (*destroy_text_lists)      (PgmBackend *backend);

  /* Rasterize a text on the viewport at position (x,y) with the color
   * (r,g,b,a) */
  void     (*raster_text)             (PgmBackend *backend,
                                       const gchar *text,
                                       gfloat x,
                                       gfloat y,
                                       gfloat r,
                                       gfloat g,
                                       gfloat b);

  /* Wait for the next VBlank */
  void     (*wait_for_vblank)         (PgmBackend *backend);

  void     (*notify_startup_complete) (PgmBackend *backend);

  /* Define the specified system cursor */
  gboolean (*set_cursor)              (PgmBackend *backend,
                                       PgmViewportCursor cursor);

  /* Define the icon */
  gboolean (*set_icon)                (PgmBackend *backend,
                                       GdkPixbuf *icon);

  /* Accept or refuse an on-going drag */
  void     (*set_drag_status)         (PgmBackend *backend,
                                       gboolean accept);

  /* Retrieve if the OpenGL context is hardware accelerated */
  gboolean (*is_accelerated)          (PgmBackend *backend);

  /* Retrieve if the backend can be embedded in another application */
  gboolean (*is_embeddable)           (PgmBackend *backend);

  /* Retrieve the ID of the window to use by other application */
  void     (*get_embedding_id)        (PgmBackend *backend,
                                       gulong *embedding_id);

  /* Retrieve if the backend handles alpha clear color */
  gboolean (*has_alpha_component)     (PgmBackend *backend);

  /* Retrieve if the backend supports system buffers */
  gboolean (*has_system_buffer)            (PgmBackend *backend);

  /* Create a system buffer object understandable by the backend from a given
   * system buffer */
  gpointer (*create_system_buffer_object)  (PgmBackend *backend,
                                            gconstpointer system_buffer,
                                            PgmImagePixelFormat format);

  /* Destroy the system buffer object created with create_system_buffer */
  void     (*destroy_system_buffer_object) (PgmBackend *backend,
                                            gpointer system_buffer_object);

  /* Bind the system buffer object created with create_system_buffer */
  void     (*bind_system_buffer_object)    (PgmBackend *backend,
                                            gconstpointer system_buffer_object);

  /* Release the system buffer object bound with bind_system_buffer */
  void     (*release_system_buffer_object) (PgmBackend *backend,
                                            gconstpointer system_buffer_object);
};

GType       pgm_backend_get_type                (void) G_GNUC_CONST;

void        pgm_backend_register                (GTypeModule *module);

gboolean    pgm_backend_create_window           (PgmBackend *backend);

gboolean    pgm_backend_destroy_window          (PgmBackend *backend);

void        pgm_backend_set_title               (PgmBackend *backend,
                                                 const gchar *title);

gboolean    pgm_backend_set_decorated           (PgmBackend *backend,
                                                 gboolean decorated);

void        pgm_backend_swap_buffers            (PgmBackend *backend);

gpointer    pgm_backend_get_proc_address        (PgmBackend *backend,
                                                 const gchar *proc_name);

gboolean    pgm_backend_set_size                (PgmBackend *backend,
                                                 gint width,
                                                 gint height);

gboolean    pgm_backend_set_fullscreen          (PgmBackend *backend,
                                                 gboolean fullscreen);

gboolean    pgm_backend_set_visibility          (PgmBackend *backend,
                                                 gboolean visible);

void        pgm_backend_get_screen_size_mm      (PgmBackend *backend,
                                                 gint *width,
                                                 gint *height);

gboolean    pgm_backend_set_screen_resolution   (PgmBackend *backend,
                                                 gint width,
                                                 gint height);

void        pgm_backend_get_screen_resolution   (PgmBackend *backend,
                                                 gint *width,
                                                 gint *height);

gboolean    pgm_backend_build_text_lists        (PgmBackend *backend);

gboolean    pgm_backend_destroy_text_lists      (PgmBackend *backend);

void        pgm_backend_raster_text             (PgmBackend *backend,
                                                 const gchar *text,
                                                 gfloat x,
                                                 gfloat y,
                                                 gfloat r,
                                                 gfloat g,
                                                 gfloat b);

void        pgm_backend_wait_for_vblank         (PgmBackend *backend);

void        pgm_backend_notify_startup_complete (PgmBackend *backend);

gboolean    pgm_backend_set_cursor              (PgmBackend *backend,
                                                 PgmViewportCursor cursor);

gboolean    pgm_backend_set_icon                (PgmBackend *backend,
                                                 GdkPixbuf *icon);

void        pgm_backend_set_drag_status         (PgmBackend *backend,
                                                 gboolean accept);

gboolean    pgm_backend_is_accelerated          (PgmBackend *backend);

gboolean    pgm_backend_is_embeddable           (PgmBackend *backend);

void        pgm_backend_get_embedding_id        (PgmBackend *backend,
                                                 gulong *embedding_id);

gboolean    pgm_backend_has_alpha_component     (PgmBackend *backend);

gboolean    pgm_backend_has_system_buffer            (PgmBackend *backend);

gpointer    pgm_backend_create_system_buffer_object  (PgmBackend *backend,
                                                      gconstpointer system_buffer,
                                                      PgmImagePixelFormat format);

void        pgm_backend_destroy_system_buffer_object (PgmBackend *backend,
                                                      gpointer system_buffer_object);

void        pgm_backend_bind_system_buffer_object    (PgmBackend *backend,
                                                      gconstpointer system_buffer_object);

void        pgm_backend_release_system_buffer_object (PgmBackend *backend,
                                                      gconstpointer system_buffer_object);

G_END_DECLS

#endif /* __PGM_BACKEND_H__ */
