# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2007-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL.EXCEPTION" in the root of this distribution
# including a special exception to use Elisa with Fluendo's plugins and
# about affiliation parameters.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>

from elisa.core.media_uri import MediaUri
from elisa.core import common
from elisa.core.utils.i18n import install_translation

from elisa.plugins.poblesec.actions import Action
from elisa.plugins.poblesec.base.hierarchy import HierarchyController
from elisa.plugins.poblesec.base.list_switcher import ListSwitcherController
from elisa.plugins.poblesec.base.preview_list import \
                                        MenuItemPreviewListController, \
                                        DoubleLineMenuItemPreviewListController
from elisa.plugins.poblesec.base.coverflow import \
                                        ImageWithReflectionCoverflowController
from elisa.plugins.poblesec.base.grid import GridItemGridController

from elisa.plugins.database.actions import TrackPlayAllAction
from elisa.plugins.database.music_controller import GenericTracksDbController
from elisa.plugins.database.music_controller import DBTracksViewMode, \
                                                    DBTracksInAlbumViewMode
from elisa.plugins.database.music_controller import ArtistsDbController
from elisa.plugins.database.music_controller import ArtistsDbViewMode
from elisa.plugins.database.music_controller import GenericAlbumsDbController
from elisa.plugins.database.music_controller import AlbumsDbViewMode

from elisa.core.utils import defer


_ = install_translation('yesfm')


class BrowseAlbumAction(Action):
    def run(self, album):
        frontend = self.controller.frontend
        browser = frontend.retrieve_controllers('/poblesec/browser')[0]
        dfr = browser.history.append_controller('/poblesec/yesfm/tracks',
                                                album.name,
                                                album=album)
        return dfr

class EnterTrackAction(Action):
    def run(self, track):
        frontend = self.controller.frontend
        browser = frontend.retrieve_controllers('/poblesec/browser')[0]
        tracks = filter(lambda x: not isinstance(x, Action), self.controller.model)
        dfr = browser.history.append_controller('/poblesec/yesfm/track_actions',
                                                track.title,
                                                track=track,
                                                tracks=tracks)
        return dfr


class ArtistsController(ArtistsDbController):
    artist_controller_path = '/poblesec/yesfm/albums'

class ArtistsPreviewListController(ArtistsController, MenuItemPreviewListController):
    view_mode = ArtistsDbViewMode

class ArtistsCoverflowController(ArtistsController, ImageWithReflectionCoverflowController):
    view_mode = ArtistsDbViewMode

class ArtistsGridController(ArtistsController, GridItemGridController):
    view_mode = ArtistsDbViewMode

class ArtistsListSwitcherController(ListSwitcherController):
    modes = [ArtistsPreviewListController,
             ArtistsCoverflowController,
             ArtistsGridController]
    default_mode = ArtistsPreviewListController


class AlbumsController(GenericAlbumsDbController):
    album_controller_path = '/poblesec/yesfm/tracks'

    def initialize(self, artist=None, albums=None):
        dfr = super(AlbumsController, self).initialize(artist=artist)

        self.albums = albums

        def extract_albums(artist):
            return artist.albums

        def retrieve_albums(controller):
            artist_name = artist.name
            # Yes.fm's API specifies that spaces have to be replaced with
            # dashes
            artist_name = artist_name.replace(" ", "-")
            uri = MediaUri("http://api.yes.fm/v1/artist/%s" % artist_name)
            model, dfr = common.application.resource_manager.get(uri)
            dfr.addCallback(extract_albums)
            return dfr

        def insert_albums(albums):
            self.model.extend(albums)
            return self

        if albums is None:
            dfr.addCallback(retrieve_albums)
            dfr.addCallback(insert_albums)
        else:
            dfr.addCallback(lambda x: insert_albums(albums))
        return dfr

    def clean(self):
        dfr = super(AlbumsController, self).clean()
        self.albums = None
        return dfr


# FIXME: this is temporary, we should do that in a unified manner so that it
#        works for any kind of albums controller (including local DB).
class AlbumsYesfmViewMode(AlbumsDbViewMode):

    def get_sublabel(self, item):
        if isinstance(item, Action):
            return defer.succeed(item.subtitle)
        return defer.succeed(item.artist.name)


class AlbumsPreviewListController(AlbumsController, DoubleLineMenuItemPreviewListController):
    view_mode = AlbumsYesfmViewMode

class AlbumsCoverflowController(AlbumsController, ImageWithReflectionCoverflowController):
    view_mode = AlbumsYesfmViewMode

class AlbumsGridController(AlbumsController, GridItemGridController):
    view_mode = AlbumsYesfmViewMode

class AlbumsListSwitcherController(ListSwitcherController):
    modes = [AlbumsPreviewListController,
             AlbumsCoverflowController,
             AlbumsGridController]
    default_mode = AlbumsPreviewListController

class TracksController(GenericTracksDbController):

    track_controller_path = "/poblesec/yesfm/track_actions"

    def initialize(self, tracks=None, album=None, artist=None):
        dfr = super(TracksController, self).initialize(tracks=tracks,
                                                       album=album,
                                                       artist=artist)

        def retrieve_tracks(controller):
            if tracks is None:
                return album.get_tracks()
            return tracks                

        def insert_tracks(tracks):
            self.model.extend(tracks)
            return self

        dfr.addCallback(retrieve_tracks)
        dfr.addCallback(insert_tracks)
        return dfr

class TracksPreviewListController(TracksController, DoubleLineMenuItemPreviewListController):
    view_mode = DBTracksInAlbumViewMode
    fastscroller = False

class TracksCoverflowController(TracksController, ImageWithReflectionCoverflowController):
    view_mode = DBTracksInAlbumViewMode
    fastscroller = False

class TracksGridController(TracksController, GridItemGridController):
    view_mode = DBTracksInAlbumViewMode
    fastscroller = False

class TracksListSwitcherController(ListSwitcherController):
    modes = [TracksPreviewListController,
             TracksCoverflowController,
             TracksGridController]
    default_mode = TracksPreviewListController


class TrackActionsController(HierarchyController, MenuItemPreviewListController):
    view_mode = DBTracksViewMode
    fastscroller = False

    def initialize(self, track, tracks):
        self.track = track
        self.tracks = tracks
        dfr = super(TrackActionsController, self).initialize()

        action = TrackPlayAllAction(self)
        action.title = "%s '%s'" % (_('Play'), self.track.title)
        self.actions.append(action)
        return dfr

    def node_clicked(self, widget, item):
        widget_index = self.nodes._widget_index_from_item_index(widget.model.index(item))
        selected_item = self.nodes._widgets[widget_index]

        if selected_item != self._previous_clicked:
            selected_item.activate(previous=self._previous_clicked)
            self._previous_clicked = selected_item
            item.run()
