/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: OOXMLContext.hxx,v $
 *
 *  $Revision: 1.33 $
 *
 *  last change: $Author: hbrinkm $ $Date: 2007/06/04 08:45:18 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#ifndef INCLUDED_OOXML_CONTEXT_HXX
#define INCLUDED_OOXML_CONTEXT_HXX

#ifndef _COM_SUN_STAR_XML_SAX_SAXEXCEPTION_HPP_
#include <com/sun/star/xml/sax/SAXException.hpp>
#endif

#ifndef _COM_SUN_STAR_UNO_RUNTIME_EXCEPTION_HPP_
#include <com/sun/star/uno/RuntimeException.hpp>
#endif

#include <rtl/ustring.hxx>
#include <doctok/WW8ResourceModel.hxx>
#include <ooxml/OOXMLDocument.hxx>

#ifndef INCLUDED_NAMESPACEMAPPER_HXX
#include "NamespaceMapper.hxx"
#endif

#ifndef INCLUDED_OOXML_PROPERTY_IMPL_SET_HXX
#include "OOXMLPropertySetImpl.hxx"
#endif

namespace ooxml
{
using namespace doctok;


class OOXMLParserState
{
    bool mbInSectionGroup;
    bool mbInParagraphGroup;
    bool mbInCharacterGroup;
    bool mbLastParagraphInSection;
    bool mbForwardEvents;
    unsigned int mnContexts;
    OOXMLDocument * mpDocument;
    rtl::OUString msXNoteId;

public:
    typedef boost::shared_ptr<OOXMLParserState> Pointer_t;

    OOXMLParserState();
    virtual ~OOXMLParserState();

    void setInSectionGroup(bool bInSectionGroup);
    bool isInSectionGroup() const;
    
    void setLastParagraphInSection(bool bLastParagraphInSection);
    bool isLastParagraphInSection() const;

    void setInParagraphGroup(bool bInParagaphGroup);
    bool isInParagraphGroup() const;

    void setInCharacterGroup(bool bInCharacterGroup);
    bool isInCharacterGroup() const;

    void setForwardEvents(bool bForwardEvents);
    bool isForwardEvents() const;
    
    void incContextCount();
    const string getHandle() const;
    unsigned int getContextCount() const;

    void setDocument(OOXMLDocument * pDocument);
    OOXMLDocument * getDocument() const;

    void setXNoteId(const rtl::OUString & rId);
    const rtl::OUString & getXNoteId() const;

    string toString() const;
};

/**
   Root class of the context hierarchy.

   Each context represents one XML-element. Each context represents a
   resource fitting in the resource/handler concept described in
   WW8ResourceModel.hxx.

   The contexts are put on a stack. For each SAX startElement event a
   new context is generated and put on the stack. For each SAX
   endElement the current top context is pulled from the stack and
   endElement is called on its parent stack.

   Typical chain of events:

<pre>
   SAX: startElement
   Context[n]: handleNamespaceMappings
   Context[n]: element =>  Context[n+1], push on stack
   Context[n+1]: handleAttributes
   SAX: characters
   Context[n+1]: characters
   SAX: endElement
   Context[n]: pop Context[n+1] from stack
   Context[n]: endElement
   Context[n+1]: destroy
</pre>
   
 */
class OOXMLContext
{
protected:
    // the stream to send the stream events to.
    Stream & mrStream;

    // the namespace mapper
    NamespaceMapper mNamespaceMapper;

    // the table depth of this context
    unsigned int mnTableDepth;

    // Returns string for resource of this context. (debug)
    string getResourceString() const;

    TokenEnum_t meToken;

    OOXMLParserState::Pointer_t mpParserState;
    
    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const rtl::OUString & rValue);
    virtual void lcl_characters(const rtl::OUString & str); 

public:
    typedef boost::shared_ptr <OOXMLContext> Pointer_t;
    // resources from WW8ResourceModel
    enum ResourceEnum_t { UNKNOWN, STREAM, PROPERTIES, TABLE };

    OOXMLContext(Stream & rStream);
    OOXMLContext(const OOXMLContext & rContext); 

    virtual ~OOXMLContext();
        
    /**
       Returns the token for a string.

       @param nType          type of token to search for
       @param rTokenString   the string to search the token for

       @see NamespaceMapper

       @return token for the given string
     */
    virtual TokenEnum_t getToken(TokenMap::TokenType_t nType,
                                 const rtl::OUString & rTokenString);

    /**
       Adds a namespace mapping to this context.

       The namespace mapping is only valid in this context and all
       contexts created by the element method of this context. If
       there is a mapping prefix->URL valid in this context a token
       string prefix:token will be interpreted as URL:token by
       getToken.

       @param rPrefix     prefix for the mapping
       @param rUrl        URL for the mapping
     */
    virtual void addNamespaceMapping(const rtl::OUString & rPrefix,
                                     const rtl::OUString & rUrl);
        
    /**
       Creates a subcontext for an element.

       @param nToken      token to create the context for

       @return the subcontext
    */
    virtual Pointer_t element(TokenEnum_t nToken);

    /**
       Handle end of a subcontext.

       This method is called when SAX sends the endElement event of
       the current subcontext.

       @param nToken      the token of the element to end
       @param pContext    the context that handled the element
     */
    virtual void endElement(TokenEnum_t nToken,
                            OOXMLContext::Pointer_t pContext);


    /**
       Creates a subcontext for an element using the <ref/> decendants
       of the <define/>.

       This is a fall back method called by element.

       @param nToken      token to create the context for

       @return the subcontext
     */
    virtual Pointer_t elementFromRefs(TokenEnum_t nToken);

    /**
       Handle SAX attribute.

       @param nToken     token for the attribute name
       @param rValue     string value of the attribute

       @return true if attribute was handled successfully, false otherwise
     */
    bool attribute(TokenEnum_t nToken, 
                   const rtl::OUString & rValue);

    /**
       Handle SAX characters.

       @param str       the characters to handle
     */
    void characters(const rtl::OUString & str); 

    /**
       Return value of this context(element).

       @return  the value
     */
    virtual OOXMLValue::Pointer_t getValue() const;

    /**
       Get id of this context(element).

       This id is used to call Stream.attribute and Stream.sprm.

       @param nToken   token to get id for
       
       @return   the id
     */
    virtual doctok::Id getId(TokenEnum_t nToken);

    /**
       Get id of this context(element).

       This is a fall back method called by getId. It tries to get a
       context for nToken iterating through all <ref/> decendants of
       the <define/> element of the according context class until a
       match is found.

       @param nToken   token to get id for
       
       @return   the id       
     */
    virtual doctok::Id getIdFromRefs(TokenEnum_t nToken);
    
    /**
       Set the token. 

       @param nToken         the token to set
    */

    void setToken(TokenEnum_t nToken);

    /**
       Processes the token. 
       Put all handling that involves element internal knowledge
       here. E.g. if a single element has a member caching the token
       it would send text specific for the token here.

       @param nToken         the token to process
    */
    virtual void processToken(TokenEnum_t nToken);

    /**
       Returns a string describing the type of the context.

       This is the name of the define normally.

       @return type string
     */
    virtual string getType() const { return "??"; }

    virtual ResourceEnum_t getResource() const { return UNKNOWN; }

    virtual string toString() const;

    virtual void setParentResource(ResourceEnum_t eResource);

    /**
       Actions to be executing when the context's XML element starts
    */
    virtual void startAction();

    /**
       Actions to be carried out when the context's XML element ends
    */
    virtual void endAction();

    virtual void resolvePropertySetAttrs();

    void sendTableDepth() const;

    void startSectionGroup();
    void setLastParagraphInSection();
    void endSectionGroup();
    void startParagraphGroup();
    void endParagraphGroup();
    void startCharacterGroup();
    void endCharacterGroup();

    void setDocument(OOXMLDocument * pDocument);
    void setXNoteId(const rtl::OUString & rId);
    const rtl::OUString & getXNoteId() const;
    void setForwardEvents(bool bForwardEvents);
    bool isForwardEvents() const;

    void resolveFootnote(const rtl::OUString & rId);
    void resolveEndnote(const rtl::OUString & rId);
    void resolveComment(const rtl::OUString & rId);
    void resolveHeader(const sal_Int32 type, const rtl::OUString & rId);
    void resolveFooter(const sal_Int32 type, const rtl::OUString & rId);
};

/**
   Root class for all context classes that represent a resource of
   type Stream.
 */
class OOXMLContextStream : public OOXMLContext
{
    mutable OOXMLPropertySet::Pointer_t mpPropertySetAttrs;
protected:
    virtual void  lcl_characters(const rtl::OUString & /*str*/);
    void newProperty(TokenEnum_t nToken, OOXMLValue::Pointer_t pVal);

public:
    OOXMLContextStream(Stream & rStream);
    OOXMLContextStream(const OOXMLContext & rContext);
    virtual ~OOXMLContextStream();

    virtual ResourceEnum_t getResource() const { return STREAM; }

    OOXMLPropertySet::Pointer_t getPropertySetAttrs() const;
    void setPropertySetAttrs(OOXMLPropertySet::Pointer_t pPropertySetAttrs);
    virtual void resolvePropertySetAttrs();
};

/**
   Root class for all context classes that represent a resource of
   type Properties.

   These classes represent a property set whose elements are sent via
   Stream::props to the inherited stream handler.
 */
class OOXMLContextProperties : public OOXMLContext
{
protected:
    /// the properties
    OOXMLPropertySet::Pointer_t mpPropertySet;
    
    /**
       Inserts a property to the property set.

       The given value is inserted as attribute to the property set.

       @param nToken   token for the property
       @param pValue   value of the property
     */
    void newProperty(TokenEnum_t nToken, OOXMLValue::Pointer_t pVal);

    bool mbResolve;

public:
    typedef boost::shared_ptr<OOXMLContextProperties> Pointer_t;
    OOXMLContextProperties(Stream & rStream);
    OOXMLContextProperties(const OOXMLContext & rContext);
    virtual ~OOXMLContextProperties();

    virtual OOXMLValue::Pointer_t getValue() const;
    virtual ResourceEnum_t getResource() const { return PROPERTIES; }

    /**
       Resolves the property set to the inherited stream handler.
       
       @param nToken    token to process
     */
    virtual void processToken(TokenEnum_t nToken);

    /**
       Ends an element.

       Takes the property value of pContext and inserts it as sprm to
       the property set.

       @param nToken    token for the property
       @param pContext  context providing the value for the property
     */
    virtual void endElement(TokenEnum_t nToken,
                            OOXMLContext::Pointer_t pContext);

    virtual string toString() const;

    virtual void setParentResource(ResourceEnum_t eResource);
    void setPropertySet(OOXMLPropertySet::Pointer_t pPropertySet);
};

/**
   Root class of all context classes that repesent a resource of type
   Property.

   The accordings elements are children of elements with resource type
   Properties.
*/
class OOXMLContextProperty : public OOXMLContext
{
public:
    OOXMLContextProperty(Stream & rStream);
    OOXMLContextProperty(const OOXMLContext & rContext);
    virtual ~OOXMLContextProperty();
};

/**
   Root class of all context classes that represent a resource of type
   SingleElement.   
 */
class OOXMLContextSingleElement : public OOXMLContext
{
protected:
    OOXMLPropertySet::Pointer_t mpPropertySet;

public:
    OOXMLContextSingleElement(Stream & rStream);
    OOXMLContextSingleElement(const OOXMLContext & rContext);
    virtual ~OOXMLContextSingleElement();
    
    virtual void newProperty(TokenEnum_t nToken, OOXMLValue::Pointer_t pVal);
    virtual void processToken(TokenEnum_t nToken);
};

/**
   Root class of all context classes that represent a resource of type
   Boolean.

   The value is set in the attribute method.
*/	
class OOXMLContextBooleanValue : public OOXMLContext
{
protected:
    /// the value
    bool mbValue;

    /**
       Handle SAX attribute event.

       The value of this object is determined by the attribute name
       token and the string value.

       If the rng::grammar allows an attribute A with boolean value,
       this object's value is set accroding to the string value of
       A. The last call of attribute that changes the value determines
       the final value of this object.

       @param nToken   attribute name token
       @param rString  the string value of the attribute

       @return true if setting the value was successfull, false otherwise
     */
    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const ::rtl::OUString & rString);

public:
    OOXMLContextBooleanValue(const OOXMLContextBooleanValue & rSrc);
    OOXMLContextBooleanValue(Stream & rStream);
    OOXMLContextBooleanValue(const OOXMLContext & rContext);
    virtual ~OOXMLContextBooleanValue();

    /**
       Returns the OOXMLBooleanValue of this object.

       @return the value
     */
    virtual OOXMLValue::Pointer_t getValue() const;
};

/**
   Root class of all context classes that represent a resource with
   type Integer.
 */
class OOXMLContextIntegerValue : public OOXMLContext
{
protected:
    /// the value
    sal_Int32 mnValue;

    /**
       Handle SAX attribute.

       If the rng:grammar allows an attribute matching the given
       attribute name token the value this object's value is set to
       the integer parsed from the given attribute value. The final
       value of this object is determined by the last attribute that
       changes the value.

       @param nToken   the given attribute name token
       @param rValue   the given attribute valuegetToken

       @return true if attribute was handled successfully, false otherwise
     */
    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const ::rtl::OUString & rString);

public:
    OOXMLContextIntegerValue(Stream & rStream);
    OOXMLContextIntegerValue(const OOXMLContext & rContext);
    virtual ~OOXMLContextIntegerValue();

    /**
       Return the OOXMLIntegerValue of this object.

       @return the value
     */
    virtual OOXMLValue::Pointer_t getValue() const;
};

/**
   Root class for context classes that represent a resource with type HexValue.
 */
class OOXMLContextHexValue : public OOXMLContext
{
protected:
    sal_uInt32 mnValue;

    /**
       Handle SAX attribute.

       If the rng:grammar allows an attribute matching the given
       attribute name token the value this object's value is set to
       the integer parsed from the given attribute value. The value
       has to be denoted in 0x.. syntax. The final value of this
       object is determined by the last attribute that changes the
       value.

       @param nToken   the given attribute name token
       @param rValue   the given attribute value

       @return true if attribute was handled successfully, false otherwise
     */
    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const ::rtl::OUString & rString);

public:
    OOXMLContextHexValue(const OOXMLContextHexValue & rSrc);
    OOXMLContextHexValue(Stream & rStream);
    OOXMLContextHexValue(const OOXMLContext & rContext);
    virtual ~OOXMLContextHexValue();
    /**
       Return the OOXMLIntegerValue of this object.

       @return the value
     */
    virtual OOXMLValue::Pointer_t getValue() const;
};

class OOXMLContextStringValue : public OOXMLContext
{
protected:
    ::rtl::OUString msValue;

    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const ::rtl::OUString & rString);

public:
    OOXMLContextStringValue(const OOXMLContext & rSrc);
    OOXMLContextStringValue(Stream & rStream);
    virtual ~OOXMLContextStringValue();

    virtual OOXMLValue::Pointer_t getValue() const;
};

class OOXMLContextListValue : public OOXMLContext
{
protected:
    sal_Int32 mnValue;

    virtual bool lcl_attribute(TokenEnum_t nToken, 
                               const ::rtl::OUString & rString);

public:
    OOXMLContextListValue(const OOXMLContext & rSrc);
    OOXMLContextListValue(Stream & rStream);
    virtual ~OOXMLContextListValue();

    virtual OOXMLValue::Pointer_t getValue() const;
};

class OOXMLContextTable : public OOXMLContext
{
protected:
    doctok::Id mId;
    OOXMLTableImpl mTable;

public:
    OOXMLContextTable(Stream & rStream);
    OOXMLContextTable(const OOXMLContext & rContext);
    virtual ~OOXMLContextTable();

    virtual void setId(doctok::Id nId);
    virtual void endElement(TokenEnum_t nToken,
                            OOXMLContext::Pointer_t pContext);

    virtual void newPropertySet(OOXMLPropertySet::Pointer_t pPropertySet);
};

class OOXMLContextTextTable : public OOXMLContextStream
{    
public:
    typedef boost::shared_ptr<OOXMLContextTextTable> Pointer_t;
    OOXMLContextTextTable(Stream & rStream);
    OOXMLContextTextTable(const OOXMLContext & rContext);
    virtual ~OOXMLContextTextTable();

    unsigned int getTableDepth() const;
};

class OOXMLContextTextTableRow : public OOXMLContextStream
{
public:
    OOXMLContextTextTableRow(Stream & rStream);
    OOXMLContextTextTableRow(const OOXMLContext & rContext);
    virtual ~OOXMLContextTextTableRow();

    void startRow();
    void endRow();
};

class OOXMLContextTextTableCell : public OOXMLContextStream
{
public:
    OOXMLContextTextTableCell(Stream & rStream);
    OOXMLContextTextTableCell(const OOXMLContext & rContext);
    virtual ~OOXMLContextTextTableCell();

    void startCell();
    void endCell();
};

class OOXMLContextXNote : public OOXMLContextStream
{
    bool mbForwardEventsSaved;

public:
    OOXMLContextXNote(Stream & rStream);
    OOXMLContextXNote(const OOXMLContext & rContext);
    virtual ~OOXMLContextXNote();
    
    void checkId(const rtl::OUString & rId);
};

class OOXMLFootnoteHandler : public Properties
{
    OOXMLContext * mpContext;
public:
    OOXMLFootnoteHandler(OOXMLContext * pContext);
    virtual ~OOXMLFootnoteHandler();

    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

class OOXMLEndnoteHandler : public Properties
{
    OOXMLContext * mpContext;
public:
    OOXMLEndnoteHandler(OOXMLContext * pContext);
    virtual ~OOXMLEndnoteHandler();

    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

class OOXMLFooterHandler : public Properties
{
    OOXMLContext * mpContext;
    ::rtl::OUString msStreamId;
    sal_Int32 mnType;
public:
    OOXMLFooterHandler(OOXMLContext * pContext);
    virtual ~OOXMLFooterHandler();
    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

class OOXMLHeaderHandler : public Properties
{
    OOXMLContext * mpContext;
    ::rtl::OUString msStreamId;
    sal_Int32 mnType;
public:
    OOXMLHeaderHandler(OOXMLContext * pContext);
    virtual ~OOXMLHeaderHandler();
    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

class OOXMLCommentHandler : public Properties
{
    OOXMLContext * mpContext;
    ::rtl::OUString msStreamId;
public:
    OOXMLCommentHandler(OOXMLContext * pContext);
    virtual ~OOXMLCommentHandler();
    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

class OOXMLBreakHandler : public Properties
{
    OOXMLContext * mpContext;
    sal_Int32 mnType, mnClear;
    Stream & mrStream;
public:
    OOXMLBreakHandler(Stream & rStream, OOXMLContext * pContext);
    virtual ~OOXMLBreakHandler();
    virtual void attribute(Id name, Value & val);
    virtual void sprm(Sprm & sprm);
};

}

#endif // INCLUDED_OOXML_CONTEXT_HXX
