#!/bin/bash
# Remove an existing new Axyl website instance.
# This script will remove an Axyl website which is on the path given
# on the command-line. Some checks will be made to make sure the
# given path is indeed an Axyl website root.
#
# Questions are then asked as to whether database, Apache configs etc.
# should also be removed, in a step-by-step approach.
#
# Paul Waite
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
set -e

echo ""
echo "REMOVE AXYL WEBSITE"
echo ""

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ] ; then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# Get path to website to remove..
WWW_PATH=
if [ "$1" != "" ] ; then
  WWW_PATH=$1
else
  echo "Enter path to the Axyl website which is to be REMOVED"
  echo -n "? "
  read ANS
  if [ "$ANS" = "" ] ; then
    echo "Exiting.. no path specified."
    exit 0
  else
    WWW_PATH=$ANS
  fi
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# SANITY CHECK
if [ ! -f ${WWW_PATH}/application.xml ] ; then
  echo "ERROR: application.xml not found. No Axyl website found at"
  echo "$WWW_PATH !"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INSTALLATION NAME
INST_NAME=`perl -ne 's/[\s]*<definition name="APP_PREFIX">([\w]+)<.*$/$1/ && print;' ${WWW_PATH}/application.xml`
echo ""
echo "Confirm the name of the Axyl website installation to be removed."
echo "(this is the APP_PREFIX as defined in application.xml)"
echo ""
echo -n "Axyl website installation name [${INST_NAME}] :"
read ANS
[ "$ANS" = "" ] && ANS=$INST_NAME
INST_NAME=$ANS
if [ "$INST_NAME" = "" ] ; then
  echo "A name is mandatory. Aborting.."
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# DATABASE DETAILS
# WARNING: Only copes with a SINGLE DATABASE defined for the website. If
# you have multiple definitions then the DB removal will need to be done
# manually.

DBTYPE=`grep "add_database" -A 1 ${WWW_PATH}/application.xml | egrep -E -o "\"type\"(.*?)>(.*?)<" | egrep -E -o ">(.*?)<" | tr -d "<>"`
DBNAME=`grep "add_database" -A 2 ${WWW_PATH}/application.xml | egrep -E -o "\"name\"(.*?)>(.*?)<" | egrep -E -o ">(.*?)<" | tr -d "<>"`
DBUSER=`grep "\"user\"" ${WWW_PATH}/application.xml | egrep -E -o ">(.*?)<" | tr -d "<>"`
DBPASSWD=`grep "\"password\"" ${WWW_PATH}/application.xml | egrep -E -o ">(.*?)<" | tr -d "<>"`
DBHOST=`grep "\"host\"" ${WWW_PATH}/application.xml | egrep -E -o ">(.*?)<" | tr -d "<>"`
DBPORT=`grep "\"port\"" ${WWW_PATH}/application.xml | egrep -E -o ">(.*?)<" | tr -d "<>"`	

case $DBTYPE in
  postgres) DBTYPE=1 ;;
  mysql)    DBTYPE=2 ;;
  odbc)     DBTYPE=3 ;;
  oracle)   DBTYPE=4 ;;
  mssql)    DBTYPE=5 ;;
  *)        DBTYPE=6 ;;
esac

# Assert host and port to sensible values
if [ "$DBHOST" = "" ] ; then
  DBHOST=direct
  DBPORT=
fi

# Assert password sensible value
if [ "$DBPASSWD" = "" ] ; then
  DBPASSWD=none
fi

echo ""
echo "Database name is $DBNAME"
if [ "$DBPORT" != "" ] ; then
  echo " - on remote server $DBHOST : $DBPORT"
fi
echo -n "Drop database '${DBNAME}' Y or N [N] :"
read ANS
if [ "$ANS" = "Y" -o "$ANS" = "y" ] ; then
  ${AXYL_HOME}/install/remove-db.sh "$DBTYPE" "$DBNAME" "$DBUSER" "$DBPASSWD" "$DBHOST" "$DBPORT"
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# LUCENE and LUCENE DOMAIN PROPERTIES FILE
if [ -f $AXYL_DATA/lucene/servers/axyl/etc/application/${INST_NAME}.properties ] ; then
  echo ""
  echo -n "Remove Lucene domain properties file? Y or N [N] :"
  read ANS
  if [ "$ANS" = "Y" -o "$ANS" = "y" ] ; then
    [ -x /etc/init.d/axyl-lucene ] && /etc/init.d/axyl-lucene stop
    rm -f $AXYL_DATA/lucene/servers/axyl/etc/application/${INST_NAME}.properties
    [ -x /etc/init.d/axyl-lucene ] && /etc/init.d/axyl-lucene start
    echo "done."
  fi
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# REMOVE APACHE VIRTUALHOST

# Find out what the servername is..
VSERVERNAME=`grep "^${INST_NAME}," ${AXYL_DATA}/system/axyl_websites.csv | cut -d',' -f10`
if [ "$VSERVERNAME" != "" ] ; then
  ${AXYL_HOME}/install/remove-apache-vh.sh $VSERVERNAME
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# The Website docroot itself..
echo ""
echo -n "Deleteing the whole ${INST_NAME} hirearchy..."
rm -rf ${WWW_PATH}
echo "done."

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Update Axyl websites database
echo ""
echo "updating Axyl website database.."
${AXYL_HOME}/install/unregister-axyl-website.sh $INST_NAME

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# FINISHED
echo ""
echo "Axyl website $INST_NAME has been removed."

# END