# This file is part of the Falcon repository manager
# Copyright (C) 2005-2008 Dennis Kaarsemaker
# See the file named COPYING in the root of the source tree for license details
#
# debchecker.py - Plugin that can check all packages for policy compliance 
#                 before building and installing

import falcon
from gettext import gettext as _
from email.MIMEText import MIMEText
import smtplib, os

class DebcheckerPlugin(falcon.plugin.FalconPlugin):
    """Package checking plugin"""
    name = _("Package checker")
    desc = _("Run packages through linda/lintian before installing them into the archive")
    api  = (1,1)

    def __init__(self):
        self.conf.register('linda', True, falcon.questions.Boolean(_("Run packages through linda")))
        self.conf.register('linda_args', '', falcon.questions.Entry(_("Arguments for linda"), [falcon.validators.shellsafe]))
        self.conf.register('lintian', True, falcon.questions.Boolean(_("Run packages through lintian")))
        self.conf.register('lintian_args', '', falcon.questions.Entry(_("Arguments for lintian"), [falcon.validators.shellsafe]))
        self.conf.register('refuse', False, falcon.questions.Boolean(_("Refuse to install packages that fail tests")))
        self.conf.register('email', True, falcon.questions.Boolean(_("Send linda/lintian output to maintainer")))
        # Copied from the mail_maintainer plugin
        self.conf.register('frommail', '', falcon.questions.Entry(_("Sender address"), [falcon.validators.mailaddress]))
        self.conf.register('extramail', '', falcon.questions.Entry(_("Extra mailaddresses to send to"),
                           [falcon.validators.blank_ok(falcon.validators.commaseparated(falcon.validators.mailaddress))]))
        self.conf.register('whitelist', '', falcon.questions.Entry(_("File with whitelist of recipients")))
        self.conf.register('server', 'localhost', falcon.questions.Entry(_("Hostname of the smtp server to use")))

    def pre_install(self, component, package):
        if not self.check_files(package.packagename, package.version, package.controlfields,
                [os.path.join(component.poolpath,f) for f in [package.filename] + [b.filename for b in package.binaries.all()]]):
            if self.conf.refuse:
                falcon.util.warning(_("There are errors in the %s package. Not installing") % package.packagename)
                package.delete()
                raise falcon.plugin.DontRun()

    def pre_run(self, queueitem, firstrun):
        if not firstrun:
            return
        control, controlfields, files = falcon.package.parse_dsc(os.path.join('.falcon', 'build','building',queueitem.dscfile))
        if not self.check_files(queueitem.packagename, queueitem.version, controlfields,
                [os.path.join('.falcon', 'build','building',x.name) for x in queueitem.files if x.name.endswith('.dsc')]):
            if self.conf.refuse:
                falcon.util.warning(_("There are errors in the %s package. Not installing") % package.packagename)
                package.delete()
                raise falcon.plugin.DontRun()

    def check_files(self, packagename, version, controlfields, files):
        flaws_detected = False
        errors = ''
        for f in files:
            falcon.util.output("Checking %s" % os.path.basename(f))
            #f = os.path.join(component.poolpath, f)
            if self.conf.linda:
                try:
                    err = falcon.util.run(['linda'] + self.conf.linda_args.split() + [f])
                    err = err.strip()
                    if err:
                        flaws_detected = True
                        errors += err + '\n'
                except RuntimeError, e:
                    flaws_detected = True
                    errors += e.stdout
            if self.conf.lintian:
                try:
                    err = falcon.util.run(['lintian'] + self.conf.lintian_args.split() + [f])
                    err = err.strip()
                    if err:
                        flaws_detected = True
                        errors += err + '\n'
                except RuntimeError, e:
                    flaws_detected = True
                    errors += e.stdout

        if not flaws_detected:
            return True

        # Largely copied from mail_maintainer.py
        if self.conf.email and self.conf.whitelist:
            whitelist = falcon.util.readfile(self.conf.whitelist)
            addresses = []
            maintainer = controlfields['Maintainer']
            if '<' in maintainer:
                maintainer = maintainer[maintainer.find('<')+1:maintainer.find('>')]
            if maintainer in whitelist:
                addresses.append(maintainer)
            for m in self.conf.extramail.split(','):
                addresses.append(m.strip())
            uploader = controlfields['Changed-By']
            if uploader:
                if '<' in uploader:
                    uploader = uploader[uploader.find('<')+1:uploader.find('>')]
                if uploader in whitelist:
                    addresses.append(uploader)
            for m in self.conf.extramail.split(','):
                addresses.append(m.strip())
            addresses = set(addresses)
            if addresses:
                msg = MIMEText(_("Package %s (%s) fails linda/lintian checks:\n%s") % (packagename, version, errors))
                msg['Subject'] = _("Package %s %s fails linda/lintian checks") % (packagename, version)
                msg['From'] = '"%s" <%s>' % (falcon.conf.description.replace('"',''), self.conf.frommail)
                msg['To'] = ', '.join(addresses)
                try:
                    s = smtplib.SMTP()
                    if falcon.conf.debug:
                        s.set_debuglevel(1)
                    s.connect(self.conf.server)
                    s.sendmail(self.conf.frommail, addresses, msg.as_string())
                    s.quit()
                except:
                    falcon.util.warning(_("Sending e-mail failed"))
                    falcon.util.debug_exception()
        elif self.conf.email:
            falcon.util.warning(_("No whitelist found for mailaddresses, not sending mail"))
        return False

