
/*
 * Copyright (C) 2006 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: vdr.c 1302 2006-10-11 15:33:54Z mschwerin $
 *
 */
#include "config.h"

#include "codeset.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"
#include "vdr.h"

#include <errno.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>

#ifdef HAVE_VDR

extern oxine_t *oxine;

#define BUFSIZE     2048

static int read_cnt;
static char *read_ptr;
static char read_buf[BUFSIZE];

static size_t
_read (int fd, char *ptr)
{
    if (read_cnt <= 0) {
      again:
        if ((read_cnt = recv (fd, read_buf, sizeof (read_buf), 0)) < 0) {
            if (errno == EINTR)
                goto again;
            return (-1);
        } else if (read_cnt == 0)
            return (0);
        read_ptr = read_buf;
    }

    read_cnt--;
    *ptr = *read_ptr++;
    return (1);
}


static size_t
readline (int fd, void *vptr, size_t maxlen)
{
    size_t n;
    size_t rc;
    char c;
    char *ptr;

    ptr = vptr;
    for (n = 1; n < maxlen; n++) {
        if ((rc = _read (fd, &c)) == 1) {
            *ptr++ = c;
            if (c == '\n')
                break;          /* newline is stored, like fgets() */
        } else if (rc == 0) {
            *ptr = 0;
            return (n - 1);     /* EOF, n - 1 bytes were read */
        } else
            return (-1);        /* error, errno set by read() */
    }

    *ptr = 0;                   /* null terminate like fgets() */
    return (n);
}


static bool
send_command (int sd, const char *command)
{
    debug ("->VDR: %s", command);

    if (send (sd, command, strlen (command), 0) == -1) {
        error (_("Could not send command to server: %s!"), strerror (errno));
        return false;
    }
    if (send (sd, "\n", 1, 0) == -1) {
        error (_("Could not send command to server: %s!"), strerror (errno));
        return false;
    }

    return true;
}


static bool
recv_answer (int sd, char *buf, int buf_size, char *expected_code)
{
    if (readline (sd, buf, buf_size) == -1) {
        error (_("Could not read from server: %s!"), strerror (errno));
        return false;
    }

    int len = strlen (buf) - 1;
    if ((buf[len] == '\n')
        || (buf[len] == '\r'))
        buf[len--] = '\0';

    if (strncmp (buf, expected_code, strlen (expected_code)) != 0) {
        error (_("Invalid SVDRP response: %s"), buf);
        return false;
    }

    debug ("<-VDR: %s", buf);

    return true;
}


static bool
simple_command (int sd, const char *command, char *expected_code)
{
    if (!send_command (sd, command)) {
        close (sd);
        return false;
    }

    char buf[BUFSIZE];
    if (!recv_answer (sd, buf, BUFSIZE, expected_code)) {
        close (sd);
        return false;
    }

    return true;
}


static int
svdrp_connect (const char *host, int port)
{
    char buf[BUFSIZE];
    int sd;
    struct hostent *hp;
    struct sockaddr_in pin;

    /* go find out about the desired host machine */
    if ((hp = gethostbyname (host)) == 0) {
        error (_("Could not resolve %s: %s!"), host, strerror (errno));
        return 0;
    }

    /* fill in the socket structure with host information */
    memset (&pin, 0, sizeof (pin));
    pin.sin_family = AF_INET;
    pin.sin_addr.s_addr = ((struct in_addr *) (hp->h_addr))->s_addr;
    pin.sin_port = htons (port);

    /* grab an Internet domain socket */
    if ((sd = socket (AF_INET, SOCK_STREAM, 0)) == -1) {
        error (_("Could not create socket: %s!"), strerror (errno));
        return 0;
    }

    /* connect to PORT on HOST */
    if (connect (sd, (struct sockaddr *) &pin, sizeof (pin)) == -1) {
        error (_("Could not connect to %s: %s!"), host, strerror (errno));
        close (sd);
        return 0;
    }

    /* wait server to say hello */
    if (!recv_answer (sd, buf, BUFSIZE, "220")) {
        close (sd);
        return 0;
    }

    return sd;
}


static int
vdr_swap_cb (void *p1, void *p2)
{
    fileitem_t *f1 = (fileitem_t *) p1;
    fileitem_t *f2 = (fileitem_t *) p2;

    struct tm t1;
    struct tm t2;

    t1.tm_year = atoi (f1->title + 6) + 100;
    t1.tm_mon = atoi (f1->title + 3) - 1;
    t1.tm_mday = atoi (f1->title);
    t1.tm_hour = atoi (f1->title + 9);
    t1.tm_min = atoi (f1->title + 12);
    t1.tm_sec = 0;

    t2.tm_year = atoi (f2->title + 6) + 100;
    t2.tm_mon = atoi (f2->title + 3) - 1;
    t2.tm_mday = atoi (f2->title);
    t2.tm_hour = atoi (f2->title + 9);
    t2.tm_min = atoi (f2->title + 12);
    t1.tm_sec = 0;

    if (mktime (&t2) > mktime (&t1))
        return 1;

    return 0;
}


bool
vdr_recordings_read (filelist_t * filelist)
{
    xine_cfg_entry_t hentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.host", &hentry);
    xine_cfg_entry_t pentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.port", &pentry);
    int sd = svdrp_connect (hentry.str_value, pentry.num_value);
    if (!sd) {
        return false;
    }

    bool result = false;
    char *codeset = get_system_encoding ();
    recode_t *xr = recode_init ("ISO-8859-1", codeset);

    if (!send_command (sd, "LSTR"))
        goto error;

    char buf[BUFSIZE];
    do {
        if (!recv_answer (sd, buf, BUFSIZE, "250"))
            goto error;

        int num = atoi (buf + 4);
        char *title = recode (xr, index (buf + 4, ' ') + 1);
        char *mrl = ho_strdup_printf ("%s%d", FILELIST_VDR_RECORDINGS_MRL,
                                      num);
        filelist_add (filelist, title, mrl, FILE_TYPE_VDR_RECORDING);
        ho_free (mrl);
        ho_free (title);
    } while (strncmp (buf, "250 ", 4) != 0);

    filelist_sort (filelist, vdr_swap_cb);

    // close connection
    if (!simple_command (sd, "QUIT", "221"))
        goto error;

    result = true;
  error:
    close (sd);
    recode_done (xr);
    ho_free (codeset);

    return result;
}


bool
vdr_play_recording (const char *mrl)
{
    if (strncmp (mrl, FILELIST_VDR_RECORDINGS_MRL,
                 strlen (FILELIST_VDR_RECORDINGS_MRL)) != 0) {
        error (_("This is not a valid VDR recordings MRL!"));
        return false;
    }

    int num = atoi (mrl + strlen (FILELIST_VDR_RECORDINGS_MRL));
    debug ("starting VDR recording %d", num);

    xine_cfg_entry_t hentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.host", &hentry);
    xine_cfg_entry_t pentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.port", &pentry);
    int sd = svdrp_connect (hentry.str_value, pentry.num_value);
    if (!sd) {
        return false;
    }

    bool result = false;
    char command[32];
    sprintf (command, "PLAY %d", num);
    if (!simple_command (sd, command, "250"))
        goto error;

    // pause the recording
    if (!simple_command (sd, "HITK Pause", "250"))
        goto error;

    // make sure recordings is showing
    if (!simple_command (sd, "HITK Recordings", "250"))
        goto error;

    // hide the menu
    if (!simple_command (sd, "HITK Menu", "250"))
        goto error;

    // show the position
    if (!simple_command (sd, "HITK Ok", "250"))
        goto error;

    // close connection
    if (!simple_command (sd, "QUIT", "221"))
        goto error;

    result = true;
  error:
    close (sd);
    return result;
}


bool
vdr_remove_recording (const char *mrl)
{
    if (strncmp (mrl, FILELIST_VDR_RECORDINGS_MRL,
                 strlen (FILELIST_VDR_RECORDINGS_MRL)) != 0) {
        error (_("This is not a valid VDR recordings MRL!"));
        return false;
    }

    int num = atoi (mrl + strlen (FILELIST_VDR_RECORDINGS_MRL));
    debug ("removing VDR recording %d", num);

    xine_cfg_entry_t hentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.host", &hentry);
    xine_cfg_entry_t pentry;
    xine_config_lookup_entry (oxine->xine, "vdr.svdrp.port", &pentry);
    int sd = svdrp_connect (hentry.str_value, pentry.num_value);
    if (!sd) {
        return false;
    }

    bool result = false;
    char command[32];
    sprintf (command, "DELR %d", num);
    if (!simple_command (sd, command, "250"))
        goto error;

    // close connection
    if (!simple_command (sd, "QUIT", "221"))
        goto error;

    result = true;
  error:
    close (sd);
    return result;
}


void
vdr_config_register_all (void)
{
    xine_config_register_num (oxine->xine, "vdr.svdrp.port", 2001,
                              _("port for SVDRP connection"),
                              _("port for SVDRP connection"), 10, NULL, NULL);

    xine_config_register_string (oxine->xine, "vdr.svdrp.host", "localhost",
                                 _("hostname for SVDRP connection"),
                                 _("hostname for SVDRP connection"), 10, NULL,
                                 NULL);
}

#endif
