# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.component import Component
from elisa.core.input_event import *

from elisa.plugins.pigment.pigment_controller import PigmentController
from elisa.plugins.pigment.widgets import notifying_list
from elisa.plugins.poblesec.widgets.nothing_to_display import NothingToDisplay

from twisted.internet import defer


class ListController(PigmentController):

    def initialize(self):
        # FIXME: Tying this "nothing to display" widget to the list controller
        #        is a sacrilege, this should be refactored in a clean way.
        self.nothing_to_display_widget = NothingToDisplay()
        self.widget.add(self.nothing_to_display_widget)
        self.nothing_to_display_widget.visible = False
        dfr = super(ListController, self).initialize()
        return dfr

    def __init__(self):
        super(ListController, self).__init__()
        self.model = notifying_list.List()
        self.nodes = None

        self._previous_clicked = None

    def set_frontend(self, frontend):
        super(ListController, self).set_frontend(frontend)

        self.nodes_setup()

        self.nodes.connect('item-clicked', self.node_clicked)
        self.nodes.connect('selected-item-changed', self.node_selected)
        self.nodes.set_model(self.model)
        self.nodes.set_renderer(self.node_renderer)

        # by default the focus is forwarded to the list widget
        self.nodes.focus = self.widget.focus
        self.widget.connect('focus', self._on_focus)

    def stop_loading_animation(self):
        # FIXME: this uses some knowledge of the widgets that are used in the
        # controller: shouldn't be done like this. But now it allows to avoid
        # copy&paste code.
        if self._previous_clicked and hasattr(self._previous_clicked, '_toggle_loading_animation'):
            self._previous_clicked._toggle_loading_animation(False)
            self._previous_clicked = None

    def removed(self):
        self.stop_loading_animation()

    def _on_focus(self, widget, focus):
        # always forward the focus to the list widget
        if focus:
            self.nodes.focus = focus

    def nodes_setup(self):
        pass

    def node_renderer(self, item, widget):
        pass

    def node_clicked(self, widget, item):
        pass

    def node_selected(self, widget, item, previous_item):
        pass

    def handle_input(self, manager, input_event):
        if input_event.value == EventValue.KEY_OK:
            try:
                item = self.nodes.model[self.nodes.selected_item_index]
                self.node_clicked(self.nodes, item)
            except IndexError:
                self.debug("selected-item-index out of range: %s" % self.nodes.selected_item_index)
            return True
        else:
            return super(ListController, self).handle_input(manager, input_event)


class GenericListViewMode(Component):

    """
    Generic view mode API.

    It defines a common API for clients. All one has to do is inherit from this
    class and implement the following methods:

      * get_label(item)
      * get_sublabel(item)
      * get_default_image(item)
      * get_image(item, theme)
      * get_preview_image(item, theme)
    """

    def get_label(self, item):
        """
        Return a text to display in a label to represent an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the text of the label.

        @param item: a list item
        @type item:  a subclass of L{elisa.core.components.model.Model}

        @return:     a cancellable deferred
        @rtype:      L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_sublabel(self, item):
        """
        Return a text to display in a sublabel to represent an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the text of the sublabel.

        @param item: a list item
        @type item:  a subclass of L{elisa.core.components.model.Model}

        @return:     a cancellable deferred
        @rtype:      L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_default_image(self, item):
        """
        Return the path of a theme resource to display as a default image for
        an item.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}

        @return:      the path of a theme resource to display as a default
                      image for the item
        @rtype:       C{str}
        """
        raise NotImplementedError()

    def get_image(self, item, theme):
        """
        Return the path to an image file to display as an image for an item.

        This call is asynchronous, it should return a
        L{elisa.core.utils.cancellable_defer.CancellableDeferred} that, when
        triggered, returns the path to an image file on disk (downloaded and
        cached if necessary).

        If no other image than the default one is necessary/available, this
        method should return C{None}.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}
        @param theme: the frontend's current theme
        @type theme:  L{elisa.plugins.pigment.widgets.theme.Theme}

        @return:      a cancellable deferred or C{None}
        @rtype:       L{elisa.core.utils.cancellable_defer.CancellableDeferred}
        """
        return defer.fail(NotImplementedError())

    def get_preview_image(self, item, theme):
        """
        Return the path to an image file to display as a preview image for an
        item.

        This call is synchronous, if no preview image is available yet for the
        item or if no other image than the default one is necessary, it should
        return C{None}.

        @param item:  a list item
        @type item:   a subclass of L{elisa.core.components.model.Model}
        @param theme: the frontend's current theme
        @type theme:  L{elisa.plugins.pigment.widgets.theme.Theme}

        @return:      the path to an image file on disk or C{None}
        @rtype:       C{str} or C{None}
        """
        raise NotImplementedError()
