# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.

from elisa.core.utils import defer
from elisa.core.log import Loggable
from elisa.core import common

from twisted.python import failure
import gobject

class History(gobject.GObject, Loggable):

    # FIXME: signal name should contain a dash, not an underscore
    __gsignals__ = {'push_controller': 
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_PYOBJECT,
                               gobject.TYPE_PYOBJECT,)),
                    'pop_controller': 
                              (gobject.SIGNAL_RUN_LAST,
                              gobject.TYPE_BOOLEAN,
                              (gobject.TYPE_PYOBJECT,
                               gobject.TYPE_PYOBJECT,))
                              }

    def __init__(self, frontend):
        super(History, self).__init__()

        self.frontend = frontend
        self.current = None
        self.index = -1
        self._controllers = []
        self._creation_dfr = None

    def append_controller(self, path, display_name, **kwargs):
        def _append_controller(controller):
            self._creation_dfr = None
            controller.display_name = display_name
            self._controllers.append(controller)
            previous = self.current
            self.current = controller
            self.index = self._controllers.index(self.current)
            
            return controller
        
        def _emit_push_controller(controller):
            index = self._controllers.index(self.current)
            if index > 0:
                previous = self._controllers[index-1]
            else:
                previous = None
            self.emit('push_controller', previous, self.current)
            
            return controller

        def _log_controller_creation_error(error, controller_path):
            self._creation_dfr = None
            log_path = common.application.log_failure(error)
            self.warning("Entering controller with path %s failed. Full" \
                         " failure log at %s" % (controller_path, log_path))
            return error

        # cancel any creation of controller if there was one
        if self._creation_dfr != None and not self._creation_dfr.called:
            self._creation_dfr.cancel()

        self._creation_dfr = defer.Deferred()
        self._creation_dfr.addCallback(_append_controller)
        self._creation_dfr.addCallback(_emit_push_controller)
        self._creation_dfr.addErrback(_log_controller_creation_error, path)

        try:
            dfr = self.frontend.create_controller(path, **kwargs)
        except:
            error = failure.Failure()
            dfr = defer.fail(error)

        dfr.chainDeferred(self._creation_dfr)

        return self._creation_dfr

    def go_back(self):
        # cancel any creation of controller if there was one
        if self._creation_dfr != None and not self._creation_dfr.called:
            self._creation_dfr.cancel()

        if not self.current:
            self.warning("No current controller in this history.")
            return

        # remove the current controller from the history
        try:
            index = self._controllers.index(self.current)
        except ValueError:
            self.warning("Current controller is unknown to this history.")
            return
        self._controllers.pop(index)

        # update the current index
        self.index = index-1

        if index > 0:
            poped_controller = self.current
            previous = self.current
            self.current = self._controllers[index-1]
            self.emit('pop_controller', previous, self.current)

            # call clean method of the controller before losing last reference
            # to it
            previous.clean()
