/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: PaneController.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: rt $ $Date: 2007/04/03 15:58:04 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include "precompiled_sd.hxx"

#include "framework/PaneController.hxx"

#include "framework/FrameworkHelper.hxx"
#include "framework/FactoryContainer.hxx"

#include "DrawController.hxx"
#include "ViewShellBase.hxx"

#ifndef _COM_SUN_STAR_DRAWING_FRAMEWORK_PANECONTROLLER_HPP_
#include <com/sun/star/drawing/framework/PaneController.hpp>
#endif
#ifndef _COM_SUN_STAR_DRAWING_FRAMEWORK_XCONTROLLERMANAGER_HPP_
#include <com/sun/star/drawing/framework/XControllerManager.hpp>
#endif
#ifndef _COM_SUN_STAR_UNO_XCOMPONENTCONTEXT_HPP_
#include <com/sun/star/uno/XComponentContext.hpp>
#endif
#ifndef _COM_SUN_STAR_FRAMME_XCONTROLLER_HPP_
#include <com/sun/star/frame/XController.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XMULTISERVICEFACTORY_HPP_
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XUNOTUNNEL_HPP_
#include <com/sun/star/lang/XUnoTunnel.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_ILLEGALARGUMENTEXCEPTION_HPP_
#include <com/sun/star/lang/IllegalArgumentException.hpp>
#endif

#include <comphelper/stl_types.hxx>
#include <vcl/svapp.hxx>
#include <vos/mutex.hxx>
#include <set>
#include <boost/bind.hpp>

using namespace ::com::sun::star;
using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::drawing::framework;
using ::rtl::OUString;

#undef VERBOSE

namespace sd { namespace framework {

Reference<XInterface> SAL_CALL PaneController_createInstance (
    const Reference<XComponentContext>& rxContext)
{
    return Reference<XInterface>((XWeak*)new PaneController(rxContext), UNO_QUERY);
}




::rtl::OUString PaneController_getImplementationName (void) throw(RuntimeException)
{
    return ::rtl::OUString(
        RTL_CONSTASCII_USTRINGPARAM("com.sun.star.comp.Draw.framework.pane.PaneController"));
}




Sequence<rtl::OUString> SAL_CALL PaneController_getSupportedServiceNames (void)
    throw (RuntimeException)
{
	static const ::rtl::OUString sServiceName(
        ::rtl::OUString::createFromAscii("com.sun.star.drawing.framework.PaneController"));
	return Sequence<rtl::OUString>(&sServiceName, 1);
}




namespace {
class PaneDescriptor
{
public:
    Reference<XResourceId> mxPaneId;
    Reference<XPane> mxPane;
    Reference<awt::XWindow> mxWindow;
    Reference<XPaneFactory> mxFactory;
    static bool CompareResourceId (
        const PaneDescriptor& rD, const Reference<XResourceId>& rxResourceId)
    { return rD.mxPaneId->compareTo(rxResourceId)==0; }
    static bool ComparePane (const PaneDescriptor& rD, const Reference<XPane>& rxPane)
    { return rD.mxPane == rxPane; }
    static bool CompareWindow (const PaneDescriptor& rD, const Reference<awt::XWindow>& rxWindow)
    { return rD.mxWindow == rxWindow; }
};

}

// PaneContainer

class PaneController::PaneContainer
    : public ::std::vector<PaneDescriptor>
{
public:
    PaneContainer (void) {}
};


typedef ::std::set<Reference<XResourceId> > ResourceIdSet;


//===== PaneController ========================================================

PaneController::PaneController (
    const Reference<XComponentContext>&rxContext)
    throw()
    : PaneControllerInterfaceBase(MutexOwner::maMutex),
      mxConfigurationController(),
      mpFactoryContainer(new FactoryContainer()),
      mpPaneContainer(new PaneContainer())
{
    (void)rxContext;
}




PaneController::~PaneController (void)
    throw()
{
}




void SAL_CALL PaneController::disposing (void)
{
    ::osl::MutexGuard aGuard (maMutex);

    while ( ! mpPaneContainer->empty())
        ReleasePane(mpPaneContainer->begin()->mxPane);

    if (mxConfigurationController.is())
    {
        Reference<XComponent> xComponent (mxConfigurationController, UNO_QUERY);
        if (xComponent.is())
            xComponent->removeEventListener(this);
    }

    mpFactoryContainer.reset();
    mpPaneContainer.reset();
}




/** This method tries to guard its access to its members with a mutex while
    notifying listeners with the mutex unlocke.
*/
bool PaneController::CreatePane (const Reference<XResourceId>& rxPaneId)
{
    bool bSuccess (false);
    
    bool bNotify (false);
    {
        ::osl::MutexGuard aGuard (maMutex);

        Reference<XPaneFactory> xFactory (
            mpFactoryContainer->GetFactory(rxPaneId->getResourceURL()),
            UNO_QUERY);
        if (xFactory.is())
        {
#ifdef VERBOSE
            OSL_TRACE("creating pane %s",
                OUStringToOString(
                    FrameworkHelper::ResourceIdToString(rxPaneId),
                    RTL_TEXTENCODING_UTF8).getStr());
#endif
            Reference<XPane> xPane;
            try
            {
                xPane = xFactory->createPane(rxPaneId);
            }
            catch (lang::DisposedException)
            {
                // The factory is disposed and can be removed from the list
                // of registered factories.
                xPane = NULL;
                mpFactoryContainer->RemoveFactoryForReference(xFactory);
            }
            
            if (xPane.is())
            {
                // Store the pane together with the factory that was used to
                // create it.
                PaneDescriptor aDescriptor;
                aDescriptor.mxPaneId = rxPaneId;
                aDescriptor.mxPane = xPane;
                aDescriptor.mxFactory = xFactory;
                aDescriptor.mxWindow = xPane->getWindow();
                mpPaneContainer->push_back(aDescriptor);

                // Add this PaneController as window listener.  This is done so
                // that a pane can be released when its window is hidden or
                // destroyed without the use of the drawing framework.
                if (aDescriptor.mxWindow.is())
                    aDescriptor.mxWindow->addWindowListener(this);

                bSuccess = true;
            }
            else
            {
#ifdef VERBOSE
                OSL_TRACE("    creating pane failed");
#endif
            }
        }
    }

    if (bNotify)
    {
        // Notify the new pane to listeners of the ConfigurationController.
        ConfigurationChangeEvent aEvent;
        aEvent.Type = FrameworkHelper::msResourceActivationEvent;
        aEvent.ResourceId = rxPaneId;
        if (mxConfigurationController.is())
        {
            try
            {
                mxConfigurationController->notifyEvent(aEvent);
            }
            catch (lang::DisposedException)
            {
                mxConfigurationController = NULL;
            }
        }
    }
    
    return bSuccess;
}




/** This method tries to guard its access to its members with a mutex while
    notifying listeners with the mutex unlocke.
*/
bool PaneController::ReleasePane (const Reference<XPane>& rxPane)
{
    bool bSuccess (false);
    
    if (rxPane.is())
    {
        PaneContainer::iterator iPane;
        Reference<XPaneFactory> xFactory;
        {
            ::osl::MutexGuard aGuard (maMutex);
            iPane = ::std::find_if(
                mpPaneContainer->begin(),
                mpPaneContainer->end(),
                ::boost::bind(&PaneDescriptor::ComparePane, _1, rxPane));
            if (iPane != mpPaneContainer->end())
                xFactory = iPane->mxFactory;
        }
        if (xFactory.is())
        {
            // Notifiy the pane being deactivated to listeners of the ConfigurationController.
            ConfigurationChangeEvent aEvent;
            aEvent.Type = FrameworkHelper::msResourceDeactivationEvent;
            aEvent.ResourceId = iPane->mxPaneId;
            if (mxConfigurationController.is())
            {
                try
                {
                    mxConfigurationController->notifyEvent(aEvent);
                }
                catch (lang::DisposedException)
                {
                    mxConfigurationController = NULL;
                }
            }

            // We are no longer interested in events from the pane window.
            Reference<awt::XWindow> xWindow (rxPane->getWindow());
            if (xWindow.is())
                xWindow->removeWindowListener(this);

            // Release the pane with the factory that has created it.
            try
            {
                xFactory->releasePane(rxPane);
            }
            catch (lang::DisposedException)
            {
                // The factory is disposed and can be removed from the list
                // of registered factories.
                mpFactoryContainer->RemoveFactoryForReference(xFactory);
            }

            ::osl::MutexGuard aGuard (maMutex);
            mpPaneContainer->erase(iPane);

            bSuccess = true;
        }
    }

    return bSuccess;
}




//----- XPaneController -------------------------------------------------------

void SAL_CALL PaneController::addPaneFactory(
    const ::rtl::OUString& rsPaneURL,
    const Reference<drawing::framework::XPaneFactory >& rxPaneFactory)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->AddFactory(rsPaneURL, rxPaneFactory);
}
    



void SAL_CALL PaneController::removePaneFactoryForURL (
    const ::rtl::OUString& rsPaneURL)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForURL(rsPaneURL);
}




void SAL_CALL PaneController::removePaneFactoryForReference(
    const Reference<XPaneFactory>& rxPaneFactory)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    mpFactoryContainer->RemoveFactoryForReference(rxPaneFactory);
}




void SAL_CALL PaneController::updateStart (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToDeactivate)
    throw (RuntimeException)
{
    ThrowIfDisposed();

    if (rxRequestedConfiguration.is())
    {
        // Release the panes that are no longer requested.
        sal_Int32 nCount = rResourcesToDeactivate.getLength();
        for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
        {
            const Reference<XResourceId> xResourceId (rResourcesToDeactivate[nIndex]);
            if (xResourceId->getResourceURL().match(FrameworkHelper::msPaneURLPrefix))
                if (ReleasePane(getPane(xResourceId)))
                    rxCurrentConfiguration->removeResource(xResourceId);
        }

        // Check the remaining panes, that they are still used by at least
        // one bound resource.  Note that for indentifying the unused panes
        // the requested configuration is used because the current
        // configuration is in a transitional state.
        if (rxRequestedConfiguration.is())
        {
            Sequence<Reference<XResourceId> > aPanes (
                rxRequestedConfiguration->getResources(
                    NULL,
                    FrameworkHelper::msPaneURLPrefix,
                    AnchorBindingMode_DIRECT));
            for (sal_Int32 nIndex=0; nIndex<aPanes.getLength(); ++nIndex)
            {
                const Reference<XResourceId> xPaneId (aPanes[nIndex]);
                if ( ! IsPaneUsed(xPaneId, rxRequestedConfiguration))
                {
                    // The pane is not used.  Request its deactivation.
                    // This request will be processed (not before) the next
                    // update.
                    if (mxConfigurationController.is())
                        mxConfigurationController->requestResourceDeactivation(xPaneId);
                }
            }
        }
    }
}




void SAL_CALL PaneController::updateEnd (
    const Reference<XConfiguration>& rxRequestedConfiguration,
    const Reference<XConfiguration>& rxCurrentConfiguration,
    const Sequence<Reference<XResourceId> >& rResourcesToActivate)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    ::osl::MutexGuard aGuard (maMutex);

    if (rxRequestedConfiguration.is())
    {
        sal_Int32 nCount = rResourcesToActivate.getLength();
        for (sal_Int32 nIndex=0; nIndex<nCount; ++nIndex)
        {
            const Reference<XResourceId> xResourceId (rResourcesToActivate[nIndex]);
            if (xResourceId->getResourceURL().match(FrameworkHelper::msPaneURLPrefix))
            {
                // Create the requested pane only when it is used by another
                // resource.
                if (IsPaneUsed(xResourceId, rxRequestedConfiguration))
                    if (CreatePane(xResourceId))
                        rxCurrentConfiguration->addResource(xResourceId);
            }
        }
    }
}




Reference<XPane> SAL_CALL PaneController::getPane (const Reference<XResourceId>& rxPaneId)
    throw (RuntimeException)
{
    ThrowIfDisposed();
    ::osl::MutexGuard aGuard (maMutex);

    PaneContainer::const_iterator iPane (
        ::std::find_if(
            mpPaneContainer->begin(),
            mpPaneContainer->end(),
            ::boost::bind(&PaneDescriptor::CompareResourceId, _1, rxPaneId)));
    if (iPane != mpPaneContainer->end())
        return iPane->mxPane;
    else
        return Reference<XPane>();
}




//----- XInitialization -------------------------------------------------------

void SAL_CALL PaneController::initialize (const Sequence<Any>& aArguments)
    throw (Exception, RuntimeException)
{
    ThrowIfDisposed();

    if (aArguments.getLength() > 0)
    {
        try
        {
            // Get the XController from the first argument.
            Reference<frame::XController> xController (aArguments[0], UNO_QUERY_THROW);

            // Get the configuration controller.
            Reference<XControllerManager> xControllerManager(xController, UNO_QUERY_THROW);
            mxConfigurationController = xControllerManager->getConfigurationController();
            if ( ! mxConfigurationController.is())
            {
                throw RuntimeException();
            }
            else
            {
                Reference<XComponent> xComponent (mxConfigurationController, UNO_QUERY);
                if (xComponent.is())
                    xComponent->addEventListener(this);
            }

            mpFactoryContainer->SetModuleController(xControllerManager->getModuleController());
        }
        catch (RuntimeException&)
        {
            DBG_ASSERT(false, "PaneController::initialize(): caught exception");
        }
    }
}




//----- XWindowListener -------------------------------------------------------

void SAL_CALL PaneController::windowResized (const awt::WindowEvent& rEvent)
    throw (RuntimeException)
{
    (void)rEvent;
}




void SAL_CALL PaneController::windowMoved (const awt::WindowEvent& rEvent)
    throw (RuntimeException)
{
    (void)rEvent;
}
    



void SAL_CALL PaneController::windowShown (const lang::EventObject& rEvent)
    throw (RuntimeException)
{
    (void)rEvent;

    ThrowIfDisposed();

    // A window has been shown.  This may cause the requested configuration
    // to differ from the, now changed, current configuration, or a
    // previously requested but not available pane has become available. In
    // any case the ConfigurationController is asked to update the current
    // configuration.
    if (mxConfigurationController.is())
        mxConfigurationController->update();
}
    



void SAL_CALL PaneController::windowHidden (const lang::EventObject& rEvent)
    throw (RuntimeException)
{
    (void)rEvent;
    
    ThrowIfDisposed();

    // A window has been hidden.  This may have caused the requested
    // configuration to differ from the, now changed, current
    // configuration.  Call the ConfigurationController to update the
    // current configuration.
    if (mxConfigurationController.is())
        mxConfigurationController->update();
}




//----- XEventListener --------------------------------------------------------

void SAL_CALL PaneController::disposing (const lang::EventObject& rEvent)
    throw (RuntimeException)
{
    ThrowIfDisposed();

    if (rEvent.Source == mxConfigurationController)
    {
        mxConfigurationController = NULL;
    }
    else
    {
        Reference<XPane> xPane (GetPaneForPaneOrWindow(rEvent.Source));
        if (xPane.is())
        {
            // The window of a pane has been disposed.  Release the pane.
            ReleasePane(xPane);
        }
    }
}




//-----------------------------------------------------------------------------

void PaneController::TraceController (void) const
{
    OSL_TRACE("PaneController {");
    PaneContainer::const_iterator iPane;
    for (iPane=mpPaneContainer->begin(); iPane!=mpPaneContainer->end(); ++iPane)
    {
        OSL_TRACE("    %s, %p created by factory %p",
            ::rtl::OUStringToOString(
                FrameworkHelper::ResourceIdToString(iPane->mxPaneId),
                RTL_TEXTENCODING_UTF8).getStr(),
            iPane->mxPane.get(),
            iPane->mxFactory.get());
    }
    OSL_TRACE("}");
}




Reference<XPane> PaneController::GetPaneForPaneOrWindow (
    const Reference<XInterface>& rxPaneOrWindow)
{
    PaneContainer::iterator iPane (mpPaneContainer->end());
    Reference<XPane> xPane (rxPaneOrWindow, UNO_QUERY);
    if (xPane.is())
        iPane = ::std::find_if(
            mpPaneContainer->begin(),
            mpPaneContainer->end(),
            ::boost::bind(&PaneDescriptor::ComparePane, _1, xPane));
    else
    {
        Reference<awt::XWindow> xWindow (rxPaneOrWindow, UNO_QUERY);
        if (xWindow.is())
            iPane = ::std::find_if(
                mpPaneContainer->begin(),
                mpPaneContainer->end(),
                ::boost::bind(&PaneDescriptor::CompareWindow, _1, xWindow));
        
    }

    if (iPane != mpPaneContainer->end())
        return iPane->mxPane;
    else
        return Reference<XPane>();
}




bool PaneController::IsPaneUsed (
    const Reference<XResourceId>& rxPaneId,
    const Reference<XConfiguration>& rxConfiguration)
{
    bool bPaneIsUsed (false);
    
    if (rxConfiguration.is())
    {
        Sequence<Reference<XResourceId> > aResources (
            rxConfiguration->getResources(rxPaneId, ::rtl::OUString(), AnchorBindingMode_DIRECT));
        if (aResources.getLength() > 0)
            bPaneIsUsed = true;
    }

    return bPaneIsUsed;
}




void PaneController::ThrowIfDisposed (void) const
    throw (::com::sun::star::lang::DisposedException)
{
	if (rBHelper.bDisposed || rBHelper.bInDispose)
	{
        throw lang::DisposedException (
            OUString(RTL_CONSTASCII_USTRINGPARAM(
                "PaneController object has already been disposed")),
            const_cast<uno::XWeak*>(static_cast<const uno::XWeak*>(this)));
    }
}


} } // end of namespace sd::framework
