/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */
/* NetworkManager Connection editor -- Connection editor for NetworkManager
 *
 * Dan Williams <dcbw@redhat.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * (C) Copyright 2008 Red Hat, Inc.
 */

#include <string.h>
#include <unistd.h>

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include <nm-setting-connection.h>
#include <nm-setting-gsm.h>
#include <nm-setting-cdma.h>

#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <string.h>
#include <nm-utils.h>
#include "../applet.h"
#include "../marshallers/nma-marshal.h"
#include "../applet-dbus-manager.h"

#include "page-mobile.h"
#include "nm-connection-editor.h"
#include "gconf-helpers.h"


G_DEFINE_TYPE (CEPageMobile, ce_page_mobile, CE_TYPE_PAGE)

#define CE_PAGE_MOBILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), CE_TYPE_PAGE_MOBILE, CEPageMobilePrivate))


#define MM_DBUS_SERVICE              "org.freedesktop.ModemManager"
#define MM_DBUS_PATH                 "/org/freedesktop/ModemManager"
#define MM_DBUS_INTERFACE            "org.freedesktop.ModemManager"
#define MM_DBUS_INTERFACE_MODEM      "org.freedesktop.ModemManager.Modem"

#define MM_DBUS_INTERFACE_MODEM_GSM_CARD    "org.freedesktop.ModemManager.Modem.Gsm.Card"
#define MM_DBUS_INTERFACE_MODEM_GSM_NETWORK "org.freedesktop.ModemManager.Modem.Gsm.Network"

#define NM_DBUS_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), \
										   NM_TYPE_DBUS_MANAGER, \
										   NMDBusManagerPrivate))

#define MM_MODEM_ERROR "org.freedesktop.ModemManager.Modem.Gsm"
#define MM_MODEM_ERROR_SIM_PUK MM_MODEM_ERROR ".SimPukRequired"
#define MM_MODEM_ERROR_SIM_WRONG MM_MODEM_ERROR ".SimWrong"
#define MM_MODEM_ERROR_SIM_PIN MM_MODEM_ERROR ".SimPinRequired"


typedef struct {
	NMSetting *setting;

	/* Common to GSM and CDMA */
	GtkEntry *number;
	GtkEntry *username;
	GtkEntry *password;

	/* GSM only */
	GtkEntry *apn;
	GtkEntry *network_id;
	GtkComboBox *network_type;
	GtkComboBox *band;
	GtkEntry *pin;
	GtkEntry *puk;


	gboolean disposed;
} CEPageMobilePrivate;

#define NET_TYPE_ANY         0
#define NET_TYPE_3G          1
#define NET_TYPE_2G          2
#define NET_TYPE_PREFER_3G   3
#define NET_TYPE_PREFER_2G   4

static void
mobile_private_init (CEPageMobile *self)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);
	GladeXML *xml;

	xml = CE_PAGE (self)->xml;

	priv->number = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_number"));
	priv->username = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_username"));
	priv->password = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_password"));

	priv->apn = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_apn"));
	priv->network_id = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_network_id"));
	priv->network_type = GTK_COMBO_BOX (glade_xml_get_widget (xml, "mobile_network_type"));
	priv->band = GTK_COMBO_BOX (glade_xml_get_widget (xml, "mobile_band"));

	priv->pin = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_pin"));
	priv->puk = GTK_ENTRY (glade_xml_get_widget (xml, "mobile_puk"));

}

static GHashTable *
get_secrets (NMConnection *connection, const char *setting_name)
{
	GError *error = NULL;
	GHashTable *secrets;

	secrets = nm_gconf_get_keyring_items (connection,
	                                      setting_name,
	                                      FALSE,
	                                      &error);
	if (!secrets && error)
		g_error_free (error);

	return secrets;
}

static void
populate_gsm_ui (CEPageMobile *self, NMConnection *connection)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);
	NMSettingGsm *setting = NM_SETTING_GSM (priv->setting);
	int type_idx;
	GHashTable *secrets;
	GValue *value;

	if (setting->number)
		gtk_entry_set_text (priv->number, setting->number);

	if (setting->username)
		gtk_entry_set_text (priv->username, setting->username);

	if (setting->apn)
		gtk_entry_set_text (priv->apn, setting->apn);

	if (setting->network_id)
		gtk_entry_set_text (priv->network_id, setting->network_id);

	switch (setting->network_type) {
	case NM_GSM_NETWORK_UMTS_HSPA:
		type_idx = NET_TYPE_3G;
		break;
	case NM_GSM_NETWORK_GPRS_EDGE:
		type_idx = NET_TYPE_2G;
		break;
	case NM_GSM_NETWORK_PREFER_UMTS_HSPA:
		type_idx = NET_TYPE_PREFER_3G;
		break;
	case NM_GSM_NETWORK_PREFER_GPRS_EDGE:
		type_idx = NET_TYPE_PREFER_2G;
		break;
	case NM_GSM_NETWORK_ANY:
	default:
		type_idx = NET_TYPE_ANY;
		break;
	}
	gtk_combo_box_set_active (priv->network_type, type_idx);

	secrets = get_secrets (connection, nm_setting_get_name (priv->setting));

	if (setting->password)
		gtk_entry_set_text (priv->password, setting->password);
	else if (secrets) {
		value = g_hash_table_lookup (secrets, NM_SETTING_GSM_PASSWORD);
		if (value)
			gtk_entry_set_text (priv->password, g_value_get_string (value));
	}

	if (setting->pin)
		gtk_entry_set_text (priv->pin, setting->pin);
	else if (secrets) {
		value = g_hash_table_lookup (secrets, NM_SETTING_GSM_PIN);
		if (value)
			gtk_entry_set_text (priv->pin, g_value_get_string (value));
	}

	if (setting->puk)
		gtk_entry_set_text (priv->pin, setting->puk);
	else if (secrets) {
		value = g_hash_table_lookup (secrets, NM_SETTING_GSM_PUK);
		if (value)
			gtk_entry_set_text (priv->puk, g_value_get_string (value));
	}

	if (secrets)
		g_hash_table_destroy (secrets);
}

static void
populate_cdma_ui (CEPageMobile *self, NMConnection *connection)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);
	NMSettingCdma *setting = NM_SETTING_CDMA (priv->setting);
	GHashTable *secrets;
	GValue *value;

	if (setting->number)
		gtk_entry_set_text (priv->number, setting->number);

	if (setting->username)
		gtk_entry_set_text (priv->username, setting->username);

	secrets = get_secrets (connection, nm_setting_get_name (priv->setting));

	if (setting->password)
		gtk_entry_set_text (priv->password, setting->password);
	else if (secrets) {
		value = g_hash_table_lookup (secrets, NM_SETTING_CDMA_PASSWORD);
		if (value)
			gtk_entry_set_text (priv->password, g_value_get_string (value));
	}

	if (secrets)
		g_hash_table_destroy (secrets);

	/* Hide GSM specific widgets */
	gtk_widget_hide (glade_xml_get_widget (CE_PAGE (self)->xml, "mobile_basic_label"));
	gtk_widget_hide (glade_xml_get_widget (CE_PAGE (self)->xml, "mobile_advanced_vbox"));
}

static void
populate_ui (CEPageMobile *self, NMConnection *connection)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);

	if (NM_IS_SETTING_GSM (priv->setting))
		populate_gsm_ui (self, connection);
	else if (NM_IS_SETTING_CDMA (priv->setting))
		populate_cdma_ui (self, connection);
	else
		g_error ("Invalid setting");
}

static void
stuff_changed (GtkWidget *w, gpointer user_data)
{
	ce_page_changed (CE_PAGE (user_data));
}

static void
show_passwords (GtkToggleButton *button, gpointer user_data)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (user_data);
	gboolean active;

	active = gtk_toggle_button_get_active (button);

	gtk_entry_set_visibility (priv->password, active);
	gtk_entry_set_visibility (priv->pin, active);
	gtk_entry_set_visibility (priv->puk, active);
}

// --------------------------------------------------------------------------------------------------
typedef enum {
	NOTHING,
	CHANGE_PIN,
	ENABLE_PIN,
	DISABLE_PIN
} EPIN_FUNC;

typedef struct {
	GtkWidget *dialog;
	GladeXML *xml;
	DBusGConnection *bus;
	DBusGProxy *proxy;
	char *path;

	EPIN_FUNC pin_func;
	
	GtkEntry *pin;
	GtkEntry *new_pin;
	GtkEntry *puk;
	GtkEntry *status;

	char *pin_str;
	char *new_pin_str;
	char *puk_str;

	int bUsePuk;
	
} CEPagePinPrivate;

static void do_change_pin (CEPagePinPrivate* pin_priv);

static void
change_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer data)
{
	GError *error = NULL;
	GtkWidget *dialog;
	CEPagePinPrivate *pin_priv = (CEPagePinPrivate *) data;
	char response[100];
	
	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_INVALID)) {
		switch (pin_priv->pin_func) {
			case CHANGE_PIN: strcpy (response, _("Wrong PIN! Couldn't change pin.")); break;
			case ENABLE_PIN: strcpy (response, _("Wrong PIN! Couldn't enable pin.")); break;
			case DISABLE_PIN: strcpy (response, _("Wrong PIN! Couldn't disable pin.")); break;
			case NOTHING:
			default:
				strcpy(response, _("Unknown function"));
		}
		if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_PIN)) {
			g_debug("MM_MODEM_ERROR_SIM_PIN");
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_PUK)) {
			strcat (response, _(" PUK code is needed!"));
			g_debug("MM_MODEM_ERROR_SIM_PUK");
			if (!pin_priv->bUsePuk) {
				pin_priv->bUsePuk = 1;
				g_warning ("%s", error->message);
				g_error_free (error);
				if (strlen(pin_priv->puk_str) != 8) {
					strcpy (response, _("PUK code is needed. \nPlease, provide the correct 8 digit PUK code."));
				} else {
				do_change_pin (pin_priv);
				return;
			}
			}
		} else if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_WRONG)) {
			g_debug("MM_MODEM_ERROR_SIM_WRONG");
		}
		dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, response);
		gtk_window_set_title (GTK_WINDOW (dialog), "Error");
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_warning ("%s", error->message);
		g_error_free (error);

	} else {
		switch (pin_priv->pin_func) {
			case CHANGE_PIN: strcpy (response, _("Pin changed!")); break;
			case ENABLE_PIN: strcpy (response, _("SIM lock enabled! Active after turning of device")); break;
			case DISABLE_PIN: strcpy (response, _("SIM lock removed!")); break;
			case NOTHING:
			default:
				strcpy(response, _("Unknown function"));
		}
		dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, response);
		gtk_window_set_title (GTK_WINDOW (dialog), _("Information"));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);

	}
	g_free(pin_priv->path);
	g_free(pin_priv->pin_str);
	g_free(pin_priv->new_pin_str);
	g_free(pin_priv->puk_str);
	gtk_widget_destroy(pin_priv->dialog);
}

static void do_change_pin (CEPagePinPrivate* pin_priv)
{
	DBusGProxy *gsm_proxy;

	g_debug("do_change_pin");
	
	gsm_proxy = dbus_g_proxy_new_for_name (pin_priv->bus,
										   MM_DBUS_SERVICE, pin_priv->path, MM_DBUS_INTERFACE_MODEM_GSM_CARD);

	g_debug("after gsm_proxy");
	

	g_debug("pin = %s, new_pin = %s, puk = %s", pin_priv->pin_str, pin_priv->new_pin_str, pin_priv->puk_str);

	sleep(5);
	
	if (pin_priv->bUsePuk) {
		g_debug("SendPuk");
		dbus_g_proxy_begin_call (gsm_proxy, "SendPuk", change_done,
								 pin_priv, NULL,
								 G_TYPE_STRING, pin_priv->puk_str,
								 G_TYPE_STRING, pin_priv->new_pin_str,
								 G_TYPE_INVALID);

		gtk_entry_set_text(pin_priv->status, _("Started PIN change using PUK..."));
	} else
	{
		g_debug("ChangePin");	
		dbus_g_proxy_begin_call (gsm_proxy, "ChangePin", change_done,
								 pin_priv, NULL,
								 G_TYPE_STRING, pin_priv->pin_str,
								 G_TYPE_STRING, pin_priv->new_pin_str,
								 G_TYPE_INVALID);
		gtk_entry_set_text(pin_priv->status, _("Started PIN change..."));
	}
}

static void do_enable_pin (CEPagePinPrivate* pin_priv, gboolean enabled)
{
	DBusGProxy *gsm_proxy;

	g_debug("do_enable_pin");

	gsm_proxy = dbus_g_proxy_new_for_name (pin_priv->bus,
										   MM_DBUS_SERVICE, pin_priv->path, MM_DBUS_INTERFACE_MODEM_GSM_CARD);

	g_debug("after gsm_proxy");


	g_debug("pin = %s, new_pin = %s", pin_priv->pin_str, pin_priv->new_pin_str);

	dbus_g_proxy_begin_call (gsm_proxy, "EnablePin", change_done,
							 pin_priv, NULL,
							 G_TYPE_STRING, pin_priv->pin_str,
							 G_TYPE_BOOLEAN, enabled,
							 G_TYPE_INVALID);
	gtk_entry_set_text(pin_priv->status, "Started SIM lock change");
}

static void
send_pin_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer data)
{
	GError *error = NULL;
	CEPagePinPrivate *pin_priv = (CEPagePinPrivate *) data;
	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_INVALID)) {

		if (!strcmp(error->message, "Incorrect password")) {
			g_debug("wrong pin!!!");
		}
		if (dbus_g_error_has_name (error, MM_MODEM_ERROR_SIM_PUK)) {
			pin_priv->bUsePuk = 1;
		}
			
		g_warning ("%s", error->message);
		g_error_free (error);
	}

	if (pin_priv->bUsePuk && (!CHANGE_PIN || !strlen(pin_priv->puk_str))) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, "You must change PIN using your PUK code!");
		gtk_window_set_title (GTK_WINDOW (dialog), _("Error"));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_free(pin_priv->path);
		g_free(pin_priv->pin_str);
		g_free(pin_priv->new_pin_str);
		g_free(pin_priv->puk_str);
		gtk_widget_destroy(pin_priv->dialog);
		return;
	}
	switch (pin_priv->pin_func) {
		case CHANGE_PIN:	do_change_pin(pin_priv); break;
		case ENABLE_PIN:	do_enable_pin(pin_priv, TRUE); break;
		case DISABLE_PIN:	do_enable_pin(pin_priv, FALSE); break;
		case NOTHING: g_debug("NOTHING!");
	}
}

static void do_send_pin (CEPagePinPrivate* pin_priv)
{
	DBusGProxy *gsm_proxy;

	g_debug("do_send_pin");

	gsm_proxy = dbus_g_proxy_new_for_name (pin_priv->bus,
										   MM_DBUS_SERVICE, pin_priv->path, MM_DBUS_INTERFACE_MODEM_GSM_CARD);


	g_debug("pin = %s, new_pin = %s", pin_priv->pin_str, pin_priv->new_pin_str);

	dbus_g_proxy_begin_call (gsm_proxy, "SendPin", send_pin_done,
							 pin_priv, NULL,
							 G_TYPE_STRING, pin_priv->pin_str,
							 G_TYPE_INVALID);
}


static void
   enable_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer data)
{
	GError *error = NULL;
	GtkWidget *dialog;
	CEPagePinPrivate *pin_priv = (CEPagePinPrivate *) data;
	
	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_INVALID)) {
		g_warning ("%s", error->message);
		g_error_free (error);
		dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, "Couldn't enable device!");
		gtk_window_set_title (GTK_WINDOW (dialog), _("Error"));
		gtk_dialog_run (GTK_DIALOG (dialog));
		gtk_widget_destroy (dialog);
		g_free(pin_priv->path);
		g_free(pin_priv->pin_str);
		g_free(pin_priv->new_pin_str);
		g_free(pin_priv->puk_str);
		gtk_widget_destroy(pin_priv->dialog);
		return;
	} else {
		g_debug("before pin_func");
		gtk_entry_set_text(pin_priv->status, _("Device enabled...."));
		do_send_pin(pin_priv);
	}
}

static void
   enumerate_devices_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer data)
{
//	NMModemManager *manager = NM_MODEM_MANAGER (data);
	GPtrArray *modems;
	GError *error = NULL;
	CEPagePinPrivate *pin_priv = (CEPagePinPrivate *) data;
	
	if (!dbus_g_proxy_end_call (proxy, call_id, &error,
								dbus_g_type_get_collection ("GPtrArray", DBUS_TYPE_G_OBJECT_PATH), &modems,
								G_TYPE_INVALID)) {
		g_debug ("Could not get modem list: %s", error->message);
		g_error_free (error);
	} else {
		int i;

		if (modems->len == 0) {
			GtkWidget *dialog;
			g_debug("No modems found");
			dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, _("No modems found!"));
			gtk_window_set_title (GTK_WINDOW (dialog), _("Error"));
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
			g_free(pin_priv->path);
			g_free(pin_priv->pin_str);
			g_free(pin_priv->new_pin_str);
			g_free(pin_priv->puk_str);
			gtk_widget_destroy(pin_priv->dialog);
			return;
		} else
		if (modems->len > 1) {
			GtkWidget *dialog;
			g_debug("%d modems found", modems->len);
			dialog = gtk_message_dialog_new (GTK_WINDOW (pin_priv->dialog), GTK_DIALOG_MODAL, GTK_MESSAGE_INFO, GTK_BUTTONS_OK, _("Several modems found! using the first..."));
			gtk_window_set_title (GTK_WINDOW (dialog), _("Warning"));
			gtk_dialog_run (GTK_DIALOG (dialog));
			gtk_widget_destroy (dialog);
		}
		
		for (i = 0; i < modems->len; i++) {
			DBusGProxy *gsm_proxy;
			char *path = (char *) g_ptr_array_index (modems, i);

			g_debug("path = %s", path);

			gsm_proxy = dbus_g_proxy_new_for_name (pin_priv->bus,
											   MM_DBUS_SERVICE, path, MM_DBUS_INTERFACE_MODEM);

			pin_priv->path = g_strdup(path);
			dbus_g_proxy_begin_call (gsm_proxy, "Enable", enable_done,
									 pin_priv, NULL,
									 G_TYPE_BOOLEAN, TRUE,
									 G_TYPE_INVALID);

			g_debug("after enable");
			
//			create_modem (manager, path);
			g_free (path);
		}

		g_ptr_array_free (modems, TRUE);
	}
}


static void
   enumerate_devices (CEPagePinPrivate *pin_priv)
{
	GError *error = NULL;

	/* DBus */
	pin_priv->bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (!pin_priv->bus) {
		g_error ("Couldn't connect to DBus: %s", error->message);
		g_error_free (error);

		return;
	}
	pin_priv->proxy = dbus_g_proxy_new_for_name (pin_priv->bus,
							   MM_DBUS_SERVICE, MM_DBUS_PATH, MM_DBUS_INTERFACE);

	
	dbus_g_proxy_begin_call (pin_priv->proxy, "EnumerateDevices", enumerate_devices_done, pin_priv, NULL, G_TYPE_INVALID);
}


static void
   pin_utils (GtkButton *button, gpointer user_data)
{
	static CEPagePinPrivate pin_priv;
//	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (user_data);

	gint result;
	
	pin_priv.xml = glade_xml_new (GLADEDIR "/ce-page-pin.glade", NULL, NULL);
	pin_priv.dialog = glade_xml_get_widget (pin_priv.xml, "dialog1");

//	glade_xml_signal_autoconnect (xml);

	pin_priv.pin = GTK_ENTRY (glade_xml_get_widget (pin_priv.xml, "mobile_pin_pin"));
	pin_priv.new_pin = GTK_ENTRY (glade_xml_get_widget (pin_priv.xml, "mobile_pin_new_pin"));
	pin_priv.puk = GTK_ENTRY (glade_xml_get_widget (pin_priv.xml, "mobile_pin_puk"));
	pin_priv.status = GTK_ENTRY (glade_xml_get_widget (pin_priv.xml, "mobile_pin_status"));

	gtk_window_set_title (GTK_WINDOW (pin_priv.dialog), _("PIN functions"));
	gtk_widget_show_all(pin_priv.dialog);

	pin_priv.bUsePuk = 0;
	result = gtk_dialog_run (GTK_DIALOG(pin_priv.dialog));

	pin_priv.pin_str = g_strdup(gtk_entry_get_text(pin_priv.pin));
	pin_priv.new_pin_str = g_strdup(gtk_entry_get_text(pin_priv.new_pin));
	pin_priv.puk_str = g_strdup(gtk_entry_get_text(pin_priv.puk));

	
	switch (result) {
		case 1: pin_priv.pin_func = CHANGE_PIN; 	gtk_entry_set_text(pin_priv.status, _("Changing pin....")); break;
		case 2: pin_priv.pin_func = ENABLE_PIN; gtk_entry_set_text(pin_priv.status, _("Enabling pin...."));break;
		case 3: pin_priv.pin_func = DISABLE_PIN; gtk_entry_set_text(pin_priv.status, _("Disabling pin...."));break;
		default:
			pin_priv.pin_func = NOTHING;
	}
	
	if (!pin_priv.pin_func == NOTHING)
		enumerate_devices(&pin_priv);
	else {
		g_free(pin_priv.path);
		g_free(pin_priv.pin_str);
		g_free(pin_priv.new_pin_str);
		g_free(pin_priv.puk_str);
		gtk_widget_destroy(pin_priv.dialog);
	}

	g_debug ("result = %d", result);
}

CEPageMobile *
ce_page_mobile_new (NMConnection *connection)
{
	CEPageMobile *self;
	CEPageMobilePrivate *priv;
	CEPage *parent;

	self = CE_PAGE_MOBILE (g_object_new (CE_TYPE_PAGE_MOBILE, NULL));
	parent = CE_PAGE (self);

	parent->xml = glade_xml_new (GLADEDIR "/ce-page-mobile.glade", "MobilePage", NULL);
	if (!parent->xml) {
		g_warning ("%s: Couldn't load mobile page glade file.", __func__);
		g_object_unref (self);
		return NULL;
	}

	parent->page = glade_xml_get_widget (parent->xml, "MobilePage");
	if (!parent->page) {
		g_warning ("%s: Couldn't load mobile page from glade file.", __func__);
		g_object_unref (self);
		return NULL;
	}
	g_object_ref_sink (parent->page);

	parent->title = g_strdup (_("Mobile Broadband"));

	mobile_private_init (self);
	priv = CE_PAGE_MOBILE_GET_PRIVATE (self);

	priv->setting = nm_connection_get_setting (connection, NM_TYPE_SETTING_GSM);
	if (!priv->setting)
		priv->setting = nm_connection_get_setting (connection, NM_TYPE_SETTING_CDMA);

	if (!priv->setting) {
		/* FIXME: Support add. */
		g_warning ("Adding mobile conneciton not supported yet.");
		g_object_unref (self);
		return NULL;
	}

	populate_ui (self, connection);

	g_signal_connect (priv->number, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->username, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->password, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->apn, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->network_id, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->network_type, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->pin, "changed", G_CALLBACK (stuff_changed), self);
	g_signal_connect (priv->puk, "changed", G_CALLBACK (stuff_changed), self);

	g_signal_connect (glade_xml_get_widget (parent->xml, "mobile_show_passwords"),
					  "toggled", G_CALLBACK (show_passwords), self);

	g_signal_connect (glade_xml_get_widget (parent->xml, "mobile_pin_button"),
					  "clicked", G_CALLBACK (pin_utils), self);
	return self;
}

static const char *
nm_entry_get_text (GtkEntry *entry)
{
	const char *txt;

	txt = gtk_entry_get_text (entry);
	if (txt && strlen (txt) > 0)
		return txt;

	return NULL;
}

static void
gsm_ui_to_setting (CEPageMobile *self)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);
	int net_type;

	switch (gtk_combo_box_get_active (priv->network_type)) {
	case NET_TYPE_3G:
		net_type = NM_GSM_NETWORK_UMTS_HSPA;
		break;
	case NET_TYPE_2G:
		net_type = NM_GSM_NETWORK_GPRS_EDGE;
		break;
	case NET_TYPE_PREFER_3G:
		net_type = NM_GSM_NETWORK_PREFER_UMTS_HSPA;
		break;
	case NET_TYPE_PREFER_2G:
		net_type = NM_GSM_NETWORK_PREFER_GPRS_EDGE;
		break;
	case NET_TYPE_ANY:
	default:
		net_type = NM_GSM_NETWORK_ANY;
		break;
	}

	
	g_object_set (priv->setting,
				  NM_SETTING_GSM_NUMBER,   nm_entry_get_text (priv->number),
				  NM_SETTING_GSM_USERNAME, nm_entry_get_text (priv->username),
				  NM_SETTING_GSM_PASSWORD, nm_entry_get_text (priv->password),
				  NM_SETTING_GSM_APN, nm_entry_get_text (priv->apn),
				  NM_SETTING_GSM_NETWORK_ID, nm_entry_get_text (priv->network_id),
				  NM_SETTING_GSM_NETWORK_TYPE, net_type,
				  NM_SETTING_GSM_PIN, nm_entry_get_text (priv->pin),
				  NM_SETTING_GSM_PUK, nm_entry_get_text (priv->puk),
				  NULL);
}

static void
cdma_ui_to_setting (CEPageMobile *self)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);

	g_object_set (priv->setting,
				  NM_SETTING_CDMA_NUMBER,   nm_entry_get_text (priv->number),
				  NM_SETTING_CDMA_USERNAME, nm_entry_get_text (priv->username),
				  NM_SETTING_CDMA_PASSWORD, nm_entry_get_text (priv->password),
				  NULL);
}

static void
ui_to_setting (CEPageMobile *self)
{
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);

	if (NM_IS_SETTING_GSM (priv->setting))
		gsm_ui_to_setting (self);
	else if (NM_IS_SETTING_CDMA (priv->setting))
		cdma_ui_to_setting (self);
	else
		g_error ("Invalid setting");
}

static gboolean
validate (CEPage *page, NMConnection *connection, GError **error)
{
	CEPageMobile *self = CE_PAGE_MOBILE (page);
	CEPageMobilePrivate *priv = CE_PAGE_MOBILE_GET_PRIVATE (self);

	ui_to_setting (self);
	return nm_setting_verify (priv->setting, NULL, error);
}

static void
ce_page_mobile_init (CEPageMobile *self)
{
}

static void
ce_page_mobile_class_init (CEPageMobileClass *mobile_class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (mobile_class);
	CEPageClass *parent_class = CE_PAGE_CLASS (mobile_class);

	g_type_class_add_private (object_class, sizeof (CEPageMobilePrivate));

	/* virtual methods */
	parent_class->validate = validate;
}
