/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*- */

#include <string.h>
#include <gtk/gtk.h>
#include <glade/glade.h>
#include <dbus/dbus-glib.h>
#include <gconf/gconf-client.h>
#include <nm-connection.h>
#include <nm-setting-connection.h>
#include <nm-setting-gsm.h>
#include <nm-setting-serial.h>
#include <nm-setting-ppp.h>
#include <nm-utils.h>
#include "gconf-helpers.h"

#define MM_DBUS_SERVICE              "org.freedesktop.ModemManager"
#define MM_DBUS_PATH                 "/org/freedesktop/ModemManager"
#define MM_DBUS_INTERFACE            "org.freedesktop.ModemManager"
#define MM_DBUS_INTERFACE_MODEM      "org.freedesktop.ModemManager.Modem"

#define MM_DBUS_INTERFACE_MODEM_GSM_CARD    "org.freedesktop.ModemManager.Modem.Gsm.Card"
#define MM_DBUS_INTERFACE_MODEM_GSM_NETWORK "org.freedesktop.ModemManager.Modem.Gsm.Network"

#define MM_MODEM_TYPE_UNKNOWN  0
#define MM_MODEM_TYPE_GSM      1
#define MM_MODEM_TYPE_CDMA     2

#define SCAN_COL_NAME    0
#define SCAN_COL_STATUS  1
#define SCAN_COL_OPER_ID 2

typedef struct {
	/* UI */
	GladeXML *glade_xml;
	GtkDialog *main_dialog;
	GtkTreeView *network_list;
	GtkListStore *network_store;
	GtkWidget *scan_button;
	GtkWidget *create_net_button;

	GtkWidget *scan_dialog;
	GtkProgressBar *scan_progress_bar;

	GtkDialog *create_network_dialog;
	GtkEntry *create_network_name;

	/* DBus */
	DBusGConnection *bus;
	DBusGProxy *proxy;
	DBusGProxy *gsm_net_proxy;

	GMainLoop *main_loop;
} AppData;

AppData *_app_data = NULL;

static void
get_str_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	char *result = NULL;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_STRING, &result, G_TYPE_INVALID)) {
		g_warning ("%s", error->message);
		g_error_free (error);
	} else {
		gtk_label_set_text (GTK_LABEL (user_data), result);
		g_free (result);
	}
}

static void
get_card_info_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	GladeXML *glade_xml = GLADE_XML (user_data);
	char *manufacturer = NULL;
	char *model = NULL;
	char *version = NULL;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (proxy, call_id, &error,
								G_TYPE_STRING, &manufacturer,
								G_TYPE_STRING, &model,
								G_TYPE_STRING, &version,
								G_TYPE_INVALID)) {
		g_warning ("Couldn't get modem information: %s", error->message);
		g_error_free (error);
	} else {
		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (glade_xml, "vendor_label")), manufacturer);
		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (glade_xml, "model_label")), model);
		gtk_label_set_text (GTK_LABEL (glade_xml_get_widget (glade_xml, "version_label")), version);

		if (_app_data)
			gtk_widget_set_sensitive (_app_data->scan_button, TRUE);

		g_free (manufacturer);
		g_free (model);
		g_free (version);
	}
}

static void
get_property_done (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	GValue value = { 0, };
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_VALUE, &value, G_TYPE_INVALID)) {
		g_warning ("%s", error->message);
		g_error_free (error);
	} else {
		gtk_label_set_text (GTK_LABEL (user_data), g_value_get_string (&value));
		g_value_unset (&value);
	}
}

static gboolean
get_info (gpointer data)
{
	AppData *app_data = (AppData *) data;

	_app_data = app_data;

	dbus_g_proxy_set_interface (app_data->proxy, MM_DBUS_INTERFACE_MODEM_GSM_CARD);
	dbus_g_proxy_begin_call (app_data->proxy, "GetImsi", get_str_done,
							 glade_xml_get_widget (app_data->glade_xml, "imsi_label"), NULL,
							 G_TYPE_INVALID);

	dbus_g_proxy_begin_call (app_data->proxy, "GetImei", get_str_done,
							 glade_xml_get_widget (app_data->glade_xml, "imei_label"), NULL,
							 G_TYPE_INVALID);

	dbus_g_proxy_begin_call (app_data->proxy, "GetInfo", get_card_info_done,
							 app_data->glade_xml, NULL,
							 G_TYPE_INVALID);

	dbus_g_proxy_set_interface (app_data->proxy, "org.freedesktop.DBus.Properties");

	dbus_g_proxy_begin_call (app_data->proxy, "Get", get_property_done,
							 glade_xml_get_widget (app_data->glade_xml, "driver_label"), NULL,
							 G_TYPE_STRING, MM_DBUS_INTERFACE_MODEM,
							 G_TYPE_STRING, "Driver",
							 G_TYPE_INVALID);

	dbus_g_proxy_begin_call (app_data->proxy, "Get", get_property_done,
							 glade_xml_get_widget (app_data->glade_xml, "device_label"), NULL,
							 G_TYPE_STRING, MM_DBUS_INTERFACE_MODEM,
							 G_TYPE_STRING, "DataDevice",
							 G_TYPE_INVALID);

	return FALSE;
}

static void
got_signal_quality (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	guint32 quality = 0;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_UINT, &quality, G_TYPE_INVALID)) {
		g_warning ("%s", error->message);
		g_error_free (error);
	} else {
		char *tmp;

		tmp = g_strdup_printf ("%d%%", quality);
		gtk_label_set_text (GTK_LABEL (user_data), tmp);
		g_free (tmp);
	}
}

static void
signal_quality_changed (DBusGProxy *proxy,
						guint32 signal_quality,
						gpointer user_data)
{
	char *tmp;

	tmp = g_strdup_printf ("%d%%", signal_quality);
	gtk_label_set_text (GTK_LABEL (user_data), tmp);
	g_free (tmp);
}

static gboolean
monitor_signal_quality (gpointer data)
{
	AppData *app_data = (AppData *) data;
	GtkWidget *label;

	label = glade_xml_get_widget (app_data->glade_xml, "signal_quality_label");

	dbus_g_proxy_add_signal (app_data->gsm_net_proxy, "SignalQuality", G_TYPE_UINT, G_TYPE_INVALID);
	dbus_g_proxy_connect_signal (app_data->gsm_net_proxy, "SignalQuality",
								 G_CALLBACK (signal_quality_changed),
								 label, NULL);

	dbus_g_proxy_begin_call (app_data->gsm_net_proxy, "GetSignalQuality",
							 got_signal_quality, label, NULL, G_TYPE_INVALID);

	return FALSE;
}

static void
got_scan_results (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;
	GPtrArray *array = NULL;
	GError *error = NULL;
	GType type;

	type = dbus_g_type_get_collection ("GPtrArray", DBUS_TYPE_G_STRING_STRING_HASHTABLE);

	if (!dbus_g_proxy_end_call (proxy, call_id, &error, type, &array, G_TYPE_INVALID)) {
		g_warning ("Couldn't scan: %s", error->message);
		g_error_free (error);
	} else {
		GtkTreeIter iter;
		int i;

		for (i = 0; i < array->len; i++) {
			GHashTable *hash = (GHashTable *) g_ptr_array_index (array, i);
			char *status;
			const char *status_str;

			status = g_hash_table_lookup (hash, "status");
			if (!strcmp (status, "1"))
				status_str = "Available";
			else if (!strcmp (status, "2"))
				status_str = "Current";
			else if (!strcmp (status, "3"))
				status_str = "Forbidden";
			else
				status_str = "Unknown";

			gtk_list_store_append (app_data->network_store, &iter);
			gtk_list_store_set (app_data->network_store, &iter,
								SCAN_COL_NAME, g_hash_table_lookup (hash, "operator-long"),
								SCAN_COL_STATUS, status_str,
								SCAN_COL_OPER_ID, g_hash_table_lookup (hash, "operator-num"),
								-1);

			g_hash_table_destroy (hash);
		}

		g_ptr_array_free (array, TRUE);
	}

	gtk_widget_hide (app_data->scan_dialog);
	gtk_widget_set_sensitive (app_data->scan_button, TRUE);
}

static gboolean
scan_pulse (gpointer data)
{
	GtkProgressBar *bar = GTK_PROGRESS_BAR (data);

	gtk_progress_bar_pulse (bar);
	return gdk_window_is_visible (gtk_widget_get_parent_window (GTK_WIDGET (bar)));
}

static void
scan (GtkButton *button, gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;

	dbus_g_proxy_begin_call_with_timeout (app_data->gsm_net_proxy, "Scan", got_scan_results,
										  app_data, NULL, 120000, G_TYPE_INVALID);

	gtk_widget_set_sensitive (app_data->scan_button, FALSE);
	gtk_list_store_clear (app_data->network_store);

	g_timeout_add (200, scan_pulse, app_data->scan_progress_bar);
	gtk_widget_show (app_data->scan_dialog);
}

static void
modem_enabled (DBusGProxy *proxy, DBusGProxyCall *call_id, gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;
	GError *error = NULL;

	if (!dbus_g_proxy_end_call (proxy, call_id, &error, G_TYPE_INVALID)) {
		g_warning ("Couldn't enable modem: %s", error->message);
		g_error_free (error);
		g_main_loop_quit (app_data->main_loop);
		return;
	}

	g_idle_add (get_info, app_data);
	g_idle_add (monitor_signal_quality, app_data);
}

static void
modem_enable (AppData *app_data)
{
	dbus_g_proxy_begin_call (app_data->proxy, "Enable", modem_enabled,
 							 app_data, NULL,
 							 G_TYPE_BOOLEAN, TRUE, G_TYPE_INVALID);
}

static void
create_network (const char *name, const char *oper_code)
{
	NMConnection *connection;
	NMSettingGsm *s_gsm;
	NMSettingSerial *s_serial;
	NMSettingPPP *s_ppp;
	NMSettingConnection *s_con;
	GConfClient *gconf_client;

	connection = nm_connection_new ();

	s_gsm = NM_SETTING_GSM (nm_setting_gsm_new ());
	s_gsm->number = g_strdup ("*99#"); /* This should be a sensible default as it's seems to be quite standard */
	s_gsm->network_id = g_strdup (oper_code);
	nm_connection_add_setting (connection, NM_SETTING (s_gsm));

	/* Serial setting */
	s_serial = (NMSettingSerial *) nm_setting_serial_new ();
	s_serial->baud = 115200;
	s_serial->bits = 8;
	s_serial->parity = 'n';
	s_serial->stopbits = 1;
	nm_connection_add_setting (connection, NM_SETTING (s_serial));

	s_ppp = (NMSettingPPP *) nm_setting_ppp_new ();
	nm_connection_add_setting (connection, NM_SETTING (s_ppp));

	s_con = NM_SETTING_CONNECTION (nm_setting_connection_new ());
	s_con->id = g_strdup (name);
	s_con->type = g_strdup (NM_SETTING (s_gsm)->name);
	s_con->autoconnect = FALSE;
	s_con->uuid = nm_utils_uuid_generate ();
	nm_connection_add_setting (connection, NM_SETTING (s_con));

	gconf_client = gconf_client_get_default ();
	if (gconf_client) {
		char *dir = NULL;
		int i;

		/* Find free GConf directory */
		for (i = 0; i < G_MAXUINT32; i++) {
			char buf[255];

			snprintf (&buf[0], 255, GCONF_PATH_CONNECTIONS"/%d", i);
			if (!gconf_client_dir_exists (gconf_client, buf, NULL)) {
				dir = g_strdup (buf);
				break;
			}
		}

		nm_gconf_write_connection (connection, gconf_client, dir);
		gconf_client_notify (gconf_client, dir);
		gconf_client_suggest_sync (gconf_client, NULL);
		g_free (dir);
		g_object_unref (gconf_client);
	} else
		g_warning ("Writing conneciton failed");

	g_object_unref (connection);
}

static void
create_network_clicked (GtkButton *button, gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;
	GtkTreeSelection *selection;
	GList *selected_rows;
	GtkTreeModel *model = NULL;
	GtkTreeIter iter;

	selection = gtk_tree_view_get_selection (app_data->network_list);
	selected_rows = gtk_tree_selection_get_selected_rows (selection, &model);
	if (!selected_rows)
		return;

	if (gtk_tree_model_get_iter (model, &iter, (GtkTreePath *) selected_rows->data)) {
		char *oper_name = NULL;
		char *oper_id = NULL;
		gint result;

		gtk_tree_model_get (model, &iter, SCAN_COL_NAME, &oper_name, -1);
		gtk_tree_model_get (model, &iter, SCAN_COL_OPER_ID, &oper_id, -1);

		gtk_entry_set_text (app_data->create_network_name, oper_name);
		gtk_editable_select_region (GTK_EDITABLE (app_data->create_network_name), 0, -1);
		gtk_widget_grab_focus (GTK_WIDGET (app_data->create_network_name));

		result = gtk_dialog_run (app_data->create_network_dialog);
		gtk_widget_hide (GTK_WIDGET (app_data->create_network_dialog));

		if (result == GTK_RESPONSE_OK)
			create_network (gtk_entry_get_text (app_data->create_network_name), oper_id);

		g_free (oper_name);
		g_free (oper_id);
	}

	/* free memory */
	g_list_foreach (selected_rows, (GFunc) gtk_tree_path_free, NULL);
	g_list_free (selected_rows);
}

static void
network_list_selection_changed (GtkTreeSelection *selection, gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;
	GtkTreeIter iter;
	GtkTreeModel *model;

	if (gtk_tree_selection_get_selected (selection, &model, &iter))
		gtk_widget_set_sensitive (app_data->create_net_button, TRUE);
	else
		gtk_widget_set_sensitive (app_data->create_net_button, FALSE);
}


static void
app_data_destroy (AppData *app_data)
{
	if (app_data->bus)
		dbus_g_connection_unref (app_data->bus);

	if (app_data->proxy)
		g_object_unref (app_data->proxy);

	if (app_data->gsm_net_proxy)
		g_object_unref (app_data->gsm_net_proxy);

	if (app_data->glade_xml)
		g_object_unref (app_data->glade_xml);

	if (app_data->main_loop)
		g_main_loop_unref (app_data->main_loop);

	g_slice_free (AppData, app_data);
}

static void
close_cb (GtkDialog *dialog,
		  gint response_id,
		  gpointer user_data)
{
	AppData *app_data = (AppData *) user_data;

	dbus_g_proxy_set_interface (app_data->proxy, MM_DBUS_INTERFACE_MODEM);
//Not for MBM	dbus_g_proxy_call_no_reply (app_data->proxy, "Enable", G_TYPE_BOOLEAN, FALSE, G_TYPE_INVALID);

	g_main_loop_quit (app_data->main_loop);
	app_data_destroy (app_data);
}


static GtkListStore *
prepare_network_list (GtkTreeView *treeview)
{
	GtkListStore *store;

	store = gtk_list_store_new (3, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING);
	gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (store));
	g_object_unref (store);

	gtk_tree_view_insert_column_with_attributes (treeview,
	                                             -1, "Name", gtk_cell_renderer_text_new (),
	                                             "text", SCAN_COL_NAME,
	                                             NULL);

	gtk_tree_view_insert_column_with_attributes (treeview,
	                                             -1, "Status", gtk_cell_renderer_text_new (),
	                                             "text", SCAN_COL_STATUS,
	                                             NULL);

	return store;
}

static AppData *
app_data_create (const char *udi)
{
	AppData *app_data;
	GtkTreeSelection *selection;
	GError *error = NULL;

	app_data = g_slice_new0 (AppData);

	/* DBus */
	app_data->bus = dbus_g_bus_get (DBUS_BUS_SYSTEM, &error);
	if (!app_data->bus) {
		g_error ("Couldn't connect to DBus: %s", error->message);
		g_error_free (error);
		g_slice_free (AppData, app_data);

		return NULL;
	}

	app_data->proxy = dbus_g_proxy_new_for_name (app_data->bus, MM_DBUS_SERVICE, udi, MM_DBUS_INTERFACE_MODEM);
	app_data->gsm_net_proxy = dbus_g_proxy_new_from_proxy (app_data->proxy, MM_DBUS_INTERFACE_MODEM_GSM_NETWORK, NULL);

	/* UI */
	app_data->glade_xml = glade_xml_new (GLADEDIR "/nm-modem-properties.glade", NULL, NULL);
	if (!app_data->glade_xml) {
		g_error ("Could not load Glade file");
		g_slice_free (AppData, app_data);

		return NULL;
	}

	app_data->main_dialog = GTK_DIALOG (glade_xml_get_widget (app_data->glade_xml, "main_dialog"));
	g_signal_connect (app_data->main_dialog, "response", G_CALLBACK (close_cb), app_data);

	app_data->network_list = GTK_TREE_VIEW (glade_xml_get_widget (app_data->glade_xml, "network_list"));
	app_data->network_store = prepare_network_list (app_data->network_list);
	app_data->scan_button = glade_xml_get_widget (app_data->glade_xml, "scan_button");
	g_signal_connect (app_data->scan_button, "clicked", G_CALLBACK (scan), app_data);

	app_data->scan_dialog = glade_xml_get_widget (app_data->glade_xml, "scan_dialog");
	gtk_window_set_transient_for (GTK_WINDOW (app_data->scan_dialog), GTK_WINDOW (app_data->main_dialog));

	app_data->scan_progress_bar = GTK_PROGRESS_BAR (glade_xml_get_widget (app_data->glade_xml, "scan_progress_bar"));

	app_data->create_net_button = glade_xml_get_widget (app_data->glade_xml, "create_connection_button");
	gtk_widget_set_sensitive (app_data->create_net_button, FALSE);
	g_signal_connect (app_data->create_net_button, "clicked", G_CALLBACK (create_network_clicked), app_data);
	selection = gtk_tree_view_get_selection (app_data->network_list);
	g_signal_connect (selection, "changed", G_CALLBACK (network_list_selection_changed), app_data);

	app_data->create_network_dialog = GTK_DIALOG (glade_xml_get_widget (app_data->glade_xml, "create_network_dialog"));
	app_data->create_network_name = GTK_ENTRY (glade_xml_get_widget (app_data->glade_xml, "create_network_name"));

	gtk_widget_set_sensitive (app_data->scan_button, FALSE);

	app_data->main_loop = g_main_loop_new (NULL, FALSE);

	return app_data;
}

int
main (int argc, char *argv[])
{
	//const char *udi = "/org/freedesktop/Hal/devices/usb_device_12d1_1003_noserial_if0_serial_usb_0";
	AppData *app_data;

	if (argc != 2) {
		g_print ("Usage: %s <udi>\n", argv[0]);
		return 1;
	}

	gtk_init (&argc, &argv);

	app_data = app_data_create (argv[1]);
	if (app_data) {
		modem_enable (app_data);
		g_main_loop_run (app_data->main_loop);
	}

	return 0;
}
