/*
 * Copyright (C) 2007 Intel
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Authored by Neil Jagdish Patel <njp@o-hand.com>
 *
 */

#include <glib.h>
#include <glib/gstdio.h>
#include <gtk/gtk.h>
#include <gio/gio.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

#include "launcher-config.h"
#include "launcher-defines.h"

G_DEFINE_TYPE (LauncherConfig, launcher_config, G_TYPE_OBJECT)

#define LAUNCHER_CONFIG_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE(obj, \
        LAUNCHER_TYPE_CONFIG, LauncherConfigPrivate))

#define FONT_DIR              "/desktop/gnome/interface"
#define FONT_PATH             "/desktop/gnome/interface/font_name"

#define RESUME_DIR  "/tmp/.launcher"
#define RESUME_FILE "launcher-resume-notify"


struct _LauncherConfigPrivate
{
  GdkScreen *screen;  

  GFile *watch_dir;
  GFileMonitor *dir_monitor;
};

enum
{
  CONFIG_CHANGED,
  RESUME_EVENT,

  LAST_SIGNAL
};
static guint _config_signals[LAST_SIGNAL] = { 0 };


static gint
get_rows_for_height (gint height)
{
  gint rows = 3;

  if (height < 600)
    rows = 3;
  else if (height < 1025)
    rows = 4;
  else
    rows = 5;

  return rows;
}

static gint
get_cols_for_width (gint width)
{
  gint cols = 3;

  if (width < 801)
    cols = 3;
  else if (width < 1025)
    cols = 4;
  else
    cols = 5;

  return cols;
}

static void
calculate_sizes (LauncherConfig *cfg)
{
  gint          width;
  gint          height;

  width = CSW();
  height = CSH();
  
  cfg->win_width = width * 0.98 ;
  cfg->win_height = CSH () - (PANEL_HEIGHT*2.3);
  cfg->bar_width = cfg->win_width/5;
  
  cfg->iconview_rows = get_rows_for_height (height);
  cfg->iconview_cols = get_cols_for_width (width);
  cfg->iconview_width = cfg->bar_width * 3;;
  cfg->iconview_height = cfg->win_height - (PANEL_HEIGHT*2.3);
  cfg->iconview_padding = 0;
  
  cfg->icon_width = (cfg->iconview_width - ((cfg->iconview_cols+1)*cfg->iconview_padding)) / cfg->iconview_cols;
  
  cfg->icon_height = (cfg->iconview_height - ((cfg->iconview_rows+1)*cfg->iconview_padding))/cfg->iconview_rows;

  cfg->shortcut_height = (cfg->win_height/N_CATS);
}


static void
set_font (const gchar *font_name, LauncherConfig *config)
{
  PangoFontDescription *desc;
  gchar *temp;

  desc = pango_font_description_from_string (font_name);

  if (pango_font_description_get_size_is_absolute (desc))
  {
    if (pango_font_description_get_size (desc) >= 10)
      pango_font_description_set_size (desc, 
                                     pango_font_description_get_size (desc) -2);

  }
  else
  {
    if (pango_units_to_double (pango_font_description_get_size (desc)) >= 10.0)
      pango_font_description_set_size (desc, 
        pango_font_description_get_size (desc) 
          - (pango_units_from_double (2.0)));
   
  }
  //pango_font_description_set_weight (desc, PANGO_WEIGHT_BOLD);

  temp = pango_font_description_to_string (desc);

  memcpy (config->font_name, temp, strlen (temp)+1);

  g_debug ("Font changed: %s\n", config->font_name);

  pango_font_description_free (desc);
}

/* Callbacks */
static void
on_font_changed (GConfClient    *client,
                 guint           cid,
                 GConfEntry     *entry,
                 LauncherConfig *config)
{
  GConfValue *value = NULL;

  value = gconf_entry_get_value (entry);

  set_font (gconf_value_get_string (value), config);

  g_signal_emit (config, _config_signals[CONFIG_CHANGED], 0);
}

/*
 * Watch for resume
 */
static void
on_resume_dir_changed (GFileMonitor      *monitor,
                       GFile             *file,
                       GFile             *other_file,
                       GFileMonitorEvent  event,
                       LauncherConfig    *config)
{
  LauncherConfigPrivate *priv = config->priv;
    
  g_return_if_fail (LAUNCHER_IS_CONFIG (config));
  priv = config->priv;

  if (event != G_FILE_MONITOR_EVENT_CREATED)
    return;

  g_print ("%s\n", g_file_get_path (file));

  g_signal_emit (config, _config_signals[RESUME_EVENT], 0);
}

static void
watch_links (LauncherConfig *config)
{
  LauncherConfigPrivate *priv = config->priv;
  GFile *dir;

  if (g_file_test (RESUME_DIR, G_FILE_TEST_EXISTS))
  {
    g_spawn_command_line_sync ("rm -rf "RESUME_DIR, NULL, NULL, NULL, NULL);

  }

  g_mkdir_with_parents (RESUME_DIR, 0777);

  dir = priv->watch_dir = g_file_new_for_path (RESUME_DIR);

  priv->dir_monitor = g_file_monitor_directory (dir, 0, NULL, NULL);

  g_signal_connect (priv->dir_monitor, "changed",
                    G_CALLBACK (on_resume_dir_changed), config);
}

/* GObject functions */
static void
launcher_config_dispose (GObject *object)
{
  LauncherConfig *config = LAUNCHER_CONFIG (object);
  LauncherConfigPrivate *priv;
  
  g_return_if_fail (LAUNCHER_IS_CONFIG (config));
  priv = LAUNCHER_CONFIG (config)->priv;

  g_object_unref (priv->dir_monitor);
  g_object_unref (priv->watch_dir);

  G_OBJECT_CLASS (launcher_config_parent_class)->dispose (object);
}

static void
launcher_config_class_init (LauncherConfigClass *klass)
{
  GObjectClass *obj_class = G_OBJECT_CLASS (klass);

  obj_class->dispose = launcher_config_dispose;

	_config_signals[CONFIG_CHANGED] =
		g_signal_new ("config-changed",
			      G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, config_changed),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);

	_config_signals[RESUME_EVENT] =
		g_signal_new ("resume-event",
        G_OBJECT_CLASS_TYPE (obj_class),
			      G_SIGNAL_RUN_LAST,
			      G_STRUCT_OFFSET (LauncherConfigClass, resume_event),
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID, 
			      G_TYPE_NONE, 0);


 
  g_type_class_add_private (obj_class, sizeof (LauncherConfigPrivate)); 
}

static void
launcher_config_init (LauncherConfig *config)
{
  LauncherConfigPrivate *priv;
  GConfClient           *client = gconf_client_get_default ();
      
  priv = config->priv = LAUNCHER_CONFIG_GET_PRIVATE (config);
  
  /* Load in gconf values first */
  gconf_client_add_dir (client, FONT_DIR, GCONF_CLIENT_PRELOAD_NONE, NULL);
  set_font (gconf_client_get_string (client, FONT_PATH, NULL), 
            config);
  gconf_client_notify_add (client, FONT_PATH,
                           (GConfClientNotifyFunc)on_font_changed,
                           config, NULL, NULL);

 
  /* Now load in all the calculations */
  calculate_sizes (config);

  /* Resume watching */
  watch_links (config);

  g_object_unref (client);
}

LauncherConfig*
launcher_config_get_default (void)
{
  static LauncherConfig *config = NULL;
  
  if (config == NULL)
    config = g_object_new (LAUNCHER_TYPE_CONFIG, 
                            NULL);

  return config;
}
