/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#include "task-title.h"

#include <libwnck/libwnck.h>
#include <panel-applet.h>
#include <panel-applet-gconf.h>


G_DEFINE_TYPE (TaskTitle, task_title, GTK_TYPE_EVENT_BOX);

#define TASK_TITLE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  TASK_TYPE_TITLE, \
  TaskTitlePrivate))

#define LOGOUT "gnome-session-save --kill --gui"

struct _TaskTitlePrivate
{
  WnckScreen *screen;
  GtkWidget *align;
  GtkWidget *box;
  GtkWidget *image;
  GtkWidget *label;
  GtkWidget *button;
  GtkWidget *button_image;
};

static gboolean
on_close_clicked (GtkButton *button, 
                  GdkEventButton *event, 
                  TaskTitle *title)
{
  TaskTitlePrivate *priv;
  WnckWindow *window;

  g_return_val_if_fail (TASK_IS_TITLE (title), FALSE);
  priv = title->priv;

  if (event->button != 1)
    return FALSE;

  window = wnck_screen_get_active_window (priv->screen);

  if (!WNCK_IS_WINDOW (window)
        || wnck_window_get_window_type (window) == WNCK_WINDOW_DESKTOP)
  {
    gdk_spawn_command_line_on_screen (gdk_screen_get_default (),
                                      LOGOUT, NULL);
  }
  else
  {
    wnck_window_close (window, GDK_CURRENT_TIME);
  }
  gtk_widget_queue_draw (GTK_WIDGET (title));

  return TRUE;
}

static void
on_name_changed (WnckWindow *window, TaskTitle *title)
{
  if (WNCK_IS_WINDOW (window) && TASK_IS_TITLE (title))
  {
    gtk_label_set_text (GTK_LABEL (title->priv->label), 
                        wnck_window_get_name (window));
    gtk_widget_set_tooltip_text (GTK_WIDGET (title), 
                                 wnck_window_get_name (window));
    gtk_widget_queue_draw (GTK_WIDGET (title));
  }
}


static void
on_icon_changed (WnckWindow *window, TaskTitle *title)
{
  if (WNCK_IS_WINDOW (window) && TASK_IS_TITLE (title))
  {
    gtk_image_set_from_pixbuf (GTK_IMAGE (title->priv->image), 
                               wnck_window_get_mini_icon (window));
    gtk_widget_queue_draw (GTK_WIDGET (title));
  }
}

static void
on_state_changed (WnckWindow *window, 
                  WnckWindowState changed_mask,
                  WnckWindowState new_state,
                  TaskTitle *title)
{
  TaskTitlePrivate *priv;

  g_return_if_fail (TASK_IS_TITLE (title));
  priv = title->priv;

  if (!WNCK_IS_WINDOW (window))
    return;

  if (wnck_window_is_maximized (window))
  {
    gtk_widget_set_state (GTK_WIDGET (title), GTK_STATE_ACTIVE);
    gtk_widget_show (priv->box);
  }
  else
  {
    gtk_widget_set_state (GTK_WIDGET (title), GTK_STATE_NORMAL);
    gtk_widget_hide (priv->box);
  }  
}
static void
on_active_window_changed (WnckScreen *screen, 
                          WnckWindow *old_window,
                          TaskTitle   *title)
{
  TaskTitlePrivate *priv;
  WnckWindow *act_window;
  WnckWindowType type = WNCK_WINDOW_NORMAL;
  
  g_return_if_fail (TASK_IS_TITLE (title));
  priv = title->priv;

  act_window = wnck_screen_get_active_window (screen);
  if (act_window)
    type = wnck_window_get_window_type (act_window);

  if (WNCK_IS_WINDOW (act_window) 
      && wnck_window_is_skip_tasklist (act_window)
      && type != WNCK_WINDOW_DESKTOP)
    return;

  if (type == WNCK_WINDOW_DOCK
      || type == WNCK_WINDOW_SPLASHSCREEN
      || type == WNCK_WINDOW_MENU)
    return;
 
  if (old_window)
  {
    g_signal_handlers_disconnect_by_func (old_window,
                                          on_name_changed, title);
    g_signal_handlers_disconnect_by_func (old_window,
                                          on_icon_changed, title);
    g_signal_handlers_disconnect_by_func (old_window,
                                          on_state_changed, title);   
  }

  if (!WNCK_IS_WINDOW (act_window) 
        || wnck_window_get_window_type (act_window) == WNCK_WINDOW_DESKTOP)
  { 
    gtk_image_set_from_stock (GTK_IMAGE (priv->image), 
                              GTK_STOCK_HOME, GTK_ICON_SIZE_MENU);
    gtk_label_set_text (GTK_LABEL (priv->label), _("Home"));
    gtk_image_set_from_stock (GTK_IMAGE (priv->button_image), 
                              GTK_STOCK_QUIT, GTK_ICON_SIZE_MENU);
    gtk_widget_set_tooltip_text (priv->button,
                                 _("Log off, switch user, lock screen or power "
                                   "down the computer"));
    gtk_widget_set_tooltip_text (GTK_WIDGET (title),
                                 _("Home"));
    gtk_widget_show (priv->box);
  }
  else
  {
    gtk_image_set_from_pixbuf (GTK_IMAGE (priv->image), 
                               wnck_window_get_mini_icon (act_window));
    gtk_label_set_text (GTK_LABEL (priv->label), 
                        wnck_window_get_name (act_window));
    gtk_image_set_from_stock (GTK_IMAGE (priv->button_image), 
                              GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);

    gtk_widget_set_tooltip_text (GTK_WIDGET (title),
                                 wnck_window_get_name (act_window));      
    gtk_widget_set_tooltip_text (priv->button, _("Close window"));

    g_signal_connect (act_window, "name-changed",
                      G_CALLBACK (on_name_changed), title);
    g_signal_connect (act_window, "icon-changed",
                      G_CALLBACK (on_icon_changed), title);
    g_signal_connect_after (act_window, "state-changed",
                            G_CALLBACK (on_state_changed), title);
    gtk_widget_show (priv->box);  
  }

  if (WNCK_IS_WINDOW (act_window)
      && !wnck_window_is_maximized (act_window) 
      && type != WNCK_WINDOW_DESKTOP)
  {
    gtk_widget_set_state (GTK_WIDGET (title), GTK_STATE_NORMAL);
    gtk_widget_hide (priv->box);
  }
  else
    gtk_widget_set_state (GTK_WIDGET (title), GTK_STATE_ACTIVE);

  gtk_widget_queue_draw (GTK_WIDGET (title));
}

static gboolean
on_button_release (GtkWidget *title, GdkEventButton *event)
{
  TaskTitlePrivate *priv;
  WnckWindow *window;
  GtkWidget *menu;

  g_return_val_if_fail (TASK_IS_TITLE (title), FALSE);
  priv = TASK_TITLE_GET_PRIVATE (title);

  window = wnck_screen_get_active_window (priv->screen);

  if (event->button != 3)
    return FALSE;

  if (!WNCK_IS_WINDOW (window)
      || wnck_window_get_window_type (window) == WNCK_WINDOW_DESKTOP)
    return FALSE;

  menu = wnck_action_menu_new (window);
  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL, 
                  event->button, event->time);
  return TRUE;
}


static gboolean
on_expose (GtkWidget *eb, GdkEventExpose *event)
{
  gtk_paint_box (eb->style, eb->window,
                 eb->state, GTK_SHADOW_NONE,
                 NULL, eb, "button",
                 0, 0, 
                 eb->allocation.width, eb->allocation.height);
  
  
  gtk_container_propagate_expose (GTK_CONTAINER (eb), 
                                  gtk_bin_get_child (GTK_BIN (eb)),
                                  event);
  return TRUE;
}

/* GObject stuff */
static void
task_title_finalize (GObject *object)
{
  TaskTitlePrivate *priv;

  priv = TASK_TITLE_GET_PRIVATE (object);

  G_OBJECT_CLASS (task_title_parent_class)->finalize (object);
}

static void
task_title_class_init (TaskTitleClass *klass)
{
  GObjectClass        *obj_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass      *wid_class = GTK_WIDGET_CLASS (klass);

  obj_class->finalize = task_title_finalize;
  wid_class->expose_event = on_expose;

  g_type_class_add_private (obj_class, sizeof (TaskTitlePrivate));
}

static void
task_title_init (TaskTitle *title)
{
  TaskTitlePrivate *priv;
    	
  priv = title->priv = TASK_TITLE_GET_PRIVATE (title);

  priv->screen = wnck_screen_get_default ();

  gtk_widget_add_events (GTK_WIDGET (title), GDK_ALL_EVENTS_MASK);

  priv->align = gtk_alignment_new (0.0, 0.5, 1.0, 1.0);
  gtk_alignment_set_padding (GTK_ALIGNMENT (priv->align), 
                             0, 0, 6, 6);
  gtk_container_add (GTK_CONTAINER (title), priv->align);

  priv->box = gtk_hbox_new (FALSE, 2);
  gtk_container_add (GTK_CONTAINER (priv->align), priv->box);
  gtk_widget_show (priv->box);

  priv->image = gtk_image_new_from_stock (GTK_STOCK_HOME, GTK_ICON_SIZE_MENU);
  gtk_box_pack_start (GTK_BOX (priv->box), priv->image, FALSE, FALSE, 0);
  gtk_widget_show (priv->image);

  priv->label = gtk_label_new (_("Home"));
  gtk_label_set_ellipsize (GTK_LABEL (priv->label), PANGO_ELLIPSIZE_END);
  gtk_misc_set_alignment (GTK_MISC (priv->label), 0.0, 0.5);
  gtk_box_pack_start (GTK_BOX (priv->box), priv->label, TRUE, TRUE, 0);
  gtk_widget_show (priv->label);

  priv->button = g_object_new (GTK_TYPE_EVENT_BOX, 
                               "visible-window", FALSE,
                               "above-child", TRUE,
                               NULL);
  gtk_box_pack_start (GTK_BOX (priv->box), priv->button, FALSE, FALSE, 0);
  gtk_widget_show (priv->button);
  g_signal_connect (priv->button, "button-release-event", 
                    G_CALLBACK (on_close_clicked), title);

  priv->button_image = gtk_image_new_from_stock (GTK_STOCK_QUIT, 
                                                 GTK_ICON_SIZE_MENU);
  gtk_container_add (GTK_CONTAINER (priv->button), priv->button_image);
  gtk_widget_show (priv->button_image);

  gtk_widget_set_tooltip_text (priv->button,
                               _("Log off, switch user, lock screen or power "
                                 "down the computer"));
  gtk_widget_set_tooltip_text (GTK_WIDGET (title), _("Home"));  

  gtk_widget_set_state (GTK_WIDGET (title), GTK_STATE_ACTIVE);
 

  gtk_widget_add_events (GTK_WIDGET (title), GDK_ALL_EVENTS_MASK);
 
  g_signal_connect (priv->screen, "active-window-changed",
                    G_CALLBACK (on_active_window_changed), title);
  g_signal_connect (title, "button-press-event",
                    G_CALLBACK (on_button_release), NULL);
}

GtkWidget *
task_title_new (void)

{
  GtkWidget *title = NULL;

  title = g_object_new (TASK_TYPE_TITLE, 
                        "border-width", 0, 
                        "name", "tasklist-button",
                        NULL);

  return title;
}
