# -*- coding: utf-8 -*-
# Elisa - Home multimedia server
# Copyright (C) 2006-2008 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Elisa with Fluendo's plugins.
#
# The GPL part of Elisa is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Elisa" in the root directory of this distribution package
# for details on that license.


__maintainer__ = 'Florian Boucault <florian@fluendo.com>'


from elisa.core.observers.list import ListObserver
from elisa.core import common
from elisa.extern.translation import Translatable
import constants

import pgm
from pgm.graph.group import Group
from pgm.graph.image import Image
from pgm.graph.text import Text
from pgm.timing import implicit

from raval_widgets.status_bar import StatusBar

class PathElement(Group):

    def __init__(self):
        Group.__init__(self)

        self._background = Image()
        self._background.fg_color = (255, 255, 255, 255)
        self._background.bg_color = (100, 200, 100, 0)
        self._background.size = (1.0, 1.0)
        self._background.layout = pgm.IMAGE_SCALED
 
        self._background.visible = True

        self._label = Text()
        self._label.font_family = "Bitstream DejaVu"
        self._label.weight = pgm.TEXT_WEIGHT_BOLD
        self._label.fg_color = (255, 255, 255, 255)
        self._label.bg_color = (100, 100, 200, 0)
        self._label.ellipsize = pgm.TEXT_ELLIPSIZE_END
        self._label.size = (1.0, 1.0)

        self.add(self._background)
        self.add(self._label)

        def loaded(image):
            ratio = float(image.aspect_ratio[0])/float(image.aspect_ratio[1])
            image.width = image.height*ratio

            margin = 0.20
            self._label.width = image.width*(1.0-margin)
            self._label.x = image.width*margin

            height = 2.0/3.0
            self._label.y = image.height*(1.0-height)/2.0
            self._label.height = image.height*height
            self._label.regenerate()

            self._label.z = 0.01
            self._label.visible = True

        self._background.connect("pixbuf-loaded", loaded)

    def label__set(self, label):
        self._label.label = label

    def background_file__set(self, background_file):
        self._background.set_from_file(background_file)

plugin_registry = common.application.plugin_registry
PigmentView = plugin_registry.get_component_class('pigment:pigment_view')

class LocationView(PigmentView, ListObserver):

    supported_controllers = ('raval:location_controller',)
    
    def clean(self):
        super(LocationView, self).clean()
        self._status_bar.canvas = None
        
    def create_widgets(self):
        theme = self.frontend.theme
        canvas = self.frontend.context.viewport_handle.get_canvas()
        
        self._status_bar = StatusBar()
        self._status_bar.background = theme.get_media("location_bg")
        
        self._status_bar.size = (canvas.width, canvas.height *
                                 (constants.status_bar_height/100))
        self._status_bar.y = canvas.height * \
            (constants.context_bar_height/100.0)-self._status_bar.height
        # -0.3 is chosen here because the first location path element is set at
        # -0.2. see raval_widgets/status_bar.py
        self._status_bar.z = -0.3
        self._status_bar.opacity = 0
        self._status_bar.visible = True
        
        self.context_handle.add(self._status_bar)
        
        self._animated_status_bar = implicit.AnimatedObject(self._status_bar)
        settings = {'duration': 500,
                    'transformation': implicit.DECELERATE}
        self._animated_status_bar.setup_next_animations(**settings)
        self._animated_status_bar.mode = implicit.REPLACE

    def inserted(self, elements, position):
        # append elements in the status bar
        for model in elements:
            item = self.create_item(model)
            self._status_bar.append(item)

        if len(self._status_bar) == 0:
            self._hide_location_bar()
        else:
            self._show_location_bar()

    def removed(self, elements, position):
        if len(self._status_bar) > 0:
            for model in elements:
                self._status_bar.pop(position)
            
        if len(self._status_bar) == 0:
            self._hide_location_bar()
        else:
            self._show_location_bar()

    def modified(self, position, value):
        print "modified", position, value
 
    def dirtied(self):
        print "dirtied"

    def controller_changed(self, old_controller, new_controller):
        if new_controller == None:
            return

        self.inserted(self.controller.model, 0)

    def create_item(self, model):
        theme = self.frontend.theme
        background_file = theme.get_media("location_item")
        item = PathElement()
        item.label = self.frontend.translate_text(model.text)
        item.background_file = background_file
        item.visible = True
        return item

    def _hide_location_bar(self):
        self.debug("Hiding location bar")
        canvas = self.frontend.context.viewport_handle.get_canvas()
        self._animated_status_bar.y = canvas.height * \
        (constants.context_bar_height/100.0)-self._status_bar.height
        self._animated_status_bar.opacity = 0
        
    def _show_location_bar(self):
        self.debug("Showing location bar")
        canvas = self.frontend.context.viewport_handle.get_canvas()
        self._animated_status_bar.y = canvas.height * \
        (constants.context_bar_height/100.0)
        self._animated_status_bar.opacity = 255
