/**
 * GMyth Library
 *
 * @file gmyth/gmyth_http.c
 * 
 * @brief <p> GMythHttp library provides a wrapper to access
 * data from the database using http+xml
 *
 * Copyright (C) 2007 INdT - Instituto Nokia de Tecnologia.
 * @author Artur Duque de Souza <artur.souza@indt.org.br>
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <assert.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/uri.h>

#include "gmyth_http.h"
#include "gmyth_debug.h"
#include "gmyth_socket.h"

xmlXPathObjectPtr
getnodeset(xmlDocPtr doc, xmlChar * xpath)
{

    xmlXPathContextPtr context;
    xmlXPathObjectPtr result;

    context = xmlXPathNewContext(doc);
    result = xmlXPathEvalExpression(xpath, context);

    if (xmlXPathNodeSetIsEmpty(result->nodesetval)) {
        g_fprintf(stderr, "Error: No result at XPath\n");
        return NULL;
    }

    xmlXPathFreeContext(context);
    return result;
}


xmlDocPtr
XMLParse(const char *content, int length)
{
    xmlDocPtr       doc;        /* the resulting document tree */

    doc = xmlReadMemory(content, length, NULL, NULL, 0);
    if (doc == NULL) {
        g_fprintf(stderr, "Error: Failed to parse XML document\n");
        return NULL;
    }

    return doc;
}

xmlXPathObjectPtr
getXPath(xmlChar * xpath, xmlDocPtr doc)
{
    xmlXPathObjectPtr result;

    result = getnodeset(doc, xpath);
    return result;
}


/** Retrieves the Progam List from the Channel
 *
 * @param nodeTab A pointer to a node inside the XML
 * @return A GSList containing a list of all the programs
 */
GSList         *
get_Program_List(xmlNodePtr node)
{
    GSList         *program_list = NULL;

    while (node != NULL) {
        if (g_ascii_strcasecmp((char *) node->name, "text") != 0) {
            GMythProgram   *program = (GMythProgram *)
                g_malloc(sizeof(struct _GMythProgram));

            program->title = g_strdup((char *)
                                      xmlGetProp(node,
                                                 (xmlChar *) "title"));

            program->subtitle = g_strdup((char *)
                                         xmlGetProp(node, (xmlChar *)
                                                    "subtitle"));

            program->catType = g_strdup((char *)
                                        xmlGetProp(node, (xmlChar *)
                                                   "catType"));

            program->category = g_strdup((char *)
                                         xmlGetProp(node, (xmlChar *)
                                                    "category"));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "repeat"),
                   "%d", &(program->repeat));

            program->startTime = gmyth_util_string_to_time_val
                ((char *) xmlGetProp(node, (xmlChar *) "startTime"));

            program->endTime = gmyth_util_string_to_time_val
                ((char *) xmlGetProp(node, (xmlChar *) "endTime"));

            program_list = g_slist_append(program_list, program);
        }

        node = node->next;
    }

    return program_list;
}

/** Retrieves the Channel List from the ProgramGuide
 *
 * @param node A pointer to a node inside the XML
 * @param epg The struct where is the current epg
 * @return The epg from "param" updated
 */
void
get_Channel_List(xmlNodePtr node, GMythEpg * epg)
{
    epg->channelList = NULL;

    while (node != NULL) {

        if (g_ascii_strcasecmp((char *) node->name, "text") != 0) {
            GMythChannel   *channel = (GMythChannel *) g_malloc
                (sizeof(struct _GMythChannel));

            channel->channelName = g_strdup((char *)
                                            xmlGetProp(node, (xmlChar *)
                                                       "channelName"));

            channel->chanNum = g_strdup((char *)
                                        xmlGetProp(node, (xmlChar *)
                                                   "chanNum"));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "chanId"),
                   "%d", &(channel->chanId));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "callSign"),
                   "%d", &(channel->callSign));

            channel->programList = get_Program_List(node->children);

            epg->channelList = g_slist_append(epg->channelList, channel);

        }

        node = node->next;
    }
}

/** Retrieves the properties from the ProgramGuide
 *
 * @param nodeTab A pointer to a node inside the XML
 * @param epg The struct where is the current epg
 * @return The epg from "param" updated
 */
void
get_ProgramGuide_Properties(xmlNodePtr nodeTab, GMythEpg * epg)
{

    xmlNode        *ptr = nodeTab->children->next->children;

    epg->startTime = gmyth_util_string_to_time_val((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    epg->endTime = gmyth_util_string_to_time_val((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->startChanId));

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->endChanId));

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->numOfChannels));

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->details));

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->totalCount));

    ptr = ptr->parent->next->next->children;
    epg->asOf = gmyth_util_string_to_time_val((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    epg->version = g_strdup((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(epg->protoVer));

    ptr = ptr->parent->next->next->children;
    // go to Channel section and retrieve Channels and Programs
    if (epg->numOfChannels > 0)
        get_Channel_List(ptr, epg);
    else
        epg->channelList = NULL;
}

/** Aux function to retrieve the Eletronic Program Guide
 *
 * @param doc An XML document (xmlDocPtr)
 * @return The epg
 */
void
getEpg(xmlDocPtr doc, GMythEpg * epg)
{
    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlChar        *keyword;

    int             i;

    result = getXPath((xmlChar *) "/*", doc);

    if (result) {
        nodeset = result->nodesetval;
        for (i = 0; i < nodeset->nodeNr; i++) {
            keyword = (xmlChar *) nodeset->nodeTab[i]->name;
            if (g_ascii_strcasecmp
                ((char *) keyword, "GetProgramGuideResponse") == 0) {
                get_ProgramGuide_Properties(nodeset->nodeTab[i], epg);
                break;
            }
        }
        xmlXPathFreeObject(result);
    }

}



/** Retrieves the Eletronic Program Guide from the backend
 *
 * @param doc An XML document (xmlDocPtr)
 * @return The epg
 */
GMythEpg
gmyth_http_retrieve_epg(GMythBackendInfo * backend_info,
                        GTimeVal * StartTime, GTimeVal * EndTime,
                        gint StartChanId, gint NumOfChannels,
                        gchar * Details)
{
    GMythEpg        epg;
    MemoryStruct    chunk;

    chunk.memory = NULL;        /* we expect realloc(NULL, size) to work */
    chunk.size = 0;             /* no data at this point */

    gchar          *starttime;

    starttime = (gchar *) xmlURIEscapeStr((const xmlChar *)
                                          gmyth_util_time_to_mythformat_from_time_val
                                          (StartTime), NULL);

    gchar          *endtime;

    endtime = (gchar *) xmlURIEscapeStr((const xmlChar *)
                                        gmyth_util_time_to_mythformat_from_time_val
                                        (EndTime), NULL);

    GString        *command = g_string_new("");

    g_string_printf(command,
                    "GetProgramGuide?StartTime=%s&EndTime=%s&StartChanId=%d"
                    "&NumOfChannels=%d&Details=%s", starttime, endtime,
                    StartChanId, NumOfChannels, Details);
    gmyth_debug("HTTP Request command = %s\n", command->str);

    chunk = gmyth_http_request(backend_info, command);
    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        getEpg(doc, &epg);
        free(chunk.memory);
    }

    return epg;
}


GMythRecorded_Recording
retrieve_recorded_recording(xmlNodePtr node)
{
    GMythRecorded_Recording recording;

    if (g_ascii_strcasecmp((char *) node->name, "text") != 0) {

        sscanf((char *) xmlGetProp(node, (xmlChar *) "dupInType"),
               "%d", &(recording.dupInType));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "dupMethod"),
               "%d", &(recording.dupMethod));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "recStatus"),
               "%d", &(recording.recStatus));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "encoderId"),
               "%d", &(recording.encoderId));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "recordId"),
               "%d", &(recording.recordId));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "recType"),
               "%d", &(recording.recType));

        recording.playGroup = g_strdup((char *)
                                       xmlGetProp(node, (xmlChar *)
                                                  "playGroup"));

        recording.recGroup = g_strdup((char *)
                                      xmlGetProp(node,
                                                 (xmlChar *) "recGroup"));

        recording.recProfile = g_strdup((char *)
                                        xmlGetProp(node, (xmlChar *)
                                                   "recProfile"));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "recPriority"),
               "%d", &(recording.recPriority));

        recording.recStartTs = gmyth_util_string_to_time_val
            ((char *) xmlGetProp(node, (xmlChar *) "recStartTs"));

        recording.recEndTs = gmyth_util_string_to_time_val
            ((char *) xmlGetProp(node, (xmlChar *) "recEndTs"));
    }

    return recording;
}


GMythRecorded_Channel
retrieve_recorded_channel(xmlNodePtr node)
{
    GMythRecorded_Channel channel;

    if (g_ascii_strcasecmp((char *) node->name, "text") != 0) {

        channel.chanFilters = g_strdup((char *)
                                       xmlGetProp(node, (xmlChar *)
                                                  "chanFilters"));

        channel.channelName = g_strdup((char *)
                                       xmlGetProp(node, (xmlChar *)
                                                  "channelName"));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "chanNum"),
               "%d", &(channel.chanNum));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "sourceId"),
               "%d", &(channel.sourceId));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "commFree"),
               "%d", &(channel.commFree));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "inputId"),
               "%d", &(channel.inputId));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "chanId"),
               "%d", &(channel.chanId));

        sscanf((char *) xmlGetProp(node, (xmlChar *) "callSign"),
               "%d", &(channel.callSign));
    }

    return channel;
}



/** Retrieves all the programs from Recorded XML
 *
 * @param nodeTab A pointer to a node inside the XML
 * @param recorded The struct where is the current epg
 * @return list with all the recorded programs
 */
GSList         *
get_Recorded_Programs(xmlNodePtr node)
{
    GSList         *programList = NULL;

    while (node != NULL) {

        if (g_ascii_strcasecmp((char *) node->name, "text") != 0) {

            GMythRecorded_Program *program = (GMythRecorded_Program *)
                g_malloc(sizeof(struct _GMythRecorded_Program));

            sscanf((char *)
                   xmlGetProp(node, (xmlChar *) "programFlags"), "%d",
                   &(program->programFlags));

            program->title = g_strdup((char *)
                                      xmlGetProp(node,
                                                 (xmlChar *) "title"));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "programId"),
                   "%d", &(program->programId));

            program->catType = g_strdup((char *)
                                        xmlGetProp(node, (xmlChar *)
                                                   "catType"));

            program->category = g_strdup((char *)
                                         xmlGetProp(node, (xmlChar *)
                                                    "category"));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "seriesId"),
                   "%d", &(program->seriesId));


            program->startTime = gmyth_util_string_to_time_val
                ((char *) xmlGetProp(node, (xmlChar *) "startTime"));

            program->endTime = gmyth_util_string_to_time_val
                ((char *) xmlGetProp(node, (xmlChar *) "endTime"));

            program->lastModified = gmyth_util_string_to_time_val((char *)
                                                                  xmlGetProp
                                                                  (node,
                                                                   (xmlChar
                                                                    *)
                                                                   "lastModified"));

            /*
             * TODO: FIX ME at gmyth_util program->asOf =
             * gmyth_util_string_to_time_val\ ((char *)xmlGetProp(node,
             * (xmlChar *)"airdate")); 
             */

            program->subTitle = g_strdup((char *)
                                         xmlGetProp(node, (xmlChar *)
                                                    "subTitle"));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "stars"),
                   "%d", &(program->stars));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "repeat"),
                   "%d", &(program->repeat));

            sscanf((char *) xmlGetProp(node, (xmlChar *) "fileSize"),
                   "%d", &(program->repeat));

            program->hostname = g_strdup((char *)
                                         xmlGetProp(node, (xmlChar *)
                                                    "hostname"));

            program->channel = retrieve_recorded_channel(node->children);

            // Skip the \n
            program->recording =
                retrieve_recorded_recording(node->children->next->next);

            // add to the list
            programList = g_slist_append(programList, program);
        }

        node = node->next;
    }

    return programList;
}

/** Retrieves the properties from Recorded XML
 *
 * @param nodeTab A pointer to a node inside the XML
 * @param recorded The struct where is the current epg
 * @return "recorded" from "param" updated
 */
void
get_Recorded_Properties(xmlNodePtr nodeTab, GMythRecorded * recorded)
{
    xmlNode        *ptr = nodeTab->children->next->children;

    sscanf((char *) ptr->content, "%d", &(recorded->totalCount));


    ptr = ptr->parent->next->next->children;
    recorded->asOf = gmyth_util_string_to_time_val((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    recorded->version = g_strdup((char *) ptr->content);

    ptr = ptr->parent->next->next->children;
    sscanf((char *) ptr->content, "%d", &(recorded->protoVer));

    ptr = ptr->parent->next->next->children;
    if (recorded->totalCount > 0)
        recorded->programList = get_Recorded_Programs(ptr->children);

}


/** Aux function to retrieve Recorded programs
 *
 * @param doc An XML document (xmlDocPtr)
 * @return The recorded var updated
 */
void
getRecorded(xmlDocPtr doc, GMythRecorded * recorded)
{
    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlChar        *keyword;

    int             i;

    result = getXPath((xmlChar *) "/*", doc);

    if (result) {
        nodeset = result->nodesetval;
        for (i = 0; i < nodeset->nodeNr; i++) {
            keyword = (xmlChar *) nodeset->nodeTab[i]->name;
            if (g_ascii_strcasecmp
                ((char *) keyword, "GetRecordedResponse") == 0) {
                get_Recorded_Properties(nodeset->nodeTab[i], recorded);
                break;
            }
        }
        xmlXPathFreeObject(result);
    }

}


/** Function to retrieve the files that are recorded
 *
 */
GMythRecorded
gmyth_http_retrieve_recorded(GMythBackendInfo * backend_info)
{
    GMythRecorded   recorded;
    MemoryStruct    chunk;

    chunk.memory = NULL;
    chunk.size = 0;

    GString        *command = g_string_new("");

    g_string_printf(command, "GetRecorded");

    chunk = gmyth_http_request(backend_info, command);
    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        getRecorded(doc, &recorded);
        free(chunk.memory);
    }

    return recorded;
}



/** Function to retrieve jobqueue status
 *
 */
gint
gmyth_http_retrieve_job_status(GMythBackendInfo * backend_info,
                               gint chanid, GTimeVal * start)
{
    gint            status = 0;
    gint            count = 0;
    gint            temp_chanid = 0;
    GTimeVal       *temp_start = NULL;
    int             i;

    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlNodePtr      node;
    MemoryStruct    chunk;

    chunk.memory = NULL;
    chunk.size = 0;

    GString        *command = g_string_new("");

    g_string_printf(command, "GetStatus");

    chunk = gmyth_http_request(backend_info, command);

    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        result = getXPath((xmlChar *) "/Status/JobQueue", doc);
        if (result) {
            nodeset = result->nodesetval;
            node = nodeset->nodeTab[0];
            sscanf((char *) xmlGetProp(node, (xmlChar *) "count"),
                   "%d", &count);

            if (count > 0) {

                // Get the first child
                node = node->children->next;

                for (i = 0; i < count; i++) {

                    sscanf((char *)
                           xmlGetProp(node, (xmlChar *) "chanId"), "%d",
                           &temp_chanid);

                    if (chanid == temp_chanid) {
                        temp_start = gmyth_util_string_to_time_val((char *)
                                                                   xmlGetProp
                                                                   (node,
                                                                    (xmlChar
                                                                     *)
                                                                    "startTime"));

                        if ((temp_start->tv_sec == start->tv_sec) &&
                            (temp_start->tv_usec == start->tv_usec))
                            sscanf((char *)
                                   xmlGetProp(node,
                                              (xmlChar *) "status"),
                                   "%d", &status);
                    }
                    // Escape "text" node
                    node = node->next->next;
                }
            }

        }

        xmlXPathFreeObject(result);
        free(chunk.memory);

    }

    return status;
}



/** Function to retrieve settings on the backend
 *
 * @param backend_info infos about the backend
 * @param key the key you want to retrieve
 * @param hostname the hostname that the key is set up
 * @return the value of the key
 */
gchar          *
gmyth_http_retrieve_setting(GMythBackendInfo * backend_info,
                            gchar * key, gchar * hostname)
{
    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlChar        *keyword;
    MemoryStruct    chunk;
    gchar          *value = NULL;

    chunk.memory = NULL;
    chunk.size = 0;

    GString        *command = g_string_new("");

    g_string_printf(command, "GetSetting?Key=%s&HostName=%s&Default=NULL",
                    key, hostname);

    chunk = gmyth_http_request(backend_info, command);

    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        result = getXPath((xmlChar *) "/GetSettingResponse/Values/*", doc);

        if (result) {
            nodeset = result->nodesetval;
            keyword = (xmlChar *) nodeset->nodeTab[0]->name;
            if (g_ascii_strcasecmp((char *) keyword, "Value") == 0) {
                // Here we have the value
                value = (gchar *) nodeset->nodeTab[0]->children->content;
            }
            xmlXPathFreeObject(result);
        }

        free(chunk.memory);
    }

    return value;
}

/** Common steps for rec_profile's functions
 *
 * @param backend_info infos about the backend
 * @param id the profile's id that you want to delete
 * @return 0 if OK
 */
gint
rec_profile_common(GMythBackendInfo * backend_info, GString * command)
{
    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlChar        *keyword;
    MemoryStruct    chunk;

    chunk.memory = NULL;
    chunk.size = 0;

    int             ret = -1;

    chunk = gmyth_http_request(backend_info, command);

    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        result = getXPath((xmlChar *) "/*", doc);

        if (result) {
            nodeset = result->nodesetval;
            keyword = (xmlChar *) nodeset->nodeTab[0]->name;

            if (g_ascii_strcasecmp((char *) keyword, "Success") == 0)
                ret = 0;

            xmlXPathFreeObject(result);
        }

        free(chunk.memory);
    }

    return ret;
}


/** Function to delete recording profiles
 *
 * @param backend_info infos about the backend
 * @param id the profile's id that you want to delete
 * @return 0 if OK
 */
gint
gmyth_http_del_rec_profile(GMythBackendInfo * backend_info, gint id)
{

    GString        *command = g_string_new("");

    g_string_printf(command, "delRecProfiles?id=%d", id);


    return rec_profile_common(backend_info, command);
}

/** Function to create recording profiles
 *
 * @param backend_info infos about the backend
 * @param profilename the name of profile you want to use
 * @param groupname the name of groupname you want to use
 * @param vcodec the name of the video codec you want to use
 * @param acodec the name of the audo codec you want to use
 * @return 0 if OK
 */
gint
gmyth_http_create_rec_profile(GMythBackendInfo * backend_info,
                              GMythRecProfile * profile)
{

    if (profile->name != NULL && profile->group != NULL &&
        profile->vcodec && profile->acodec && profile->options != NULL) {
        GString        *command = g_string_new("");

        g_string_printf(command, "createRecProfiles?profilename=%s&"
                        "groupname=%s&vcodec=%s&acodec=%s&"
                        "transcodelossless=%d&transcoderesize=%d&"
                        "width=%d&height=%d&rtjpegquality=%d&"
                        "rtjpeglumafilter=%d&rtjpegchromafilter=%d&"
                        "mpeg4bitrate=%d&mpeg4maxquality=%d&"
                        "mpeg4minquality=%d&mpeg4qualdiff=%d&"
                        "mpeg4scalebitrate=%d&mpeg4optionvhq=%d&"
                        "mpeg4option4mv=%d&mpeg4optionidct=%d&"
                        "mpeg4optionime=%d&hardwaremjpegquality=%d&"
                        "hardwaremjpeghdecimation=%d&hardwaremjpegvdecimation=%d&"
                        "mpeg2streamtype=%s&mpeg2aspectratio=%s&"
                        "mpeg2bitrate=%d&mpeg2maxbitrate=%d&"
                        "samplerate=%d&mp3quality=%d&"
                        "volume=%d&mpeg2audtype=%s&"
                        "mpeg2audbitratel1=%d&mpeg2audbitratel2=%d&"
                        "mpeg2audvolume=%d",
                        profile->name, profile->group,
                        profile->vcodec, profile->acodec,
                        profile->options->transcodelossless,
                        profile->options->transcoderesize,
                        profile->options->width,
                        profile->options->height,
                        profile->options->rtjpegquality,
                        profile->options->rtjpeglumafilter,
                        profile->options->rtjpegchromafilter,
                        profile->options->mpeg4bitrate,
                        profile->options->mpeg4maxquality,
                        profile->options->mpeg4minquality,
                        profile->options->mpeg4qualdiff,
                        profile->options->mpeg4scalebitrate,
                        profile->options->mpeg4optionvhq,
                        profile->options->mpeg4option4mv,
                        profile->options->mpeg4optionidct,
                        profile->options->mpeg4optionime,
                        profile->options->hardwaremjpegquality,
                        profile->options->hardwaremjpeghdecimation,
                        profile->options->hardwaremjpegvdecimation,
                        profile->options->mpeg2streamtype,
                        profile->options->mpeg2aspectratio,
                        profile->options->mpeg2bitrate,
                        profile->options->mpeg2maxbitrate,
                        profile->options->samplerate,
                        profile->options->mp3quality,
                        profile->options->volume,
                        profile->options->mpeg2audtype,
                        profile->options->mpeg2audbitratel1,
                        profile->options->mpeg2audbitratel2,
                        profile->options->mpeg2audvolume);


        return rec_profile_common(backend_info, command);
    } else
        return -1;
}

/** Function to retrieve recording profiles
 *
 * @param backend_info infos about the backend
 * @param groupname the name of group you want to retrieve
 * @return the list of profiles
 */
GSList         *
gmyth_http_retrieve_rec_profiles(GMythBackendInfo * backend_info,
                                 gchar * groupname)
{
    xmlXPathObjectPtr result;
    xmlNodeSetPtr   nodeset;
    xmlChar        *keyword;
    MemoryStruct    chunk;
    GSList         *profiles = NULL;

    chunk.memory = NULL;
    chunk.size = 0;

    GString        *command = g_string_new("");

    g_string_printf(command, "GetRecProfiles?groupname=%s", groupname);

    chunk = gmyth_http_request(backend_info, command);

    if (chunk.memory != NULL) {
        xmlDocPtr       doc = XMLParse(chunk.memory, strlen(chunk.memory));

        result = getXPath((xmlChar *) "/*", doc);

        if (result) {
            nodeset = result->nodesetval;
            keyword = (xmlChar *) nodeset->nodeTab[0]->name;

            if (g_ascii_strcasecmp((char *) keyword, "Profiles") == 0) {
                xmlNodePtr      node = nodeset->nodeTab[0]->children->next;
                GMythRecProfile *profile;

                while (node != NULL) {
                    if (g_ascii_strcasecmp((char *) node->name, "text") !=
                        0) {
                        profile = gmyth_recprofile_new();

                        sscanf((char *) xmlGetProp(node, (xmlChar *)
                                                   "id"), "%d",
                               &(profile->id));

                        profile->name = g_strdup((char *)
                                                 xmlGetProp(node,
                                                            (xmlChar *)
                                                            "name"));

                        profile->vcodec = g_strdup((char *)
                                                   xmlGetProp(node,
                                                              (xmlChar *)
                                                              "vcodec"));

                        profile->acodec = g_strdup((char *)
                                                   xmlGetProp(node,
                                                              (xmlChar *)
                                                              "acodec"));

                        profile->group = g_strdup(groupname);

                        profiles = g_slist_append(profiles, profile);
                    }
                    node = node->next;
                }

            }
            xmlXPathFreeObject(result);
        }

        free(chunk.memory);
    }

    return profiles;
}



/*
 * Aux functions got from libcurl 
 */
void           *
myrealloc(void *ptr, size_t size)
{
    /*
     * There might be a realloc() out there that doesn't like reallocing
     * NULL pointers, so we take care of it here 
     */
    if (ptr)
        return realloc(ptr, size);
    else
        return malloc(size);
}

size_t
WriteMemoryCallback(void *ptr, size_t size, size_t nmemb, void *data)
{
    size_t          realsize = size * nmemb;
    MemoryStruct   *mem = (struct _MemoryStruct *) data;

    mem->memory =
        (char *) myrealloc(mem->memory, mem->size + realsize + 1);
    if (mem->memory) {
        memcpy(&(mem->memory[mem->size]), ptr, realsize);
        mem->size += realsize;
        mem->memory[mem->size] = 0;
    }

    return realsize;
}


/** Send HTTP Command and receives the result of it
 *
 * @return A string with the response from the server
 *          NULL if there is no response.
 */
MemoryStruct
gmyth_http_request(GMythBackendInfo * backend_info, GString * command)
{
    LIBXML_TEST_VERSION
        size_t size =
        strlen(backend_info->hostname) + strlen(command->str) + 20;

    gchar          *URL = (gchar *) g_malloc(sizeof(gchar) * size);
    gchar          *mid = (gchar *) g_malloc(sizeof(gchar) * 6);

    mid = "";

    if (g_ascii_strcasecmp(command->str, "GetStatus") &&
        g_ascii_strcasecmp(command->str, "GetStatusHTML")) {
        mid = "Myth/";
    }

    g_snprintf(URL, size, "http://%s:%d/%s%s",
               backend_info->hostname, backend_info->status_port, mid,
               command->str);

    CURL           *curl_handle;

    MemoryStruct    chunk;

    chunk.memory = NULL;        /* we expect realloc(NULL, size) to work */
    chunk.size = 0;             /* no data at this point */

    curl_global_init(CURL_GLOBAL_ALL);

    /*
     * init the curl session 
     */
    curl_handle = curl_easy_init();

    /*
     * specify URL to get 
     */
    curl_easy_setopt(curl_handle, CURLOPT_URL, URL);

    /*
     * send all data to this function 
     */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEFUNCTION,
                     WriteMemoryCallback);

    /*
     * we pass our 'chunk' struct to the callback function 
     */
    curl_easy_setopt(curl_handle, CURLOPT_WRITEDATA, (void *) &chunk);

    /*
     * some servers don't like requests that are made without a user-agent
     * field, so we provide one 
     */
    curl_easy_setopt(curl_handle, CURLOPT_USERAGENT, "libcurl-agent/1.0");

    /*
     * set timeout 
     */
    curl_easy_setopt(curl_handle, CURLOPT_CONNECTTIMEOUT, 20);

    /*
     * get it! 
     */
    curl_easy_perform(curl_handle);

    /*
     * cleanup curl stuff 
     */
    curl_easy_cleanup(curl_handle);

    return chunk;
}
