/**
  @file btcond-hci.c

  Functions for monitoring BT HCI events, and sending
  BT_ON/BT_OFF HCI commands.

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2006 Nokia Corporation. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA

*/  
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <termios.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>

#include <bluetooth/bluetooth.h>
#include <bluetooth/hci.h>
#include <bluetooth/hci_lib.h>

#include <glib.h>

#include "log.h"
#include "btcond-bt.h"
#include "state.h"
#include "bt-error.h"
#include "btcond-signals.h"
#include "btcond-hci.h"

/* hci0 */
#define DEV_NUMBER 0

static int hci_sock = -1;
static int dev_sock = -1;

static int device_socket(GError **err)
{
    struct sockaddr_hci addr;
    struct hci_filter flt;
    int sock;

    /* Create HCI socket */
    sock = socket(AF_BLUETOOTH, SOCK_RAW, BTPROTO_HCI);
    if (sock < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't create HCI socket: %s (%d)",
                    g_strerror(errno), errno);
        return -1;
    }

    /* Setup filter */
    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);
    hci_filter_set_event(EVT_STACK_INTERNAL, &flt);

    if (setsockopt(sock, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't set HCI filter: %s (%d)",
                    g_strerror(errno), errno);
        return -1;
    }

    /* Bind socket to the HCI device */
    addr.hci_family = AF_BLUETOOTH;
    addr.hci_dev = HCI_DEV_NONE;
    if (bind(sock, (struct sockaddr *)&addr, sizeof(addr)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't bind hci socket: %s (%d)",
                    g_strerror(errno), errno);
        return -1;
    }

    return sock;
}

static con_evt_t *new_con_evt(uint8_t type) {
    con_evt_t *evt = g_new0(con_evt_t, 1);
    evt->type = type;
    return evt;
}

static void connection_complete(char *data, con_evt_t **event)
{
    evt_conn_complete *evt = (evt_conn_complete *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_CC);
    (*event)->handle   = evt->handle;
    (*event)->status   = evt->status;
    (*event)->bda      = g_strdup(addr);

    if (evt->status != 0x00) {
        debug("Connection failed. status: 0x%02x", evt->status);
        if (connection_status(addr) != CONN_STATUS_CONNECTED) {
            debug("Marking device capability as old");
            expire_device_capability(addr);
        }
    }
}

static void pin_request(char *data, con_evt_t **event)
{
    evt_pin_code_req *evt = (evt_pin_code_req *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_PIN_REQ);
    (*event)->bda = g_strdup(addr);
}

static void link_key_notify(char *data, con_evt_t **event)
{
    evt_link_key_notify *evt = (evt_link_key_notify *)data;
    char addr[18];

    ba2str(&evt->bdaddr, addr);

    *event = new_con_evt(EVT_TYPE_LINK_KEY);
    (*event)->bda = g_strdup(addr);
}

static void auth_complete(char *data, con_evt_t **event)
{
    evt_auth_complete *evt = (evt_auth_complete *)data;

    debug("auth_complete(): handle=%u, status=0x%02X",
            evt->handle, evt->status);

    if (evt->status != 0x00) {
        const gchar *bda;

        bda = get_handle_bda(evt->handle);
        if (bda == NULL) {
            error("auth_complete: Unable to get BDA for handle %u",
                  g_ntohs(evt->handle));
            return;
        }

        *event = new_con_evt(EVT_TYPE_AUTH_FAILED);
        (*event)->status = evt->status;
        (*event)->bda = g_strdup(bda);
    }
}

static void disconnection_complete(char *data, con_evt_t **event)
{
    evt_disconn_complete *evt = (evt_disconn_complete *)data;

    if (evt->status != 0x00) {
        debug("disconnection complete error. status: 0x%02x", evt->status);
        return;
    }

    if (!handle_is_connected(evt->handle)) {
        debug("disconnection of unknown connection handle (%u)", evt->handle);
        return;
    }

    *event = new_con_evt(EVT_TYPE_DC);
    (*event)->handle = evt->handle;
    (*event)->bda = g_strdup(get_handle_bda(evt->handle));

    (*event)->dc_reason = evt->reason;

    if (evt->reason == HCI_CONNECTION_TERMINATED)
        debug("BT connection to %s was closed by local event", (*event)->bda);
    else {
        switch (evt->reason) {
            case HCI_OE_USER_ENDED_CONNECTION:
                debug("Connection closed by %s", (*event)->bda);
                break;
            case HCI_CONNECTION_TIMEOUT:
                debug("Supervision timeout to %s. Device out of rage?",
                        (*event)->bda);
                break;
            default:
                debug("Disconnection Complete from %s, reason: 0x%02x",
                        (*event)->bda, evt->reason);
                break;
        }
        debug("Marking device capability as old");
        expire_device_capability((*event)->bda);
    }
}

/* Bluez does not provide this definition */
#define EVT_HW_ERROR 0x10

static con_evt_t *process_data(char *data, int data_len)
{
    uint8_t type = data[0];
    con_evt_t *event = NULL;

    if (type == HCI_EVENT_PKT) {
        hci_event_hdr *hdr = (hci_event_hdr *) &data[1];
        char *body = &data[HCI_EVENT_HDR_SIZE + 1];

        switch (hdr->evt) {
            case EVT_CONN_COMPLETE:
                connection_complete(body, &event);
                break;
            case EVT_DISCONN_COMPLETE:
                disconnection_complete(body, &event);
                break;
            case EVT_PIN_CODE_REQ:
                pin_request(body, &event);
                break;
            case EVT_LINK_KEY_NOTIFY:
                link_key_notify(body, &event);
                break;
            case EVT_AUTH_COMPLETE:
                auth_complete(body, &event);
                break;
            case EVT_HW_ERROR:
                bt_hw_error(*body);
                break;
            default:
                error("Invalid event type (0x%02x)", hdr->evt);
                break;
        }
    }
    else
        error("Invalid packet type (0x%02x)", type);

    return event;
}

static void free_con_evt(con_evt_t *evt)
{
    if (evt->bda)
        g_free(evt->bda);
    g_free(evt);
}

static void device_event(evt_stack_internal *si)
{
    evt_si_device *sd = (void *) &si->data;
    char dev[6];

    snprintf(dev, sizeof(dev), "hci%d", sd->dev_id);

     switch (sd->event) {
         case HCI_DEV_REG:
             report("HCI dev %d registered", sd->dev_id);
             break;

         case HCI_DEV_UNREG:
             report("HCI dev %d unregistered", sd->dev_id);
             break;

         case HCI_DEV_UP:
             report("HCI dev %d up", sd->dev_id);
             update_dev_state(sd->dev_id, FALSE);
             send_dbus_hci_dev_up(dev);
             break;

         case HCI_DEV_DOWN:
             report("HCI dev %d down", sd->dev_id);
             update_dev_state(sd->dev_id, TRUE);
             send_dbus_hci_dev_down(dev);
             break;
     }
}

static gboolean socket_cb(GIOChannel *chan, GIOCondition cond, gpointer data)
{
    con_evt_t *event;
    hci_event_hdr *eh;
    char buf[HCI_MAX_EVENT_SIZE], *ptr;
    int buf_len, sock, type;

    ptr = buf;

    if (cond != G_IO_IN) {
        error("HUP or error on HCI socket. Unable to receive more HCI events.");
        g_io_channel_unref(chan);
        return FALSE;
    }
       
    sock = g_io_channel_unix_get_fd(chan);
    g_assert(sock >= 0);

    buf_len = read(sock, buf, sizeof(buf));
    if (buf_len < 0) {
        error("reading HCI socket failed: %s", g_strerror(errno));
        g_io_channel_unref(chan);
        return FALSE;
    }

    type = *ptr++;

    if (type != HCI_EVENT_PKT)
        return TRUE;

    eh = (hci_event_hdr *) ptr;
    if (eh->evt == EVT_STACK_INTERNAL) {
        evt_stack_internal *si = (evt_stack_internal *) (ptr + HCI_EVENT_HDR_SIZE);
        if (si->type == EVT_SI_DEVICE)
            device_event(si);
        return TRUE;
    }

    event = process_data(buf, buf_len);
    if (event != NULL) {
        switch (event->type) {
            case EVT_TYPE_CC:
                if (event->status == 0)
                    update_state(event);
                else
                    send_dbus_cc_failed(event);
                break;
            case EVT_TYPE_DC:
                update_state(event);
                break;
            case EVT_TYPE_PIN_REQ:
                send_dbus_pin_req(event);
                break;
            case EVT_TYPE_LINK_KEY:
                send_dbus_link_key_ok(event);
                break;
            case EVT_TYPE_AUTH_FAILED:
                send_dbus_auth_failed(event);
                break;
            default:
                debug("Invalid event type (0x%02x)", event->type);
                break;
        }

        free_con_evt(event);
    }

    return TRUE;
}

gboolean monitor_connections(GError **err)
{
    struct hci_filter flt;
    GIOChannel *gio;

    dev_sock = device_socket(err);
    if (dev_sock < 0)
        return FALSE;

    gio = g_io_channel_unix_new(dev_sock);
    g_io_channel_set_close_on_unref(gio, TRUE);
    g_io_add_watch(gio, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                   socket_cb, NULL);

    hci_sock = hci_open_dev(DEV_NUMBER);
    if (hci_sock < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "hci_open_dev(%d): %s (%d)",
                    DEV_NUMBER, g_strerror(errno), errno);
        g_io_channel_unref(gio);
        dev_sock = -1;
        return FALSE;
    }

    hci_filter_clear(&flt);
    hci_filter_set_ptype(HCI_EVENT_PKT, &flt);
    hci_filter_set_event(EVT_CONN_COMPLETE, &flt);
    hci_filter_set_event(EVT_DISCONN_COMPLETE, &flt);
    hci_filter_set_event(EVT_PIN_CODE_REQ, &flt);
    hci_filter_set_event(EVT_LINK_KEY_NOTIFY, &flt);
    hci_filter_set_event(EVT_AUTH_COMPLETE, &flt);
    hci_filter_set_event(EVT_HW_ERROR, &flt);

    if (setsockopt(hci_sock, SOL_HCI, HCI_FILTER, &flt, sizeof(flt)) < 0) {
        g_set_error(err, BT_COND_ERROR,
                    BT_COND_ERROR_FAILED,
                    "Can't set HCI filter: %s (%d)",
                    g_strerror(errno), errno);
        g_io_channel_unref(gio);
        close(hci_sock);
        hci_sock = -1;
    }

    gio = g_io_channel_unix_new(hci_sock);
    g_io_channel_set_close_on_unref(gio, TRUE);
    g_io_add_watch(gio, G_IO_IN | G_IO_ERR | G_IO_HUP | G_IO_NVAL,
                   socket_cb, NULL);

    return TRUE;
}

gboolean hci_bt_up(void)
{
    if (ioctl(dev_sock, HCIDEVUP, DEV_NUMBER) < 0) {
        if (errno == EALREADY)
            return TRUE;

        error("HCIDEVUP for hci%d failed: %s (%d)", DEV_NUMBER,
              g_strerror(errno), errno);

        return FALSE;
    }

    return TRUE;
}

gboolean hci_bt_down(void)
{
    if (ioctl(dev_sock, HCIDEVDOWN, DEV_NUMBER) < 0) {
        error("HCIDEVDOWN for hci%d failed: %s (%d)", DEV_NUMBER,
              g_strerror(errno), errno);
        return FALSE;
    }

    return TRUE;
}
