/*
 * tagged collection - Experimental programs to test and study tagged collections
 *
 * Copyright (C) 2003,2004,2005,2006  Enrico Zini
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Index location is (in order of importance):
 *  1. The directory specified with the -I commandline option
 *  2. The directory specified with the TAGIDX environment variable
 *  3. ~/.tagcoll/index
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
#warning No config.h found: using fallback values
#define VERSION "unknown"
#endif

#define APPNAME "tagidx"

#include "TagidxParser.h"

#include <BasicStringDiskIndex.h>
#include <tagcoll/coll/fast.h>
#include <tagcoll/coll/patched.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <pwd.h>
#include <dirent.h>
#include <time.h>
#include <stdlib.h>	// getenv
#include <errno.h>

#include <tagcoll/input/stdio.h>
#include <tagcoll/TextFormat.h>

#include <iostream>
#include <sstream>

#if 0
#include <tagcoll/Consumer.h>
#include <tagcoll/Filter.h>
#include <tagcoll/InputMerger.h>
#include <tagcoll/Implications.h>
#include <tagcoll/Filters.h>
#include <tagcoll/Patches.h>
#include <tagcoll/DerivedTags.h>
#include <tagcoll/Expression.h>
#include <algorithm>
#endif

using namespace std;
using namespace tagcoll;

string get_username()
{
	struct passwd* pw = getpwuid(getuid());
	if (pw != NULL)
		return pw->pw_name;
	else
		return string();
}

void mkpath(const std::string& dir)
{
	size_t sep = dir.rfind('/');

	//cerr << "Ensuring existance of " << dir << endl;

	// First ensure that the parent directory exists
	if (sep != string::npos && sep != 0)
		mkpath(dir.substr(0, sep));

	// Then create this directory if needed
	struct stat st;
	if (stat(dir.c_str(), &st) == -1)
	{
		if (errno == ENOENT)
		{
			if (mkdir(dir.c_str(), 0777) == -1)
				throw wibble::exception::System("creating directory " + dir);
		} else
			throw wibble::exception::System("getting informations about " + dir);
	} else if (!S_ISDIR(st.st_mode))
		throw wibble::exception::Consistency("performing sanity checks on path components",
				dir + " is not a directory");
}


#if 0
void printItems(const set<string>& items, const string& prefix = "")
{
	for (set<string>::const_iterator i = items.begin();
			i != items.end(); i++)
	{
		printf("%.*s%.*s\n", PFSTR(prefix), PFSTR(*i));
	}
}

void printNode(HierarchyNode<string, string>* node, string prefix = "")
{
	OpSet<string> items = node->getItems();
	if (!items.empty())
	{
		printItems(items, prefix + ": ");
//	} else {
//		printf("%.*s: no items\n", PFSTR(prefix));
	}
	
	if (prefix.size() > 0 && prefix[prefix.size() - 1] != '/')
		prefix += '/';

	for (HierarchyNode<string, string>::iterator i = node->begin();
			i != node->end(); i++)
	{
		printNode(*i, prefix + (*i)->tag());
	}
}

OpSet<string> getItems(HierarchyNode<string, string>* node)
{
	OpSet<string> items = node->getItems();
	
	for (HierarchyNode<string, string>::iterator i = node->begin();
			i != node->end(); i++)
		items += getItems(*i);

	return items;
}
#endif

template<typename OUT>
void readCollection(const string& file, const OUT& out)
{
	if (file == "-")
	{
		input::Stdio input(stdin, "<stdin>");
		textformat::parse(input, out);
	}
	else
	{
		input::Stdio input(file);
		textformat::parse(input, out);
	}
}

PatchList<string, string> readPatchFromFile(const std::string& filename)
{
	TrivialConverter<string, string> conv;
	StdioParserInput input(filename);
	return TextFormat<string, string>::parsePatch(conv, conv, input);

}

/**
 * Only read files that do not start with a '.' and are not regular files or
 * symlinks
 */
PatchList<string, string> readPatchesFromDir(const std::string& dirname)
{
	PatchList<string, string> res;

	DIR* dir = opendir(dirname.c_str());
	if (dir == NULL)
		throw wibble::exception::System("opening directory " + dirname);
	
	struct dirent* ent;
	while ((ent = readdir(dir)) != NULL)
	{
		// Skip files starting with a dot
		if (ent->d_name[0] == '.')
			continue;

		string file = dirname + "/" + ent->d_name;
		struct stat st;
		if (stat(file.c_str(), &st) == -1)
			throw wibble::exception::System("getting informations about " + file);

		// We only want regular files or symlinks
		if (!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode)))
			continue;

		res.addPatch(readPatchFromFile(file));
	}
	closedir(dir);

	return res;
}

void removePatchesFromDir(const std::string& dirname)
{
	DIR* dir = opendir(dirname.c_str());
	if (dir == NULL)
		throw wibble::exception::System("opening directory " + dirname);
	
	struct dirent* ent;
	while ((ent = readdir(dir)) != NULL)
	{
		// Skip files starting with a dot
		if (ent->d_name[0] == '.')
			continue;

		string file = dirname + "/" + ent->d_name;
		struct stat st;
		if (stat(file.c_str(), &st) == -1)
			throw wibble::exception::System("getting informations about " + file);

		// We only want regular files or symlinks
		if (!(S_ISREG(st.st_mode) || S_ISLNK(st.st_mode)))
			continue;

		if (unlink(file.c_str()) == -1)
			throw wibble::exception::System("removing file " + file);
	}
	closedir(dir);
}

class Index : public PatchCollection<std::string, std::string>
{
protected:
	BasicStringDiskIndex ro_idx;

public:
	Index(const std::string& file)
		: PatchCollection<std::string, std::string>(ro_idx),
			ro_idx(file)
	{
		// Read the patches
		setChanges(readPatchesFromDir(file + ".patches"));
	}

	virtual ~Index() {}

	void addPatch(const PatchList<std::string, std::string>& patch, const std::string& name, bool overwrite = false)
	{
		TrivialConverter<string, string> conv;
		string patchfile = ro_idx.filename() + ".patches/" + name;
		
		int fd = open(patchfile.c_str(), O_CREAT | O_WRONLY | (overwrite ? O_TRUNC : O_EXCL), 0666);
		if (fd == -1)
			throw wibble::exception::System("creating file " + patchfile);

		FILE* out = fdopen(fd, "w");
		if (out == NULL)
			throw wibble::exception::System("fdopening file " + patchfile);

		TextFormat<string, string>::outputPatch(conv, conv, patch, out);
		fclose(out);

		addChanges(patch);
	}

	static void remove(const std::string& file)
	{
		string patchdir = file + ".patches";
		removePatchesFromDir(patchdir);

		if (rmdir(patchdir.c_str()) == -1)
			if (errno == ENOTEMPTY)
				cerr << "Directory " + patchdir + " was not empty: skipping removing it";
			else
				throw wibble::exception::System("Removing directory " + patchdir);

		unlink(file.c_str());
	}
};

class Indexer : public BasicStringDiskIndexer
{
public:
	virtual ~Indexer() {}

	void write(const std::string& file)
	{
		BasicStringDiskIndexer::write(file);

		// Create the patches directory if it does not exist yet
		mkpath(file + ".patches");
	}
};

void addPatch(Index& idx, PatchList<string, string>& patch)
{
	static int seq = 1;

	if (patch.size() > 0)
	{
		// Compute a default name from username, current time and pid
		char timestr[50];
		time_t t = time(NULL);
		strftime(timestr, 50, "%Y%m%d-%H%M%S", localtime(&t));

		std::stringstream ss;
		ss << get_username() << "-" << timestr << "-" << getpid() << "-" << seq++;
	
		idx.addPatch(patch, ss.str(), false);
	} else
		cerr << "ignoring request to add an empty patch" << endl;
}

#if 0
PatchList<string, string> readPatches(const string& file)
	throw (FileException, ParserException)
{
	Converter<string, string> conv;

	if (file == "-")
	{
		StdioParserInput input(stdin, "<stdin>");
		return TextFormat<string, string>::parsePatch(conv, conv, input);
	}
	else
	{
		StdioParserInput input(file);
		return TextFormat<string, string>::parsePatch(conv, conv, input);
	}
}

void parseDerivedTags(ParserInput& in, DerivedTags& output)
{
	string tag;
	string expr;

	int c;
	enum {TAG, TAGCOLON, SEXPR, EXPR} state = TAG;
	int line = 1;
	while ((c = in.nextChar()) != ParserInput::Eof)
	{
		if (c == '\n')
		{
			if (tag.size() > 0 && expr.size() > 0)
				output.add(tag, expr);
			else
				fprintf(stderr, "In derived tags file, ignoring incomplete line %d.\n", line);
			tag = string();
			expr = string();
			state = TAG;
			line++;
		} else
			switch (state)
			{
				// Read item
				case TAG:
					switch (c)
					{
						case ':':
							state = TAGCOLON;
							break;
						default:
							tag += c;
							break;
					}
					break;
				// After colon on item
				case TAGCOLON:
					switch (c)
					{
						case ' ':
						case '\t':
							state = SEXPR;
							break;
						case ':':
							tag += c;
							break;
						default:
							tag += ':';
							tag += c;
							state = EXPR;
							break;
					}
					break;
				// Space before tag
				case SEXPR:
					switch (c)
					{
						case ' ':
						case '\t':
							break;
						default:
							expr += c;
							state = EXPR;
							break;
					}
					break;
				// Read tag
				case EXPR:
					expr += c;
					break;
			}
	}
}

void readDerivedTags(const string& file, DerivedTags& derivedTags)
	throw (FileException)
{
	if (file == "-")
	{
		StdioParserInput input(stdin, "<stdin>");
		parseDerivedTags(input, derivedTags);
	}
	else
	{
		StdioParserInput input(file);
		parseDerivedTags(input, derivedTags);
	}
}

template<typename ITEM, typename TAG>
class ItemsOnly : public Filter<ITEM, TAG>
{
protected:
	virtual void consumeItemUntagged(const ITEM& item)
	{
		this->consumer->consume(item);
	}
	virtual void consumeItem(const ITEM& item, const OpSet<TAG>& tags)
	{
		this->consumer->consume(item);
	}
	virtual void consumeItemsUntagged(const OpSet<ITEM>& items)
	{
		this->consumer->consume(items);
	}
	virtual void consumeItem(const OpSet<ITEM>& items, const OpSet<TAG>& tags)
	{
		this->consumer->consume(items);
	}

public:
	ItemsOnly() {}
	ItemsOnly(Consumer<ITEM, TAG>& cons) : Filter<ITEM, TAG>(cons) {}
};
#endif

#if 0
class Reader
{
	// Prepare the input filter chain
	FilterChain<string, string> filters;
	Substitute<string, string> substitutions;
	PatchList<string, string> patches;
	Implications<string> implications;
	DerivedTags derivedTags;

	AddImplied<string, string> addImplied;
	RemoveImplied<string, string> removeImplied;
	AddDerived<string> addDerived;
	RemoveDerived<string> removeDerived;
	UnfacetedRemover<string> unfacetedRemover;
	FilterTagsByExpression<string, string> filterByExpression;

public:
	Reader(CommandlineParserWithCommand& opts, valid_command cmd)
		: addImplied(implications), removeImplied(implications),
		  addDerived(derivedTags), removeDerived(derivedTags),
		  unfacetedRemover("::"), filterByExpression("")
	{
		if (opts.get("rename").defined())
		{
			readCollection(opts.get("rename").stringVal(), substitutions.substitutions());
			filters.appendFilter(substitutions);
		}
		if (opts.get("patch").defined())
		{
			patches = readPatches(opts.get("patch").stringVal());
			filters.appendFilter(patches);
		}

		if (opts.get("extimpl").defined())
		{
			readCollection(opts.get("extimpl").stringVal(), implications);
			// Pack the structure for faster expansions
			implications.pack();
		}
		if (opts.get("derived").defined())
			readDerivedTags(opts.get("derived").stringVal(), derivedTags);

		// Intermix implications and derived tags as seems best
		bool compressOutput = (cmd == COPY && !opts.get("expanded").defined());
		bool hasImpl = opts.get("extimpl").defined();
		bool hasDerv = opts.get("derived").defined();

		if (compressOutput)
		{
			if (hasDerv)
			{
				// Expand implications
				if (hasImpl) filters.appendFilter(addImplied);

				// Remove derived tags computing them using the expanded tag set
				filters.appendFilter(removeDerived);
			}

			// Compress implications
			if (hasImpl) filters.appendFilter(removeImplied);
		} else {
			// Expand implications
			if (hasImpl) filters.appendFilter(addImplied);

			// Add derived tags computing them using the expanded tag set
			if (hasDerv)
			{
				filters.appendFilter(addDerived);

				// Add further tags implicated by the derived tags
				if (hasImpl) filters.appendFilter(addImplied);
			}
		}

		if (opts.get("remove-unfaceted").defined())
			filters.appendFilter(unfacetedRemover);

		if (opts.get("remove-tags").defined())
		{
			filterByExpression.setExpression(not Expression(opts.get("remove-tags").stringVal())); 
			filters.appendFilter(filterByExpression);
		}
	}
	void output(const string& file, Consumer<string, string>& cons)
	{
		filters.setConsumer(cons);
		readCollection(file, filters);
	}
	void output(CommandlineArgs args, Consumer<string, string>& cons)
	{
		if (args.hasNext())
			while (args.hasNext())
				output(args.next(), cons);
		else
			output("-", cons);
	}
};
#endif

void output(wibble::commandline::TagidxParser& opts, ReadonlyCollection<string, string>& coll)
{
	if (opts.out_quiet->boolValue())
	{
		Sink<string, string> sink;
		coll.output(sink);
	} else {
		TrivialConverter<string, string> conv;
		TextFormat<string, string> writer(conv, conv, stdout);

		if (opts.out_group->boolValue())
		{
			ItemGrouper<string, string> grouper;
			coll.output(grouper);
			grouper.output(writer);
		} else
			coll.output(writer);
	}
}

string indexname(wibble::commandline::TagidxParser& opts)
{
	if (opts.idx_index->boolValue())
		return opts.idx_index->stringValue();
	if (getenv("TAGIDX") != NULL)
		return getenv("TAGIDX");
	struct passwd* pw = getpwuid(getuid());
	if (pw != NULL)
		return string(pw->pw_dir) + "/.tagcoll/index";
	throw wibble::exception::Consistency("computing the pathname for the index directory",
			"cannot find a suitable place for the index directory"
			"(tried the --index commandline option, "
			"the $TAGIDX environment variable and the home directory)");
}

int main(int argc, const char* argv[])
{
	wibble::commandline::TagidxParser opts;

	try {
		// Parse commandline.
		// If the parser took care of the action, we can exit just fine.
		if (opts.parse(argc, argv))
			return 0;

		if (opts.foundCommand() == opts.cmd_create)
		{
			string fname = indexname(opts);

			Indexer indexer;

			if (opts.hasNext())
				while (opts.hasNext())
					readCollection(opts.next(), indexer);
			else
				readCollection("-",  indexer);

			indexer.write(fname);
		}
		else if (opts.foundCommand() == opts.cmd_remove)
		{
			string fname = indexname(opts);
			Index::remove(fname);
		}
		else if (opts.foundCommand() == opts.cmd_addpatch)
		{
			string fname = indexname(opts);
			Index idx(fname);

			if (opts.hasNext())
			{
				while (opts.hasNext())
				{
					string filename = opts.next();
					size_t pos = filename.rfind('/');
					string name = pos == string::npos ? filename : filename.substr(pos + 1);

					PatchList<string, string> patch = readPatchFromFile(filename);
					if (patch.size() > 0)
					{
						idx.addPatch(patch, name, opts.misc_force->boolValue());
					} else {
						cerr << "skipped empty patch file " << filename << endl;
					}
				}
			} else {
				TrivialConverter<string, string> conv;
				StdioParserInput input(stdin, "(stdin)");
				PatchList<string, string> patch = TextFormat<string, string>::parsePatch(conv, conv, input);

				addPatch(idx, patch);
			}
		}
		else if (opts.foundCommand() == opts.cmd_compact)
		{
			string fname = indexname(opts);

			Indexer indexer;

			// Output the patched collection to the indexer
			{
				Index idx(fname);
				idx.output(indexer);
			}

			// Delete the old index and patches
			Index::remove(fname);

			// Write the new index
			indexer.write(fname);
		}
		else if (opts.foundCommand() == opts.cmd_cat)
		{
			string fname = indexname(opts);
			Index idx(fname);

			output(opts, idx);
		}
		else
			throw wibble::exception::BadOption(string("unhandled command ") +
						(opts.foundCommand() ? opts.foundCommand()->name() : "(null)"));

		return 0;
	} catch (wibble::exception::BadOption& e) {
		cerr << e.desc() << endl;
		opts.outputHelp(cerr);
		return 1;
	} catch (std::exception& e) {
		cerr << e.what() << endl;
		return 1;
	}
}

// vim:set ts=4 sw=4:
