/*
 * Copyright 1999-2005 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* $Id$ */

package org.apache.fop.layoutmgr;

import org.apache.fop.datatypes.PercentBase;
import org.apache.fop.fo.flow.Marker;
import org.apache.fop.fo.pagination.Flow;
import org.apache.fop.area.Area;
import org.apache.fop.area.BlockParent;

import java.util.List;
import org.apache.fop.traits.MinOptMax;

/**
 * LayoutManager for an fo:flow object.
 * Its parent LM is the PageSequenceLayoutManager.
 * This LM is responsible for getting columns of the appropriate size
 * and filling them with block-level areas generated by its children.
 */
public class FlowLayoutManager extends BlockStackingLayoutManager {
    
    private Flow fobj;
    
    /** List of break possibilities */
    protected List blockBreaks = new java.util.ArrayList();

    /** Array of areas currently being filled stored by area class */
    private BlockParent[] currentAreas = new BlockParent[Area.CLASS_MAX];

    private int iStartPos = 0;

    /**
     * Used to count the number of subsequent times to layout child areas on
     * multiple pages.
     */
    private int numSubsequentOverflows = 0;
    
    /**
     * This is the top level layout manager.
     * It is created by the PageSequence FO.
     * @param node Flow object
     */
    public FlowLayoutManager(Flow node) {
        super(node);
        fobj = node;
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#getNextBreakPoss(LayoutContext)
     */
    public BreakPoss getNextBreakPoss(LayoutContext context) {

        // currently active LM
        LayoutManager curLM;
        MinOptMax stackSize = new MinOptMax();

        fobj.setLayoutDimension(PercentBase.BLOCK_IPD, context.getRefIPD());
        fobj.setLayoutDimension(PercentBase.BLOCK_BPD, context.getStackLimit().opt);

        while ((curLM = getChildLM()) != null) {
            if (curLM.generatesInlineAreas()) {
                log.error("inline area not allowed under flow - ignoring");
                curLM.setFinished(true);
                continue;
            }

            // Make break positions and return page break
            // Set up a LayoutContext
            MinOptMax bpd = context.getStackLimit();
            BreakPoss bp;

            LayoutContext childLC = new LayoutContext(0);
            boolean breakPage = false;
            childLC.setStackLimit(MinOptMax.subtract(bpd, stackSize));
            childLC.setRefIPD(context.getRefIPD());

            if (!curLM.isFinished()) {
                if ((bp = curLM.getNextBreakPoss(childLC)) != null) {
                    stackSize.add(bp.getStackingSize());
                    blockBreaks.add(bp);
                    // set stackLimit for remaining space
                    childLC.setStackLimit(MinOptMax.subtract(bpd, stackSize));

                    if (bp.isForcedBreak() || bp.nextBreakOverflows()) {
                        if (log.isDebugEnabled()) {
                            log.debug("BreakPoss signals " + (bp.isForcedBreak() 
                                    ? "forced break" : "next break overflows"));
                        }
                        breakPage = true;
                    }
                }
            }

            // check the stack bpd and if greater than available
            // height then go to the last best break and return
            // break position
            if (stackSize.opt > context.getStackLimit().opt) {
                breakPage = true;
            }
            if (breakPage) {
                numSubsequentOverflows++;
                if (numSubsequentOverflows > 50) {
                    log.error("Content overflows available area. Giving up after 50 attempts.");
                    setFinished(true);
                    return null;
                }
                return new BreakPoss(
                      new LeafPosition(this, blockBreaks.size() - 1));
            }
            numSubsequentOverflows = 0; //Reset emergency counter
        }
        setFinished(true);
        if (blockBreaks.size() > 0) {
            return new BreakPoss(
                             new LeafPosition(this, blockBreaks.size() - 1));
        }
        return null;
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#addAreas(PositionIterator, LayoutContext)
     */
    public void addAreas(PositionIterator parentIter, LayoutContext layoutContext) {

        LayoutManager childLM;
        LayoutContext lc = new LayoutContext(0);
        while (parentIter.hasNext()) {
            LeafPosition lfp = (LeafPosition) parentIter.next();
            // Add the block areas to Area
            PositionIterator breakPosIter =  new BreakPossPosIter(
                    blockBreaks, iStartPos, lfp.getLeafPos() + 1);
            iStartPos = lfp.getLeafPos() + 1;
            while ((childLM = breakPosIter.getNextChildLM()) != null) {
                childLM.addAreas(breakPosIter, lc);
            }
        }

        flush();
    }

    /**
     * Add child area to a the correct container, depending on its
     * area class. A Flow can fill at most one area container of any class
     * at any one time. The actual work is done by BlockStackingLM.
     * @see org.apache.fop.layoutmgr.LayoutManager#addChildArea(Area)
     */
    public void addChildArea(Area childArea) {
        addChildToArea(childArea,
                          this.currentAreas[childArea.getAreaClass()]);
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#getParentArea(Area)
     */
    public Area getParentArea(Area childArea) {
        // Get an area from the Page
        BlockParent parentArea = (BlockParent)parentLM.getParentArea(childArea);
        this.currentAreas[parentArea.getAreaClass()] = parentArea;
        setCurrentArea(parentArea);
        return parentArea;
    }

    /**
     * @see org.apache.fop.layoutmgr.LayoutManager#resetPosition(Position)
     */
    public void resetPosition(Position resetPos) {
        if (resetPos == null) {
            reset(null);
        }
    }

    /**
     * Retrieve marker is not allowed in the flow so this reports an
     * error and returns null.
     *
     * @see org.apache.fop.layoutmgr.LayoutManager
     */
    public Marker retrieveMarker(String name, int pos, int boundary) {
        // error cannot retrieve markers in flow
        log.error("Cannot retrieve a marker from the flow");
        return null;
    }
}

