/*
 * easy_spice - a spice front end
 * Copyright (C) 2001 Routoure Jean-Marc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

/* This part of the code concerns the spice interface */
/* there is also the test of the different entry */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include "easyspice.h"


/* ----------------------------------- */
/* Read the spice result and put       */
/* the result into easy_spice          */
/* ----------------------------------- */
void 
spice2easy(EasySpice *es)
{
  FILE *fp;
  gchar line[FICHIER_MAX_LINE], **tokens;
  gint from, to;  gchar *over;

  enum ParserStatus {ERROR=0, DEVICES, DEV_RESULTS, OP_RESULTS};
  gint parser=ERROR;

  GList *device_result=NULL, *tmp, *nodes=NULL;
  KeyValue *res;
  KeyValue *op_res;
  SpiceDevice *dev_current=NULL;

  if ((fp=fopen(es->file_spice_results,"rt")) == NULL) {
    write_log("ERROR: unable to open the result file\n");
    return;
  }
  /* Free the op_result list */
  for (tmp=g_list_first(es->netlist->op_result);
       tmp != NULL;
       tmp = g_list_next(tmp)) {
    g_free(((KeyValue*)(tmp->data))->key);
    g_free(((KeyValue*)(tmp->data))->value);
    g_free(tmp->data); /* KeyValue structure */
  }
  g_list_free(es->netlist->op_result);
  es->netlist->op_result=NULL;
	   
  /*this is how the model section looks like:
   *
   *result_show:
   * Resistor: Simple linear resistor
   *     device                    r5
   *      model                     R
   * resistance                  1000
   *         ac                  1000
   *      dtemp                     0
   *      noisy                     1
   *          i           5.26316e-05
   *          p           2.77008e-06
   *
   */
  
  /*this is how the result section looks like
   *
   *result_op:
   *n0 = 1.000000e+00
   *n1 = 9.473684e-01
   *n2 = 8.947368e-01
   *n3 = 6.315789e-01
   *n4 = 5.789474e-01
   *n5 = 5.263158e-01
   *v1#branch = -5.26316e-05
   */

  while (fgets(line, FICHIER_MAX_LINE, fp) != NULL) {
    /* replace the newline and the equal sign with spaces */
    g_strdelimit(line,"=\n",' ');
    tokens= g_strsplit(line, " ", -1);
    
    /* now we have tokens={"","tok1","","","","tok2",""}
     * lets move the tokens to the start of the list 
     */
    to=0;
    for (from=0; tokens[from] != NULL; from++) {
      if (strlen(tokens[from]) != 0) {
	if (from != to) { /* exchange the tokens */
	  over=tokens[from];
	  tokens[from]=tokens[to];
	  tokens[to]=over;
	}
	to++;
      }
    }
    /* "to" contains the number of non empty elements after this for-loop */

    if (DEBUG) {
      printf("  line=\"%s\", from=%d, to=%d, parser=%d\n", line, 
	     from, to, parser);
    }
    
    /* each device is followed by an empty line 
     * if we are inside a device and theres a line which contains only
     * the newline, we can copy the device to the array
     */
    if (to == 0) {
      if (parser == DEV_RESULTS) {
	/* end of device results */
	if (dev_current != NULL)
	  dev_current->result = device_result;
	device_result = NULL;
	dev_current = NULL;
	parser = DEVICES;
      }
      else if (parser == OP_RESULTS) {
	parser = ERROR;
      }
    }
    else if (to == 1) {
      if (strcmp(tokens[0],"result_show:")==0)
	parser = DEVICES;
      if (strcmp(tokens[0],"result_op_start:") == 0)
	parser = OP_RESULTS;
      if (strcmp(tokens[0],"result_op_end") == 0)
	parser = ERROR;
    }
    else if (parser==DEVICES) {
      if (VERBOSE3)
	printf("  device section: tok1=%s, tok2=%s\n",tokens[0], tokens[1]);
      if (strcmp(tokens[0], "device") == 0) {  /* device line */
	/* search for the device in the device list */
	tmp=g_list_find_custom(es->netlist->devices, tokens[1],
			       (GCompareFunc) spicedevice_compare_name);
	if (tmp == NULL) {
	  if (VERBOSE3)
	    printf("    device not found\n");
	}
	else {
	  dev_current = tmp->data;
	  parser = DEV_RESULTS;
	}
      }
    }
    else if (parser == DEV_RESULTS) {
      /* fill the device structure */
      if (VERBOSE3) {
	printf("Device_Result section\n");
	printf("  device=%s, value=%s\n",tokens[0], tokens[1]);
      }
      res=g_malloc(sizeof(KeyValue));
      res->key=g_strdup(tokens[0]);
      res->value=g_strdup(tokens[1]);
      device_result=g_list_append(device_result, res);
    }
    else if (parser == OP_RESULTS) {
      if (VERBOSE3) {
	printf("OP_Results section\n");
	printf("  nodename=%s, value=%s\n",tokens[0], tokens[1]);
      }
      op_res= g_malloc(sizeof(KeyValue));
      op_res->key = g_strdup(tokens[0]);
      op_res->value = g_strdup(tokens[1]);
      es->netlist->op_result = g_list_append(es->netlist->op_result, op_res);
    }
    g_strfreev(tokens);
  }
  if (VERBOSE3) {
    printf("operation point results:\n");
    keyvalue_print_list(es->netlist->op_result);
  }
  fclose(fp);

  /* get the result_names and the push it to the interface */
  for (tmp=g_list_first(es->netlist->op_result); tmp != NULL;
       tmp=g_list_next(tmp)) {
    op_res=tmp->data;
    nodes=g_list_append(nodes, op_res->key);
  }
  
  gtk_combo_set_popdown_strings(GTK_COMBO(lookup_widget(es->win_main,"op_node_combo")), nodes);
  /* tell the gui that the values have changed */
  on_op_device_changed(NULL, es);
  on_op_node_changed(NULL, es);
}


/* ------------------------------------------ */
/* test the different field for all the simul */
/* return 1 if OK.                            */
/* -------------------------------------------*/
int spice_test_field(EasySpice *es, gint type_simul)
{
  SpiceVariables *sv;
  int res=1;

  sv=es->spice_vars;

  switch (type_simul)
    {
    case SIMUL_DC:
      if (!(strcmp(sv->dc_source1,"") && strcmp(sv->dc_start1,"")
	    && strcmp(sv->dc_stop1,"") && strcmp(sv->dc_inc1,""))) {
        write_log(_("Error: one field empty in first DC variable\n"));
        res=0;
      }
      /* the second source is always set by the listbox 
	 lets don't check the other values */
//      if ((strcmp(sv->dc_source2,"") != 0)
//	  && (!(strcmp(sv->dc_start2,"")
//		&& strcmp(sv->dc_stop2,"") && strcmp(sv->dc_inc2,"")))) {
//	write_log(_("Error: one field empty in the second DC variable\n"));
//        res=0;
//      }
      break;
    case SIMUL_AC:
      if (strcmp(sv->ac_points,"") && strcmp(sv->ac_fmin,"")
	  && strcmp(sv->ac_fmax,"")==0) {
        write_log(_("Error: One field empty in AC variables\n"));
        res=0;
      }
      break;
    case SIMUL_TRAN:
      if (strcmp(sv->tran_tstep,"") && strcmp(sv->tran_tstop,"")==0) {
        write_log(_("Error: One field empty in TRAN variables\n"));
        res=0;
      }
      break;
    case SIMUL_TF:
      if (strcmp(sv->tf_source,"") && strcmp(sv->tf_output,"")==0) {
        write_log(_("Error: One field empty in TF variables\n"));
        res=0;
      }
      break;
    case SIMUL_NOISE:  
      if (strcmp(sv->noise_points,"") && strcmp(sv->noise_fmin,"")
	  && strcmp(sv->noise_fmax,"") && strcmp(sv->noise_node,"")
	  && strcmp(sv->noise_source,"")==0) {
        write_log(_("Error: One field empty in NOISE variables\n"));
        res=0;
      }
      break;
    default:
      write_log(_("Error in spice_test_field\n"));
      res=0;
      break;
    }
  if (res==0) 
    led_false(es->win_main); 
  else 
    led_ok(es->win_main);
  return res;
}

/*  ------------ Mise  jour de la fentre d'erreur ---------------------*/

void 
fatal_error(EasySpice *es)
{
  GtkTextBuffer *buffer;

  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(lookup_widget(es->win_main,"text_spice_error")));
  textfile2buffer(es->file_spice_error, buffer);
}


/* ------------------------------------------------------------------------- */
/* After a simulation: count the errors in the output spice text file        */
/* ------------------------------------------------------------------------- */
/* return O if OK              */
/* return 1 if error (warning) */
/* return 2 if abort           */

int 
spice_parse_error_file(gchar *filename)
{
  int n_erreur=0, n_aborted=0;
  int return_value=0;
  FILE *file;
  gchar tmp[FICHIER_MAX_LINE];
  gchar *error_msg=NULL, *err_tmp;   

  
  if ((file=fopen(filename,"r"))==NULL) {
    create_quickmessage(_("Error"),_("Failed to open the\nspice error file!"));
    return_value=2;
  }
  else {
    while (fscanf(file,"%s",tmp)==1) {
      if ((strcasecmp(tmp,"error")==0) || (strcasecmp(tmp,"error:")==0)) 
	n_erreur++;
      if ((strcasecmp(tmp,"aborted")==0) 
	  || (strcasecmp(tmp,"segmentation")==0))
	n_aborted++;
    }
    fclose(file);

    if (n_erreur==0) {
      write_log(_("No error\n"));
      return_value=0;
    } else if (n_erreur==1) {
      error_msg=g_strdup_printf(_("One error found\n"));
      return_value=1;
    } else {
      error_msg=g_strdup_printf(_("%d errors founded\n"),n_erreur);
      return_value=1;
    }

    if (n_aborted>0) {
      err_tmp = error_msg;
      error_msg=g_strdup_printf(_("%sSimulation was not successful\n"),
				error_msg);
      g_free(err_tmp);
      return_value=2;
    }
    if ((n_erreur>0) || (n_aborted>0))   
      write_log(error_msg);
    g_free(error_msg);
  }
  return return_value;
}

/*  ------------------------------- spice txt results ----------------------------*/

void spice_txt_results(EasySpice *es)
{
  GtkTextBuffer *buffer;
  
  buffer=gtk_text_view_get_buffer(GTK_TEXT_VIEW(lookup_widget(es->win_main,"text_spice_output")));
  textfile2buffer(es->file_spice_results, buffer);
  spice2easy(es);
}

/* -----------------------test if errors occurs during the simulation ------------ */
/* return 0 of OK  */
/* 1 if warning    */
/* 2 if aborted    */

int 
spice_test_error(EasySpice *es)
{
  int error1, error2, return_value=0;
  
  /* test of the result file */
  write_log(_("Test of the results file\n"));
  error1=spice_parse_error_file(es->file_spice_results);

  if (DEBUG)
    printf("spice_test_error: Error1: %d\n",error1);

  /* test of the standard error file */
  write_log(_("Test of the standard error ouput\n"));
  error2=spice_parse_error_file(es->file_spice_error);

  if (DEBUG) 
    printf("spice_test_error: Error2: %d\n",error2);

  if (error2>0)
    fatal_error(es);
  if ((error1>1)||(error2>1))
    return_value=1;
  else
    return_value=0;
  return return_value;
}

/* -------------------------------------------------------------------------------- */
/* to be removed: the returned pointer is dangling as the text is on the stack
 * but the memory of the strings is already freed
 */
//gchar* print_sweep_type(gint type_sweep)
//{
//  switch(type_sweep) {
//  case SWEEP_DEC:
//    return "dec";
//    break;
//  case SWEEP_LIN:
//    return "lin";
//    break;
//  case SWEEP_OCT:
//    return "oct";
//    break;
//  default:
//    return "";
//  }
//}

/* ------------------------------------------------------------------------ */
void 
spice_write_simul(FILE *fp_simul, gint type_simul, EasySpice *es)
{
  gchar *sweeptype[]={"dec","lin","oct",""};
  SpiceDevice *dev;
  GList *tmp;
  SpiceVariables *sv;
  SpiceNetlist *netlist;

  sv=es->spice_vars;
  netlist=es->netlist;
  
  switch (type_simul) {
  case SIMUL_OP:
    fprintf(fp_simul,"  op\n");
    if (sv->op_show_netlist) {
      fprintf(fp_simul,"  echo -----------------------------------------------------\n");
      fprintf(fp_simul,_("  echo -----listing of the circuit--------------------------\n"));
      fprintf(fp_simul,"  echo -----------------------------------------------------\n");
      fprintf(fp_simul,"listing\n");
    }

    fprintf(fp_simul,"echo result_show: \n");
    /* show all devices for later processing */
    for (tmp=g_list_first(netlist->devices); tmp != NULL;
	 tmp=g_list_next(tmp)) {
      dev=tmp->data;
      fprintf(fp_simul,"  show %s \n", dev->name);
    }
    
    if (sv->op_show_models != FALSE) {
      fprintf(fp_simul,"  echo ---------------------------------------------\n");
      fprintf(fp_simul,_("  echo ------- parameters of the devices -----------\n"));
      fprintf(fp_simul,"  echo ---------------------------------------------\n");
      fprintf(fp_simul,"  showmod all\n");
    }
    
    fprintf(fp_simul,"  echo ---------------------------------------------\n");
    fprintf(fp_simul,_("  echo -------   operation point    ----------------\n"));
    fprintf(fp_simul,"  echo ---------------------------------------------\n");
    fprintf(fp_simul,"  echo result_op_start: \n");
    fprintf(fp_simul,"  print all\n");
    fprintf(fp_simul,"  echo result_op_end \n");
    break;
  case SIMUL_DC:
    fprintf(fp_simul,"  set filetype=ascii\n");
    fprintf(fp_simul,"  dc %s %s %s %s",sv->dc_source1,
	    sv->dc_start1, sv->dc_stop1, sv->dc_inc1);
    /* test whether the second set of the dc parameter is complete */
    if (strcmp(sv->dc_source2, "") && strcmp(sv->dc_start2, "")
	&& strcmp(sv->dc_stop2, "") && strcmp(sv->dc_inc2, ""))
      fprintf(fp_simul," %s %s %s %s", sv->dc_source2,
	      sv->dc_start2, sv->dc_stop2, sv->dc_inc2);
    fprintf(fp_simul,"\n");
    break;
  case SIMUL_AC:
    fprintf(fp_simul,"  set filetype=ascii\n");
    fprintf(fp_simul,"  ac %s %s %s %s \n", sweeptype[sv->ac_type],
	    sv->ac_points, sv->ac_fmin, sv->ac_fmax);

    break;
  case SIMUL_TRAN:
    fprintf(fp_simul,"  set filetype=ascii\n");
    fprintf(fp_simul,"  tran %s %s %s %s \n", sv->tran_tstep,
	    sv->tran_tstop, sv->tran_tmax, sv->tran_tstart);

    break;
  case SIMUL_TF:
    fprintf(fp_simul,"  tf %s %s\n",sv->tf_output,sv->tf_source);
    fprintf(fp_simul,"  print all\n");
	
    break;
  case SIMUL_NOISE:
    fprintf(fp_simul,"  set filetype=ascii\n");
    fprintf(fp_simul,"  ac %s %s %s %s \n",
	    sweeptype[sv->noise_type],
	    sv->noise_points,sv->noise_fmin,sv->noise_fmax);
    fprintf(fp_simul,"  noise %s %s %s %s %s %s \n",
	    sv->noise_node, sv->noise_source,
	    sweeptype[sv->noise_type], sv->noise_points,
	    sv->noise_fmin, sv->noise_fmax);

    // Modif pour ngspice14
    //  fprintf(fp_simul,"print all\n");
    //  fprintf(fp_simul,"destroy\n");
    break;
  default:
    printf("spice_write_simul: FIXME: illegal value in simulation type\n");
  }

  fprintf(fp_simul,"%s\n",es->postproc->commands);
}


void 
spice_view_result(EasySpice *es, int type_simul, int n_step)
{
  char *buf;

  switch(type_simul) {
  case SIMUL_OP:
    /* TODO: generate a result list for the interface*/

    /*show the results in a temporary schematic */
    if (es->spice_vars->op_show_results_schematic != FALSE) {
      visu_result_gschem(es);
      buf=g_strdup_printf("gschem %s &",es->file_schematic_tmp);
      if (VERBOSE1)
	printf("executing: \"%s\"\n",buf);
      system(buf);
      g_free(buf);
    }
    break;
  case SIMUL_DC:
    create_spice_plot_file(es, n_step, type_simul);
    if (es->spice_vars->plot_autoupdate != FALSE) 
      start_plot(es);
    break;
  case SIMUL_AC:
    create_spice_plot_file(es, n_step, type_simul);
    if (es->spice_vars->plot_autoupdate != FALSE) 
      start_plot(es);
    break;
  case SIMUL_NOISE:
    create_spice_plot_file(es, n_step, type_simul);
    if (es->spice_vars->plot_autoupdate != FALSE) 
      start_plot(es);
    break;
  case SIMUL_TRAN:
    create_spice_plot_file(es, n_step,type_simul);
    if (es->spice_vars->plot_autoupdate != FALSE)
      start_plot(es);
    break;
  case SIMUL_TF:
    break;
  }
}

/* ----------------------------------------------------------------------- */

gint spice_write_cir(EasySpice *es, FILE *fp_simul, int type_simul)
{
  gchar **param_list=NULL;
  gboolean fin_param=FALSE;
  gint n_step=0;
  double var_step=0;
  GList *tmp;

  SpiceVariables *sv= es->spice_vars;

  fprintf(fp_simul,"* simulation of %s\n",es->project);
  fprintf(fp_simul,".control\n");
  if (sv->param_activ == TRUE) {
    if (sv->param_type==SWEEP_LIST) 
      param_list=g_strsplit(g_strstrip(sv->param_values)," ",-1);
    do {
      n_step++;
      if ((sv->param_type==SWEEP_LIN)||(sv->param_type==SWEEP_DEC)) {
	if (sv->param_type==SWEEP_LIN)
	  var_step = (spice2double(sv->param_start)+
		      spice2double(sv->param_step)*(n_step-1));
	if (sv->param_type==SWEEP_DEC)
	  var_step = (spice2double(sv->param_start)*
		      pow(10,(n_step-1)/spice2double(sv->param_step)));
	if (var_step>spice2double(sv->param_stop))
	  fin_param=TRUE;
      }
      if (sv->param_type==SWEEP_LIST) {
	if (param_list[n_step-1]==NULL) 
	  fin_param=TRUE;
	else {
	  if (VERBOSE2)
	    printf("Parameter %d from list: %s\n", n_step, param_list[n_step-1]);
	  var_step=spice2double(param_list[n_step-1]);
	}
      }
      if (fin_param==FALSE) {
	if (VERBOSE2)
	  printf("parametric simulation step %d\n",n_step);
	fprintf(fp_simul,"alter %s %s = %e\n",sv->param_name, 
		sv->param_param, var_step);
	fprintf(fp_simul,"echo parameter %s %s = %e\n",sv->param_name, 
		sv->param_param, var_step);
	fprintf(fp_simul,"let para=%e\n", var_step);
	spice_write_simul(fp_simul, type_simul, es);
	fprintf(fp_simul,"write rawspice%d.raw\n",n_step);
      }
    } while (fin_param==FALSE);
    
    if (sv->param_type==SWEEP_LIST)
      g_strfreev(param_list);
  } /* parametric simulation */
  else {  /* only a single simulation */
    spice_write_simul(fp_simul, type_simul, es);
    fprintf(fp_simul,"write\n");
  }

  /* ending the control section */
  fprintf(fp_simul,".endc\n");

  /* include the libraries */
  for (tmp=g_list_first(es->spicelibrary->libs); tmp != NULL;
       tmp=g_list_next(tmp)) {
    fprintf(fp_simul,".include %s\n", (gchar*)(tmp->data));
  }
  /* finaly include the spice netlist */
  fprintf(fp_simul,".include %s\n", es->file_spice_netlist);
  /* FIXME: what is this return value good for? */
  return n_step-1; 
}


/*  ------------------------- start the simulation -------------------- */
void spice_start_spice(EasySpice *es)
{
  gchar *buf;
  buf=g_strdup_printf("%s -b %s >%s 2>%s",
  		      es->commands->spice_command, es->file_spice_simul, 
  		      es->file_spice_results, es->file_spice_error);
  if (VERBOSE1)
    printf("executing: \"%s\"\n",buf);
  system(buf);
  g_free(buf);
  if (DEBUG)
    printf("spice.c: simulation done\n");
}


/*  ------------------------- create the file and do the simulation -------------------- */

void simul(EasySpice *es, gint type_simul)
{
  FILE *fp_simul=NULL, *fp_tmp;
  gint n_step;

  if (DEBUG) 
    printf("spice.c: Entering Simulation\n");

  clean_log(es);
  clean_std_err(es);
  clean_std_output(es);
  led_off(es->win_main);

  if ((fp_tmp=fopen(es->file_spice_netlist,"r")) == NULL) {
    write_log(_("Error: No netlist"));
    led_false(es->win_main);
    return;
  }
  fclose(fp_tmp);

  if ((fp_simul = fopen(es->file_spice_simul, "w")) == NULL) {
    write_log(_("Error: Problem with spice .cir file.\n"));
    led_false(es->win_main);
  }
  else {
    n_step=spice_write_cir(es, fp_simul, type_simul);
    fclose(fp_simul);
    es->current_n_step=n_step;
    es->current_simul=type_simul;
    if (DEBUG)
      printf("spice.c: Step number %d\n",n_step);
    gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "op_hbox_polarisation"),FALSE);
    gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "op_hbox_device"),FALSE);
    gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "plot_vbox"),FALSE);
    spice_start_spice(es);

    if (spice_test_error(es)==0) { /* NO ERROR */
      gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "op_hbox_polarisation"),TRUE);
      gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "op_hbox_device"),TRUE);
      gtk_widget_set_sensitive(lookup_widget (GTK_WIDGET (es->win_main), "plot_vbox"),TRUE);
      led_ok(es->win_main);

      spice_txt_results(es);
      if (DEBUG) 
	printf("done\n");
      /*  Show the simulation results */
      spice_view_result(es, type_simul, es->current_n_step);
    }
    else {
      led_false(es->win_main);
      spice_txt_results(es);
      fatal_error(es);
    }
  }
}









