// ==============================================================
//	This file is part of Glest (www.glest.org)
//
//	Copyright (C) 2001-2008 Martio Figueroa
//
//	You can redistribute this code and/or modify it under 
//	the terms of the GNU General Public License as published 
//	by the Free Software Foundation; either version 2 of the 
//	License, or (at your option) any later version
// ==============================================================

#include "server_interface.h"

#include <cassert>
#include <stdexcept>

#include "platform_util.h"
#include "conversion.h"
#include "config.h"

using namespace std;
using namespace Shared::Platform;
using namespace Shared::Util;

namespace Glest{ namespace Game{

// =====================================================
//	class ServerInterface
// =====================================================

ServerInterface::ServerInterface(){
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		slots[i]= NULL;
	}
	serverSocket.setBlock(false);
	serverSocket.bind(GameConstants::serverPort);
}

ServerInterface::~ServerInterface(){
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		delete slots[i];
	}
}

void ServerInterface::addSlot(int playerIndex){
	assert(playerIndex>=0 && playerIndex<GameConstants::maxPlayers);

	delete slots[playerIndex];
	slots[playerIndex]= new ConnectionSlot(this, playerIndex);
	updateListen();
}

void ServerInterface::removeSlot(int playerIndex){
	delete slots[playerIndex];
	slots[playerIndex]= NULL;
	updateListen();
}

ConnectionSlot* ServerInterface::getSlot(int playerIndex){
	return slots[playerIndex];
}

int ServerInterface::getConnectedSlotCount(){
	int connectedSlotCount= 0;
	
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		if(slots[i]!= NULL){
			++connectedSlotCount;
		}
	}
	return connectedSlotCount;
}

void ServerInterface::update(){
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		if(slots[i]!= NULL){
			slots[i]->update();
		}
	}
}

void ServerInterface::waitUntilReady(Checksum* checksum){
	
	Chrono chrono;
	bool allReady= false;

	chrono.start();

	//wait until we get a ready message from all clients
	while(!allReady){
		
		allReady= true;
		for(int i= 0; i<GameConstants::maxPlayers; ++i){
			ConnectionSlot* connectionSlot= slots[i];

			if(connectionSlot!=NULL){
				if(!connectionSlot->isReady()){
					NetworkMessageType networkMessageType= connectionSlot->getNextMessageType();
					NetworkMessageReady networkMessageReady;

					if(networkMessageType==nmtReady && connectionSlot->receiveMessage(&networkMessageReady)){
						connectionSlot->setReady();
					}
					else if(networkMessageType!=nmtInvalid){
						throw runtime_error("Unexpected network message: " + intToStr(networkMessageType));
					}

					allReady= false;
				}
			}
		}

		//check for timeout
		if(chrono.getMillis()>readyWaitTimeout){
			throw runtime_error("Timeout waiting for clients");
		}
	}

	//send ready message after, so clients start delayed
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		NetworkMessageReady networkMessageReady(checksum->getSum());
		ConnectionSlot* connectionSlot= slots[i];

		if(connectionSlot!=NULL){
			connectionSlot->sendMessage(&networkMessageReady);
		}
	}
}

void ServerInterface::launchGame(const GameSettings* gameSettings){
	NetworkMessageLaunch networkMessageLaunch(gameSettings);
	
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		ConnectionSlot* connectionSlot= slots[i];

		if(connectionSlot!= NULL){
			if(connectionSlot->isConnected()){
				connectionSlot->sendMessage(&networkMessageLaunch);
			}
			else{
				removeSlot(i);
			}
		}
	}
}

void ServerInterface::quitGame(){
	NetworkMessageQuit networkMessagequit;
	
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		ConnectionSlot* connectionSlot= slots[i];

		if(connectionSlot!= NULL){
			if(connectionSlot->isConnected()){
				connectionSlot->sendMessage(&networkMessagequit);
			}
			else{
				removeSlot(i);
			}
		}
	}
}


void ServerInterface::addCommand(const NetworkCommand* networkCommand){
	commands.push_back(*networkCommand);
}

void ServerInterface::sendCommands(int frameCount){
	NetworkMessageCommandList networkMessageCommandList(frameCount);
	
	// build command list
	while(!commands.empty()){
		if( networkMessageCommandList.addCommand(&commands.front()) ){
			commands.erase(commands.begin());
		}
		else{
			break;
		}
	}

	// broadcast commands
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		ConnectionSlot* connectionSlot= slots[i];
		if(connectionSlot!= NULL && connectionSlot->isConnected()){
			connectionSlot->sendMessage(&networkMessageCommandList);
		}
	}
}

void ServerInterface::updateListen(){
	int openSlotCount= 0;
	
	for(int i= 0; i<GameConstants::maxPlayers; ++i){
		if(slots[i]!= NULL && !slots[i]->isConnected()){
			++openSlotCount;
		}
	}

	serverSocket.listen(openSlotCount);
}

}}//end namespace
