/*
 * Copyright (c) 2000-2005 CyberFOX Software, Inc. All Rights Reserved.
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Library General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the
 *  Free Software Foundation, Inc.
 *  59 Temple Place
 *  Suite 330
 *  Boston, MA 02111-1307
 *  USA
 */

import java.util.*;
import java.io.File;

/*!@class DeletedManager
 * @brief Manage deleted items, so they are not added again.
 */
public class DeletedManager implements XMLSerialize {
  private Map _deletedItems = null;

  public DeletedManager() {
    _deletedItems = new TreeMap();
  }

  public int clearDeleted() {
    int rval = _deletedItems.size();

    _deletedItems = new TreeMap();

    return rval;
  }

  public boolean isDeleted(String id) {
    if(JConfig.queryConfiguration("deleted.ignore", "true").equals("true")) {
      return _deletedItems.containsKey(id);
    }
    undelete(id);
    return false;
  }

  public void undelete(String id) {
    _deletedItems.remove(id);
  }

  private void killFiles(String id) {
    String outPath = JConfig.queryConfiguration("auctions.savepath");
    String imgPath = outPath + System.getProperty("file.separator") + id;

    File thumb = new File(imgPath+".jpg");
    if(thumb.exists()) thumb.delete();

    File realThumb = new File(imgPath + "_t.jpg");
    if (realThumb.exists()) realThumb.delete();

    File badBlocker = new File(imgPath + "_b.jpg");
    if (badBlocker.exists()) badBlocker.delete();

    File html = new File(imgPath+".html.gz");
    if(html.exists()) html.delete();

    File htmlBackup = new File(imgPath+".html.gz~");
    if(htmlBackup.exists()) htmlBackup.delete();
  }

  public void delete(String id) {
    if(!isDeleted(id)) {
      killFiles(id);
      _deletedItems.put(id, new Long(System.currentTimeMillis()));
    }
  }

  public void fromXML(XMLElement deletedEntries) {
    //  A lack of deleted entries is not a failure.
    if(deletedEntries == null) return;
    long now = System.currentTimeMillis();
    //  60 days ago is the cut off point for keeping deleted entries around.
    long past = now - (Constants.ONE_DAY * 60);

    //  Walk over all the deleted entries and store them.
    Iterator delWalk = deletedEntries.getChildren();
    while(delWalk.hasNext()) {
      XMLElement delEntry = (XMLElement)delWalk.next();
      if(delEntry.getTagName().equals("delentry")) {
        long when;
        String strDel = delEntry.getProperty("when");
        if(strDel == null) {
          when = now;
        } else {
          when = Long.parseLong(strDel);
        }
        if(when > past) {
          _deletedItems.put(delEntry.getContents(), new Long(when));
        }
      }
    }
  }

  public XMLElement toXML() {
    XMLElement xmlResult = new XMLElement("deleted");
    for(Iterator it=_deletedItems.keySet().iterator(); it.hasNext(); ) {
      String delId = (String)it.next();
      XMLElement entry = new XMLElement("delentry");
      entry.setProperty("when", _deletedItems.get(delId).toString());
      entry.setContents(delId);
      xmlResult.addChild(entry);
    }

    return xmlResult;
  }
}
