/***************************************************************************
 *   Copyright (C) 2003-2005 by Kevin Hessels                              *
 *   khessels@shaw.ca                                                      *
 *   Copyright (C) 2002, 2003 Richard P. Howell IV.                        * 
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#ifndef QT_CLEAN_NAMESPACE
#define QT_CLEAN_NAMESPACE
#endif
 
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <ctime>

#include <qgl.h>
#include <qtimer.h>
#include <qtooltip.h>

#ifdef HAVE_GL_GL_H
#include <GL/gl.h>				// has to be after qgl.h
#endif
#ifdef HAVE_GL_GLU_H
#include <GL/glu.h>
#endif
#include <GL/glx.h> 

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "kfoldingconfigdata.h" 
#include "kfoldingmoleculewidget.h"


#ifndef DLIST
# define DLIST		1			/* 1 if OpenGL should create a display list */
#endif

#define IZOOM		3.0			/* Initial zoom factor (Z distance) */
#define FOV		45.0			/* Normal (not stereo) field of view */
#define ZOOMF		1.25			/* Zoom in/out factor */
#define MAXZOOM		10
#define DEPF		1.3			/* Stereo depth more/less factor */
#define SDEP		9.0			/* Initial stereo depth */
#define SSEP		0.5			/* Stereo separation (fraction of FOV) */
#define DQUAL		5000.0			/* Medium display quality */
#define BSBSIZ		0.20			/* Ball size in Ball and Stick model */

#define XYZFILE		"work/current.xyz"
#define MAXBONDS	8			/* Maximum bonds on one  */
#define MAXATOMS	100000			/* Maximum atoms in molecule */

#define FONT		"10x20"			/* Default font name */
#define M_LW		2			/* Line width separating menu buttons */

#define ETYPES		"?HCNOPS"		/* Known elements (in table order) */

/* Atom flag values */
#define AF_H	0x0001				/* Hydrogen */
#define AF_H2O	0x0002				/* Water */
#define AF_CA	0x0004				/* Alpha Carbon */
#define AF_BB	0x0008				/* Backbone */
#define AF_CN	0x0010				/* Carbon in carbonyl of peptide bond or amide */
#define AF_DL	0x4000				/* Atom is in display list */

#define MAX_BOND_LEN		2.4
#define CH_BOND_LEN		1.095
#define CH_BOND_ID		21
#define BOND_RADIUS		0.07
#define ALPHA_BOND_RADIUS	0.48
#define LARGEST_ATOM		2.10

/* Current option state */

#define OF_RL	0x0002	/* Stereo in [R L] mode */		// Note: oflags | oF_RL = StereoRL, oflags & ~OF_RL = StereoLR.  MB_ST to enable stereo.
#define OF_BO	0x0800	/* Display backbone only */
#define OF_IH	0x1000	/* Don't display hydrogen atoms */
#define OF_IW	0x2000	/* Don't display water molecules */
#define OF_BI	0x4000	/* Derive bond info from XYZ location only */
#define OF_QT	0x8000	/* Queue file type explicitly specified */

/* KH - option states which I have removed */
#define OF_DM	0x0001	/* Switch to default model on rotation */ 
#define OF_RR	0x0004	/* Redraw only on REDRAW or NEW VIEW */
#define OF_VF	0x0008	/* NEW VIEW can select SF model */
#define OF_VB	0x0010	/* NEW VIEW can select BS model */
#define OF_VW	0x0020	/* NEW VIEW can select WF model */
#define OF_VA	0x0040	/* NEW VIEW can select AT model */
#define OF_VS	0x0080	/* NEW VIEW can select Stereo */
#define OF_VR	0x0100	/* NEW VIEW can change Rotation */
#define OF_VZ	0x0200	/* NEW VIEW can reset Zoom */
#define OF_VO	0x0400	/* NEW VIEW can reset Origin */



/* Menu definitions [0-19] */
#define MB_SF	0			/* SF */
#define MB_BS	1			/* BS */
#define MB_WF	2			/* WF */
#define MB_AT	3			/* AT */
#define MB_ST	4			/* STEREO */
#define MB_RE	5			/* REDRAW */
#define MB_HE	6			/* HELP */
#define MB_QU	7			/* QUIT */
#define MB_ZI	8			/* ZOOM IN */
#define MB_ZO	9			/* ZOOM OUT */
#define MB_NV	10			/* NEW VIEW */
#define MB_OP	11			/* OPTIONS */
#define MB_QI	12			/* QI */
#define MB_N	13			/* Total number of menu buttons */


/* Menu structure flag values */
#define BF_E	0x000F		/* Special effects */
#define BF_B	0x0010		/* Region is a normal button */
#define BF_I	0x0020		/* Region is a button inactive in toggled state */
#define BF_R	0x0040		/* Region starts a new line */
#define BF_C	0x0080		/* Region starts a new line with button clearance */
#define BF_S	0x0100		/* Region starts a new line with extra space */
#define BF_Z	0x0200		/* Region should be cleared before text is drawn */
#define BF_A	0x4000		/* Button is active even if toggled */
#define BF_T	0x8000		/* Button in toggled state */

struct bnd
{	
	int		toward;			/* Atom bonded to */
	float		len;			/* Bond length (/2) */
	float		theta;			/* Z-axis rotation */
	float		phi;			/* Y-axis rotation */
};


struct atm
{	
	short		element;		/* Element */
	short		flags;			/* Flags */
	float		x;			/* X coordinate */
	float		y;			/* Y coordinate */
	float		z;			/* Z coordinate */
	struct bnd	achc;			/* Alpha carbon peptide bond toward carbonyl */
	struct bnd	acha;			/* Alpha carbon peptide bond toward imino */
	struct bnd	bond[MAXBONDS];		/* Bond descriptions */
};

struct adist
{	float		r;
	struct atm *pa;
};


/* Atomic radii, bond lengths, and similar things are in Angstroms.
** The display sizes of the atoms are, to some extent, a fudge.
*/
struct atmprop						/* Atom properties (in ETYPES order) */
{
	float		d_radius;			/* Display radius of this element type */
	float		b_radius;			/* Bonding radius (max) of this element type */
	float		color[4];			/* Color (R G B A) */
} aprops[] =
{	
 	{ 1.50, 1.00, { 0.80, 0.10, 0.80, 1.00 } },	/* ? purple */
	{ 1.17, 0.37, { 0.50, 0.50, 0.50, 1.00 } },	/* H light grey */
	{ 1.75, 0.77, { 0.30, 0.30, 0.30, 1.00 } },	/* C dark grey */
	{ 1.55, 0.77, { 0.20, 0.20, 0.80, 1.00 } },	/* N blue */
	{ 1.40, 0.74, { 0.80, 0.10, 0.20, 1.00 } },	/* O red */
	{ 2.04, 1.11, { 0.10, 0.60, 0.10, 1.00 } },	/* P green */
	{ 1.84, 1.06, { 0.60, 0.60, 0.15, 1.00 } }	/* S yellow */
 };

struct atmprop acprops[] =				/* Pseudo-atom properties in alpha-carbon chain */
{	
 	{ 0.80, 0.00, { 0.60, 0.65, 0.30, 1.00 } },	/* Normal gold */
	{ 0.80, 0.00, { 0.15, 0.65, 0.60, 1.00 } },	/* Amino end turquoise */
	{ 0.80, 0.00, { 0.70, 0.40, 0.40, 1.00 } },	/* Carboxyl end pink */
};

/* Initialize OpenGL environment */
float front_specular[] = { 0.6, 0.6, 0.6, 1.0 };
float ambient0[] = { 0.3, 0.3, 0.3, 1.0 };
float diffuse0[] = { 1.0, 1.0, 1.0, 1.0 };
float position0[] = { 1.5, 1.5, 1.5, 0.0 };
float ambient1[] = { 0.15, 0.15, 0.15, 1.0 };
float diffuse1[] = { 0.5, 0.5, 0.5, 1.0 };
float position1[] = { -1.5, -1.5, 1.5, 0.0 };
float lmodel_ambient[] = { 0.5, 0.5, 0.5, 1.0 };
float lmodel_twoside[] = { GL_TRUE };
 


/* Comparison function for sorting so we render nearest first */
int cmpf( struct adist *pa, struct adist *pb )
{
	if (pa->r < pb->r) return (-1);
	if (pa->r > pb->r) return (1);
	return (0);
} // cmpf
 
 

kfoldingMoleculeWidget::kfoldingMoleculeWidget( QWidget* parent, const char* name, kfoldingConfigData* config ) 
	: QGLWidget( QGLFormat( DoubleBuffer | Rgba | DirectRendering ), parent, name ),
	  _config( config ),
	  _zoomLevel( 0 )
{
	if ( !_config )
		return;

	// set up the initial parameters, as per fpd
	_zoom = IZOOM;
	xyzfile = ( char* ) XYZFILE;
	xyztime = 0;
	
	_model = MB_SF;
	_atom = NULL;
	
	_xsiz = width();
	_ysiz = height();
	_rotx = 0.0f;
	_roty = 0.0f;
	_rotz = 0.0f;
	_dvx = 0.0f;
	_dvy = 0.0f;
	_dvz = 1.0f;
	_ox = 0.0f;
	_oy = 0.0f;
	_oz = 0.0f;
		
	_numAtoms = 0;
	_nh2o = 0;
	
#ifdef HAVE_GL_GLU_H
	qobj = NULL;
	qobj = gluNewQuadric();
	gluQuadricDrawStyle(qobj, GLU_FILL);
#endif

	setOptions();

	QToolTip::add( this, i18n( "Left-mouse drag: rotate\nRight-mouse drag: move" ) );
		
	connect( this, SIGNAL( leftDrag( QPoint, QPoint ) ), this, SLOT( slotRotate( QPoint, QPoint ) ) );
	connect( this, SIGNAL( rightDrag( QPoint, QPoint ) ), this, SLOT( slotTranslate( QPoint, QPoint ) ) );
	return;
} // kfoldingMoleculeWidget ctor


kfoldingMoleculeWidget::~kfoldingMoleculeWidget()
{	
	if ( _atom != NULL )
		free( _atom );
	
#ifdef HAVE_GL_GLU_H
	gluDeleteQuadric( qobj );
#endif
	return;
} // kfoldingMoleculeWidget dtor



void kfoldingMoleculeWidget::setOptions( )
{
	char *fdir;

	
	if ( !_config )
		return;
		
	memset( filname, '\0', sizeof( filname ) );
	memset( xyzname, '\0', sizeof( xyzname ) );

	const char *tmpwd = _config->workingDir().latin1();
	fdir = ( char* ) malloc ( ( strlen( tmpwd ) + 1 ) * sizeof( char ) );
	strncpy( fdir, tmpwd, strlen( tmpwd ) );
	fdir[ strlen( tmpwd )  ] = '\0';

	int i = 0;
	if (fdir != NULL)
	{	strncpy(filname, fdir, sizeof(filname) - 20);
		filname[sizeof(filname) - 20] = '\0';
		i = strlen(filname);
		if (filname[i - 1] != '/')
		{	filname[i] = '/';
			filname[++i] = '\0';
		}
		strcpy(xyzname, filname);
		free( fdir );
	}

	if (xyzfile[0] == '/')
		i = 0;
	strncpy(&xyzname[i], xyzfile, sizeof(xyzname) - 1 - i);
	xyzname[sizeof(xyzname) - 1] = '\0';
	
	_oflags = 0;
	
	switch( _config->model() ) {
		case kfoldingConfigData::SpaceFilling:
			_model = MB_SF;
			break;
		case kfoldingConfigData::BallandStick:
			_model = MB_BS;
			break;
		case kfoldingConfigData::AlphaCarbonTrace:
			_model = MB_AT;
			break;
		case kfoldingConfigData::WireFrame:
			_model = MB_WF;
			break;
		default:
			break;
	} // switch
	
	if ( !_config->showH2O() )
		_oflags |= OF_IW;
	if ( !_config->showH2() )
		_oflags |= OF_IH;
	if ( !_config->nonBackbone() )
		_oflags |= OF_BO;
		
	return;
} // setOptions


void kfoldingMoleculeWidget::update()
{
	paintGL();
	return;
} // paintGL



void kfoldingMoleculeWidget::zoomIn()
{
	if ( _zoomLevel <= MAXZOOM ) {
		_zoom /= ZOOMF;
		_zoomLevel++;
		paintGL();
	} else {
		emit drawComplete();
	}
	return;
} // slotZoomIn



void kfoldingMoleculeWidget::zoomOut()
{
	if ( _zoomLevel >= -MAXZOOM ) {
		_zoom *= ZOOMF;
		_zoomLevel--;
		paintGL();
	} else {
		emit drawComplete();
	}
	return;
} // slotZoomOut



void kfoldingMoleculeWidget::mousePressEvent( QMouseEvent* event )
{
	switch( event->button() ) {
		case Qt::LeftButton:
			_leftDragStartPos = event->pos();
			break;
		case Qt::RightButton:
			_rightDragStartPos = event->pos();
			break;
		default:
			QGLWidget::mousePressEvent( event );
			break;
	}  // switch
	return;
} // mousePressEvent



void kfoldingMoleculeWidget::mouseReleaseEvent( QMouseEvent* event )
{
	QPoint releasePos = event->pos();

	if ( event->button() == Qt::LeftButton ) {
		if ( _leftDragStartPos.isNull() )
			return;
			
		if ( event->pos() != _leftDragStartPos ) {
			emit leftDrag( _leftDragStartPos, releasePos );
		} // if
	}
	else if ( event->button() == Qt::RightButton ) {
		if ( _rightDragStartPos.isNull() )
			return;
			
		if ( event->pos() != _rightDragStartPos ) {
			emit rightDrag( _rightDragStartPos, releasePos );
		}
	}
	else {
		QGLWidget::mouseReleaseEvent( event );
	}
	return;
} // mouseReleaseEvent



#ifndef QT_NO_WHEELEVENT

void kfoldingMoleculeWidget::wheelEvent( QWheelEvent* event )
{
	event->accept();
	
	if ( event->delta() > 0 ) {
		zoomIn();
	}
	else if ( event->delta() < 0 ) {
		zoomOut();
	} // else
	return;
} // wheelEvent

#endif



void kfoldingMoleculeWidget::initializeGL()
{
	glClearColor(0.0, 0.0, 0.0, 0.0);
	glColor3f( 1.0, 0.0, 0.0 );
	glClearDepth(1.0);				// Doesn't hurt 
	glDepthFunc(GL_LESS);				// Doesn't hurt 
	glEnable(GL_DEPTH_TEST); 
	
	// Lighting 
	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient0);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse0);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient1);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse1);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);

	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);
	glShadeModel(GL_SMOOTH);

	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular); 
	//glDrawBuffer(GL_FRONT_LEFT);	// Change to GL_FRONT_AND_BACK for animation  
	glDrawBuffer( GL_FRONT_AND_BACK );

	// Window size 
	return;
} // initializeGL



void kfoldingMoleculeWidget::resizeGL( int w,int h )
{
	_xsiz = w;
	_ysiz = h;

	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	glViewport(0, 0, ( GLint ) _xsiz, ( GLint) _ysiz );
	glMatrixMode( GL_MODELVIEW );
	
	updateGL();
	return;
} // resizeGL



void kfoldingMoleculeWidget::paintGL()
{
	if ( !_config )
		return;

	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

	draw_scene();
	glFlush();
	emit drawComplete();
	return;
} // paintGL



void kfoldingMoleculeWidget::slotRotate( QPoint startPos, QPoint endPos  )
{
	float by1, by2, by3, bz1, bz2, bz3, c1, c2, c3, dd, ee;
	float dx, dy, dz, d;
	int x1, x2, y1, y2;
	
	x1 = startPos.x();
	y1 = startPos.y();
	x2 = endPos.x();
	y2 = endPos.y();

	dx = 0.0f;
	dy = 0.0f;
	dz = 0.0f;
	
//	if (debug >= 4)
//		printf("Left button released: (%d,%d), (%d,%d)\n", x1, y1, x2, y2);
			
	d = sqrt((float) (sq(x1 + x2 - _xsiz ) + sq(y1 + y2 - _ysiz)) / (float) (_xsiz * _ysiz));
	dz = M_PI * d * ((x2 * y1 - x1 * y2) * 2 - (x2 - x1) * _ysiz + (y2 - y1) * _xsiz) / (float) (sq(_xsiz) + sq(_ysiz));
	
	if ((d = M_PI/2 * (1 - d)) < 0) 
		d = 0.0;
	
	dx = d * (float) (y2 - y1) / (float) _ysiz;
	dy = d * (float) (x2 - x1) / (float) _xsiz;
	
	if ((d = sqrt(dx * dx + dy * dy + dz * dz)) == 0.0)
		return;		/* No motion, no need to recalculate */
	
	dx /= d;
	dy /= d;
	dz /= d;
			
//	if (debug >= 4)
//		printf("Current (x, y, z) = (%g, %g, %g)\n", rotx, roty, rotz);


	/* Define Y and Z basis vectors, rotate to match molecule on screen */
	by1 = 0.0;
	by2 = 1.0;
	by3 = 0.0;
			
	rot2(_rotz * M_PI/180.0, &by1, &by2);
	rot2(_roty * M_PI/180.0, &by3, &by1);
	rot2(_rotx * M_PI/180.0, &by2, &by3);

	bz1 = 0.0;
	bz2 = 0.0;
	bz3 = 1.0;
	rot2(_roty * M_PI/180.0, &bz3, &bz1);
	rot2(_rotx * M_PI/180.0, &bz2, &bz3);

	/* Rotate basis vectors according to the mouse motion */

//	if (debug >= 4) {	
//		printf("Protein Y = (%g, %g, %g), Z = (%g, %g, %g)\n", by1, by2, by3, bz1, bz2, bz3);
//		printf("Mouse rotation d around (dx, dy, dz) = %g around (%g, %g, %g)\n", d, dx, dy, dz);
//	}
			
	dd = atan2(dz, dy);
	c1 = dx;
	c2 = dy;
	c3 = dz;
	rot2(-dd, &c2, &c3);
	ee = atan2(c2, c1);
//	if (debug >= 4)
//	{	printf("dd = %g, ee = %g\n", dd, ee);
//		rot2(-ee, &c1, &c2);
//		printf("Rotated c (dx, dy, dz) = (%g, %g, %g)\n", c1, c2, c3);
//	}

	rot2(-dd, &by2, &by3);
	rot2(-dd, &bz2, &bz3);
	rot2(-ee, &by1, &by2);
	rot2(-ee, &bz1, &bz2);
	rot2(d, &by2, &by3);
	rot2(d, &bz2, &bz3);
	rot2(ee, &by1, &by2);
	rot2(ee, &bz1, &bz2);
	rot2(dd, &by2, &by3);
	rot2(dd, &bz2, &bz3);
//	if (debug >= 4)
//		printf("-> target Y = (%g, %g, %g), Z = (%g, %g, %g)\n", by1, by2, by3, bz1, bz2, bz3);
		
	_dvx = 0.0;				/* Also calculate direction of view */
	_dvy = 0.0;
	_dvz = 1.0;

	/* B is now in the target rotation, read it out into the protein view */

	dd = -atan2(bz2, bz3);
//	if (debug >= 4)
//		printf("Rot x = -atan2(%g, %g) = %g\n", bz2, bz3, dd);
			
	rot2(-dd, &by2, &by3);
	rot2(-dd, &bz2, &bz3);
	rot2(-dd, &_dvy, &_dvz);
	
//	if (debug >= 4)
//		printf("Y = (%g, %g, %g), Z = (%g, %g, %g)\n", by1, by2, by3, bz1, bz2, bz3);
			
	_rotx = 180.0/M_PI * dd;

	dd = atan2(bz1, bz3);
	rot2(-dd, &by3, &by1);
	rot2(-dd, &_dvz, &_dvx);
	
//	if (debug >= 4)
//	{	printf("Rot y = atan2(%g, %g) = %g\n", bz1, bz3, dd);
//		rot2(-dd, &bz3, &bz1);
//		printf("Y = (%g, %g, %g), Z = (%g, %g, %g)\n", by1, by2, by3, bz1, bz2, bz3);
//	}
	_roty = 180.0/M_PI * dd;

	dd = -atan2(by1, by2);
	rot2(-dd, &_dvx, &_dvy);
//	if (debug >= 4)
//	{	printf("Rot z = -atan2(%g, %g) = %g\n", by1, by2, dd);
///		rot2(-dd, &by1, &by2);
//		rot2(-dd, &bz1, &bz2);
//		printf("Y = (%g, %g, %g), Z = (%g, %g, %g)\n", by1, by2, by3, bz1, bz2, bz3);
//	}
	_rotz = 180.0/M_PI * dd;

//	if (debug >= 4)
//		printf("New (x, y, z) = (%g, %g, %g)\n", rotx, roty, rotz);

	// redraw scene
	paintGL();
	return;
} // slotRotate



/* Mouse right button dragged over molecule, do translation */
void kfoldingMoleculeWidget::slotTranslate( QPoint startPos, QPoint endPos )
{
	float bz1, bz2, bz3;
	int x1, x2, y1, y2, x1r, y1r;
	
	x1 = 0;
	y1 = -1;
	x1r = startPos.x();
	y1r = startPos.y();
	x2 = endPos.x();
	y2 = endPos.y();
	
	bz1 = _xspan * (float) (x2 - x1r) / (float) _xsiz;
	bz2 = _yspan * (float) (y1r - y2) / (float) _ysiz;
	bz3 = 0.0;
	if (bz1 * bz1 + bz2 * bz2 == 0.0)
		return;		/* No motion, no need to redisplay */
				
	//if (debug >= 4)
	//	printf("Right button released: _xspan %g, _yspan %g, dx %g, dy %g, o (%g, %g, %g)\n",
	//		_xspan, _yspan, bz1, bz2, _ox, _oy, _oz);

	rot2(-_rotx * M_PI/180.0, &bz2, &bz3);		/* Unrotate translation to match molecule coordinates */
	rot2(-_roty * M_PI/180.0, &bz3, &bz1);
	rot2(-_rotz * M_PI/180.0, &bz1, &bz2);

	_ox -= bz1;			/* Adjust origin in molecule coordinates */
	_oy -= bz2;
	_oz -= bz3;
		
	// redraw scene
	paintGL();
	return;
} // slotTranslate



int kfoldingMoleculeWidget::sq( int x )
{	
	return x * x;
} // sq



void kfoldingMoleculeWidget::rot2( float angle, float *x, float *y )
{
	float a, b, s, c;

	//kdDebug() << "rot2( " << angle << ", " << *x << ", " << *y << " )" << endl;
	a = *x;
	b = *y;
	s = sin(angle);
	c = cos(angle);
	*x = a * c - b * s;
	*y = b * c + a * s;
	//kdDebug() << "-> ( " << *x << ", " << *y << " )" << endl;
	return;
} // rot2


/* Calculate bond length, theta, and phi */
void kfoldingMoleculeWidget::bondparm( struct atm *p1, struct atm *p2, struct bnd *pb )
{
	float x, y, z, r, t;

	x = p2->x - p1->x;
	y = p2->y - p1->y;
	z = p2->z - p1->z;
	r = sqrt(x * x + y * y + z * z);
	t = (x == 0.0) ? 90.0 : acos(x / sqrt(x * x + y * y)) * 180.0/M_PI;
	pb->len = r / 2.0;
	pb->theta = (y < 0.0) ? -t: t;
	pb->phi = (r == 0.0) ? 0.0 : acos(z / r) * 180.0/M_PI;
	return;
} // bondparm



/* Calculate scaling from XYZ data only */
float kfoldingMoleculeWidget::biscale()
{
	int i;
	struct atm *p0, *p1;
	float min, best, lim;
	float d, f, g;
	char *p;
	struct adist *q0, *q1, *qb, *qe;

	qe = patom;
	for (i = _numAtoms + 1; --i > 0; )
	{	p0 = &_atom[i];
		qe->pa = p0;
		qe->r = p0->z;		// Sort on Z position
		++qe;
	}
	qsort(patom, _numAtoms, sizeof(struct adist), (int (*)(const void *, const void *)) &cmpf);

	p = ( char* )ETYPES;
	min = best = lim = 1e10;
	qb = patom;
	for (q0 = qb; q0 < qe; ++q0)
	{	if (p[(p0 = q0->pa)->element] != 'C')
			continue;
		while (qb->r < q0->r - lim) ++qb;
		for (q1 = qb; q1 < qe; ++q1)
		{	if (q1->r > q0->r + lim)
				break;
			p1 = q1->pa;
			if ((i = p[p1->element]) == 'H')
				f = 1.000;
			else if ((i == 'C') && (q1 > q0))
				f = 0.708;
			else if (i == 'N')
				f = 0.734;
			else if (i == 'O')
				f = 0.758;
			else continue;
			if ((d = (p0->x - p1->x) * f) < 0) d = -d;
			if (d > lim) continue;
			g = d * d;
			if ((d = (p0->y - p1->y) * f) < 0) d = -d;
			if (d > lim) continue;
			g += d * d;
			d = (p0->z - p1->z) * f;
			g += d * d;
			if ((g = sqrt(g)) > lim) continue;
			if (g == 0.0) continue;			// Yeah, some files are that bad
			if (g > best) best = g;
			else if (g < min)
			{	lim = g * 1.3;
				if (best > lim)
					best = min;
				min = g;
			}
			else continue;
			//kdDebug() << "min = " << min << " best = " << best << ", (" << p0 - _atom << " to " << p1 - _atom << ")" << endl;
		}
	}
	if (best > lim)
		best = min;
	return (best);
} // biscale




/* Construct bonds from scaled XYZ positions */
void kfoldingMoleculeWidget::bibond(float s)
{
	int n0, n1;
	int b0, b1;
	struct atm *p0, *p1;
	struct adist *q0, *q1;
	float d, g, r0, r, mbl;

	s *= 1.28;						// Margin seems to be from 1.13 to 1.45	***** RECHECK THIS *****
	mbl = MAX_BOND_LEN * s;
	for (q0 = patom; q0 < &patom[_numAtoms]; ++q0)
	{	r0 = aprops[(p0 = q0->pa)->element].b_radius;
		n0 = p0 - _atom;
		for (b0 = 0; b0 < MAXBONDS; ++b0)
			if (p0->bond[b0].toward == 0)
				break;
		for (q1 = q0; ++q1 < &patom[_numAtoms]; )
		{	p1 = q1->pa;
			if ((d = p0->z - p1->z) < 0) d = -d;
			if (d > mbl) break;
			r = (r0 + aprops[p1->element].b_radius) * s;
			if (d > r) continue;
			g = d * d;
			if ((d = p0->x - p1->x) < 0) d = -d;
			if (d > r) continue;
			g += d * d;
			if ((d = p0->y - p1->y) < 0) d = -d;
			if (d > r) continue;
			g += d * d;
			if (g == 0.0) continue;
			if ((g = sqrt(g)) > r) continue;
			n1 = p1 - _atom;
			if (g < r * 0.5)
			{	
				//kdDebug() << "Atoms " << n0 << " and " << n1 << " too close for bond (distance " << g << ", limit " << r * 0.5 << ")" << endl;
				continue;
			}
			if (b0 >= MAXBONDS)
			{	
				//kdDebug() << "Too many bonds to atom " << n0 << endl;
				break;
			}
			for (b1 = 0; b1 < MAXBONDS; ++b1)
				if (p1->bond[b1].toward == 0)
					break;
			if (b1 >= MAXBONDS)
			{	
				//kdDebug() << "Too many bonds to atom " << n1 << endl;
				continue;
			}
			p0->bond[b0].toward = n1;
			++b0;
			p1->bond[b1].toward = n0;
		}
	}
} // bibond



/* Clean up bond data */
void kfoldingMoleculeWidget::xyzclean()
{
	int i, j;
	int n0, n1;
	int b0, b1;
	struct atm *p0, *p1;

	// Step 1: identify XYZ file type (first "bond" pointer is atom ID with Genome and Tinker)

	i = _numAtoms;
	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	p0 = &_atom[n0];
		if (((n1 = p0->bond[0].toward) <= 0) || (n1 > _numAtoms))
			continue;
		p1 = &_atom[n1];
		for (b1 = MAXBONDS; --b1 >= 0; )
			if (p1->bond[b1].toward == n0)
			{	--i;
				break;
			}
	}
	j = (i * 5 > _numAtoms) ? 1 : 0;	// Which bond pointer is really the first one? 

	// Step 2: clean up bad bond pointers 

	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	p0 = &_atom[n0];
		for (b0 = j; b0 < MAXBONDS; ++b0)
		{	if ((n1 = p0->bond[b0].toward) <= 0)
				continue;
			if (n1 == n0)
			{	
				p0->bond[b0].toward = 0;	// Atom bonded to itself 
				//kdDebug() << "Atom " << n0 << " bonded to itself (bond " << b0 - j + 1 << ")" << endl;
				continue;
			}
			if (n1 > _numAtoms)
			{	
				p0->bond[b0].toward = 0;	// Bond pointer out of range
				//kdDebug() << "Bond " << b0 - j + 1 << "on atom " << n0 << " goes to atom " << n1 << " but there are only " << _numAtoms << " atoms total" << endl;
				continue;
			}
			p1 = &_atom[n1];
			i = 0;
			for (b1 = j; b1 < MAXBONDS; ++b1)
				if	(	(p1->bond[b1].toward == n0)
					&&	(++i > 1)
					)
				{	p1->bond[b1].toward = 0;	// Redundant 
					//kdDebug() << "Redundant bond from atom " << n1 << " toward " << n0 << endl;
				}
			if (i == 0)
			{	p0->bond[b0].toward = 0;		// Not reciprocal
				//kdDebug() << "Hanging bond from atom " << n0 << " toward " << n1 << endl;
			}
		}
	}

	// Step 3: compress the remaining bond pointers to start at index 0

	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	p0 = &_atom[n0];
		i = -1;
		for (b0 = j; b0 < MAXBONDS; ++b0)
			if ((n1 = p0->bond[b0].toward) > 0)
				p0->bond[++i].toward = n1;
		while (++i < MAXBONDS)
			p0->bond[i].toward = 0;
	}
	return;
} // xyzclean



void kfoldingMoleculeWidget::achain()
{
	int i, j;
	int n0, n1, n2;
	int b0, b1;
	struct atm *p0, *p1, *p2;
	char *p;
	float pbscale;
	int pbnum;

	p = ( char* ) ETYPES;

	// Step 1: find all carbon - carbonyl - imino sequences

	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	j = p[(p0 = &_atom[n0])->element];
		if (j == 'H')
			p0->flags |= AF_H;
		if (j != 'C')
			continue;
		i = 0;
		n2 = 0;
		p2 = NULL;						// (just to make the compiler happy) 
		for (b0 = MAXBONDS; --b0 >= 0; )
		{	if ((n1 = p0->bond[b0].toward) <= 0)
				continue;
			j = p[(p1 = &_atom[n1])->element];
			if (j == 'O')
				++i;
			else if (j == 'N')
			{	n2 = n1;
				i += 10;
			}
			else if (j == 'C')
			{	p2 = p1;
				i += 100;
			}
			else i += 1000;
		}
		if (i == 111)
		{	p0->flags |= AF_CN;			// Either peptide or amide 
			p2->achc.toward = n2;			// Temporary pointer from (probably alpha) C to N
		}
		else if (((i == 101) || (i == 102)) && (p2->achc.toward <= 0))
			p2->achc.toward = -1;			// At least eligible to be alpha C 
	}

	// Step 2: find imino - alpha-carbon sequences and peptide bonds

	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	p0 = &_atom[n0];
		if ((n1 = p0->achc.toward) <= 0)	// Find our temporary pointer
			continue;
		p0->achc.toward = -1;
		p1 = &_atom[n1];
		for (b1 = MAXBONDS; --b1 >= 0; )
		{	if	(	((n2 = p1->bond[b1].toward) <= 0)
				||	((p2 = &_atom[n2])->achc.toward == 0)
				) continue;
			p0->achc.toward = n2;			// Peptide bond, carbonyl side 
			p2->acha.toward = n0;			// Peptide bond, imino side 
		}
	}

	// Step 3: calculate angles and flush isolated peptide bonds 

	pbscale = 0.0;
	pbnum = 0;
	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	p0 = &_atom[n0];
		if ((n2 = p0->achc.toward) <= 0)
			goto dc;
		p2 = &_atom[n2];
		if	(	(p0->acha.toward == 0)
			&&	(p2->achc.toward == 0)
			)
		{	p2->acha.toward = 0;			// Isolated peptide bond, not in "backbone" 
dc:			p0->achc.toward = 0;
			continue;
		}
		_atom[n0].flags |= AF_CA;			// Mark both as alpha Carbon 
		_atom[n2].flags |= AF_CA;
		bondparm(p0, p2, &p0->achc);			// Calculate angles 
		bondparm(p2, p0, &p2->acha);
//		if (debug >= 3)
//		{	for (b0 = MAXBONDS; --b0 >= 0; )
//				if	(	((n1 = p0->bond[b0].toward) > 0)
//					||	(p[(p1 = &_atom[n1])->element] == 'N')
//					)
//				{	printf("Alpha C1 #%d, C2 #%d\n", n0, n2);
//					goto nb;
//				}
//			printf("No amino alpha C1 #%d, alpha C2 #%d\n", n0, n2);
//		} 
/*nb:*/		for (b0 = MAXBONDS; --b0 >= 0; )					// Mark atoms in backbone 
		{	if	(	((n1 = p2->bond[b0].toward) > 0)
				&&	(p[(p1 = &_atom[n1])->element] == 'N')
				)							// p1 is peptide N
			{	p0 = NULL;
				for (b1 = MAXBONDS; --b1 >= 0; )
				{	if ((n1 = p1->bond[b1].toward) <= 0)
						continue;
					_atom[n1].flags |= AF_BB;			// Mark if bonded to peptide N 
					if ((_atom[n1].flags & AF_CN) != 0)		// Already identified as carbonyl or amide
						p0 = &_atom[n1];			// p0 is peptide C
				}
				if (p0 != NULL)
				{	pbscale += sqrt((p1->x - p0->x) * (p1->x - p0->x) +
							(p1->y - p0->y) * (p1->y - p0->y) + (p1->z - p0->z) * (p1->z - p0->z));
					pbnum++;
					for (b1 = MAXBONDS; --b1 >= 0; )
						if ((n1 = p0->bond[b1].toward) > 0)
							_atom[n1].flags |= AF_BB;	// Mark if bonded to peptide C
				}
			}
		}
	}
//	if ((debug >= 2) && (pbnum > 0))
//	{	pbscale /= (float) pbnum * scale;
//		printf("Average C-N in peptide bond: %g (%.2f scale)\n", pbscale, pbscale / 1.33);
//	}

	// Extra step: identify water molecules

	for (n0 = 1; n0 <= _numAtoms; ++n0)
	{	if (p[(p0 = &_atom[n0])->element] != 'O')
nw:			continue;
		i = 0;
		for (b0 = MAXBONDS; --b0 >= 0; )
		{	if ((n1 = p0->bond[b0].toward) <= 0)
				continue;
			if (((p1 = &_atom[n1])->flags & AF_H) == 0)
				goto nw;
			++i;
			for (b1 = MAXBONDS; --b1 >= 0; )
				if	(	((n2 = p1->bond[b1].toward) > 0)
					&&	(n2 != n0)
					) goto nw;
		}
		if (i != 2)
			continue;
		p0->flags |= AF_H2O;				// Mark the Water
		for (b0 = MAXBONDS; --b0 >= 0; )
		{	if ((n1 = p0->bond[b0].toward) > 0)
				_atom[n1].flags |= AF_H2O;	// Mark the Hydrogens
		}
		++_nh2o;
	}
	return;
} // achain



int kfoldingMoleculeWidget::readxyz()
{
	int i, j, k, n;
	char *p, *q;
	FILE *fp;
	struct atm *pa;
	int na;
	int zi, zn, zz;
	float xa, xb, ya, yb, za, zb;
	float chdst;
	float d, g;
	struct stat st;

	p = ( char* )ETYPES;
	if (stat(xyzname, &st) != 0)
		goto co;
	if (st.st_mtime == xyztime)
		return (0);				// File hasn't changed since last time
	_nh2o = 0;
	if ((fp = fopen(xyzname, "r")) == NULL)
	{
co:		//sprintf(errmsg, "Can't open XYZ file \"%s\"", xyzname);
		//kdDebug() << "can't open XYZ file " << xyzname << endl;
		return (1);
	}
	na = 0;
	if (fgets(wbuf, sizeof(wbuf), fp) != NULL)
	{	pname[0] = '\0';
		pname[30] = '\0';
		sscanf(wbuf, "%d %30c", &na, pname);
		for (i = 0; i < 30; ++i)
			if (((j = pname[i]) == '\012') || (j == '\015'))
			{	pname[i] = '\0';
				break;
			}
	}
	if (pname[0] == '\0')						// Genome 
		strcpy(pname, "Genome");

	n = 0;
	_numAtoms = 0;
	++na;								// Tinker reports one too few atoms 
	if (na >= MAXATOMS)
	{	sprintf(errmsg, "Molecule too large.  Max = %d atoms", MAXATOMS);
		n = 2;
		goto cf;
	}
	if (_atom != NULL)
		free(_atom);
	i = sizeof(struct atm) * (na + 3);
	j = i + sizeof(struct adist) * (na + 3);
	if ((_atom = ( struct atm* ) malloc(j)) == NULL)
	{	
		sprintf(errmsg, "Can't assign memory.  Need %d bytes.", j);
		n = 3;
		goto cf;
	}
	memset(_atom, 0, j);					// Zeroing the array is important 
	patom = (struct adist *) ((char *) _atom + i);
	pa = &_atom[1];						// Atom 0 isn't used

	while (fgets(wbuf, sizeof(wbuf), fp) != NULL)
	{	
		if (sscanf(wbuf, "%d%7s%f%f%f%n", &zi, ebuf, &pa->x, &pa->y, &pa->z, &zn) < 5)
			continue;
		if ((_numAtoms = pa - _atom) != zi)
		{	if (_numAtoms >= na) break;		// Don't complain if last atom 
			sprintf(errmsg, "Sequence error in XYZ file expecting atom number %d", _numAtoms);
			n = 4;
			break;
		}
		pa->element = ((q = strchr(p, ebuf[0])) == NULL) ? 0 : (q - p);
		if ((_oflags & OF_BI) == 0)
		{	zi = 0;
			for (i = 0; ; ++i)
			{	zn += zi;
				if (sscanf(&wbuf[zn], "%d%n", &zz, &zi) != 1)
					break;
				if (i >= MAXBONDS)			// Too much garbage
				{	sprintf(errmsg, "Too many fields in XYZ file at atom number %d", pa - _atom);
					n = 5;
					goto cf;
				}
				pa->bond[i].toward = zz;
			}
		}
		if (_numAtoms >= na) break;			// End of atom list, according the the header line
		++pa;
	}
cf:	fclose(fp);
	if (n != 0)
		return (n);

	if (_numAtoms == 0)
	{	sprintf(errmsg, "No atoms in molecule");
		return (6);
	}

	if ((_oflags & OF_BI) == 0)
		xyzclean();			// Clean up bond data

	g = 0.0;				// Calculate scaling
	chdst = xa = ya = za = 1e10;
	xb = yb = zb = -1e10;
	for (i = _numAtoms + 1; --i > 0; )
	{	pa = &_atom[i];
		if (xa > pa->x) xa = pa->x;
		if (xb < pa->x) xb = pa->x;
		if (ya > pa->y) ya = pa->y;
		if (yb < pa->y) yb = pa->y;
		if (za > pa->z) za = pa->z;
		if (zb < pa->z) zb = pa->z;
		if ((_oflags & OF_BI) != 0)
			continue;
		for (j = MAXBONDS; --j >= 0; )
		{	if ((k = pa->bond[j].toward) <= 0)
				continue;
			if (pa->element * 10 + _atom[k].element != CH_BOND_ID)
				continue;				// Just look at lengths of C-H bonds
			bondparm(pa, &_atom[k], &pa->bond[j]);		// Temporarily calculate len
			g += pa->bond[j].len;
			d = g * 1.2 / (float) ++n;
			if (d > 1.44 * chdst)
				g = chdst * (float) n;
			if	(	(chdst > d)
				||	(	(pa->bond[j].len > chdst)
					&&	(pa->bond[j].len < d)
					)
				) chdst = pa->bond[j].len;		// Best guess for "typical" C-H bond
		}
	}
	
	if ((_oflags & OF_BI) == 0) {
		chdst *= 2;
	}
	if (chdst > 1e5)					// Either no C-H bonds, or we've been told to calculate it all
		chdst = biscale();
	if (chdst > 1e5)					// Not a reasonable value, set to default 
		chdst = CH_BOND_LEN;

	chdst *= 1.0177;					// This seems to give the best match in peptide bonds 

	xa = (xb + xa) / 2;
	xb -= xa;
	ya = (yb + ya) / 2;
	yb -= ya;
	za = (zb + za) / 2;
	zb -= za;
	if (xb < yb) xb = yb;
	if (xb < zb) xb = zb;
	xb = 1 / xb;
	_scale = xb * chdst / CH_BOND_LEN;

	for (i = _numAtoms + 1; --i > 0; )
	{	pa = &_atom[i];
		pa->x = (pa->x - xa) * xb;			// Adjust to fit in box (-1, -1, -1) to (+1, +1, +1) 
		pa->y = (pa->y - ya) * xb;
		pa->z = (pa->z - za) * xb;
	}
	if ((_oflags & OF_BI) != 0)
		bibond(_scale);					// Scaling seems to work from 1.13 to 1.45 
	for (i = _numAtoms + 1; --i > 0; )
	{	pa = &_atom[i];
		for (j = MAXBONDS; --j >= 0; )
		{	if ((k = pa->bond[j].toward) <= 0)
				continue;
			bondparm(pa, &_atom[k], &pa->bond[j]);	// Calculate len, theta, phi
			d = pa->bond[j].len * 2.0 / _scale;
			g = (aprops[pa->element].b_radius + aprops[_atom[k].element].b_radius) * 1.46;
			if (d > g)
			{	pa->bond[j].toward = 0;		// Something is wrong, bond is too long 
			//	if ((debug >= 2) && (i < k))
			//		printf("Bond from atom %d to atom %d too long (length %g, limit %g)\n", i, k, d, g);
			}
			//if	(	(debug >= 4)
			//	&&	(pa->element * 10 + _atom[k].element == CH_BOND_ID)
			//	) printf("C-H scaled bond length %g (%d to %d)\n", d, i, k);
		}
	}

	achain();		// Find peptide bonds and set up the alpha carbon chain 
	
	//kdDebug() << "Atom size scale: "  << _scale / xb << ", molecule scale: " << _scale << endl;
	//kdDebug() << "pname = " << pname << ", _numAtoms = " << _numAtoms << endl;

/*	if (debug >= 1)
	{	printf("Atom size scale %g, molecule scale %g\npname = \"%s\", _numAtoms = %d\n",
				scale / xb, _scale, pname, _numAtoms);
		if (_nh2o != 0)
			printf("Number of water molecules = %d\n", _nh2o);
	}
	if (debug >= 5)
		for (i = 1; i <= _numAtoms; ++i)
		{	pa = &_atom[i];
			printf(" Atom %03d, element %d (%c): (%g, %g, %g), bonds:", i,
					pa->element, p[pa->element], pa->x, pa->y, pa->z);
			for (na = 0; ; na = k)
			{	k = _numAtoms + 1;
				for (j = 0; j < MAXBONDS; ++j)
					if	(	((n = pa->bond[j].toward) > na)
						&&	(n < k)
						) k = n;
				if (k > _numAtoms) break;
				printf(" %d", k);
			}
			if ((pa->flags & AF_H2O) != 0)
				printf(" H2O");
			printf("\n");
		} */
	xyztime = st.st_mtime; 
	return 0; 
} // readxyz




int kfoldingMoleculeWidget::dobond( struct bnd *pb, int n )
{
	//makeCurrent();
	if	(	(pb->toward <= 0)				// No bond 
		||	(	((_oflags & OF_IH) != 0)		// Don't display bonds to hydrogen 
			&&	((_atom[pb->toward].flags & AF_H) != 0)	// and it's to hydrogen 
			)
		) return (0);
	glPushMatrix();
	glRotatef(pb->theta, 0, 0, 1);
	glRotatef(pb->phi, 0, 1, 0);
#ifndef HAVE_GL_GLU_H
	dCylinder(_bond_radius, pb->len, n);
#else
	gluCylinder(qobj, _bond_radius, _bond_radius, pb->len, n, 1);
#endif	
	if ((_atom[pb->toward].flags & AF_DL) == 0)	// The atom it's bonded to isn't being displayed 
	{	glTranslatef(0.0, 0.0, pb->len);	// Cap it so we don't see "inside" the bond 
#ifndef HAVE_GL_GLU_H
		dSphere(_bond_radius, n);
#else
		gluSphere(qobj, _bond_radius, n, n);
#endif
	}
	glPopMatrix();
	return 1;
} // dobond


/* Render the molecule */
void kfoldingMoleculeWidget::draw_molecule()
{
	int i, j, n;
	struct atm *pa;
	struct adist *pd, *pe;
	struct atmprop *pp;
	float sscale, d, g, h;
	float dqual;
	float shiny;

	glPushMatrix();
	glPushAttrib(GL_ALL_ATTRIB_BITS);

	glRotatef(_rotx, 1.0, 0.0, 0.0);
	glRotatef(_roty, 0.0, 1.0, 0.0);
	glRotatef(_rotz, 0.0, 0.0, 1.0);

	// Set feature scales suitable for the model type 

	sscale = 1.0;
	_bond_radius = BOND_RADIUS * _scale;
	if (_model == MB_BS)			// Ball and Stick 
		sscale = _config->ballSize() * 0.01f;
	else if (_model == MB_AT)		// Alpha Carbon Trace 
		_bond_radius = ALPHA_BOND_RADIUS * _scale;
	else if (_model == MB_WF)		// Wire Frame 
		sscale = 0.0;
	sscale *= _scale;
	dqual = DQUAL * pow(1.2, (double) _config->quality()) * FOV / _fov;
	
	// Select the atoms to be displayed and sort the list so closest are first

	pe = patom;
	for (i = _numAtoms + 1; --i > 0; )
	{	pa = &_atom[i];
		pa->flags &= ~AF_DL;
		if	(	(	((_oflags & OF_BO) != 0)	// Display backbone only 
				&&	((pa->flags & AF_BB) == 0)	// This isn't backbone 
				)
			||	(	((_oflags & OF_IH) != 0)	// Don't display hydrogen 
				&&	((pa->flags & AF_H) != 0)	// This is hydrogen 
				)
			||	(	((_oflags & OF_IW) != 0)	// Don't display water 
				&&	((pa->flags & AF_H2O) != 0)	// This is water 
				)
			||	(	(_model == MB_AT)		// Alpha trace model 
				&&	(pa->achc.toward == 0)		// No bond toward carboxyl end 
				&&	(pa->acha.toward == 0)		// No bond toward amino end 
				)
			) continue;					// Skip this atom
		pe->pa = pa;
		d = pa->x - _ox - _dvx * _zdist;
		g = d * d;
		h = d * _dvx;
		d = pa->y - _oy - _dvy * _zdist;
		g += d * d;
		h += d * _dvy;
		d = pa->z - _oz - _dvz * _zdist;
		g += d * d;
		h += d * _dvz;
		pe->r = sqrt(g);
		h = -h / pe->r;		// Cosine of the angle off the direction of view
		if	(	(h < 0)
			||	(	(h < 0.75)
				&&	(pe->r > sscale * 10 * MAX_BOND_LEN)
				)
			) continue;		// Atom and its bonds are out of the field of view
		if (pe->r < nearest / h + sscale * aprops[pa->element].d_radius)
			continue;		// Atom is too close to display properly 
		pa->flags |= AF_DL;
		++pe;
	} 
	//if (debug >= 2)
	//	printf("Atoms in display list: %d\n", pe - patom);
	//kdDebug() << "Atoms in display list: " << pe - patom << endl;
	qsort(patom, pe - patom, sizeof(struct adist), (int (*)(const void *, const void *)) &cmpf);

	// Display each atom in sorted list

	for (pd = patom; pd < pe; ++pd)
	{	pa = pd->pa;
		if (_model == MB_AT)
		{	if (pa->achc.toward == 0)
				pp = &acprops[2];	// Special color for carboxyl end 
			else if (pa->acha.toward == 0)
				pp = &acprops[1];	// Special color for amino end 
			else
				pp = &acprops[0];
		}
		else 
			pp = &aprops[pa->element];
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, pp->color);
		glPushMatrix();
		glTranslatef(pa->x - _ox, pa->y - _oy, pa->z - _oz);
		h = dqual / pd->r;
		if (_model == MB_SF)
			goto ds;
		d = 3.0 * sqrt(h * _bond_radius);
		if ((n = (int) d) < 8) n = 8;
		if (n > 32) n = 32;
		shiny = (float) n * 4.0 - 15.0;
		glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shiny);
		if (_model == MB_AT)
		{	dobond(&pa->achc, n);
			dobond(&pa->acha, n);
			goto ds;
		}
		else
		{	j = 0;
			for (i = MAXBONDS; --i >= 0; )
				j += dobond(&pa->bond[i], n);
			if ((j == 0) && (_model == MB_WF))
				goto ns;
ds:			g = sscale * pp->d_radius;
			if ((_model != MB_SF) && (g < _bond_radius))
				g = _bond_radius;
			d = sqrt(g * h);
			if ((n = (int) d) < 10) n = 10;
			if (n > 48) n = 48;
			shiny = (float) n * 4.0 - 15.0;
			glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, &shiny);
#ifndef HAVE_GL_GLU_H
			dSphere(g, n);
#else
			gluSphere(qobj, g, n, n);
#endif
		}
ns:		glPopMatrix();
	} 

	glPopAttrib();
	glPopMatrix(); 
	//kdDebug() << "kfoldingMoleculeWidget::draw_molecule() - done!" << endl;
	return;
} // draw_molecule



/* Render an OpenGL image of the molecule */
void kfoldingMoleculeWidget::draw_scene()
{
	float d, e, g;
	float aspect;
	float sdep;
	#if DLIST
	GLint dlist;
	#endif
	double cplane[4];

	//kdDebug() << "kfoldingMoleculeWidget::draw_scene" << endl;
	//makeCurrent();
//	if (((_oflags ^ coflags) & OF_BI) != 0)
		xyztime = 0;
	cmodel = _model;
	czoom = _zoom;
	//cstflg = menu[MB_ST].flags;
	csdctr = _config->stereoDepth();
	cdqctr = _config->quality();
	cbsbsiz = _config->ballSize() * 0.01f;
	coflags = _oflags;

	
	_xsiz = width();
	_ysiz = height();
	// Make sure the current pixmap is the right size for the screen

	/*if	(	(ywsiz != ytop + height())
		||	(xwsiz != width())
		)					// Assign a new pixmap
	{	//_xsiz = xwsiz;
		//_ysiz = ywsiz - ytop; */

		//makeCurrent();
		//initializeGL();
	//}

	if (readxyz() != 0)				// Read in the XYZ file data 
	{	//demsg("Error reading XYZ file:", errmsg);
		return;
	}
	
#if DLIST
	dlist = glGenLists(1);
	glNewList(dlist, GL_COMPILE);
#endif

	// Set perspective based on scaling and current view 

	sdep = SDEP / pow(DEPF, (double) _config->stereoDepth() );
	if ((_oflags & OF_RL) == 0)
		sdep *= 1.5;						// Viewpoint is relatively closer to the screen in [R L] mode 
	aspect = (float) _xsiz / (float) _ysiz;
	_fov = FOV;
	_zdist = _zoom;
	if ( ( _config->display() == kfoldingConfigData::StereoLR ) || ( _config->display() == kfoldingConfigData::StereoRL ) ) 	// Stereo
	{	_fov /= sdep;
		_zdist *= sdep * 1.3;
	}
	e = 1.732 + LARGEST_ATOM * _scale;		// This adjustment should really be in readxyz
	d = _zdist + e;
	g = sqrt(d * d + e);
	if ((d /= cos(_fov * M_PI/360.0)) < g) g = d;
	if ((d = _zdist - e) < g * 0.01) d = g * 0.01;
	
	//kdDebug() << "_zoom  = " << _zoom <<  ", _zdist = " << _zdist << ", near =  " << d << ", far = " << g << endl;
	
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	e = d * tan(_fov * M_PI/360.0 / sqrt(aspect));
	glFrustum(-e * aspect, e * aspect, -e, e, d, g);
	_yspan = 2.0 * e * sqrt(g / d);
	_xspan = _yspan * aspect;
	nearest = d;
	glMatrixMode(GL_MODELVIEW);

	// Clear the bitmap and render the requested image

	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glLoadIdentity();
	if ( ( _config->display() == kfoldingConfigData::StereoLR ) || ( _config->display() == kfoldingConfigData::StereoRL ) ) 	// Stereo 
	{	g = -(_zdist / sdep) * sqrt(aspect) * SSEP * FOV * M_PI/180.0 / 2.0;
		d = ((_oflags & OF_RL) == 0) ? _fov * SSEP : 0.0;		// Select stereo mode 
		cplane[1] = cplane[2] = cplane[3] = 0.0;
		glEnable(GL_CLIP_PLANE0);
		glPushMatrix();
		cplane[0] = 1.0;
		glClipPlane(GL_CLIP_PLANE0, cplane);
		glTranslatef(-g, 0.0, -_zdist);
		glRotatef(-d, 0.0, 1.0, 0.0);
		draw_molecule();
		glPopMatrix();
		cplane[0] = -1.0;
		glClipPlane(GL_CLIP_PLANE0, cplane);
		glTranslatef(g, 0.0, -_zdist);
		glRotatef(d, 0.0, 1.0, 0.0);
		draw_molecule();
		glDisable(GL_CLIP_PLANE0);
	}
	else
	{ 	glTranslatef(0.0, 0.0, -_zdist);
		draw_molecule();
	}
#if DLIST
	glEndList();
	glCallList(dlist);
	glDeleteLists(dlist, 1);
#endif
	glFinish();
/*	if ((cstflg & BF_T) != 0)	// Stereo   FIXME?
	{	XSetForeground(dpy, gc, yellow);
		XDrawLine(dpy, glwin, gc, M_LW / 2, 0, M_LW / 2, ysiz);
		XDrawLine(dpy, glwin, gc, xsiz / 2, 0, xsiz / 2, ysiz);
		XDrawLine(dpy, glwin, gc, xsiz - M_LW / 2, 0, xsiz - M_LW / 2, ysiz);
	} */
}

#ifndef HAVE_GL_GLU_H

/* Display cylinder (if not using GLU functions) */
void kfoldingMoleculeWidget::dCylinder(float r, float h, int n)
{
	float x, y;
	float a;
	int theta;

	//makeCurrent();
	n += n / 2;
	if (n > 100) n = 100;
	n += (100 - n) % 4;
	glBegin(GL_QUAD_STRIP);
	for (theta = 0; theta <= n; ++theta)
	{	a = (float) theta * M_PI*2.0 / (float) n;
		x = cos(a) * r;
		y = sin(a) * r;
		glNormal3f(x, y, 0.0);
		glVertex3f(x, y, h);
		glVertex3f(x, y, 0.0);
	}
	glEnd();
} // dCylinder


/* Display sphere (if not using GLU functions) */
void kfoldingMoleculeWidget::dSphere(float r, int n)
{
	float x, y, z;
	float zs;
	float cs;
	float c;
	int phi, theta;
	float st[101], ct[101];

	//makeCurrent();
	n += n / 2;
	if (n > 100) n = 100;
	n += (100 - n) % 4;
	for (theta = 0; theta <= n; ++theta)
	{	x = (float) theta * M_PI*2.0 / (float) n;
		st[theta] = sin(x);
		ct[theta] = cos(x);
	}
	c = 0.0;
	z = r;
	for (phi = 1; phi <= n / 4; ++phi)
	{	cs = c;
		zs = z;
		c = st[phi] * r;
		z = ct[phi] * r;
		glBegin((phi == 1) ? GL_TRIANGLE_FAN : GL_QUAD_STRIP);
		for (theta = 0; theta <= n; ++theta)
		{	if ((phi != 1) || (theta == 0))
			{	x = ct[theta] * cs;
				y = st[theta] * cs;
				glNormal3f(x, y, zs);
				glVertex3f(x, y, zs);
			}
			x = ct[theta] * c;
			y = st[theta] * c;
			glNormal3f(x, y, z);
			glVertex3f(x, y, z);
		}
		glEnd();
		glBegin((phi == 1) ? GL_TRIANGLE_FAN : GL_QUAD_STRIP);
		for (theta = 0; theta <= n; ++theta)
		{	if ((phi != 1) || (theta == 0))
			{	x = ct[theta] * cs;
				y = -st[theta] * cs;
				glNormal3f(x, y, -zs);
				glVertex3f(x, y, -zs);
			}
			x = ct[theta] * c;
			y = -st[theta] * c;
			glNormal3f(x, y, -z);
			glVertex3f(x, y, -z);
		}
		glEnd();
	}
} // dSphere

#endif


#include "kfoldingmoleculewidget.moc"
