/***************************************************************************
 *   Copyright (C) 2005 by Nguyen The Toan   *
 *   nguyenthetoan@gmail.com   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/* Note to self:
	Be careful when implement per_game properties. One
	needs to free additional_options properly.
*/

#include "pref.h"
#include "DisplayPref.h"
#include "kxmamelogo.h"

#include <qcheckbox.h>
#include <qstringlist.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qtextedit.h>
#include <qgroupbox.h>
#include <qslider.h>
#include <qpushbutton.h>

#include <kmessagebox.h>
#include <klocale.h>
#include <knuminput.h>
#include <kcombobox.h>
#include <klineedit.h>
#include <kapplication.h>

#include "kxmame.h"
#include "properties.h"
#include "io.h"
#include "gxmame.h"
#include "game_list.h"
#include "keyboard.h"
#include "kxmame_joy.h"
#include "options_string.h"

extern "C" {
#include <stdio.h>
#include <glib.h>
}

const char *bios_version[] = {
	I18N_NOOP("Europe, 1 Slot (also been seen on a 4 slot)"),
	I18N_NOOP("Europe, 4 Slot"),
	I18N_NOOP("US, 2 Slot"),
	I18N_NOOP("US, 6 Slot (V5?)"),
	I18N_NOOP("Asia S3 Ver 6"),
	I18N_NOOP("Japan, Ver 6 VS Bios"),
	I18N_NOOP("Japan, Older"),
	I18N_NOOP("Universe Bios v1.0 (hack)"),
	I18N_NOOP("Universe Bios v1.1 (hack)"),
	I18N_NOOP("Debug (Development) Bios"),
	I18N_NOOP("AES Console (Asia?) Bios"),
	I18N_NOOP("Universe Bios v1.2 (hack)"),
	NULL
};

#define BUFFER_SIZE 1000
static gchar title[BUFFER_SIZE];

bool isInPrefDialog;

kxmamePreferences::kxmamePreferences(RomEntry *rom)
	: KDialogBase(IconList, 0, Default|Ok|Cancel, NoDefault)
{
	setUpdatesEnabled(FALSE);
	QStringList Strs;

	if(!isInPrefDialog) joy_focus_off();
	
	mRom = rom;

	xmame_get_options (current_exec);

	//actionButton(KDialogBase::Apply)->setEnabled(FALSE);
	
	if(rom) {
		localGameOpts = load_options(rom);
		if(!localGameOpts) {
			localGameOpts = (GameOptions *)g_malloc(sizeof(default_options));
			memcpy((void *)localGameOpts, (void *)&default_options, sizeof(default_options));
			localGameOpts->additional_options = g_strdup(default_options.additional_options);
		}
		setCaption(i18n("Properties for %1").arg(rom->name_in_list));
		setButtonText(KDialogBase::Default, i18n("Reset to Defaults"));
		
	} else {
		/* make a local copy of default_options */
		localGameOpts = (GameOptions *)g_malloc(sizeof(default_options));
		memcpy((void *)localGameOpts, (void *)&default_options, sizeof(default_options));
		setCaption(i18n("Default Properties"));
		actionButton(KDialogBase::Default)->hide();
		iconSize = gui_prefs.iconSize;
		scaledScreenshot = gui_prefs.scaledScreenshot;
		GameCheck = gui_prefs.GameCheck;
		VersionCheck = gui_prefs.VersionCheck;
		use_xmame_options = gui_prefs.use_xmame_options;
		showVideoPreview = gui_prefs.showVideoPreview;
#ifdef ENABLE_JOYSTICK
		gui_joy = gui_prefs.gui_joy;
		strncpy(Joystick_in_GUI, gui_prefs.Joystick_in_GUI, 256);
		for(int i=0;i<JOYTOTAL;i++)
			joyButton[i] = gui_prefs.kxmameJoyButton[i];
#endif
	}
	
	if(rom) {
		QFrame *frame;
		frame = addPage(i18n("General"), QString(rom->name_in_list)+" - \""+
				rom->romname+"\"", infoPix());
		QHBoxLayout *gameTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
		m_gamePage = new kxmameGamePref(frame);
		gameTopLayout->addWidget(m_gamePage);

		setupGamePrefFrame();	
	}
	
	QFrame *frame;
	frame = addPage(i18n("Display"), i18n("Display Options"), displayPix());
	QHBoxLayout *topLayout1 = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
	m_displayPage = new kxmameDisplayPref(frame);
	topLayout1->addWidget(m_displayPage);

	setupDisplayPrefFrame();

	frame = addPage(i18n("Rendering"), i18n("Rendering Options"), renderPix());
	QHBoxLayout *topLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
	m_renderPage = new kxmameRenderPref(frame);
	topLayout->addWidget(m_renderPage);

	setupRenderPrefFrame();

	frame = addPage(i18n("Sound"), i18n("Sound Options"), soundPix());
	QHBoxLayout *soundTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
	m_soundPage = new kxmameSoundPref(frame);
	soundTopLayout->addWidget(m_soundPage);

	setupSoundPrefFrame();
	
	frame = addPage(i18n("Controllers"), i18n("Controllers Options"), ctrlPix());
	QHBoxLayout *ctrlTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
	m_ctrlPage = new kxmameCtrlPref(frame);
	ctrlTopLayout->addWidget(m_ctrlPage);

	setupCtrlPrefFrame();
	
	frame = addPage(i18n("Miscellaneous"), i18n("Miscellaneous Options"), miscPix());
	QHBoxLayout *miscTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
	m_miscPage = new kxmameMiscPref(frame);
	miscTopLayout->addWidget(m_miscPage);

	setupMiscPrefFrame();

	if((!rom)||(rom->vector)) {
		frame = addPage(i18n("Vector"), i18n("Vector Options"), vectorPix());
		QHBoxLayout *vectorTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
		m_vectorPage = new kxmameVectorPref(frame);
		vectorTopLayout->addWidget(m_vectorPage);
	
		setupVectorPrefFrame();
	}

	if (!rom) {
		frame = addPage(i18n("Kxmame"), i18n("Startup Options"), kxmamePix());
		QHBoxLayout *kxmameTopLayout = new QHBoxLayout( frame, 0, KDialog::spacingHint() );
		m_kxmamePage = new kxmameStartupPref(frame);
		kxmameTopLayout->addWidget(m_kxmamePage);
	
		setupKxmamePrefFrame();
	}
	
	isDefault = TRUE;
	isModified = FALSE;
	if(mRom) actionButton(KDialogBase::Default)->setEnabled(FALSE);
	
	setUpdatesEnabled(TRUE);
}

kxmamePreferences::~kxmamePreferences()
{
	if(mRom) {
		g_free(localGameOpts->additional_options);
	}
	g_free(localGameOpts);
	
	if(!isInPrefDialog) joy_focus_on();
}

void kxmamePreferences::OptModified()
{
	if((!mRom)||isModified) return;
	isModified = TRUE;
	if(isDefault) {
		isDefault = FALSE;
		actionButton(KDialogBase::Default)->setEnabled(TRUE);
		actionButton(KDialogBase::Cancel)->setEnabled(TRUE);
	}
}

void kxmamePreferences::prefJoyButtonPressed( int bt ) {
	if(m_kxmamePage->mKxmameFrame->gameSelectJoyButtonEdit ==
		  kapp->focusWidget() ) 
		m_kxmamePage->mKxmameFrame->gameSelectJoyButtonEdit->
				setText(QString::number(bt));
	if(m_kxmamePage->mKxmameFrame->panelSwitchJoyButtonEdit ==
		  kapp->focusWidget() ) 
		m_kxmamePage->mKxmameFrame->panelSwitchJoyButtonEdit->
				setText(QString::number(bt));
	if(m_kxmamePage->mKxmameFrame->execSwitchJoyButtonEdit ==
		  kapp->focusWidget() ) 
		m_kxmamePage->mKxmameFrame->execSwitchJoyButtonEdit->
				setText(QString::number(bt));
	if(m_kxmamePage->mKxmameFrame->parentJoyButtonEdit ==
		  kapp->focusWidget() ) 
		m_kxmamePage->mKxmameFrame->parentJoyButtonEdit->
				setText(QString::number(bt));
	if(m_kxmamePage->mKxmameFrame->altJoyButtonEdit ==
		  kapp->focusWidget() ) 
		m_kxmamePage->mKxmameFrame->altJoyButtonEdit->
				setText(QString::number(bt));
}

void
kxmamePreferences::combo_init (QComboBox   *combo,
	    const gchar *option, const gchar *key, gint value)
{
	QStringList combo_items;
	const gchar **values;
	int i;
	const gchar *str_value;
	bool setitem=FALSE;

	values = xmame_get_option_values (current_exec, option);
	if (values) {
		for (i=0; values[i]; i++) {
			combo_items << values[i];
			if (!key && (i == value)) {
				setitem=TRUE;
			}
		}

		combo->clear();
		combo->insertStringList(combo_items);
		combo->setEditable(FALSE);
		if(setitem)
			combo->setCurrentItem(value);

		if (key) {
			str_value = xmame_get_option_key_value (current_exec, option, key);
		} else if (value > -1) {
			str_value = xmame_get_option_value (current_exec, option, value);
		} else {
			str_value = NULL;
		}

		if (str_value) {
			for (i=0; i< combo->count(); i++)
				if(!QString::compare(combo->text(i),str_value)) {
					combo->setCurrentItem(i);
					break;
				}
		}
	}
}

void kxmamePreferences::setupGamePrefFrame()
{
	gchar *title, *value;
	
	//It's likely the sampleCheckLabel has the widest width
	//We set all other label to this length
	int preWidth = m_gamePage->mGameFrame->preSampleLabel->width();
	m_gamePage->mGameFrame->preSampleLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preYearLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preManuLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preCpuLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preSoundLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preScreenLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preColorLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preCloneLabel->setMinimumWidth(preWidth);
	m_gamePage->mGameFrame->preRomLabel->setMinimumWidth(preWidth);
	
	m_gamePage->mGameFrame->yearLabel->setText(mRom->year);
	m_gamePage->mGameFrame->manuLabel->setText(mRom->manu);
	title = get_rom_cpu_value (mRom);
	m_gamePage->mGameFrame->cpuLabel->setText(title);
	g_free (title);

	title = get_rom_sound_value (mRom);
	m_gamePage->mGameFrame->soundLabel->setText(title);
	g_free (title);
	
	/* doesn't display resolution if this is a vector game */
	if (!mRom->vector)
		value = g_strdup_printf ("%i x %i ", mRom->screen_y, mRom->screen_x);
	else
		value = g_strdup_printf (" ");
	
	title = g_strdup_printf ("%s (%s) %.2f Hz",
				 value,
				 mRom->horizontal ? "horizontal" : "vertical",
				 mRom->screen_freq);
	g_free (value);
	m_gamePage->mGameFrame->screenLabel->setText(title);
	g_free (title);

	m_gamePage->mGameFrame->colorLabel->
			setText(i18n("%1 colors").arg(QString::number(mRom->colors)));
	
	if (strcmp (mRom->cloneof, "-")) {
		title = get_rom_clone_name (mRom);
		m_gamePage->mGameFrame->cloneLabel->setText(title);
		g_free (title);
	} else {
		m_gamePage->mGameFrame->cloneLabel->hide();
		m_gamePage->mGameFrame->preCloneLabel->hide();
	}
	
	if(mRom->has_roms == NOTNEEDED) {
		m_gamePage->mGameFrame->romLabel->setText(i18n("None required"));
		m_gamePage->mGameFrame->romAuditButton->setEnabled(FALSE);
	} else {
		switch(mRom->status) {
			case CORRECT:
				m_gamePage->mGameFrame->romLabel->setText(i18n("Passed"));
				break;
			case INCORRECT:
				m_gamePage->mGameFrame->romLabel->setText(i18n("Incorrect"));
				break;
			case UNKNOWN:
			default:
				m_gamePage->mGameFrame->romLabel->setText(i18n("Unknown"));
				break;
		}
	};
	connect(m_gamePage->mGameFrame->romAuditButton, SIGNAL(clicked()),
			this, SLOT(auditRom()));

	if(mRom->nb_samples == 0) {
		m_gamePage->mGameFrame->sampleLabel->setText(i18n("None required"));
		m_gamePage->mGameFrame->sampleAuditButton->setEnabled(FALSE);
	} else {
		if (mRom->has_samples == CORRECT)
			m_gamePage->mGameFrame->sampleLabel->setText(i18n("Passed"));
		else if (mRom->has_samples == INCORRECT)
			m_gamePage->mGameFrame->sampleLabel->setText(i18n("Incorrect"));
		else 
			m_gamePage->mGameFrame->sampleLabel->setText(i18n("Not found"));
	}
	connect(m_gamePage->mGameFrame->sampleAuditButton, SIGNAL(clicked()),
			this, SLOT(auditSample()));
	
}

void kxmamePreferences::setupDisplayPrefFrame()
{
	QStringList Strs;
	
	m_displayPage->mDisplayFrame->throttleCheck->setChecked(localGameOpts->throttle);
	connect (m_displayPage->mDisplayFrame->throttleCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setThrottle(bool)) );

	m_displayPage->mDisplayFrame->scanlineCheck->setChecked(localGameOpts->scanlines);
	connect (m_displayPage->mDisplayFrame->scanlineCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setScanlines(bool)) );

	m_displayPage->mDisplayFrame->heightSpin->setValue(localGameOpts->heightscale);
	connect (m_displayPage->mDisplayFrame->heightSpin, SIGNAL(valueChanged(int)),
		 this, SLOT(setHeightScale(int)) );
	m_displayPage->mDisplayFrame->widthSpin->setValue(localGameOpts->widthscale);
	connect (m_displayPage->mDisplayFrame->widthSpin, SIGNAL(valueChanged(int)),
		 this, SLOT(setWidthScale(int)) );

	Strs.clear();
	Strs << i18n("Auto") << i18n("8 bits") << i18n("15 bits") << i18n("16 bits") << i18n("32 bits");
	m_displayPage->mDisplayFrame->bppCombo->clear();
	m_displayPage->mDisplayFrame->bppCombo->insertStringList(Strs);
	
	switch (localGameOpts->bpp) {
		case (8):
			m_displayPage->mDisplayFrame->bppCombo->setCurrentItem(1); break;
		case (15):
			m_displayPage->mDisplayFrame->bppCombo->setCurrentItem(2); break;
		case (16):
			m_displayPage->mDisplayFrame->bppCombo->setCurrentItem(3); break;
		case (32):
			m_displayPage->mDisplayFrame->bppCombo->setCurrentItem(4); break;
			/* case (0): or*/
		default:
			m_displayPage->mDisplayFrame->bppCombo->setCurrentItem(0); break;
	}
	connect (m_displayPage->mDisplayFrame->bppCombo, SIGNAL(activated(int)),
		 this, SLOT(setBpp(int)) );
	m_displayPage->mDisplayFrame->arbHeightCheck->setChecked(localGameOpts->arbheight);
	connect (m_displayPage->mDisplayFrame->arbHeightCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArbHeightToggle(bool)) );
	m_displayPage->mDisplayFrame->heightScaleIntNumInput->setEnabled(localGameOpts->arbheight);
	m_displayPage->mDisplayFrame->heightScaleIntNumInput->setValue(localGameOpts->arbheight);
	connect (m_displayPage->mDisplayFrame->heightScaleIntNumInput, SIGNAL(valueChanged(int)),
		 this, SLOT(setArbHeight(int)) );
	m_displayPage->mDisplayFrame->autoDoubleCheck->setChecked(localGameOpts->autodouble);
	connect (m_displayPage->mDisplayFrame->autoDoubleCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setAutoDouble(bool)) );
	Strs.clear();
	Strs << i18n("None") << i18n("Clockwise") << i18n("Anti-clockwise");
	m_displayPage->mDisplayFrame->rotateCombo->clear();
	m_displayPage->mDisplayFrame->rotateCombo->insertStringList(Strs);
	if (localGameOpts->ror)
		m_displayPage->mDisplayFrame->rotateCombo->setCurrentItem(1);
	else if (localGameOpts->rol)
		m_displayPage->mDisplayFrame->rotateCombo->setCurrentItem(2);
	else
		m_displayPage->mDisplayFrame->rotateCombo->setCurrentItem(0);
	connect (m_displayPage->mDisplayFrame->rotateCombo, SIGNAL(activated(int)),
		 this, SLOT(setRotate(int)) );
	m_displayPage->mDisplayFrame->norotateCheck->setChecked(localGameOpts->norotate);
	connect (m_displayPage->mDisplayFrame->norotateCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setNoRotate(bool)) );
	m_displayPage->mDisplayFrame->flipxCheck->setChecked(localGameOpts->flipx);
	connect (m_displayPage->mDisplayFrame->flipxCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setFlipx(bool)) );
	m_displayPage->mDisplayFrame->flipyCheck->setChecked(localGameOpts->flipy);
	connect (m_displayPage->mDisplayFrame->flipyCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setFlipy(bool)) );
	
	const gchar **values;
	Strs.clear();
	values = xmame_get_option_values (current_exec, "effect");

	if (values) {
		for (int i=0; values[i]; i++)
			Strs << values[i];
	}
	m_displayPage->mDisplayFrame->effectCombo->clear();
	m_displayPage->mDisplayFrame->effectCombo->insertStringList(Strs);
	m_displayPage->mDisplayFrame->effectCombo->setEditable(FALSE);
	m_displayPage->mDisplayFrame->effectCombo->setCurrentItem(localGameOpts->effect);
	connect (m_displayPage->mDisplayFrame->effectCombo, SIGNAL(activated(int)),
		 this, SLOT(setEffect(int)) );

	resize( QSize(600, 480).expandedTo(minimumSizeHint()) );

	m_displayPage->mDisplayFrame->dirtyCheck->setEnabled(xmame_has_option (current_exec, "dirty"));
	m_displayPage->mDisplayFrame->dirtyCheck->setChecked(localGameOpts->dirty);
	connect (m_displayPage->mDisplayFrame->dirtyCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setDirty(bool)) );
	m_displayPage->mDisplayFrame->sleepIdleCheck->setChecked(localGameOpts->sleepidle);
	connect (m_displayPage->mDisplayFrame->sleepIdleCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setSleepIdle(bool)) );
	
	m_displayPage->mDisplayFrame->gammaSlider->setValue(int(localGameOpts->gamma_correction * 100));
	g_snprintf (title, BUFFER_SIZE, i18n("%.2f"), localGameOpts->gamma_correction);
	m_displayPage->mDisplayFrame->gammaLabel->setText(title);
	connect (m_displayPage->mDisplayFrame->gammaSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setGamma(int)) );
	m_displayPage->mDisplayFrame->brightnessSlider->setValue(int(localGameOpts->brightness * 100));
	g_snprintf (title, BUFFER_SIZE, i18n("%.2f"), localGameOpts->brightness);
	m_displayPage->mDisplayFrame->brightnessLabel->setText(title);
	connect (m_displayPage->mDisplayFrame->brightnessSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setBrightness(int)) );
	
	m_displayPage->mDisplayFrame->autoFrameSkipCheck->setChecked(localGameOpts->autoframeskip);
	connect (m_displayPage->mDisplayFrame->autoFrameSkipCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setAutoFrameSkip(bool)) );
	m_displayPage->mDisplayFrame->frameToSkipLabel->setEnabled(!localGameOpts->autoframeskip);
	m_displayPage->mDisplayFrame->frameToSkipCombo->setEnabled(!localGameOpts->autoframeskip);
	

	Strs.clear();
	Strs << i18n("Draw every frame");
	for (int i=1; i<12; i++) {
		g_snprintf (title, BUFFER_SIZE, i18n("Skip %i of 12 frames"), i);
		Strs << title;
	}
	m_displayPage->mDisplayFrame->frameToSkipCombo->clear();
	m_displayPage->mDisplayFrame->frameToSkipCombo->insertStringList(Strs);
	m_displayPage->mDisplayFrame->frameToSkipCombo->setCurrentItem(localGameOpts->frameskip);
	connect (m_displayPage->mDisplayFrame->frameToSkipCombo, SIGNAL(activated(int)),
		 this, SLOT(setFrameToSkip(int)) );

	m_displayPage->mDisplayFrame->frameMethodLabel->setEnabled(localGameOpts->autoframeskip && xmame_has_option (current_exec, "frameskipper"));
	combo_init(m_displayPage->mDisplayFrame->frameMethodCombo, "frameskipper", NULL, localGameOpts->frameskipper);
	m_displayPage->mDisplayFrame->frameMethodCombo->setEnabled(localGameOpts->autoframeskip && xmame_has_option (current_exec, "frameskipper"));
	connect (m_displayPage->mDisplayFrame->frameMethodCombo, SIGNAL(activated(int)),
		 this, SLOT(setFrameMethod(int)) );
	m_displayPage->mDisplayFrame->maxFrameLabel->setEnabled(localGameOpts->autoframeskip);

	Strs.clear();
	Strs << "0" << "1" << "2" << "3" << "4" << "5";
	Strs << "6" << "7" << "8" << "9" << "10" << "11";
	m_displayPage->mDisplayFrame->maxFrameCombo->setEnabled(localGameOpts->autoframeskip);
	m_displayPage->mDisplayFrame->maxFrameCombo->clear();
	m_displayPage->mDisplayFrame->maxFrameCombo->insertStringList(Strs);
	m_displayPage->mDisplayFrame->maxFrameCombo->setEditable(FALSE);
	m_displayPage->mDisplayFrame->maxFrameCombo->setCurrentItem(localGameOpts->maxautoframeskip);
	connect (m_displayPage->mDisplayFrame->maxFrameCombo, SIGNAL(activated(int)),
		 this, SLOT(setMaxFrame(int)) );
}

void kxmamePreferences::setupRenderPrefFrame()
{
	QStringList Strs;
	bool foundGlres;
	
	switch (current_exec->type) {
		case XMAME_EXEC_X11:
		case XMAME_EXEC_XGL:

			combo_init(m_renderPage->mX11Render->videoCombo, "video-mode", NULL, localGameOpts->x11_mode);
			connect (m_renderPage->mX11Render->videoCombo, SIGNAL(activated(int)),
				 this, SLOT(setVideoMode(int)) );
			m_renderPage->mX11Render->fullScreenCheck->setChecked(localGameOpts->fullscreen);
			connect (m_renderPage->mX11Render->fullScreenCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setFullScreen(bool)) );
			m_renderPage->mX11Render->cursorCheck->setChecked(localGameOpts->cursor);
			connect (m_renderPage->mX11Render->cursorCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setCursor(bool)) );
			m_renderPage->mX11Render->mitCheck->setChecked(localGameOpts->mitshm);
			connect (m_renderPage->mX11Render->mitCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setMit(bool)) );
			m_renderPage->mX11Render->geometryCheck->setChecked(localGameOpts->xvgeom_flag);
			connect (m_renderPage->mX11Render->geometryCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setXVgeometry(bool)) );
			combo_init(m_renderPage->mX11Render->yuvCombo, "force-yuv", NULL, localGameOpts->force_yuv);
			connect (m_renderPage->mX11Render->yuvCombo, SIGNAL(activated(int)),
				 this, SLOT(setYuvMode(int)) );
			m_renderPage->mX11Render->xSyncCheck->setChecked(localGameOpts->xsync);
			connect (m_renderPage->mX11Render->xSyncCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setXSync(bool)) );
			m_renderPage->mX11Render->colorMapCheck->setChecked(localGameOpts->privatecmap);
			connect (m_renderPage->mX11Render->colorMapCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setColorMap(bool)) );
			m_renderPage->mX11Render->xilCheck->setChecked(localGameOpts->xil);
			connect (m_renderPage->mX11Render->xilCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setXil(bool)) );
			m_renderPage->mX11Render->xilMTCheck->setChecked(localGameOpts->mtxil);
			connect (m_renderPage->mX11Render->xilMTCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setXilMt(bool)) );
		
			m_renderPage->mX11Render->mitCheck->setEnabled(xmame_has_option (current_exec, "mitshm"));
			m_renderPage->mX11Render->cursorCheck->setEnabled(xmame_has_option (current_exec, "cursor"));
			m_renderPage->mX11Render->fullScreenCheck->setEnabled(xmame_has_option (current_exec, "fullscreen"));
			m_renderPage->mX11Render->geometryCombo->setEnabled(localGameOpts->xvgeom_flag);
			m_renderPage->mX11Render->xSyncCheck->setEnabled(xmame_has_option (current_exec, "xsync"));
			m_renderPage->mX11Render->xilCheck->setEnabled(xmame_has_option (current_exec, "xil"));
			m_renderPage->mX11Render->xilMTCheck->setEnabled(xmame_has_option (current_exec, "mtxil")&& localGameOpts->xil);
			
			Strs.clear();
			Strs << "320x240" << "640x480" << "800x600" << "1024x768" << "1280x1024" << "1600x1200";
			m_renderPage->mGlRender->glResCombo->clear();
			m_renderPage->mGlRender->glResCombo->insertStringList(Strs);
			
			foundGlres = FALSE;
			if(localGameOpts->glres[0]!='\0') {
				for(int i=0; i< m_renderPage->mGlRender->glResCombo->count(); i++) {
					if (!strcmp(localGameOpts->glres, m_renderPage->mGlRender->glResCombo->text(i))) {
						m_renderPage->mGlRender->glResCombo->setCurrentItem(i);
						foundGlres = TRUE;
						break;
					}
				}
				if (!foundGlres) {
					m_renderPage->mGlRender->glResCombo->insertItem(localGameOpts->glres);
					m_renderPage->mGlRender->glResCombo->setCurrentItem(
							m_renderPage->mGlRender->glResCombo->count()-1);
				}
			}	
			else
				m_renderPage->mGlRender->glResCombo->setCurrentItem(1);
			
			m_renderPage->mGlRender->glResCheck->setChecked(localGameOpts->glres_flag);
			m_renderPage->mGlRender->textureInput->setValue(localGameOpts->gltexture_size);
			m_renderPage->mGlRender->doubleBufferingCheck->setChecked(localGameOpts->gldblbuffer);
			m_renderPage->mGlRender->bilinearCheck->setChecked(localGameOpts->glbilinear);
			m_renderPage->mGlRender->glext78Check->setChecked(localGameOpts->glext78);
			m_renderPage->mGlRender->blitterCheck->setChecked(localGameOpts->glforceblitmode);
			m_renderPage->mGlRender->drawBitmapCheck->setChecked(localGameOpts->gldrawbitmap);
			m_renderPage->mGlRender->colorModCheck->setChecked(localGameOpts->glcolormod);
			m_renderPage->mGlRender->aliasingCheck->setChecked(localGameOpts->glantialias);
			m_renderPage->mGlRender->blendingCheck->setChecked(localGameOpts->glalphablending);
			m_renderPage->mGlRender->cabinetCheck->setChecked(localGameOpts->cabview);
			m_renderPage->mGlRender->cabinetEdit->setText(localGameOpts->cabinet);
			m_renderPage->mGlRender->cabinetEdit->setEnabled(localGameOpts->cabview);
			
			connect (m_renderPage->mGlRender->glResCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setGlResFlag(bool)) );
			connect (m_renderPage->mGlRender->glResCombo, SIGNAL(activated(int)),
				 this, SLOT(setGlRes(int)) );
			connect (m_renderPage->mGlRender->textureInput, SIGNAL(valueChanged(int)),
				 this, SLOT(setGlTextureSize(int)) );
			connect (m_renderPage->mGlRender->aliasingCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setAliasing(bool)) );
			connect (m_renderPage->mGlRender->colorModCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setColorMod(bool)) );
			connect (m_renderPage->mGlRender->doubleBufferingCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setGlDoubleBuf(bool)) );
			connect (m_renderPage->mGlRender->bilinearCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setBilinear(bool)) );
			connect (m_renderPage->mGlRender->glext78Check, SIGNAL(toggled(bool)),
				 this, SLOT(setExt78(bool)) );
			connect (m_renderPage->mGlRender->blendingCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setBlending(bool)) );
			connect (m_renderPage->mGlRender->blitterCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setGlBlitter(bool)) );
			connect (m_renderPage->mGlRender->drawBitmapCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setGlBitmap(bool)) );
			connect (m_renderPage->mGlRender->cabinetEdit, SIGNAL(textChanged(const QString &)),
				 this, SLOT(setCabinetFile(const QString &)) );
			connect (m_renderPage->mGlRender->cabinetCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setCabinet(bool)) );
			connect (m_renderPage->mGlRender->glLineEdit, SIGNAL(textChanged(const QString &)),
				 this, SLOT(setGlLib(const QString &)) );
			connect (m_renderPage->mGlRender->gluLineEdit, SIGNAL(textChanged(const QString &)),
				 this, SLOT(setGluLib(const QString &)) );

			m_renderPage->mGlRender->glLineEdit->setText(localGameOpts->gllibname);
			m_renderPage->mGlRender->gluLineEdit->setText(localGameOpts->glulibname);
			
			m_renderPage->mGlRender->glResCombo->setEnabled(localGameOpts->glres_flag);
			m_renderPage->mGlRender->glLineEdit->setEnabled (xmame_has_option (current_exec, "gllibname"));
			m_renderPage->mGlRender->gluLineEdit->setEnabled (xmame_has_option (current_exec, "glulibname"));
			m_renderPage->mGlRender->glext78Check->setEnabled (xmame_has_option (current_exec, "glext78"));
			m_renderPage->mGlRender->drawBitmapCheck->setEnabled (xmame_has_option (current_exec, "gldrawbitmap"));
			m_renderPage->mGlRender->colorModCheck->setEnabled(xmame_has_option (current_exec, "glcolormod"));
			m_renderPage->mGlRender->blitterCheck->setEnabled(xmame_has_option (current_exec, "glforceblitmode"));
			m_renderPage->mGlRender->cabinetEdit->setEnabled (localGameOpts->cabview && xmame_has_option (current_exec, "cabview"));
			
			break;
			
		case XMAME_EXEC_SDL:
			m_renderPage->mSdlRender->fullScreenCheck->setChecked(localGameOpts->fullscreen);
			connect (m_renderPage->mSdlRender->fullScreenCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setFullScreen(bool)) );
			m_renderPage->mSdlRender->doubleBufferCheck->setChecked(localGameOpts->sdl_doublebuf);
			connect (m_renderPage->mSdlRender->doubleBufferCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setSdlDoubleBuffer(bool)) );
			m_renderPage->mSdlRender->autoResCheck->setChecked(localGameOpts->sdl_auto_mode);
			connect (m_renderPage->mSdlRender->autoResCheck, SIGNAL(toggled(bool)),
				 this, SLOT(setSdlAutoMode(bool)) );

			m_renderPage->mSdlRender->sdlModeLabel->setEnabled(!(localGameOpts->sdl_auto_mode));
			m_renderPage->mSdlRender->sdlModeSpin->setValue(localGameOpts->modenumber);
			combo_init (m_renderPage->mSdlRender->sdlModeCombo, "", NULL, localGameOpts->modenumber);
			/* only show the combobox if sdl_modes exist */
			if (xmame_get_option_value_count (current_exec, "") > 0) {
				m_renderPage->mSdlRender->sdlModeCombo->setEnabled( !(localGameOpts->sdl_auto_mode));
				m_renderPage->mSdlRender->sdlModeCombo->show();
				m_renderPage->mSdlRender->sdlModeSpin->hide();
				connect (m_renderPage->mSdlRender->sdlModeCombo, SIGNAL(activated(int)),
					 this, SLOT(setSdlMode(int)) );
			} else {
				m_renderPage->mSdlRender->sdlModeSpin->setEnabled( !(localGameOpts->sdl_auto_mode));
				m_renderPage->mSdlRender->sdlModeSpin->show();
				m_renderPage->mSdlRender->sdlModeCombo->hide();
				connect (m_renderPage->mSdlRender->sdlModeSpin, SIGNAL(valueChanged(int)),
					 this, SLOT(setSdlMode(int)) );
			}

		default:
			break;
	}

	m_renderPage->aspectCheck->setChecked(localGameOpts->keepaspect);
	connect (m_renderPage->aspectCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setAspectToggle(bool)) );
	connect (m_renderPage->arDoubleNumInput, SIGNAL(valueChanged(double)),
		 this, SLOT(setAspect(double)) );
	if (localGameOpts->keepaspect)
		m_renderPage->arDoubleNumInput->setValue(localGameOpts->displayaspectratio);
	m_renderPage->arDoubleNumInput->setEnabled(localGameOpts->keepaspect);
		m_renderPage->disableModeEdit->setText(localGameOpts->disablemode);
	connect (m_renderPage->disableModeEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setDisableMode(const QString &)) );
}

void kxmamePreferences::setupSoundPrefFrame()
{
	QStringList Strs;
	
	m_soundPage->mSoundFrame->soundEnableCheck->setChecked(localGameOpts->sound);
	connect (m_soundPage->mSoundFrame->soundEnableCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setSound(bool)) );
	m_soundPage->mSoundFrame->sampleCheck->setChecked(localGameOpts->samples);
	connect (m_soundPage->mSoundFrame->sampleCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setSoundSample(bool)) );
	m_soundPage->mSoundFrame->soundFakeCheck->setChecked(localGameOpts->timer);
	connect (m_soundPage->mSoundFrame->soundFakeCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setSoundFake(bool)) );
	Strs << "8000" << "11025" << "16000" << "22050" << "44100" << "48000";
	m_soundPage->mSoundFrame->sampleFreqCombo->clear();
	m_soundPage->mSoundFrame->sampleFreqCombo->insertStringList(Strs);
	m_soundPage->mSoundFrame->sampleFreqCombo->setCurrentItem(3);
	for (int i=0; i<m_soundPage->mSoundFrame->sampleFreqCombo->count();i++) {
		if(!strcmp(QString::number(localGameOpts->samplefre), 
		    m_soundPage->mSoundFrame->sampleFreqCombo->text(i)))
				  m_soundPage->mSoundFrame->sampleFreqCombo->setCurrentItem(i);
	}
	connect (m_soundPage->mSoundFrame->sampleFreqCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setSampleFreq(const QString &)) );
	m_soundPage->mSoundFrame->volumeSlider->setValue(localGameOpts->volume);
	g_snprintf (title, BUFFER_SIZE, i18n("%d dB"), localGameOpts->volume);
	m_soundPage->mSoundFrame->volumeLabel->setText(title);
	connect (m_soundPage->mSoundFrame->volumeSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setVolume(int)) );
	m_soundPage->mSoundFrame->bufferSlider->setValue((int)(localGameOpts->bufsize));
	g_snprintf (title, BUFFER_SIZE, i18n("%.1f frames"), localGameOpts->bufsize);
	m_soundPage->mSoundFrame->bufferLabel->setText(title);
	connect (m_soundPage->mSoundFrame->bufferSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setSoundBuffer(int)) );
	m_soundPage->mSoundFrame->timerCheck->setChecked(localGameOpts->timer);
	connect (m_soundPage->mSoundFrame->timerCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setTimer(bool)) );
	m_soundPage->mSoundFrame->artsSpin->setValue((int)(localGameOpts->artsBufferTime));
	connect (m_soundPage->mSoundFrame->artsSpin, SIGNAL(valueChanged(int)),
		 this, SLOT(setArtsBuffer(int)) );
	m_soundPage->mSoundFrame->qnxCheck->setChecked(localGameOpts->audio_preferred);
	connect (m_soundPage->mSoundFrame->qnxCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setQnxDev(bool)) );
	
	combo_init (m_soundPage->mSoundFrame->dspPluginCombo, "dsp-plugin",
		    localGameOpts->dsp_plugin, -1);
	combo_init (m_soundPage->mSoundFrame->mixerPluginCombo, "sound-mixer-plugin",
		    localGameOpts->sound_mixer_plugin, -1);
	
	Strs.clear();
	Strs << "/dev/dsp" << "/dev/dsp0" << "/dev/dsp1" << "/dev/dsp2" << "/dev/dsp3" << "/dev/audio";
	m_soundPage->mSoundFrame->audioDevCombo->clear();
	m_soundPage->mSoundFrame->audioDevCombo->insertStringList(Strs);
	if(localGameOpts->audiodevice)
		m_soundPage->mSoundFrame->audioDevCombo->setCurrentText(localGameOpts->audiodevice);
	
	Strs.clear();
	Strs << "/dev/mixer";
	m_soundPage->mSoundFrame->mixerDevCombo->clear();
	m_soundPage->mSoundFrame->mixerDevCombo->insertStringList(Strs);
	if(localGameOpts->mixerdevice)
		m_soundPage->mSoundFrame->mixerDevCombo->setCurrentText(localGameOpts->mixerdevice);
	
	m_soundPage->mSoundFrame->soundFileEdit->setText(localGameOpts->soundfile);
	connect (m_soundPage->mSoundFrame->soundFileEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setSoundFile(const QString &)) );
	m_soundPage->file_or_device (!strcmp (localGameOpts->dsp_plugin, "waveout"));

	{
		const gchar *dsp;
		dsp = xmame_get_option_key_value (current_exec, "dsp-plugin", localGameOpts->dsp_plugin);

		m_soundPage->mSoundFrame->dspPluginCombo->setCurrentText ( dsp?dsp:localGameOpts->dsp_plugin);
	}

	{
		const gchar *mixer;
		mixer = xmame_get_option_key_value (current_exec, "sound-mixer-plugin", localGameOpts->sound_mixer_plugin);

		m_soundPage->mSoundFrame->mixerPluginCombo->setCurrentText ( mixer?mixer:localGameOpts->sound_mixer_plugin);
	}
	
	connect (m_soundPage->mSoundFrame->dspPluginCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setDspPlugin(const QString &)) );
	connect (m_soundPage->mSoundFrame->mixerPluginCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setMixerPlugin(const QString &)) );

	setSound(localGameOpts->sound);
	
	m_soundPage->mSoundFrame->alsaDevSpin->setValue(localGameOpts->alsadevice);
	m_soundPage->mSoundFrame->alsaBufferSpin->setValue(localGameOpts->alsa_buffer);
	m_soundPage->mSoundFrame->alsaCardSpin->setValue(localGameOpts->alsacard);
	m_soundPage->mSoundFrame->pcmEdit->setText(localGameOpts->alsa_pcm);
	connect (m_soundPage->mSoundFrame->alsaBufferSpin, SIGNAL(valueChanged(int)), 
		 this, SLOT(setAlsaBuffer(int)) );
	connect (m_soundPage->mSoundFrame->alsaCardSpin, SIGNAL(valueChanged(int)), 
		 this, SLOT(setAlsaCard(int)) );
	connect (m_soundPage->mSoundFrame->alsaDevSpin, SIGNAL(valueChanged(int)),
		 this, SLOT(setAlsaDevice(int)) );
	connect (m_soundPage->mSoundFrame->pcmEdit, SIGNAL(textChanged(const QString &)), 
		 this, SLOT(setAlsaPcm(const QString &)) );
	
	m_soundPage->mSoundFrame->qnxGroup->setEnabled(xmame_has_option (current_exec, "audio_preferred"));
	m_soundPage->mSoundFrame->artsGroup->setEnabled(xmame_has_option (current_exec, "artsBufferTime"));
	m_soundPage->mSoundFrame->alsaGroup->setEnabled(xmame_get_option_key_value (current_exec, "dsp-plugin", "alsa") != NULL);
	m_soundPage->mSoundFrame->alsaCardLabel->setEnabled(xmame_has_option (current_exec, "alsacard"));	
	m_soundPage->mSoundFrame->alsaCardSpin->setEnabled(xmame_has_option (current_exec, "alsacard"));
	m_soundPage->mSoundFrame->alsaDevLabel->setEnabled(xmame_has_option (current_exec, "alsadevice"));	
	m_soundPage->mSoundFrame->alsaDevSpin->setEnabled(xmame_has_option (current_exec, "alsadevice"));
	m_soundPage->mSoundFrame->alsaBufferLabel->setEnabled(xmame_has_option (current_exec, "alsa-buffer"));	
	m_soundPage->mSoundFrame->alsaBufferSpin->setEnabled(xmame_has_option (current_exec, "alsa-buffer"));
	m_soundPage->mSoundFrame->pcmLabel->setEnabled(xmame_has_option (current_exec, "alsa-pcm"));
	m_soundPage->mSoundFrame->pcmEdit->setEnabled(xmame_has_option (current_exec, "alsa-pcm"));
}

void kxmamePreferences::setupVectorPrefFrame()
{
	QStringList Strs;
	
	m_vectorPage->mVectorFrame->vectorAACheck->setChecked(localGameOpts->antialias);
	m_vectorPage->mVectorFrame->vectorTranCheck->setChecked(localGameOpts->translucency);
	m_vectorPage->mVectorFrame->aaCheck->setChecked(localGameOpts->glantialiasvec);
	m_vectorPage->mVectorFrame->glBitmapCheck->setChecked(localGameOpts->gldrawbitmapvec);
	m_vectorPage->mVectorFrame->vectorResCheck->setChecked(localGameOpts->vectorres_flag);
	Strs << "320x240" << "640x480" << "800x600" << "1024x768" << "1280x1024" << "1600x1200";
	m_vectorPage->mVectorFrame->vectorResCombo->clear();
	m_vectorPage->mVectorFrame->vectorResCombo->insertStringList(Strs);
	m_vectorPage->mVectorFrame->vectorResCombo->setCurrentItem(1);
	for (int i=0; i<m_vectorPage->mVectorFrame->vectorResCombo->count();i++) {
		if(!strcmp(localGameOpts->vectorres,
		    m_vectorPage->mVectorFrame->vectorResCombo->text(i)))
			m_vectorPage->mVectorFrame->vectorResCombo->setCurrentItem(i);
	}
	
	setVectorResCheck(localGameOpts->vectorres_flag);
	m_vectorPage->mVectorFrame->beamSlider->setValue((int)localGameOpts->beam*10);
	m_vectorPage->mVectorFrame->flickerSlider->setValue((int)localGameOpts->flicker);
	m_vectorPage->mVectorFrame->intensitySlider->setValue((int)localGameOpts->intensity*100);
	m_vectorPage->mVectorFrame->beamLabel->setText(QString::number(localGameOpts->beam,'f',2));
	m_vectorPage->mVectorFrame->flickerLabel->setText(QString::number((int)localGameOpts->flicker));
	m_vectorPage->mVectorFrame->intensityLabel->setText(QString::number(localGameOpts->intensity,'f',2));
	connect (m_vectorPage->mVectorFrame->vectorAACheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVectorAA(bool)) );
	connect (m_vectorPage->mVectorFrame->vectorTranCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVectorTran(bool)) );
	connect (m_vectorPage->mVectorFrame->glBitmapCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVectorGlBitmap(bool)) );
	connect (m_vectorPage->mVectorFrame->aaCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVectorGlAA(bool)) );
	connect (m_vectorPage->mVectorFrame->vectorResCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVectorResCheck(bool)) );
	connect (m_vectorPage->mVectorFrame->vectorResCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setVectorRes(const QString &)) );
	connect (m_vectorPage->mVectorFrame->beamSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setBeam(int)) );
	connect (m_vectorPage->mVectorFrame->flickerSlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setFlicker(int)) );
	connect (m_vectorPage->mVectorFrame->intensitySlider, SIGNAL(valueChanged(int)),
		 this, SLOT(setIntensity(int)) );

}
void kxmamePreferences::setupCtrlPrefFrame()
{
	m_ctrlPage->mCtrlFrame->padDevLabel->setText(i18n("Name of pad device:\n (FM townpad)"));
	
	int i;
	m_ctrlPage->mCtrlFrame->analogCheck->setChecked(localGameOpts->analogstick);
	connect (m_ctrlPage->mCtrlFrame->analogCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setAnalog(bool)) );
	m_ctrlPage->mCtrlFrame->hotrodCheck->setChecked(localGameOpts->hotrod);
	connect (m_ctrlPage->mCtrlFrame->hotrodCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setHotrod(bool)) );
	m_ctrlPage->mCtrlFrame->hotrodSECheck->setChecked(localGameOpts->hotrodse);
	connect (m_ctrlPage->mCtrlFrame->hotrodSECheck, SIGNAL(toggled(bool)),
		 this, SLOT(setHotrodSE(bool)) );
	m_ctrlPage->mCtrlFrame->usbPSCheck->setChecked(localGameOpts->usbpspad);
	connect (m_ctrlPage->mCtrlFrame->usbPSCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setUsbPs(bool)) );
	m_ctrlPage->mCtrlFrame->rapidCheck->setChecked(localGameOpts->rapidfire);
	connect (m_ctrlPage->mCtrlFrame->rapidCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setRapid(bool)) );
	m_ctrlPage->mCtrlFrame->ugciCheck->setChecked(localGameOpts->ugcicoin);
	connect (m_ctrlPage->mCtrlFrame->ugciCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setUGCI(bool)) );
	m_ctrlPage->mCtrlFrame->useMouseCheck->setChecked(localGameOpts->mouse);
	connect (m_ctrlPage->mCtrlFrame->useMouseCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setMouse(bool)) );
	m_ctrlPage->mCtrlFrame->grabMouseCheck->setChecked(localGameOpts->grabmouse);
	connect (m_ctrlPage->mCtrlFrame->grabMouseCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setGrabMouse(bool)) );
	m_ctrlPage->mCtrlFrame->winKeybCheck->setChecked(localGameOpts->winkeys);
	connect (m_ctrlPage->mCtrlFrame->winKeybCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setWinKeyb(bool)) );
	m_ctrlPage->mCtrlFrame->grabKeybCheck->setChecked(localGameOpts->grabkeyboard);
	connect (m_ctrlPage->mCtrlFrame->grabKeybCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setGrabKeyb(bool)) );

	m_ctrlPage->mCtrlFrame->padDevEdit->setEnabled(FALSE);
	m_ctrlPage->mCtrlFrame->xDevEdit->setEnabled(FALSE);
	m_ctrlPage->mCtrlFrame->joyDevEdit->setEnabled(FALSE);
	m_ctrlPage->mCtrlFrame->hotrodCheck->setEnabled(FALSE);
	m_ctrlPage->mCtrlFrame->hotrodSECheck->setEnabled(FALSE);
	
	QStringList Strs;
	
	{
		const gchar **joytypes;

		joytypes = xmame_get_option_values (current_exec, "joytype");

		if (joytypes) {
			i = 0;
			while (joytypes[i]) {
				Strs << joytypes[i];
				i++;
			}
		}
		// Why gxmame don't free joytypes? It's const
	}
	m_ctrlPage->mCtrlFrame->joyTypeCombo->clear();
	m_ctrlPage->mCtrlFrame->joyTypeCombo->insertStringList(Strs);
	{
		const gchar *joyname;

		joyname = xmame_get_option_value (current_exec, "joytype", localGameOpts->joytype);

		m_ctrlPage->mCtrlFrame->joyTypeCombo->setCurrentText(joyname?joyname:"No joystick");
	} // Why gxmame don't free joyname? It's const .
	
	
	setJoyType(localGameOpts->joytype);

	m_ctrlPage->mCtrlFrame->joyDevEdit->setText(localGameOpts->joydevname);
	m_ctrlPage->mCtrlFrame->padDevEdit->setText(localGameOpts->paddevname);
	m_ctrlPage->mCtrlFrame->xDevEdit->setText(localGameOpts->x11joyname);
	m_ctrlPage->mCtrlFrame->keybCfgEdit->setText(localGameOpts->cfgname);
	m_ctrlPage->mCtrlFrame->trackball1Edit->setText(localGameOpts->XInput_trackball1);
	m_ctrlPage->mCtrlFrame->trackball2Edit->setText(localGameOpts->XInput_trackball2);
	m_ctrlPage->mCtrlFrame->trackball3Edit->setText(localGameOpts->XInput_trackball3);
	m_ctrlPage->mCtrlFrame->trackball4Edit->setText(localGameOpts->XInput_trackball4);
	m_ctrlPage->mCtrlFrame->hotrodCheck->setEnabled(xmame_has_option (current_exec, "hotrod"));
	m_ctrlPage->mCtrlFrame->hotrodSECheck->setEnabled(xmame_has_option (current_exec, "hotrodse"));
	//gtk_widget_set_sensitive (GTK_WIDGET (Xinput_joy_frame), FALSE);

	m_ctrlPage->mCtrlFrame->grabMouseCheck->setEnabled(
			localGameOpts->mouse && xmame_has_option (current_exec, "grabmouse"));
	m_ctrlPage->mCtrlFrame->grabKeybCheck->setEnabled(
			xmame_has_option (current_exec, "grabkeyboard"));
	m_ctrlPage->mCtrlFrame->keybUseCfgCheck->setEnabled(
				  (xmame_has_option (current_exec, "cfgname") ||
						  xmame_has_option (current_exec, "cfg_directory")));
	m_ctrlPage->mCtrlFrame->keybCfgEdit->setEnabled(
			localGameOpts->cfgname_flag && (xmame_has_option (current_exec, "cfgname") ||
						  xmame_has_option (current_exec, "cfg_directory")));
	
	Strs.clear();
	Strs << i18n("Default layout");
	for (i=0;x11_keymaps_layout[i].shortname;i++)
	{
		Strs << i18n(x11_keymaps_layout[i].name);
	}
	m_ctrlPage->mCtrlFrame->keymapCombo->clear();
	m_ctrlPage->mCtrlFrame->keymapCombo->insertStringList(Strs);
	for (i=0;x11_keymaps_layout[i].shortname;i++)
	{
		if (strcmp (x11_keymaps_layout[i].shortname, localGameOpts->keymap) == 0)
			m_ctrlPage->mCtrlFrame->keymapCombo->setCurrentItem(i+1);
	}
	
	Strs.clear();
	/* Getting ctrlr list */
	GList *ctrlr_list = get_ctrlr_list ();
	Strs << i18n("None");
	for (GList *my_list = g_list_first (ctrlr_list);
		    (my_list != NULL);
		    my_list = g_list_next (my_list))
	{
		Strs << i18n((char *)(my_list->data));
	}
	g_list_free (ctrlr_list);
	m_ctrlPage->mCtrlFrame->ctrlCombo->clear();
	m_ctrlPage->mCtrlFrame->ctrlCombo->insertStringList(Strs);
	for (int i=0; i<m_ctrlPage->mCtrlFrame->ctrlCombo->count();i++) {
		if(!strcmp(localGameOpts->ctrlr,
		    m_ctrlPage->mCtrlFrame->ctrlCombo->text(i)))
			m_ctrlPage->mCtrlFrame->ctrlCombo->setCurrentItem(i);
	}
	m_ctrlPage->mCtrlFrame->ctrlCombo->setEnabled(xmame_has_option (current_exec, "ctrlr"));

	connect (m_ctrlPage->mCtrlFrame->joyTypeCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setJoyType(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->joyDevEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setJoyDev(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->padDevEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setPadDev(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->xDevEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setX11Joy(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->trackball1Edit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setTrackBall1(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->trackball2Edit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setTrackBall2(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->trackball3Edit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setTrackBall3(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->trackball4Edit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setTrackBall4(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->keybUseCfgCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setUseKeybCfg(bool)) );
	connect (m_ctrlPage->mCtrlFrame->keybCfgEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setKeybCfg(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->keymapCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setKeymap(const QString &)) );
	connect (m_ctrlPage->mCtrlFrame->ctrlCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setCtrl(const QString &)) );

}
void kxmamePreferences::setupMiscPrefFrame()
{
	m_miscPage->mMiscFrame->cheatCheck->setChecked(localGameOpts->cheat);
	m_miscPage->mMiscFrame->addOptsCheck->setChecked(localGameOpts->use_additional_options);
	setAddOptsCheck(localGameOpts->use_additional_options);
	m_miscPage->mMiscFrame->addOptsEdit->setText(localGameOpts->additional_options);

	m_miscPage->mMiscFrame->artCheck->setChecked(localGameOpts->artwork);
	m_miscPage->mMiscFrame->backdropCheck->setChecked(localGameOpts->use_backdrops);
	m_miscPage->mMiscFrame->bezelCheck->setChecked(localGameOpts->use_bezels);
	m_miscPage->mMiscFrame->overlayCheck->setChecked(localGameOpts->use_overlays);
	m_miscPage->mMiscFrame->artCropCheck->setChecked(localGameOpts->artwork_crop);
	setArtwork(localGameOpts->artwork);
	
	QStringList Strs;
	int i = 0;
	do {
		Strs << i18n(bios_version[i]);
		i++;
	} while (bios_version[i]);
	
	m_miscPage->mMiscFrame->biosCombo->clear();
	m_miscPage->mMiscFrame->biosCombo->insertStringList(Strs);
	m_miscPage->mMiscFrame->biosCombo->setCurrentItem(localGameOpts->bios);
	m_miscPage->mMiscFrame->biosCombo->setEnabled(xmame_has_option (current_exec, "bios"));
	m_miscPage->mMiscFrame->biosLabel->setEnabled(xmame_has_option (current_exec, "bios"));
	
	Strs.clear();
	Strs << i18n("Auto") << i18n("Standard")<<  i18n("High");
	m_miscPage->mMiscFrame->artResCombo->clear();
	m_miscPage->mMiscFrame->artResCombo->insertStringList(Strs);
	m_miscPage->mMiscFrame->artResCombo->setCurrentItem(localGameOpts->artwork_resolution);
	
	m_miscPage->mMiscFrame->keybLedCheck->setEnabled(
				  xmame_has_option (current_exec, "keyboard_leds"));
	m_miscPage->mMiscFrame->keybLedCheck->setChecked(localGameOpts->keyboard_leds);
	
	m_miscPage->mMiscFrame->skipDisclaimerCheck->setChecked(localGameOpts->skip_disclaimer);
	m_miscPage->mMiscFrame->skipDisclaimerCheck->setEnabled(
			xmame_has_option (current_exec, "skip_disclaimer"));
	m_miscPage->mMiscFrame->skipInfoCheck->setChecked(localGameOpts->skip_gameinfo);
	m_miscPage->mMiscFrame->skipInfoCheck->setEnabled(
			xmame_has_option (current_exec, "skip_gameinfo"));
	m_miscPage->mMiscFrame->logCheck->setChecked(localGameOpts->log_flag);
	m_miscPage->mMiscFrame->logEdit->setText(localGameOpts->log);
	m_miscPage->mMiscFrame->logEdit->setEnabled(localGameOpts->debug);
	m_miscPage->mMiscFrame->debugCheck->setChecked(localGameOpts->debug);
	m_miscPage->mMiscFrame->debugCheck->setEnabled(
			xmame_has_option (current_exec, "debug"));
	
	Strs.clear();
	Strs << "640x480" << "800x600" << "1024x768" << "1280x1024" << "1600x1200";
	m_miscPage->mMiscFrame->debugCombo->clear();
	m_miscPage->mMiscFrame->debugCombo->insertStringList(Strs);
	m_miscPage->mMiscFrame->debugCombo->setCurrentItem(0);
	for (int i=0; i<m_miscPage->mMiscFrame->debugCombo->count();i++) {
		if(!strcmp(localGameOpts->debug_size,
		    m_miscPage->mMiscFrame->debugCombo->text(i)))
			m_miscPage->mMiscFrame->debugCombo->setCurrentItem(i);
	}
	m_miscPage->mMiscFrame->debugCombo->setEnabled(
			xmame_has_option (current_exec, "debug") && localGameOpts->debug);
	
	connect (m_miscPage->mMiscFrame->cheatCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setCheat(bool)) );
	connect (m_miscPage->mMiscFrame->addOptsCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setAddOptsCheck(bool)) );
	connect (m_miscPage->mMiscFrame->addOptsEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setAddOpts(const QString &)) );
	connect (m_miscPage->mMiscFrame->artCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArtwork(bool)) );
	connect (m_miscPage->mMiscFrame->backdropCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArtBackdrop(bool)) );
	connect (m_miscPage->mMiscFrame->bezelCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArtBezel(bool)) );
	connect (m_miscPage->mMiscFrame->overlayCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArtOverlay(bool)) );
	connect (m_miscPage->mMiscFrame->artCropCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setArtCrop(bool)) );
	connect (m_miscPage->mMiscFrame->artResCombo, SIGNAL(activated(int)),
		 this, SLOT(setArtRes(int)) );
	connect (m_miscPage->mMiscFrame->biosCombo, SIGNAL(activated(int)),
		 this, SLOT(setBios(int)) );
	connect (m_miscPage->mMiscFrame->keybLedCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setKeybLed(bool)) );
	connect (m_miscPage->mMiscFrame->debugCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setDebug(bool)) );
	connect (m_miscPage->mMiscFrame->debugCombo, SIGNAL(activated(const QString &)),
		 this, SLOT(setDebugSize(const QString &)) );
	connect (m_miscPage->mMiscFrame->logCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setLog(bool)) );
	connect (m_miscPage->mMiscFrame->logEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setLogFile(const QString &)) );
	connect (m_miscPage->mMiscFrame->skipInfoCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setGameInfo(bool)) );
	connect (m_miscPage->mMiscFrame->skipDisclaimerCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setDisclaimer(bool)) );

}

void kxmamePreferences::setupKxmamePrefFrame()
{
	m_kxmamePage->mKxmameFrame->scaleScreenshotCheck->setChecked(gui_prefs.scaledScreenshot);
	connect (m_kxmamePage->mKxmameFrame->scaleScreenshotCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setScaledScreenshot(bool)));
	
	m_kxmamePage->mKxmameFrame->videoPreviewCheck->setChecked(gui_prefs.showVideoPreview);
	connect (m_kxmamePage->mKxmameFrame->videoPreviewCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVideoPreview(bool)));
	
	QStringList strs;
	strs << "0" << "8" << "16" << "24" << "32";
	m_kxmamePage->mKxmameFrame->iconSizeCombo->insertStringList(strs);
	m_kxmamePage->mKxmameFrame->iconSizeCombo->setCurrentItem(gui_prefs.iconSize/8);
	strs.clear();
	connect (m_kxmamePage->mKxmameFrame->iconSizeCombo, SIGNAL(activated(int)),
		 this, SLOT(setIconSize(int)) );

	m_kxmamePage->mKxmameFrame->newGameCheck->setChecked(GameCheck);
	m_kxmamePage->mKxmameFrame->versionCheck->setChecked(VersionCheck);
	m_kxmamePage->mKxmameFrame->xmameOptsCheck->setChecked(use_xmame_options);	
	
	connect (m_kxmamePage->mKxmameFrame->newGameCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setGameCheck(bool)));
	connect (m_kxmamePage->mKxmameFrame->versionCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setVersionCheck(bool)));
	connect (m_kxmamePage->mKxmameFrame->xmameOptsCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setXmameOpts(bool)));

#ifndef ENABLE_JOYSTICK
	m_kxmamePage->mKxmameFrame->kxmameJoyGroupBox->hide();
#else
	m_kxmamePage->mKxmameFrame->joyGuiCheck->setChecked(gui_joy);
	m_kxmamePage->mKxmameFrame->joyGuiEdit->setText(Joystick_in_GUI);
	m_kxmamePage->mKxmameFrame->gameSelectJoyButtonEdit->
			setText(QString::number(gui_prefs.kxmameJoyButton[JOYSELECT]));
	m_kxmamePage->mKxmameFrame->panelSwitchJoyButtonEdit->
			setText(QString::number(gui_prefs.kxmameJoyButton[JOYPANELSWITCH]));
	m_kxmamePage->mKxmameFrame->execSwitchJoyButtonEdit->
			setText(QString::number(gui_prefs.kxmameJoyButton[JOYEXECSWITCH]));
	m_kxmamePage->mKxmameFrame->parentJoyButtonEdit->
			setText(QString::number(gui_prefs.kxmameJoyButton[JOYPARENT]));
	m_kxmamePage->mKxmameFrame->altJoyButtonEdit->
			setText(QString::number(gui_prefs.kxmameJoyButton[JOYALT]));
	if(!gui_prefs.allowAlternative) {
		m_kxmamePage->mKxmameFrame->altJoyButtonEdit->hide();
		m_kxmamePage->mKxmameFrame->altJoyButtonLabel->hide();
	}
	
	m_kxmamePage->mKxmameFrame->kxmameJoyFrame->setEnabled(gui_joy);
	
	connect (m_kxmamePage->mKxmameFrame->joyGuiCheck, SIGNAL(toggled(bool)),
		 this, SLOT(setJoyGui(bool)));
	connect (m_kxmamePage->mKxmameFrame->joyGuiEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setJoyGuiDev(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->gameSelectJoyButtonEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setGameSelectJoy(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->panelSwitchJoyButtonEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setPanelSwitchJoy(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->execSwitchJoyButtonEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setExecSwitchJoy(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->parentJoyButtonEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setParentJoy(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->altJoyButtonEdit, SIGNAL(textChanged(const QString &)),
		 this, SLOT(setAltJoy(const QString &)));
	connect (m_kxmamePage->mKxmameFrame->joyApplyButton, SIGNAL(clicked()),
		 this, SLOT(applyJoy()));
#endif
}

/********************* Slots for Preferences dialog ********************/

void kxmamePreferences::slotDefault()
{
	//override the default slot to reset the options.
	if(!mRom) return;
	
	setUpdatesEnabled(FALSE);
	g_free(localGameOpts->additional_options);
	g_free(localGameOpts);
	
	gchar *filename = g_build_filename (g_get_home_dir (), DOT_GXMAME, "options", mRom->romname, NULL);
	unlink (filename);
	g_free(filename);
	
	load_options(NULL);
	
	localGameOpts = (GameOptions *)g_malloc(sizeof(default_options));
	memcpy((void *)localGameOpts, (void *)&default_options, sizeof(default_options));
	localGameOpts->additional_options = g_strdup(default_options.additional_options);
	
	setupDisplayPrefFrame();
	setupRenderPrefFrame();
	setupSoundPrefFrame();
	setupCtrlPrefFrame();
	setupMiscPrefFrame();
	
	if(mRom->vector)
		setupVectorPrefFrame();
	
	isDefault = TRUE;
	isModified = FALSE;
	actionButton(KDialogBase::Default)->setEnabled(FALSE);
	actionButton(KDialogBase::Cancel)->setEnabled(FALSE);
	
	setUpdatesEnabled(TRUE);
}

void kxmamePreferences::auditRom()
{
	FILE       *xmame_pipe = NULL;
	gchar       line [BUFFER_SIZE];
	gchar      *command, *p;
	gchar      *rompath_option;
	RomStatus   status;
	gboolean    error_during_check = FALSE;

	QTextEdit *mOutput = m_gamePage->mGameFrame->gameEdit;
	QLabel *romLabel = m_gamePage->mGameFrame->romLabel;
	RomEntry *rom = mRom;

	romLabel->setText(i18n("Checking..."));
	kapp->processEvents();
//	GtkTextIter text_iter;

	title[0] = '\0';

	if (!current_exec) {
		strcpy(title, "Can't audit game");
		//gtk_label_set_text (GTK_LABEL (rom_check_result), title);
		romLabel->setText(i18n(title));
		//game_checked = TRUE;
		return;
	}
		
	GXMAME_DEBUG ("single audit");
	if(globalIsMess)
		rompath_option = create_biospath_options_string (current_exec);
	else
		rompath_option = create_rompath_options_string (current_exec);

	command = g_strdup_printf ("%s -verifyroms %s %s 2>/dev/null",
				   current_exec->path,
				   rompath_option,
				   rom->romname);

	g_free (rompath_option);
	GXMAME_DEBUG ("running command %s", command);
	//fprintf(stderr, "%s\n", command);
	xmame_pipe = popen (command, "r");
	if (!xmame_pipe) {
		GXMAME_DEBUG ("Could not run %s", command);
		g_free (command);
		return;
	}

	status = NOTNEEDED;

	/* Loading */
	while (fgets (line, BUFFER_SIZE, xmame_pipe)) {
		/* jump the last comments */
		if (!line[0] || !line[1])	/* same as strlen <= 1 but faster */
			break;

		if (!strncmp (line, "romset", 6)) {
			for (p = line + 7; (*p && (*p != ' ')); p++);

			*p = '\0';
			p++;

			if (!strncmp (p, "correct", 7)) {
				strcpy(title, "Passed");
				status = CORRECT;
			} else if (!strncmp (p, "incorrect", 9) && error_during_check == TRUE) {
				status = INCORRECT;
				strcpy(title, "Incorrect");
			} else if (!strncmp (p, "not found", 9)) {
				status = INCORRECT;
				strcpy(title, "Not found");
			} else if (!strncmp (p, "best available", 14)) {
				status = CORRECT;
				strcpy(title, "Best available");
			} else {
				status = UNKNOWN;
				strcpy(title, "Unknown");
			}
			rom->has_roms = status;
			//rom->status = (status == CORRECT);
		
			//gtk_label_set_text (GTK_LABEL (rom_check_result), title);
			romLabel->setText(i18n(title));
			/* continue with the GUI */
			error_during_check = FALSE;
		} else if (!strncmp (line, "name", 4)) {
			/* do nothing */
		} else if (!strncmp (line, "---", 3)) {
			/* do nothing */
		} else {
			//gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter);
			//gtk_text_buffer_insert (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter, line, -1);
			mOutput->append(line);
			error_during_check = TRUE;
		}
//		while (gtk_events_pending ()) gtk_main_iteration ();
	}

	pclose (xmame_pipe);
	g_free (command);

	if(globalIsMess && (status == NOTNEEDED)) {
		rom->has_roms = status;
		romLabel->setText(i18n("None required"));
	}


	//game_checked = TRUE;
	//update_game_in_list (rom);
}


void kxmamePreferences::auditSample ()
{
	FILE       *xmame_pipe = NULL;
	gchar       line [BUFFER_SIZE];
	//const       gchar *title;
	gchar      *command, *p;
	gchar      *rompath_option;
	RomStatus   status;
	gboolean    error_during_check = FALSE;
//	GtkTextIter text_iter;
	QTextEdit *mOutput = m_gamePage->mGameFrame->gameEdit;
	QLabel *sampleLabel = m_gamePage->mGameFrame->sampleLabel;
	RomEntry *rom = mRom;
	
	sampleLabel->setText(i18n("Checking..."));
	kapp->processEvents();

// 	if (game_checked)
// 		return;
// 
// 	while (gtk_events_pending ()) gtk_main_iteration ();

	if (!current_exec) {
		sprintf(title,"%s","Can't audit game");
		if (rom->nb_samples > 0) {
			//gtk_label_set_text (GTK_LABEL (sample_check_result), title);
			sampleLabel->setText(i18n(title));
		} else {
			//gtk_label_set_text (GTK_LABEL (sample_check_result), i18n("None required"));
			sampleLabel->setText(i18n("None required"));
		}
		//game_checked = TRUE;
		return;
	}
		

	/* Samples now */
	if (rom->nb_samples > 0) {
		rompath_option = create_rompath_options_string (current_exec);

		command = g_strdup_printf ("%s %s -verifysamples %s 2>/dev/null",
					   current_exec->path,
					   rompath_option,
					   rom->romname);

		g_free (rompath_option);

		GXMAME_DEBUG ("running command %s", command);
		xmame_pipe = popen (command, "r");

		/* Loading */
		while (fgets (line, BUFFER_SIZE, xmame_pipe)) {
			/* jump the last comments */
			if (!line[0] || !line[1])
				break;

			if (!strncmp (line, "sampleset", 9)) {
				for (p = line + 10; (*p && (*p != ' ')); p++);
				*p = '\0';
				p++;

				if (!strncmp (p, "correct", 7)) {
					strcpy(title, "Passed");
					status = CORRECT;
				} else if (!strncmp (p, "incorrect", 9) && error_during_check == TRUE) {
					strcpy(title, "Incorrect");
					status = INCORRECT;
				} else {
					status = UNKNOWN;
					strcpy(title, "Not found");
				}

				//gtk_label_set_text (GTK_LABEL (sample_check_result), title);
				sampleLabel->setText(i18n(title));

				rom->has_samples = status;
				/*continue with the GUI */
				error_during_check = FALSE;
			} else {
				//gtk_text_buffer_get_end_iter (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter);
				//gtk_text_buffer_insert (GTK_TEXT_BUFFER (details_audit_result_buffer), &text_iter, line, -1);
				mOutput->append(line);
				error_during_check = TRUE;
			}
			//while (gtk_events_pending ()) gtk_main_iteration ();
		}
		pclose (xmame_pipe);
		g_free (command);
	} else {
		//gtk_label_set_text (GTK_LABEL (sample_check_result), i18n("None required"));
		sampleLabel->setText(i18n("None required"));
	}

	//game_checked = TRUE;
	//update_game_in_list (rom);
}


void kxmamePreferences::setArbHeightToggle(bool val){
	m_displayPage->mDisplayFrame->heightScaleIntNumInput->setEnabled(val);
	if (!val) localGameOpts->arbheight=0;
	OptModified();
}
void kxmamePreferences::setBpp(int val){
	if (val == 1) localGameOpts->bpp = 8;
	else if (val == 2) localGameOpts->bpp = 15;
	else if (val == 3) localGameOpts->bpp = 16;
	else if (val == 4) localGameOpts->bpp = 32;
	else localGameOpts->bpp = 0;
	OptModified();
}
void kxmamePreferences::setRotate(int val){
	if(val == 1) { localGameOpts->ror =1;localGameOpts->rol =0;}
	else if(val == 2) { localGameOpts->ror =0;localGameOpts->rol =1;}
	else { localGameOpts->ror =0;localGameOpts->rol =0;}
	OptModified();
}
void kxmamePreferences::setAutoFrameSkip(bool state){ 
	localGameOpts->autoframeskip = state;
	m_displayPage->mDisplayFrame->frameToSkipLabel->setEnabled(!state);
	m_displayPage->mDisplayFrame->frameToSkipCombo->setEnabled(!state);
	m_displayPage->mDisplayFrame->frameMethodLabel->setEnabled(state && xmame_has_option (current_exec, "frameskipper"));
	m_displayPage->mDisplayFrame->frameMethodCombo->setEnabled(state && xmame_has_option (current_exec, "frameskipper"));
	m_displayPage->mDisplayFrame->maxFrameLabel->setEnabled(state);
	m_displayPage->mDisplayFrame->maxFrameCombo->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setGamma(int val){
	localGameOpts->gamma_correction = gfloat(val)/100;
	g_snprintf (title, BUFFER_SIZE, i18n("%.2f"), localGameOpts->gamma_correction);
	m_displayPage->mDisplayFrame->gammaLabel->setText(title);
	OptModified();
}
void kxmamePreferences::setBrightness(int val){
	localGameOpts->brightness = gfloat(val)/100;
	g_snprintf (title, BUFFER_SIZE, i18n("%.2f"), localGameOpts->brightness);
	m_displayPage->mDisplayFrame->brightnessLabel->setText(title);
	OptModified();
}
void kxmamePreferences::setXVgeometry(bool state){
	localGameOpts->xvgeom_flag = state;
	m_renderPage->mX11Render->geometryCombo->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setCabinet(bool state) {
	localGameOpts->cabview = state;
	m_renderPage->mGlRender->cabinetEdit->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setGlResFlag(bool state){
	localGameOpts->glres_flag = state;
	m_renderPage->mGlRender->glResCombo->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setGlRes(int val){
	strncpy(localGameOpts->glres,
		m_renderPage->mGlRender->glResCombo->text(val), 9);
	OptModified();
}
void kxmamePreferences::setSdlAutoMode(bool state){ 
	localGameOpts->sdl_auto_mode = state;
	m_renderPage->mSdlRender->sdlModeLabel->setEnabled(!state);
	if (xmame_get_option_value_count (current_exec, "modenumber") > 0)
	{
		m_renderPage->mSdlRender->sdlModeCombo->setEnabled(! state);
	}
	else
	{
		m_renderPage->mSdlRender->sdlModeSpin->setEnabled(! state);
	}
	OptModified();
}
void kxmamePreferences::setAspectToggle(bool state){
	localGameOpts->keepaspect = state;
	m_renderPage->arDoubleNumInput->setEnabled(state);
	m_renderPage->arDoubleNumInput->setValue(localGameOpts->displayaspectratio);
	OptModified();
}
void kxmamePreferences::setSound(bool state){ 
	localGameOpts->sound = state;
	m_soundPage->mSoundFrame->sampleCheck->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->volumeGroup->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->bufferGroup->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->audioDevLabel->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->mixerDevLabel->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->audioDevCombo->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->mixerDevCombo->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->sampleFreqLabel->setEnabled(localGameOpts->sound);
	m_soundPage->mSoundFrame->sampleFreqCombo->setEnabled(localGameOpts->sound);
	OptModified();
}
void kxmamePreferences::setVolume(int val){
	localGameOpts->volume = val;
	g_snprintf (title, BUFFER_SIZE, i18n("%d dB"), localGameOpts->volume);
	m_soundPage->mSoundFrame->volumeLabel->setText(title);
	OptModified();
}
void kxmamePreferences::setSoundBuffer(int val){
	localGameOpts->bufsize = gfloat(val);
	g_snprintf (title, BUFFER_SIZE, i18n("%.1f frames"), localGameOpts->bufsize);
	m_soundPage->mSoundFrame->bufferLabel->setText(title);
	OptModified();
}

void kxmamePreferences::setDspPlugin (const QString &selected_plugin)
{
	//gchar *selected_plugin;
	const gchar *plugin;

	//selected_plugin = g_strdup(m_soundPage->mSoundFrame->dspPluginCombo->text(val));
		
	plugin = xmame_get_option_key (current_exec, "dsp-plugin", selected_plugin);
 

	m_soundPage->file_or_device (plugin && !strcmp (plugin, "waveout"));
		
	strncpy (localGameOpts->dsp_plugin, selected_plugin, 20);
	//g_free (selected_plugin);
	if (xmame_get_option_value_count (current_exec, "dsp-plugin") > 0)
	{
		const gchar *value;

		//selected_plugin = g_strdup(m_soundPage->mSoundFrame->dspPluginCombo->text(val));
		value = xmame_get_option_key (current_exec, "dsp-plugin", selected_plugin);
		if (value) strncpy (localGameOpts->dsp_plugin, value, 20);
		//g_free (selected_plugin);
	}
	OptModified();
}

void kxmamePreferences::setMixerPlugin (const QString &selected_plugin)
{
	if (xmame_get_option_value_count (current_exec, "sound-mixer-plugin") > 0)
	{
		const gchar *value;

		//used_text = gtk_editable_get_chars (GTK_EDITABLE (mixer_plugin_entry), 0, -1);
		value = xmame_get_option_key (current_exec, "sound-mixer-plugin", selected_plugin);
		if (value) strncpy (localGameOpts->sound_mixer_plugin, value, 20);
		//g_free (used_text);
	}
	OptModified();
}
void kxmamePreferences::setBeam(int val){
	localGameOpts->beam = gfloat(val)/10.0;
	m_vectorPage->mVectorFrame->beamLabel->setText(QString::number(localGameOpts->beam,'f',2));
	OptModified();
}
void kxmamePreferences::setFlicker(int val){ 
	localGameOpts->flicker = gfloat(val);
	m_vectorPage->mVectorFrame->flickerLabel->setText(QString::number(val));
	OptModified();
}
void kxmamePreferences::setIntensity(int val){
	 localGameOpts->intensity = gfloat(val)/100.0; 
	 m_vectorPage->mVectorFrame->intensityLabel->setText(QString::number(localGameOpts->intensity,'f',2));
	 OptModified();
}

void kxmamePreferences::setVectorResCheck(bool state){
	localGameOpts->vectorres_flag = state;
	m_vectorPage->mVectorFrame->vectorResCombo->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setMouse(bool state) {
	localGameOpts->mouse = state;
	m_ctrlPage->mCtrlFrame->grabMouseCheck->setEnabled(state && xmame_has_option (current_exec, "grabmouse"));
	OptModified();
}
void kxmamePreferences::setJoyType(int type) {
	switch (type)
	{
		case 0:
			m_ctrlPage->mCtrlFrame->analogCheck->setEnabled(FALSE);
			m_ctrlPage->mCtrlFrame->hotrodCheck->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->hotrodSECheck->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->usbPSCheck->setEnabled(FALSE);
			m_ctrlPage->mCtrlFrame->rapidCheck->setEnabled(FALSE);
			break;
		case 1:
			m_ctrlPage->mCtrlFrame->joyDevLabel->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->joyDevEdit->setEnabled(TRUE);
			break;
		case 2:
			m_ctrlPage->mCtrlFrame->padDevLabel->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->padDevEdit->setEnabled(TRUE);
			break;
		case 3:
			m_ctrlPage->mCtrlFrame->xDevLabel->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->xDevEdit->setEnabled(TRUE);
			break;
		case 4:
			m_ctrlPage->mCtrlFrame->joyDevLabel->setEnabled(TRUE);
			m_ctrlPage->mCtrlFrame->joyDevEdit->setEnabled(TRUE);
			break;
	}
	
	if(type>7) type = 0;
	localGameOpts->joytype = type;
	OptModified();
}
void kxmamePreferences::setKeymap(const QString &str) {
	bool found = FALSE;
	for (int i=0;x11_keymaps_layout[i].shortname;i++)
	{
		if (strcmp (i18n(x11_keymaps_layout[i].name), str)==0) {
			strncpy (localGameOpts->keymap, x11_keymaps_layout[i].shortname, 4);
			found = TRUE;
		}
	}
	if (!found && (strcmp(str,i18n("Default layout"))==0))
		strcpy (localGameOpts->keymap, " ");
	OptModified();
}
void kxmamePreferences::setJoyType(const QString &str) {
	localGameOpts->joytype = 0;
	localGameOpts->joytype = xmame_get_option_value_index (current_exec, "joytype", str);
	setJoyType(localGameOpts->joytype);
	OptModified();
}
void kxmamePreferences::setUseKeybCfg(bool state) {
	localGameOpts->cfgname_flag = state;
	m_ctrlPage->mCtrlFrame->keybCfgEdit->setEnabled(state && 
			(xmame_has_option (current_exec, "cfgname") ||
			xmame_has_option (current_exec, "cfg_directory")));
	OptModified();
}
void kxmamePreferences::setArtwork(bool state)
{	
	localGameOpts->artwork = state;
	
	m_miscPage->mMiscFrame->artResCombo->setEnabled(
			xmame_has_option (current_exec, "artwork_resolution") && state);
	m_miscPage->mMiscFrame->backdropCheck->setEnabled(
			xmame_has_option (current_exec, "use_backdrops") && state);
	m_miscPage->mMiscFrame->bezelCheck->setEnabled(
			xmame_has_option (current_exec, "use_backdrops") && state);
	m_miscPage->mMiscFrame->overlayCheck->setEnabled(
			xmame_has_option (current_exec, "use_backdrops") && state);
	m_miscPage->mMiscFrame->artCropCheck->setEnabled(
			xmame_has_option (current_exec, "artwork_crop") && state);
	OptModified();
}
void kxmamePreferences::setAddOptsCheck(bool state){
	localGameOpts->use_additional_options = state;
	m_miscPage->mMiscFrame->addOptsEdit->setEnabled(state);
	OptModified();
}
void kxmamePreferences::setAddOpts(const QString &str){
	g_free(localGameOpts->additional_options);
	localGameOpts->additional_options = g_strdup(str);
	
	// we need to do this in case the user cancel the dialog
	// and additional_options pointer needs to be freed.
	if(!mRom)
		default_options.additional_options = localGameOpts->additional_options;
	
	OptModified();
}

void kxmamePreferences::setJoyGui(bool state) {
#ifdef ENABLE_JOYSTICK
	gui_joy = state;
	m_kxmamePage->mKxmameFrame->kxmameJoyFrame->setEnabled(state);
	OptModified();
#endif 
}

void kxmamePreferences::setLog(bool state) {
	localGameOpts->log_flag = state;
	m_miscPage->mMiscFrame->logEdit->setEnabled(state);
	OptModified();
}

void kxmamePreferences::setDebug(bool state) {
	localGameOpts->debug = state;
	m_miscPage->mMiscFrame->debugCombo->setEnabled(state);
	OptModified();
}

void kxmamePreferences::applyJoy( ) {
#ifdef ENABLE_JOYSTICK
	joystick_close( joydata );
	joydata = NULL;
	joydata = joystick_new (gui_prefs.Joystick_in_GUI);
	if(!joydata)
		gxmame_message( ERROR, NULL, i18n("Could not initialize joystick"));
	else
		gxmame_message( MESSAGE, NULL, i18n("Found joystick \"%1\"").arg(joydata->device_name));
#endif
}

/******************* class implementation for Preferences Dialog pages ********************/

kxmameGamePref::kxmameGamePref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mGameFrame = new gamePrefWidget(this);
	if(globalIsMess)
		mGameFrame->preRomLabel->setText("BIOS check:");
	else
		mGameFrame->preRomLabel->setText("Rom check:");
}

kxmameDisplayPref::kxmameDisplayPref(QWidget *parent)
    : QWidget(parent)
{
    QVBoxLayout *layout = new QVBoxLayout(this);
    layout->setAutoAdd(true);

    mDisplayFrame = new displayPrefFrame(this);
}

kxmameSoundPref::kxmameSoundPref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mSoundFrame = new soundPrefFrame(this);
}

void kxmameSoundPref::file_or_device(bool file)
{
	if (file) {
		mSoundFrame->soundFileLabel->setEnabled(TRUE);
		mSoundFrame->soundFileEdit->setEnabled(TRUE);
		mSoundFrame->soundFileButton->setEnabled(TRUE);
		mSoundFrame->audioDevLabel->setEnabled(FALSE);
		mSoundFrame->audioDevCombo->setEnabled(FALSE);
	} else {
		mSoundFrame->soundFileLabel->setEnabled(FALSE);
		mSoundFrame->soundFileEdit->setEnabled(FALSE);
		mSoundFrame->soundFileButton->setEnabled(FALSE);
		mSoundFrame->audioDevLabel->setEnabled(TRUE);
		mSoundFrame->audioDevCombo->setEnabled(TRUE);
	}
}

kxmameVectorPref::kxmameVectorPref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mVectorFrame = new vectorPrefFrame(this);
}

kxmameCtrlPref::kxmameCtrlPref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mCtrlFrame = new ctrlPrefFrame(this);
}

kxmameMiscPref::kxmameMiscPref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mMiscFrame = new miscPrefFrame(this);
}


kxmameStartupPref::kxmameStartupPref(QWidget *parent)
	: QWidget(parent)
{
	QVBoxLayout *layout = new QVBoxLayout(this);
	layout->setAutoAdd(true);

	mKxmameFrame = new startupPrefWidget(this);
	//mKxmameFrame->colorGroupBox->hide();
}


kxmameRenderPref::kxmameRenderPref(QWidget *parent)
    : QWidget(parent)
{
	layout = new QHBoxLayout(this);

	QVBoxLayout *layout1 = new QVBoxLayout(layout);

	layout->setSpacing(6);
	layout1->setSpacing(6);

	switch (current_exec->type) {
		case XMAME_EXEC_X11:
		case XMAME_EXEC_XGL:
			mX11Render = new x11Render(this);
			mGlRender = new glRender(this);
			layout1->addWidget(mX11Render);
			layout->addWidget(mGlRender);
			break;
		case XMAME_EXEC_SDL:
			mSdlRender = new sdlRender(this);
			layout1->addWidget(mSdlRender);
		default:
			break;
	}
		
	groupBox5 = new QGroupBox(this, "groupBox5" );
	groupBox5->setColumnLayout(0, Qt::Vertical );
	groupBox5->layout()->setSpacing( 6 );
	groupBox5->layout()->setMargin( 11 );
	groupBox5Layout = new QHBoxLayout( groupBox5->layout() );
	groupBox5Layout->setAlignment( Qt::AlignTop );

	layout23 = new QVBoxLayout( 0, 0, 6, "layout23"); 
	layout22 = new QHBoxLayout( 0, 0, 6, "layout22"); 
	
	aspectCheck = new QCheckBox( groupBox5, "aspectCheck" );
	layout22->addWidget( aspectCheck );
	spacer15 = new QSpacerItem( 33, 17, QSizePolicy::Expanding, QSizePolicy::Minimum );
	layout22->addItem( spacer15 );
	
	arDoubleNumInput = new KDoubleNumInput( groupBox5, "arDoubleNumInput" );
	layout22->addWidget( arDoubleNumInput );
	layout23->addLayout( layout22 );
	
	layout21 = new QHBoxLayout( 0, 0, 6, "layout21"); 
	
	textLabel4 = new QLabel( groupBox5, "textLabel4" );
	layout21->addWidget( textLabel4 );
	spacer16 = new QSpacerItem( 49, 25, QSizePolicy::Expanding, QSizePolicy::Minimum );
	layout21->addItem( spacer16 );
	
	disableModeEdit = new QLineEdit( groupBox5, "disableModeEdit" );
	layout21->addWidget( disableModeEdit );
	layout23->addLayout( layout21 );
	groupBox5Layout->addLayout( layout23 );

	layout1->addWidget(groupBox5);

	groupBox5->setTitle( i18n( "Video mode:" ) );
	aspectCheck->setText( i18n( "Keep aspect ratio:" ) );
	aspectCheck->setAccel( QKeySequence( i18n( "Alt+K" ) ) );
	textLabel4->setText( i18n( "Disable mode:" ) );

}
#include "pref.moc"
