/*
**  MMU030.c
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
*/

#include <stdio.h>
#include <Gestalt.h>
#include <Quickdraw.h>
#include "MMU.h"
#include "MMU030.h"
#include "penguin_prototypes.h"

/* Prototypes */
static u32			gps, gis, gtia, gtib, gtic, gtid;

static void			record030TTMapping (u32 tt);
static void			logTo030Phys (u32 logAddr, u32 *physAddr, u32 *physLen, u32 *physAttr);
static Boolean		walk030PageTable (u32	 logAddr, 
						u32	*physAddr, 
				  		u32	*physLen, 
			  			u32	*physAttr, 
			 			u32	*pageTable, 
			  			u32	 indexLimit,
			 			u32	 indexLeft, 
						u32	 indexSize, 
						u32	 indexRight[4],
					Boolean eightBytePD,
					Boolean foundBottom);
static long			read030LongAssembly (void *physAddr, void *screenBaseAddr);
static long			read030Long (void *physAddr);

typedef long		(*Read030LongAssembly_t) (void *);

/*
 *	isCpu030
 *
 */
int isCpu030 (void)
{
	long		cpuType;
	OSErr		err;
	
	err = Gestalt (gestaltProcessorType, &cpuType);
	if (err) {
		cprintf ("This computer does not recognize the Gestalt selector `gestaltProcessorType`.\n");
		cprintf ("Are you sure it's a new-enough system software for this?\n");
		return 0;
	}
	
	if (cpuType == gestalt68030) 		return 1;
	else					return 0;
}

/*
 *	init030Mmu
 *
 */
void init030Mmu (void)
{
	u32	 tc;
	
	get030TCReg (&tc);
	
	gps = xNIBBLE (tc, 5);
	gis = xNIBBLE (tc, 4);
	
	gtia = xNIBBLE (tc, 3);
	gtib = xNIBBLE (tc, 2);
	gtic = xNIBBLE (tc, 1);
	gtid = xNIBBLE (tc, 0);
	
	if (0 == gtia) {
		gtib = 0;
		gtic = 0;
		gtid = 0;
	} else if (0 == gtib) {
		gtic = 0;
		gtid = 0;
	} else if (0 == gtic) {
		gtid = 0;
	}
	
	gMemoryMappings.index = 0;
	gMemoryMappings.cacheEmitter = NULL;
}

/*
 *	calculate030MmuMappings
 *
 */
void calculate030MmuMappings (void)
{	
	u32 	 logAddr, endAddr, phys;
	u32	 len, physAttr;
	u32	 tt;
	
	logAddr = 0;
	endAddr = BIT(32-gis);
	
	/*
	 *	Look at the transparent translation registers
	 */
	get030TT0 (&tt);
	record030TTMapping (tt);
	
	get030TT1 (&tt);
	record030TTMapping (tt);
	
	/*
	 *	Walk through all of logical address spaces
	 */
	do {
		logTo030Phys (logAddr, &phys, &len, &physAttr);

		recordMemoryMapping (logAddr, phys, len, physAttr);

		logAddr += len;
		
	} while (logAddr != endAddr);
}

/*
 *	record030TTMapping
 *
 */
void record030TTMapping (u32 tt)
{
	u32	logAddr, physAddr, len, attr;
	
	if ((tt & 0x00008000) == 0)
		return;
	
	logAddr = tt & 0xFF000000 & (~((tt & 0x00FF0000 ) << 8));
	physAddr = logAddr;
	len = (tt & 0x00FF0000) << 8;
	len += 0x01000000 -1;
	attr = (tt & 0x00000400) >> 4;
	
	recordMemoryMapping (logAddr, physAddr, len, attr);
}

#if 0 /* Inline ASM */
/*
 *	get030TCReg
 *
 */
void ASMFUNC (get030TCReg, u32 *tc)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0xF010, 0x4200		/*	pmove	%tc, %a0@	*/
ENDASM
}


/*
 *	get030SRP
 *
 */
void ASMFUNC (get030SRP, void *root)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0xF010, 0x4A00		/*	pmove	%srp, %a0@	*/
ENDASM
}

/*
 *	get030CRP
 *
 */
void ASMFUNC (get030CRP, void *root)
{
BEGINASM
	movea.l	8(A6),a0
	movea.l	0xCB4,a1
	move.l	(a1),(a0)
	move.l	4(a1),4(a0)
//	dc.w	0xF010, 0x4E00		/*	pmove	%crp, %a0@	*/
ENDASM
}

/*
 *	get030TT0
 *
 */
void ASMFUNC (get030TT0, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0xF010, 0x0A00		/*	pmove	%tt0, %a0@	*/
ENDASM
}

/*
 *	get030TT1
 *
 */
void ASMFUNC (get030TT1, void *tt)
{
BEGINASM
	movea.l	8(A6),a0
	dc.w	0xF010, 0x0E00		/*	pmove	%tt1, %a0@	*/
ENDASM
}

/*
 *	read030LongAssembly
 *
 */
#define USE_100	1
long ASMFUNC2 (read030LongAssembly, void *physAddr, void *screenBaseAddr)
{
BEGINASM
@entry:
	move.l	a2,-(SP)
	//	In the following, '..' means the high order 8 bits
	//	of the physical address the caller wants to read
	move.l	8(A6),d0
	move.l	12(A6),a2	//	screen base addr
#if 0
	bne.s	@valid_address
	lea	@entry-4,a0
	move.l	a0,d0
	bra	@exit
@valid_address:
#endif
	move.l	d0,a0
	andi.l	#0xFF000000,d0	//	D0 = 0x..000000
	addi.l	#0x01000000,d0	//	D0 = 0x+1000000
	subi.l	#0x00000001,d0	//	D0 = 0x..FFFFFF
	andi.l	#0xFF008607,d0	//	D0 = 0x..008603
	/*
	 *	D0 now contains:
	 *		The base Physical address of the location to read
	 *		The the range of 1 meg of mappable log->phys
	 *		Enable
	 *		Cache Inhibit
	 *		Only allow reads
	 *		Ignore the FC BASE field (which is 0)
	 *	This value is used to program the Transparent Translation Register
	 */
	
	move.l	d7,-(SP)
	move.w	SR,d7
	ori.w	#0x0700,SR

#if defined(USE_100)
	/*
	 *	First, save the some low memory data
	 *	to store a routine there...
	 */
	move.l	0x0100,-(SP)
	move.l	0x0104,-(SP)
	move.l	0x0108,-(SP)
	move.l	0x010C,-(SP)
	move.l	0x0110,-(SP)
	move.l	0x0114,-(SP)

	move.l	0x0100(a2),-(SP)
	move.l	0x0104(a2),-(SP)
	move.l	0x0108(a2),-(SP)
	move.l	0x010C(a2),-(SP)
	move.l	0x0110(a2),-(SP)
	move.l	0x0114(a2),-(SP)
	
	lea	@readMMUfunc,a1
	move.l	(a1)+,0x0100
	move.l	(a1)+,0x0104
	move.l	(a1)+,0x0108
	move.l	(a1)+,0x010C
	move.l	(a1)+,0x0110
	move.l	(a1)+,0x0114

	lea	@readMMUfunc,a1
	move.l	(a1)+,0x0100(a2)
	move.l	(a1)+,0x0104(a2)
	move.l	(a1)+,0x0108(a2)
	move.l	(a1)+,0x010C(a2)
	move.l	(a1)+,0x0110(a2)
	move.l	(a1)+,0x0114(a2)
#endif
	lea	0x0808,a1	//	clear data cache and instruction cache
	dc.w	0x4E7B, 0x9002	//	movec	a1,cacr
	
	move.l	d0,-(SP)	//	Pass the new tt0 for the routine
	clr.l	-(SP)		//	Place to save the existing tt0
#if defined(USE_100)
	jsr	0x0100
#else
	jsr	@readMMUfunc
#endif
	
	adda.w	#8,a7

#if defined(USE_100)
	move.l	(SP)+,0x0114(a2)
	move.l	(SP)+,0x0110(a2)
	move.l	(SP)+,0x010C(a2)
	move.l	(SP)+,0x0108(a2)
	move.l	(SP)+,0x0104(a2)
	move.l	(SP)+,0x0100(a2)

	move.l	(SP)+,0x0114
	move.l	(SP)+,0x0110
	move.l	(SP)+,0x010C
	move.l	(SP)+,0x0108
	move.l	(SP)+,0x0104
	move.l	(SP)+,0x0100
#endif
	move.w	d7,SR
	move.l	(SP)+,d7
	
	movea.l	(SP)+,a2
	
	/*
	 *	Record this read data
	 */
	move.l	d0,-(SP)
	move.l	a0,-(SP)
	jsr	addReadData
	move.l	(SP)+,a0
	move.l	(SP)+,d0
	
	bra	@exit

	/*
	 *	Here is the actual code that reads data
	 *	from the MMU memory table.  This code runs
	 *	out of some low memory address in the hopes
	 *	that it will never be within a megabyte of
	 *	the MMU tables.  Can't really gaurantee it,
	 *	but it's a good start.
	 */
@readMMUfunc:
	dc.w	0xF02F, 0x0A00, 0x0004	/*	pmove	%tt0, %0x4+a7@	*/

	//	Set our TT0 as described above in D0
	dc.w	0xF02F, 0x0800, 0x0008	/*	pmove	%0x8+a7@, %tt0	*/
	
	/*
	 *	Ladies and Gentlemen,
	 *		The moment we've been waiting for ...
	 *		(Drum roll please...)
	 */
	move.l	(a0),d0
	/*
	 *	Tadaa!!  And the read has been completed!
	 */
	
	dc.w	0xF02F, 0x0800, 0x0004	/*	pmove	%0x4+a7@, %tt0	*/
	rts
	
@exit:
ENDASM
}
#endif /* Inline ASM */

#if 0
/*
 *	read030Long
 *
 */
long read030Long (void *physAddr)
{
	static	Read030LongAssembly_t	 code_allocated = NULL;
	static	Read030LongAssembly_t	 code_callable = NULL;
	static	void			*reader;
	
	if (!code_allocated)
	{
		reader = (void *) read030LongAssembly (NULL);
		code_allocated = (Read030LongAssembly_t) NewPtrClear (0x50100);
		if (!code_allocated)
			Unsupported ("This program needs more Memory; Use 'Get Info' in the Finder to give it more memory.");
		BlockMove (reader, (void *) ((unsigned long) code_allocated + 0x00000), 0x0100);
		BlockMove (reader, (void *) ((unsigned long) code_allocated + 0x50000), 0x0100);
		/* BlockMove automatically flushes the data caches */
		
		code_callable = (Read030LongAssembly_t) ((unsigned long) code_allocated + 0x50000);
	}
	
	return (*code_callable)(physAddr);
}

#else

/*
 *	read030Long
 *
 */
long read030Long (void *physAddr)
{
	static GDHandle		 theScreen = NULL;
	static PixMapHandle	 theScreenMap;
	static char		*screenBaseAddr;
	
	if (NULL == theScreen)
	{
		theScreen = GetMainDevice();
		theScreenMap = (*theScreen)->gdPMap;
		screenBaseAddr = (char *) (*theScreenMap)->baseAddr;
	}
	
	return read030LongAssembly (physAddr, screenBaseAddr);
}

#endif

/*
 *	walk030PageTable
 *
 */
Boolean walk030PageTable (u32	 logAddr, 
			  u32	*PhysAddr, 
			  u32	*PhysLen, 
			  u32	*PhysAttr, 
			  u32	*PageTable, 
			  u32	 indexLimit,
			  u32	 indexLeft, 
			  u32	 indexSize, 
			  u32	 indexRight[4],
			  Boolean eightBytePD,
			  Boolean foundBottom)
{
	u32		pageSize, pageAddr, pageAttr, entryType;
	u32		index, entry;
	u32		*tableAddr, tableType;
	Boolean		early;
	u32		rightIndexSum;
	Boolean		pdType;
	u32		limit;

	/*
	 *	Establish the table, entry is throw-away
	 */	
	entry = read030Long (PageTable);
	tableReadData (PageTable);
	
	rightIndexSum =  indexRight[0] + indexRight[1] + indexRight[2] + indexRight[3];

	index = xBITS (logAddr,31 - indexLeft, rightIndexSum);
	pageSize = BIT (rightIndexSum);
	
	/*
	 *	If a page descriptor is found at this level ...
	 *		Is it an early terminating one?
	 */
	early = indexRight[0] > 0;
	
	/*
	 *	Are we past the end of the valid table entries?
	 */
	if (index > indexLimit) {
		if (!foundBottom) {
			foundBottom = true;
			*PhysAddr = NOT_MAPPED;
			*PhysLen  = pageSize * (BIT(indexSize) - index);
			*PhysAttr = 0;
		}
		
		return foundBottom;
	}
	
	while (index < BIT(indexSize) && index <= indexLimit)
	{
		if (eightBytePD == kEightBytePD) {
			entry = read030Long (&PageTable[(index *2) +1]);
			tableAddr = (u32 *) (entry & 0xFFFFFFF0);
			pageAddr  = entry & 0xFFFFFF00;

			entry = read030Long (&PageTable[index *2]);
			limit = (entry & 0x7FFF0000) >> 16;
		} else {
			entry = read030Long (&PageTable[index]);
			tableAddr = (u32 *) (entry & 0xFFFFFFF0);
			pageAddr  = entry & 0xFFFFFF00;
			limit = 0x7FFF;
		}

		tableType = entry & 0x00000003;
		pageAttr  = entry & 0x000000E7;
		entryType = entry & 0x00000003;
		pageAttr |= early ? 0x08 : 0x00;
		
		index ++;
#if 0		
		if (index >= 0x0A && index < 0x1F && indexRight[3] == gps) {
			cprintf ("LogAddr = %0.8lX : pageAddr = %0.8lX, entryType = %0.1X\n",
				logAddr, pageAddr, entryType);
		}
#endif		
	 	if (foundBottom		&& 
	 	    entryType == 1	&& 
	 	    pageAddr != (*PhysAddr + *PhysLen)) 
	 		{
	 			return foundBottom;
	 		}
		
		switch (entryType)
		{
		case 0x0:
			/*
			 *	Invalid page 
			 *	... continue determining length of invalid range
			 */
			if (foundBottom)
				if (*PhysAttr == 0) {
					*PhysLen += pageSize;
				} else {
					return foundBottom;
				}
			else {
				*PhysAddr = NOT_MAPPED;
				*PhysAttr = 0;
				*PhysLen  = pageSize;
				 foundBottom = true;
			}
			break;
		
		case 0x1:
			/*
			 *	Base Case: This is a page descriptor
			 */
			if (!foundBottom) {
				/*
				 *	Haven't seen a page yet, this must
				 *	be the first physical address of
				 *	the supplied logical address
				 */
				 foundBottom = true;
				*PhysAddr = pageAddr;
				*PhysAttr = pageAttr & 0xEF;
				*PhysLen  = pageSize;
			} else {
				/*
				 *	We've seen at least one page before,
				 *	so this next page must be contiguous
				 *	with the last and have the same attributes
				 *	to be counted as a continuation of it.
				 */
				if (*PhysAttr != pageAttr) {
					/*
					 *	Different page types
					 */
					return foundBottom;
				}
				
				*PhysLen += pageSize;
			}
			break;
				 
		case 0x2:
		/*
		 *	This is a 4 byte page table address: Recursive case
		 */
			pdType = kFourBytePD;
			goto recurse;
			
		case 0x3:
		/*
		 *	This is an 8 byte page table address: Recursive case
		 */
			pdType = kEightBytePD;

recurse:
			foundBottom = walk030PageTable ( logAddr, 
							 PhysAddr, 
							 PhysLen, 
							 PhysAttr,
							 tableAddr,
							 limit,
							 indexLeft + indexSize,
							 indexRight[0],
							&indexRight[1],
							 pdType,
							 foundBottom);
			
			if (foundBottom && *PhysAddr == NOT_MAPPED)
				return foundBottom;
			
			break;
			}
	}

	return foundBottom;
}


/*
 *	logTo030Phys
 *
 */
void logTo030Phys (u32 logAddr, u32 *physAddr, u32 *physLen, u32 *physAttr)
{
	u32		 crp[2];
	u32		*rootTable;
	u32		 right[8];
	Boolean		 pdSize;
	
	/*
	 *	Get the pointer to the root table
	 */
	get030CRP (&crp[0]);
	
	/*
	 *	Get the Translation Control Reg.
	 */
	//	get030TCReg (&tc);
	
	rootTable = (u32 *) (crp[1] & 0xFFFFFFF0);

	/*
	 *	Assign incrementing tix's or 0 after first 0 tix
	 */
	right[0] = gtib;
	right[1] = gtic;
	right[2] = gtid;
	right[3] = gps;		//  Number of bits per page
	right[4] = 0;
	right[5] = 0;
	right[6] = 0;
	right[7] = 0;
	
	if ((crp[0] & 0x3) == 0x2)
		pdSize = kFourBytePD;
	else if ((crp[0] & 0x3) == 0x3)
		pdSize = kEightBytePD;
	else
		Unsupported ("Invalid CRP DT");

	*physAddr = 0;
	*physLen  = 0;
	*physAttr = 0;
	
	(void) walk030PageTable (logAddr, physAddr, physLen, physAttr, 
			rootTable, (crp[0] & 0x7FFF0000) >> 16,
			gis, gtia, right, pdSize, false);
}
