# This file is part of qVamps.
#
# qVamps is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# qVamps is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with qVamps; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


use strict;
use warnings;


# common class for items in the title list
package TitleListCommonItem;
use Qt;
use Qt::isa qw (Qt::CheckListItem);
use Qt::attributes qw (titn ordn tlit);
use constant
{
  TITLE_ITEM           => 0,
  VIDEO_ITEM           => 1,
  AUDIO_ITEM           => 2,
  SUBTITLE_ITEM        => 3,
  VIDEO_ANGLE_ITEM     => 4,
  AUDIO_STREAM_ITEM    => 5,
  SUBTITLE_STREAM_ITEM => 6
};


# map from title list item type (tli_type) to check list item type
our %typemap =
  ( &TITLE_ITEM           => &Qt::CheckListItem::CheckBoxController,
    &VIDEO_ITEM           => &Qt::CheckListItem::RadioButtonController,
    &AUDIO_ITEM           => &Qt::CheckListItem::CheckBoxController,
    &SUBTITLE_ITEM        => &Qt::CheckListItem::CheckBoxController,
    &VIDEO_ANGLE_ITEM     => &Qt::CheckListItem::RadioButton,
    &AUDIO_STREAM_ITEM    => &Qt::CheckListItem::CheckBox,
    &SUBTITLE_STREAM_ITEM => &Qt::CheckListItem::CheckBox );


# TitleListCommonItem (title_nr, ordinal_nr, parent, text, tli_type)
sub NEW
{
  my $this = shift;
  my $titn = shift;
  my $ordn = shift;
  my $tlit = pop;

  push @_, $typemap {$tlit};

  $this -> SUPER::NEW (@_);

  titn = $titn;
  ordn = $ordn;
  tlit = $tlit;
}


sub DESTROY
{
  return unless (SUPER);

  titn = 0;
  ordn = 0;
  tlit = 0;

  SUPER -> DESTROY ();
}


sub remove
{
  titn = 0;
  ordn = 0;
  tlit = 0;

  while (my $item = firstChild ())
  {
    $item -> remove ();
  }

  my $parent = depth () ? parent () : listView ();

  $parent -> takeItem (this);
}


# number of title which we belong to
sub title_nr
{
  return titn;
}


# item's ordinal number - used for sorting
sub ordinal_nr
{
  return ordn;
}


# title list item type
sub tli_type
{
  return tlit;
}


# overload function of Qt::CheckListItem
sub compare
{
  my $oth = shift;
  my $col = shift;
  my $asc = shift;

  # sort by ordinal number in ascending order (regardless of $asc)
  return (ordn <=> $oth -> ordinal_nr ()) * ($asc ? 1 : -1);
}


1;


# timer used in title item for audio selection checking
package TitleItemTimer;
use Qt;
use Qt::isa qw (Qt::Timer);
use Qt::attributes qw (causers title_item);
use Qt::signals audio_check     => [ "QCheckListItem *", "QObjectList *" ];
use Qt::slots trigger_audio_check => [];


# TitleItemTimer (title_item)
sub NEW
{
  my $this       = shift;
  my $title_item = shift;

  $this -> SUPER::NEW (@_);

  causers    = [];
  title_item = $title_item;
  this -> connect (this, SIGNAL "timeout()", SLOT "trigger_audio_check()");
}


sub DESTROY
{
#  print STDERR "TitleItemTimer::DESTROY\n";

  causers    = undef;
  title_item = undef;

  SUPER -> DESTROY ();
}


# register causer and trigger one shot timer
sub trigger
{
  my $causer = shift;

  push @{&causers}, $causer;

  start (0, 1);
}


# our timeout signal gets connected here
sub trigger_audio_check : SLOT()
{
  emit audio_check (title_item, causers);

  causers = [];
}


1;


# title item (top level item of title list)
package TitleListTitleItem;
use Qt;
use Qt::isa qw (TitleListCommonItem);
use Qt::attributes qw (timer list_items);
use TitleItemTimer;


# TitleListTitleItem (title_list_items, title_nr, parent, text)
sub NEW
{
  my $this       = shift;
  my $list_items = shift;

  unshift @_, ($_ [0]);
  push @_, &TITLE_ITEM;

  $this -> SUPER::NEW (@_);

  my $timer  = TitleItemTimer (this);
  timer      = $timer;
  list_items = $list_items;
  $list_items ->
    connect ($timer,
	     SIGNAL "audio_check(QCheckListItem *, QObjectList *)",
	     SLOT   "audio_check(QCheckListItem *, QObjectList *)");
}


sub DESTROY
{
#  print STDERR "TitleListTitleItem::DESTROY\n";

  timer      = undef;
  list_items = undef;

  SUPER -> DESTROY ();
}


sub remove
{
  timer      = undef;
  list_items = undef;

  SUPER -> remove ();
}


# overload function of Qt::CheckListItem
sub stateChange
{
  my $state = state ();

  # trigger audio check if we have been turned on
  timer -> trigger (this) unless ($state == &Off);

  $state = &On unless ($state == &Off);

  list_items -> trigger_update_vts_list (&titn, $state);
}


# trigger timer
sub trigger_audio_check
{
  my $causer = shift;

  timer -> trigger ($causer);
}


# overload function of TitleListCommonItem
sub compare
{
  my $oth = shift;
  my $col = shift;

  # sort by playback time
  if ($col == 1)
  {
    return $oth -> text (1) cmp text (1);
  }

  # sort by size (strip of "MB"
  if ($col == 2)
  {
    my ($a) = $oth -> text ($col) =~ /(\S+)/;
    my ($b) =         text ($col) =~ /(\S+)/;

    return $a <=> $b;
  }

  # call base class implementation as default
  return SUPER -> compare ($oth, $col, 1);
}


# overload function of Qt::CheckListItem
sub setOpen
{
  my $o = shift;

  SUPER -> setOpen ($o);

  return unless ($o);

  for (my $item = firstChild (); $item; $item = $item -> nextSibling ())
  {
    $item -> setOpen (1);
  }
}


1;


package TitleListAngleItem;
use Qt;
use Qt::isa qw (TitleListCommonItem);
use Qt::attributes qw (list_items);


# TitleListAngleItem (title_list_items, title_nr, angle, parent, text)
sub NEW
{
  my $this       = shift;
  my $list_items = shift;

  push @_, &VIDEO_ANGLE_ITEM;

  $this -> SUPER::NEW (@_);

  list_items = $list_items;
}


sub DESTROY
{
#  print STDERR "TitleListAngleItem::DESTROY\n";

  list_items = undef;

  SUPER -> DESTROY ();
}


sub remove
{
  list_items = undef;

  SUPER -> remove ();
}


# overload function of Qt::CheckListItem
sub stateChange
{
  list_items -> trigger_update_vts_list (&titn, &NoChange);
}


1;


package TitleListAudioItem;
use Qt;
use Qt::isa qw (TitleListCommonItem);
use Qt::attributes qw (title_item);


# TitleListAudioItem (title_item, ordinal_nr, parent, text, tli_type)
sub NEW
{
  my $this       = shift;
  my $title_item = shift;

  unshift @_, ($title_item -> title_nr ());

  $this -> SUPER::NEW (@_);

  title_item = $title_item;
}


sub DESTROY
{
#  print STDERR "TitleListAudioItem::DESTROY\n";

  title_item = undef;

  SUPER -> DESTROY ();
}


sub remove
{
  title_item = undef;

  SUPER -> remove ();
}


# overload function of Qt::CheckListItem
sub stateChange
{
  # trigger the audio check if we have been turned off
  title_item -> trigger_audio_check (this) if (state () == &Off);
}


1;


package StreamToolTip;
use Qt;
use Qt::isa qw (Qt::ToolTip);
use Qt::attributes qw (list_items title_list);
use DvdIfoRead;


sub NEW
{
  my $this       = shift;
  my $list_items = shift;
  my $title_list = shift;

  $this -> SUPER::NEW ($title_list -> viewport ());

  list_items = $list_items;
  title_list = $title_list;
}


sub DESTROY
{
#  print STDERR "StreamToolTip::DESTROY\n";

  return unless (SUPER);

  list_items = undef;
  title_list = undef;

  SUPER -> DESTROY ();
}


sub maybeTip
{
  my $item = title_list -> itemAt (@_);

  return unless ($item);

  my $tip = list_items -> tip_text ($item);

  return unless ($tip);

  my $rect = title_list -> itemRect ($item);

  return unless ($rect -> isValid ());

  tip ($rect, $tip);
}


sub intact
{
  return defined (title_list);
}


1;


package TitleListItems;
use Qt;
use Qt::isa qw (Qt::Object);
use Qt::attributes qw (dev dvd title_list tool_tip vli context_menu item_refs);
use Qt::signals update_vts_list => [ "int", "int", "int" ];
use Qt::slots audio_check              => [ "QCheckListItem *",
					    "QObjectList *" ],
              item_double_clicked      => [ "QListViewItem *",
					    "const QPoint &", "int" ],
              item_preview             => [ ],
              item_details             => [ ],
              title_list_context_popup => [ "QListViewItem *",
					    "const QPoint &", "int" ],
              vts_list_context_popup   => [ "QListViewItem *",
					    "const QPoint &", "int" ],
              current_item_changed     => [ "QListViewItem *" ];
use Iso639;
use DvdIfoRead;
use QVamps qw (read_setting replace tr);
use StreamToolTip;
use TitleListCommonItem;
use TitleListTitleItem;
use TitleListAngleItem;
use TitleListAudioItem;
use VTSListItems;


our @video_attributes =
(
  "MPEG Version",		#$translate
  "Permitted Display Formats",	#$translate
  "NTSC CC Type",		#$translate
  "Bit Rate",			#$translate
  "Picture Size",		#$translate
  "Source Letterboxed",		#$translate
  "Film Mode"			#$translate
);

our @audio_attributes =
(
  "Audio application mode",	#$translate
  "Quantization",		#$translate
  "Sample Frequency",		#$translate
  "Audio Code Extension"	#$translate
);

our @subtitle_attributes =
(
  "Subtitle Code Extension"	#$translate
);


sub NEW
{
  my $this       = shift;
  my $dev        = shift;
  my $dvd        = shift;
  my $title_list = shift;
  my $name       = shift;
  my $vts_list   = shift;

  $this -> SUPER::NEW ($title_list, $name);

  dev        = $dev;
  dvd        = $dvd;
  title_list = $title_list;
  item_refs  = [];

  my $nr_of_titles = $dvd -> nr_of_titles ();

  foreach my $ttn (1 .. $nr_of_titles)
  {
    my $tsn           = $dvd -> title_set_nr ($ttn);
    my $ts_ttn        = $dvd -> title_nr_in_title_set ($ttn);
    my @playback_time = $dvd -> title_playback_time ($tsn, $ts_ttn);
    my $nr_of_sectors = $dvd -> title_size ($tsn, $ts_ttn);

    my $title = TitleListTitleItem (this, $ttn, $title_list,
				    (sprintf "%s %d", &tr ("Title"), $ttn));
    $title -> setText (1, (sprintf "%02d:%02d:%02d.%02d", @playback_time));
    $title -> setText (2, (sprintf "%.1f MB", $nr_of_sectors/512.));
    push @{&item_refs}, $title;

    my $video = TitleListCommonItem ($ttn, 0, $title, &tr ("Video"),
				     &TitleListCommonItem::VIDEO_ITEM);
    $video -> setText (1, (sprintf "%s %s",
			   &tr ($dvd -> video_format ($tsn)),
			   &tr ($dvd -> display_aspect_ratio ($tsn))));
    push @{&item_refs}, $video;

    foreach my $angle (1 .. $dvd -> nr_of_angles ($ttn))
    {
      my $v = TitleListAngleItem (this, $ttn, $angle, $video,
				  (sprintf "%s %d", &tr ("Angle"), $angle));
      push @{&item_refs}, $v;

      $v -> setOn (1) if ($angle == 1);
      #$v -> activate () if ($angle == 1);
    }

    my $nr_of_audio_streams = $dvd -> nr_of_audio_streams ($tsn);

    my $audio = TitleListAudioItem ($title, 1, $title, &tr ("Audio"),
				    &TitleListCommonItem::AUDIO_ITEM);
    push @{&item_refs}, $audio;

    foreach my $astrm (1 .. $nr_of_audio_streams)
    {
      my $code = $dvd -> audio_lang_code ($tsn, $astrm);
      my $lang = Iso639::code2lang ($code);
      $lang    = sprintf "%s \"%s\"",
                 &tr ("Language code"), $code unless ($lang);
      my $a    = TitleListAudioItem ($title, $astrm, $audio, &tr ($lang),
				     &TitleListCommonItem::AUDIO_STREAM_ITEM);
      $a -> setText (1, (sprintf "%s %s",
			 &tr ($dvd -> audio_format ($tsn, $astrm)),
			 &tr ($dvd -> audio_channels ($tsn, $astrm))));
      push @{&item_refs}, $a;
    }

    my $nr_of_subtitle_streams = $dvd -> nr_of_subtitle_streams ($tsn);

    if ($nr_of_subtitle_streams)
    {
      my $subtitles = TitleListCommonItem ($ttn, 2, $title, &tr ("Subtitles"),
					  &TitleListCommonItem::SUBTITLE_ITEM);
      push @{&item_refs}, $subtitles;

      foreach my $sstrm (1 .. $nr_of_subtitle_streams)
      {
	my $code = $dvd -> subtitle_lang_code ($tsn, $sstrm);
	my $lang = Iso639::code2lang ($code);
	$lang    = sprintf "%s \"%s\"",
	           &tr ("Language code"), $code unless ($lang);
	my $s    = TitleListCommonItem ($ttn, $sstrm, $subtitles, &tr ($lang),
				   &TitleListCommonItem::SUBTITLE_STREAM_ITEM);
	$s -> setText (1, &tr ($dvd -> subtitle_code_mode ($tsn, $sstrm)));
	push @{&item_refs}, $s;
      }
    }
  }

  $title_list -> setSorting (1);
  $title_list -> setColumnAlignment (2, &Qt::AlignRight);
  my $first_title = $title_list -> firstChild ();
  $first_title -> setOpen (1);
  $title_list  -> setCurrentItem ($first_title);

  this -> connect ($title_list,
	   SIGNAL "doubleClicked(QListViewItem *, const QPoint &, int)",
	   SLOT   "item_double_clicked(QListViewItem *, const QPoint &, int)");

  vli = VTSListItems (this, $dvd, $vts_list, "vts_list_items");
  vli -> connect (this,
		  SIGNAL "update_vts_list(int, int, int)",
		  SLOT   "update(int, int, int)");
  this -> connect ($title_list,
		   SIGNAL "currentChanged(QListViewItem *)",
		   SLOT "current_item_changed(QListViewItem *)");
  current_item_changed ($first_title);

  my $context_menu = Qt::PopupMenu ($title_list, "context_menu");
  $context_menu -> insertItem (&tr ("Preview"),
			       this, SLOT "item_preview()",
			       Qt::KeySequence (&Qt::Key_Return), 0);
  $context_menu -> insertItem (&tr ("Show Details"),
			       this, SLOT "item_details()",
			       Qt::KeySequence (&Qt::Key_F2), 1);
  context_menu = $context_menu;

  this -> connect ($title_list,
	SIGNAL "contextMenuRequested(QListViewItem *, const QPoint &, int)",
	SLOT "title_list_context_popup(QListViewItem *, const QPoint &, int)");
  this -> connect ($vts_list,
	  SIGNAL "contextMenuRequested(QListViewItem *, const QPoint &, int)",
	  SLOT "vts_list_context_popup(QListViewItem *, const QPoint &, int)");

  tool_tip = StreamToolTip (this, $title_list);
}


sub DESTROY
{
#  print STDERR "TitleListItems::DESTROY\n";

  parent () -> removeChild (this);

  dev          = undef;
  dvd          = undef;
  title_list   = undef;
  tool_tip     = undef;
  vli          = undef;
  context_menu = undef;
  item_refs    = undef;

  SUPER -> DESTROY ();
}


sub remove_all
{
  my $title_list   = title_list;
  my $tool_tip     = tool_tip;
  my $context_menu = context_menu;

  dev          = undef;
  dvd          = undef;
  title_list   = undef;
  tool_tip     = undef;
  vli          = undef;
  context_menu = undef;
  item_refs    = undef;

  $title_list -> disconnect ();
  $title_list -> removeChild ($context_menu);
  $tool_tip   -> remove ($title_list -> viewport ())
    if ($tool_tip -> intact ());
  $context_menu -> clear ();

  while (my $item = $title_list -> firstChild ())
  {
    $item -> remove ();
  }
}


sub vts_list_items
{
  return vli;
}


sub audio_check : SLOT(QCheckListItem *, QObjectList *)
{
  my $title = shift;
  my $causers = shift;

  unless ($title -> state () == &TitleListCommonItem::Off)
  {
    # title selected
    my $audio = $title -> firstChild () -> nextSibling ();

    if ($audio -> state () == &TitleListCommonItem::Off)
    {
      # audio deselected
      my @astrms = grep ($_ -> tli_type () ==
			 &TitleListCommonItem::AUDIO_STREAM_ITEM, @{$causers});

      if (@astrms == 1)
      {
        # user deselected last audio stream
        $astrms [0] -> activate ();
      }
      else
      {
        # user deselected all audio or
	# subtitle selection caused title selection with no audio
        $audio -> activate ();
      }
    }
  }
}


sub item_double_clicked : SLOT(QListViewItem *, const QPoint &, int)
{
  my $item = shift;

  play_stream ($item);
}


sub item_preview : SLOT()
{
  unless (title_list -> hasFocus ())
  {
    vli -> item_preview ();

    return;
  }

  play_stream (title_list -> currentItem ());
}


sub play_stream
{
  my $item = shift;

  return unless ($item);

  my $tlit  = $item -> tli_type ();
  my $titn  = $item -> title_nr ();
  my $ordn  = $item -> ordinal_nr ();
  my $angle = items_selected_angle ($item, $titn);
  my $astrm = first_selected_audio ($item, $titn);
  my $sstrm = first_selected_subtitle ($item, $titn);

  if ($tlit == &TitleListCommonItem::VIDEO_ANGLE_ITEM)
  {
    $angle = $ordn;
  }
  elsif ($item -> tli_type () == &TitleListCommonItem::AUDIO_STREAM_ITEM)
  {
    $astrm = $ordn;
  }
  elsif ($item -> tli_type () == &TitleListCommonItem::SUBTITLE_STREAM_ITEM)
  {
    $sstrm = $ordn;
  }

  my $vid = 0xe0;		# no representation for angle in MPEG PS
  my $aid = audio_base_id ($titn, $astrm) + $astrm - 1;
  my $sid = 0x20 + $sstrm - 1;

  my $vopt = read_setting ("/MediaPlayer/select_angle");
  my $aopt = read_setting ("/MediaPlayer/select_audio");
  my $sopt = read_setting ("/MediaPlayer/select_subtitle");

  my @opt;
  push @opt, replace ($vopt, { "n" => $angle-1,
			       "N" => $angle, "i" => $vid }) if ($vopt);
  push @opt, replace ($aopt, { "n" => $astrm-1,
			       "N" => $astrm, "i" => $aid }) if ($aopt);
  push @opt, replace ($sopt, { "n" => $sstrm-1,
			       "N" => $sstrm, "i" => $sid })
    if ($sopt && $sstrm);
  my $opt = join (" ", @opt);
  my $cmd = read_setting ("/MediaPlayer/play_dvd_title");
  $cmd    = replace ($cmd, { "d" => dev,     "o" => $opt,
			     "t" => $titn-1, "T" => $titn });
  system ("$cmd &");
}


sub items_selected_angle
{
  my $item = shift;
  my $ttn  = shift;
  my $title;

  for ($title = $item; $title -> depth (); $title = $title -> parent ())
  {
  }

  for (my $it =
       Qt::ListViewItemIterator ($title, &Qt::ListViewItemIterator::Checked);
       my $item = $it -> current (); $it++)
  {
    return $item -> ordinal_nr ()
      if ($item -> title_nr () == $ttn &&
	  $item -> tli_type () == &TitleListCommonItem::VIDEO_ANGLE_ITEM);
  }

  return 1;
}


sub first_selected_audio
{
  my $item = shift;
  my $ttn  = shift;
  my $title;

  for ($title = $item; $title -> depth (); $title = $title -> parent ())
  {
  }

  for (my $it =
       Qt::ListViewItemIterator ($title, &Qt::ListViewItemIterator::Checked);
       my $item = $it -> current (); $it++)
  {
    return $item -> ordinal_nr ()
      if ($item -> title_nr () == $ttn &&
	  $item -> tli_type () == &TitleListCommonItem::AUDIO_STREAM_ITEM);
  }

  return 1;
}


sub first_selected_subtitle
{
  my $item = shift;
  my $ttn  = shift;
  my $title;

  for ($title = $item; $title -> depth (); $title = $title -> parent ())
  {
  }

  for (my $it =
       Qt::ListViewItemIterator ($title, &Qt::ListViewItemIterator::Checked);
       my $item = $it -> current (); $it++)
  {
    return $item -> ordinal_nr ()
      if ($item -> title_nr () == $ttn &&
	  $item -> tli_type () == &TitleListCommonItem::SUBTITLE_STREAM_ITEM);
  }

  return 0;
}


sub audio_base_id
{
  my $ttn  = shift;
  my $strm = shift;

  # some crappy DVDs have titles with no audio at all
  return 0
    if ($strm > dvd -> nr_of_audio_streams (dvd -> title_set_nr ($ttn)));

  my $fmt = dvd -> vts_audio_attr (dvd -> title_set_nr ($ttn),
				   $strm) -> {audio_format};

  return 0x80 if ($fmt == 0);	# AC3
  return 0x88 if ($fmt == 6);	# DTS
  return 0xa0 if ($fmt == 4);	# LPCM
  return 0xc0;			# MPEG
}


sub item_details : SLOT()
{
  my $title_list = title_list;

  unless ($title_list -> hasFocus ())
  {
    vli -> item_details ();

    return;
  }

  my $item = $title_list -> currentItem ();

  return unless ($item);

  my $tip = tip_text ($item);

  return unless ($tip);

  my $vpos = $title_list -> itemRect ($item) -> center ();
  my $gpos = $title_list -> viewport () -> mapToGlobal ($vpos);

  Qt::WhatsThis::display ($tip, $gpos);
}


sub tip_text
{
  my $item = shift;

  return undef unless ($item);

  my $tip;
  my $dvd      = dvd;
  my $tli_type = $item -> tli_type ();

  if ($tli_type == &TitleListCommonItem::VIDEO_ITEM)
  {
    my @values;
    my $tsn = $dvd -> title_set_nr ($item -> title_nr ());
    push @values, $dvd -> mpeg_version ($tsn);
    my @permitted_display_formats =
      map &tr ($_), $dvd -> permitted_display_formats ($tsn);
    push @values, join (",", @permitted_display_formats);
    my @ntsc_cc = $dvd -> ntsc_cc ($tsn);
    push @ntsc_cc, "No CCs" unless (@ntsc_cc);			#$translate
    push @values, join (" " . &tr ("and") . " ", @ntsc_cc);
    push @values, $dvd -> constant_bit_rate ($tsn) ?
      "Constant" : "Variable";					#$translate
    push @values, join ("x", $dvd -> picture_size ($tsn));
    push @values, $dvd -> letterboxed ($tsn) ? "Yes" : "No";	#$translate
    push @values, $dvd -> film_mode ($tsn) ?
      "Film" : "Video Camera";					#$translate
    $tip = make_table (\@video_attributes, \@values);
  }
  elsif ($tli_type == &TitleListCommonItem::AUDIO_STREAM_ITEM)
  {
    my @values;
    my $tsn    = $dvd -> title_set_nr ($item -> title_nr ());
    my $stream = $item -> ordinal_nr ();
    push @values, $dvd -> audio_application_mode ($tsn, $stream);
    push @values, $dvd -> audio_quantization ($tsn, $stream);
    push @values, $dvd -> audio_sample_frequency ($tsn, $stream);
    push @values, $dvd -> audio_code_extension ($tsn, $stream);
    $tip = make_table (\@audio_attributes, \@values);
  }
  elsif ($tli_type == &TitleListCommonItem::SUBTITLE_STREAM_ITEM)
  {
    my @values;
    my $tsn    = $dvd -> title_set_nr ($item -> title_nr ());
    my $stream = $item -> ordinal_nr ();
    push @values, $dvd -> subtitle_code_extension ($tsn, $stream);
    $tip = make_table (\@subtitle_attributes, \@values);
  }

  return $tip;
}


sub make_table
{
  my @attr = @{shift ()};
  my @val  = @{shift ()};

  unshift @attr, sprintf "<b>%s</b>", &tr ("Attribute");
  unshift @val,  sprintf "<b>%s</b>", &tr ("Value");

  my $text = "<table cellspacing=0 cellpadding=0 border=1>";

  for (my $i = 0; $i < @attr; $i++)
  {
    my $a = &tr ($attr [$i]);
    my $v = &tr ($val  [$i]);

    $text .= "<tr><td><nobr>$a</nobr></td><td><nobr>$v</nobr></td></tr>";
  }

  $text .= "</table>";
}


sub title_list_context_popup : SLOT(QListViewItem *, const QPoint &, int)
{
  my $item   = shift;
  my $pos    = shift;
  my $col    = shift;

  return unless ($item);

  my $tlit = $item -> tli_type ();

  context_menu ->
    setItemEnabled (1, $tlit == &TitleListCommonItem::VIDEO_ITEM ||
		    $tlit >= &TitleListCommonItem::AUDIO_STREAM_ITEM);
  context_menu -> popup ($pos);
}


sub vts_list_context_popup : SLOT(QListViewItem *, const QPoint &, int)
{
  my $item   = shift;
  my $pos    = shift;
  my $col    = shift;

  return unless ($item &&
		 $item -> vli_type () == &VTSListCommonItem::PGM_ITEM);

  context_menu -> popup ($pos);
}


sub trigger_update_vts_list
{
  my $ttn   = shift;
  my $state = shift;

  emit update_vts_list ($ttn, selected_angle ($ttn), $state);
}


sub current_item_changed : SLOT(QListViewItem *)
{
  my $item = shift;

  my $ttn   = $item -> title_nr ();
  my $angle = items_selected_angle ($item, $ttn);

  emit update_vts_list ($ttn, $angle, &TitleListCommonItem::NoChange);
}


sub selected_titles
{
  my @rv = ();

  for (my $it = Qt::ListViewItemIterator (title_list);
       my $item = $it -> current (); $it++)
  {
    # Qt::ListViewItemIterator::Checked iterator flag does
    # not work on CheckBoxController (tristate) type items
    next if ($item -> state () == &TitleListCommonItem::Off);

    push @rv, $item -> title_nr ()
      if ($item -> tli_type () == &TitleListCommonItem::TITLE_ITEM);
  }

  return \@rv;
}


sub selected_angle
{
  my $ttn = shift;

  my $item;

  for (my $it = Qt::ListViewItemIterator (title_list);
       $item = $it -> current (); $it++)
  {
    return items_selected_angle ($item, $ttn)
      if ($item -> title_nr () == $ttn);
  }

  return 1;
}


sub selected_audio_streams
{
  my $ttn = shift;

  return selected_streams ($ttn, &TitleListCommonItem::AUDIO_STREAM_ITEM);
}


sub selected_subtitle_streams
{
  my $ttn = shift;

  return selected_streams ($ttn, &TitleListCommonItem::SUBTITLE_STREAM_ITEM);
}


sub selected_streams
{
  my $ttn  = shift;
  my $type = shift;

  my $item;
  my @rv = ();

  for (my $it = Qt::ListViewItemIterator (title_list);
       $item = $it -> current (); $it++)
  {
    last if ($item -> title_nr () == $ttn);
  }

  for (my $it = Qt::ListViewItemIterator ($item,
					  &Qt::ListViewItemIterator::Checked);
       $item = $it -> current (); $it++)
  {
    last if ($item -> title_nr () != $ttn);

    push @rv, $item -> ordinal_nr () if ($item -> tli_type () == $type);
  }

  return \@rv;
}


sub current_item
{
  return title_list -> currentItem ();
}


1;
