/*
 * snes9express
 * sound.cc
 * Copyright (C) 1998-2002  David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */

#include <stdlib.h>
#include "sound.h"

static char*SoundRates[] = {
   "0 (no sound)",
     "1 ( 8192 Hz)",
     "2 (11025 Hz)",
     "3 (16500 Hz)",
     "4 (22050 Hz)",
     "5 (29300 Hz)",
     "6 (36600 Hz)",
     "7 (44000 Hz)",
     "",
};

s9x_Sound::s9x_Sound(fr_Notebook*parent):
s9x_Notepage(parent, "Sound"),
Sound(this, "Sound", true),
Stereo(this, "Stereo", true),
Filters(this, "Filters", true),
NoCache(this, "No Cache", false),
ThreadSound(this, "Thread Sound", false),
Skip(this, "Skip Method:", 0, 0, 3, 0),
Rate(this, "Rate:", fr_Horizontal, 4, 1, 7, 0, false),
RateLabel(this, SoundRates[4]),
Envelope(this, "Envelope", false),
Sync(this, "Sync", false),
Interpolate(this, "Interpolate", false),
NoMasterVolume(this, "Play it Loud", false)
{
   SetGridSize(4, 3, true);

   Sound.Args << fr_CaseInsensitive << "-sound" << "-so";
   Sound.NotArgs << fr_CaseInsensitive << "-nosound" << "-NS";
   Sound.SetTooltip("Use sound emulation. (If your desktop environment"
		    " uses sound effects, snes9x might not be able to"
		    " access the sound device)");
   Sound.AddListener(this);
   Pack(Sound);
   addOption(Sound);

   Stereo.Args << fr_CaseInsensitive << "-stereo" << "-st";
   Stereo.NotArgs << fr_CaseInsensitive << "-mono";
   Stereo.SetTooltip("Select either stereo or mono sound");
   Pack(Stereo);
   addOption(Stereo);

   Filters.Args << fr_CaseInsensitive << "-echo" << "-e";
   Filters.NotArgs << fr_CaseInsensitive << "-ne" << "-noecho";
   Filters.SetTooltip("Echo & FIR sound filters.  Filters are slow on non-MMX processors");
   Pack(Filters);
   addOption(Filters);

   NoCache.Args << fr_CaseInsensitive << "-nsc" << "-nosamplecaching" << "-nc";
   NoCache.SetTooltip("Disable sound sample caching. "
		      "Sample caching speeds up emulation by "
		      "keeping a decompressed copy of sound samples "
		      "in memory. However, this can some times cause "
		      "distortion in looped samples.");
   Pack(NoCache, 3, 0, 4, 1);
   addOption(NoCache);

   ThreadSound.Args << fr_CaseInsensitive << "-ts" << "-threadsound";
   ThreadSound.SetTooltip("Use a separate thread for sound processing. "
                          "This may benefit multi-cpu systems "
                          "or certain sound card drivers.");

   Pack(ThreadSound, 3, 0, 4, 1);
   addOption(ThreadSound);
   
   Skip.Args << fr_CaseInsensitive << "-soundskip" << "-sk";
   Skip.SetTooltip("Sound CPU skip waiting method");
   Pack(Skip, 0, 1, 2, 2);
   addOption(Skip);

   Rate.Args << fr_CaseInsensitive << "-r" << "-soundquality" << "-sq";
   Rate.SetTooltip("sound rate: lower runs faster / higher sounds better");
   Rate.AddListener(this);
   Pack(Rate, 0, 1, 2, 2);
   addOption(Rate);

   RateLabel.SetVisibility(true);
   Pack(RateLabel);

   NoMasterVolume.Args << fr_CaseInsensitive << "-nmv" << "-nomastervolume";
   NoMasterVolume.SetTooltip("Lock the emulator's master volume at full. "
			     " This is helpful for ROMs that set the volume"
			     " rather low.");
   Pack(NoMasterVolume);
   addOption(NoMasterVolume);

   Interpolate.Args << fr_CaseInsensitive << "-is" << "-interpolatedsound";
   Interpolate.SetKeyStroke("[");
   Interpolate.SetTooltip("Sound samples played at a frequency lower "
			  "than the specified Rate will have "
			  "intermediate sample values calculated, to "
			  "improve the sound quality of low-rate "
			  "samples.");
   Pack(Interpolate);
   addOption(Interpolate);
	
   Envelope.Args << fr_CaseInsensitive << "-ex" << "-envx";
   Envelope.SetTooltip("Use envelope volume height.  Can fix some sound"
		       " problems, but can also freeze in some ROMs");
   Pack(Envelope);
   addOption(Envelope);

   Sync.Args << fr_CaseInsensitive << "-sy" << "-soundsync";
   Sync.SetKeyStroke("]");
   Sync.SetTooltip("Synchronize sound playback with instructions"
		   " from the SNES sound CPU.  This option will also"
		   " activate interpolated sound and envelope height"
		   " reading.");
   Sync.AddListener(this);
   Pack(Sync);
   addOption(Sync);
}

void s9x_Sound::Set9xVersion(float version) {
   NoMasterVolume.EnableIf(version >= 1.05);
   Interpolate.EnableIf(version >= 1.19);
   Sync.EnableIf(version >= 1.19);
   if(version <= 1.41)
   {
        NoCache.EnableIf(true);
        NoCache.SetVisibility(true);
        ThreadSound.EnableIf(false);
        ThreadSound.SetVisibility(false);
   } else {
        NoCache.EnableIf(false);
        NoCache.SetVisibility(false);
        ThreadSound.EnableIf(true);
        ThreadSound.SetVisibility(true);
   }
}

void s9x_Sound::CompileArgs(fr_ArgList& L) {
   if(!Sound.GetState()) {
      Sound.CompileArgs(L);
      Skip.CompileArgs(L);
      return;
   };
   Stereo.CompileArgs(L);
   Rate.CompileArgs(L);
   Filters.CompileArgs(L);
   Envelope.CompileArgs(L);
   NoMasterVolume.CompileArgs(L);
   Interpolate.CompileArgs(L);
   Sync.CompileArgs(L);
   NoCache.CompileArgs(L);
   ThreadSound.CompileArgs(L);
}

void s9x_Sound::EventOccurred(fr_Event*e) {
   if(e->Is(Sound))
     Toggled(Sound.GetState());
   else if(e->Is(Rate))
     RateLabel.SetLabel(SoundRates[(int)Rate.GetValue()]);
   else if(e->Is(Sync)&&Sync.GetState()) {
      /* this is just to demonstrate visually that Sync also
       * turns on Envelope and Interpolate */
      Interpolate.SetState(true);
      Envelope.SetState(true);
   };
}

void s9x_Sound::Toggled(bool s) {
   Stereo.SetEditable(s);
   Filters.SetEditable(s);
   Envelope.SetEditable(s);
   NoMasterVolume.SetEditable(s);
   RateLabel.SetEditable(s);
   Interpolate.SetEditable(s);
   Sync.SetEditable(s);
   NoCache.SetEditable(s);
   ThreadSound.SetEditable(s);
   if(s) {
      Skip.SetVisibility(false);
      Rate.SetVisibility(true);
      RateLabel.SetLabel(SoundRates[(int)Rate.GetValue()]);
   } else {
      Rate.SetVisibility(false);
      Skip.SetVisibility(true);
      RateLabel.SetLabel(SoundRates[0]);
   }
}
