/*
	lg3zoom.c - zoom the image using Lagrange polynominals algorithm

	Author:   Susumu Shiohara (shiohara@tpp.epson.co.jp)

		Copyright 1993-1997 by Susumu Shiohara

			All Rights Reserved

	------------------------------------------------------------------

	This enlargement and reduction method was proposed by

			Mr. Shinhwan KIM
			Mr. Shigeo KATO
			Mr. Yasuhiko YASUDA

			Institute of Industrial Science, University of Tokyo

			18th Image Technology Conference, 1987
*/

#include "xslideshow.h"

extern void goodbyekiss();
extern void myevent();
/*
	LG3 (Lagrange polynominals algorithm)

	M: Magnify rate
		Enlarge   > 1.0
		Reduce    < 1.0

	S0       S1       S2      S3
    .        .<- dk ->.       .
	0        1        2       3

	The pixel value which has been interpolated is: 
	Yk = LG3(dk)		(k=1,2,,,,,,n*M)


	3 degree lagrange polynominals is:
	LG3(dk) = (A * (dk)^3 + B * (dk)^2 + C * (dk)) + D

	A = ( -1*S0 +  3*S1 -  3*S2 + 1*S3)/6
	B = (  6*S0 - 15*S1 + 12*S2 - 3*S3)/6
	C = (-11*S0 + 18*S1 -  9*S2 + 2*S3)/6
	D = S0

	dk = 1 + (k-1)/M - (INT)((k-1)/M) + a	(0<= a < 1)

*/

#if defined(__STDC__) || defined(__cplusplus)
static void readLine(byte *srcptr, byte *dstptr, dword xwidth, double rzoom)
#else
static void readLine(srcptr, dstptr, xwidth, rzoom)
byte *srcptr;
byte *dstptr;
dword xwidth;
double rzoom;
#endif
{
dword x;
int iposi;
double dposi;
double dk1,dk2,dk3;
double A,B,C,D;
double r,g,b;
dword r0,r1,r2,r3;
dword g0,g1,g2,g3;
dword b0,b1,b2,b3;

	/* Calcurate LG3 */
	for (x = 1; x < (xwidth-2); x++) {

		iposi = (int)((x-1)/rzoom);
		dposi = (double)((x-1)/rzoom);

		/* Set the interpolation position */
		dk1 = 1.0 + dposi - iposi + 0.0;
		dk2 = dk1*dk1;
		dk3 = dk1*dk2;

		/* Get the pixel value */
		if(gim.subImageList->bits_per_pixel == 8){
			/* Red */
			r0 = gim.subImageList->red[*(srcptr+iposi-1)];
			r1 = gim.subImageList->red[*(srcptr+iposi+0)];
			r2 = gim.subImageList->red[*(srcptr+iposi+1)];
			r3 = gim.subImageList->red[*(srcptr+iposi+2)];

			/* Green */
			g0 = gim.subImageList->green[*(srcptr+iposi-1)];
			g1 = gim.subImageList->green[*(srcptr+iposi+0)];
			g2 = gim.subImageList->green[*(srcptr+iposi+1)];
			g3 = gim.subImageList->green[*(srcptr+iposi+2)];

			/* Blue */
			b0 = gim.subImageList->blue[*(srcptr+iposi-1)];
			b1 = gim.subImageList->blue[*(srcptr+iposi+0)];
			b2 = gim.subImageList->blue[*(srcptr+iposi+1)];
			b3 = gim.subImageList->blue[*(srcptr+iposi+2)];
		}
		else {
			/* Red */
			r0 = *(srcptr+(iposi-1)*3+0);
			r1 = *(srcptr+(iposi+0)*3+0);
			r2 = *(srcptr+(iposi+1)*3+0);
			r3 = *(srcptr+(iposi+2)*3+0);

			/* Green */
			g0 = *(srcptr+(iposi-1)*3+1);
			g1 = *(srcptr+(iposi+0)*3+1);
			g2 = *(srcptr+(iposi+1)*3+1);
			g3 = *(srcptr+(iposi+2)*3+1);

			/* Blue */
			b0 = *(srcptr+(iposi-1)*3+2);
			b1 = *(srcptr+(iposi+0)*3+2);
			b2 = *(srcptr+(iposi+1)*3+2);
			b3 = *(srcptr+(iposi+2)*3+2);
		}

		A = ( 1 * (double)r3 -  3 * (double)r2 +  3 * (double)r1 -  1 * (double)r0)/6;
		B = (-3 * (double)r3 + 12 * (double)r2 - 15 * (double)r1 +  6 * (double)r0)/6;
		C = ( 2 * (double)r3 -  9 * (double)r2 + 18 * (double)r1 - 11 * (double)r0)/6;
		D = (double)r0;
		r = A*dk3 + B*dk2 + C*dk1 + D;

		A = ( 1 * (double)g3 -  3 * (double)g2 +  3 * (double)g1 -  1 * (double)g0)/6;
		B = (-3 * (double)g3 + 12 * (double)g2 - 15 * (double)g1 +  6 * (double)g0)/6;
		C = ( 2 * (double)g3 -  9 * (double)g2 + 18 * (double)g1 - 11 * (double)g0)/6;
		D = (double)g0;
		g = A*dk3 + B*dk2 + C*dk1 + D;

		A = ( 1 * (double)b3 -  3 * (double)b2 +  3 * (double)b1 -  1 * (double)b0)/6;
		B = (-3 * (double)b3 + 12 * (double)b2 - 15 * (double)b1 +  6 * (double)b0)/6;
		C = ( 2 * (double)b3 -  9 * (double)b2 + 18 * (double)b1 - 11 * (double)b0)/6;
		D = (double)b0;
		b = A*dk3 + B*dk2 + C*dk1 + D;

		if(r > 255.0) r = 255.0; else if(r < 0.0) r = 0.0;
		if(g > 255.0) g = 255.0; else if(g < 0.0) g = 0.0;
		if(b > 255.0) b = 255.0; else if(b < 0.0) b = 0.0;

		*dstptr++ = (byte)r;
		*dstptr++ = (byte)g;
		*dstptr++ = (byte)b;
	}
}

#if defined(__STDC__) || defined(__cplusplus)
void lg3zoom(dword xzoom, dword yzoom)
#else
void lg3zoom(xzoom, yzoom)
dword xzoom, yzoom;
#endif
{ 
int i;
dword x, y;
dword xwidth, ywidth;
double rzoom;
dword srclinelen;
byte *srcptr, *srcdata, *dstptr, *dstdata;
int iposi;
double dposi;
double dk1,dk2,dk3;
double A,B,C,D;
double r,g,b;
dword r0,r1,r2,r3;
dword g0,g1,g2,g3;
dword b0,b1,b2,b3;
byte *lineBuf[5];

	rzoom= (double)xzoom / 100.0;
	xwidth= rzoom * gim.subImageList->width;
	ywidth= rzoom * gim.subImageList->height;

	if(app_data.verbose){
		fprintf(stderr,"xslideshow: lg3zoom() %d%% (%d x %d) -> (%d x %d)\n",
				(int)xzoom, (int)gim.subImageList->width,(int)gim.subImageList->height,(int)xwidth,(int)ywidth);
	}

	dstdata = (byte *)XtCalloc(xwidth * ywidth * 3, sizeof(byte));
	if(dstdata ==(byte *)NULL){
		(void) fprintf(stderr,"xslideshow: lg3zoom() memory allocation error.\n");
		goodbyekiss();
	}

	for(i=0; i<4; i++){
		lineBuf[i] = (byte *)XtCalloc(xwidth * 3, sizeof(byte));
		if(lineBuf[i] ==(byte *)NULL){
			(void) fprintf(stderr,"xslideshow: lg3zoom() memory allocation error.\n");
			goodbyekiss();
		}
	}

	/* read and write from the second pixel */
	if(gim.subImageList->bits_per_pixel == 8) {
		srcdata = gim.subImageList->image + 1;
		srclinelen = gim.subImageList->width;
	}
	else {
		srcdata = gim.subImageList->image + 3;
		srclinelen = gim.subImageList->width * 3;
	}
	dstptr = dstdata + 3;

	/* Read 3 line */
	for(y=1; y < 4; y++) {
		iposi = (int)((y-1)/rzoom);
		srcptr = srcdata + iposi * srclinelen;
		if(y < ywidth)
			readLine(srcptr, lineBuf[y-1], xwidth, rzoom);
	}

	for(y=1; y < (ywidth-2); y++) {
		iposi = (int)((y-1)/rzoom);
		dposi = (double)((y-1)/rzoom);

		/* Set the interpolation line position  (0.0 <= a < 1.0) */
		dk1 = 1.0 + dposi - iposi + 0.0;
		dk2 = dk1*dk1;
		dk3 = dk1*dk2;

		/* Read the 4th line */
		srcptr = srcdata + (int)((y-1+3)/rzoom) * srclinelen;
		readLine(srcptr, lineBuf[3], xwidth, rzoom);

		for(x=0; x < xwidth; x++) {
			r0 = *(lineBuf[0]+x*3+0); r1 = *(lineBuf[1]+x*3+0);
			r2 = *(lineBuf[2]+x*3+0); r3 = *(lineBuf[3]+x*3+0);

			g0 = *(lineBuf[0]+x*3+1); g1 = *(lineBuf[1]+x*3+1);
			g2 = *(lineBuf[2]+x*3+1); g3 = *(lineBuf[3]+x*3+1);

			b0 = *(lineBuf[0]+x*3+2); b1 = *(lineBuf[1]+x*3+2);
			b2 = *(lineBuf[2]+x*3+2); b3 = *(lineBuf[3]+x*3+2);

			A = ( 1 * (double)r3 -  3 * (double)r2 +  3 * (double)r1 -  1 * (double)r0)/6;
			B = (-3 * (double)r3 + 12 * (double)r2 - 15 * (double)r1 +  6 * (double)r0)/6;
			C = ( 2 * (double)r3 -  9 * (double)r2 + 18 * (double)r1 - 11 * (double)r0)/6;
			D = (double)r0;
			r = A*dk3 + B*dk2 + C*dk1 + D;

			A = ( 1 * (double)g3 -  3 * (double)g2 +  3 * (double)g1 -  1 * (double)g0)/6;
			B = (-3 * (double)g3 + 12 * (double)g2 - 15 * (double)g1 +  6 * (double)g0)/6;
			C = ( 2 * (double)g3 -  9 * (double)g2 + 18 * (double)g1 - 11 * (double)g0)/6;
			D = (double)g0;
			g = A*dk3 + B*dk2 + C*dk1 + D;

			A = ( 1 * (double)b3 -  3 * (double)b2 +  3 * (double)b1 -  1 * (double)b0)/6;
			B = (-3 * (double)b3 + 12 * (double)b2 - 15 * (double)b1 +  6 * (double)b0)/6;
			C = ( 2 * (double)b3 -  9 * (double)b2 + 18 * (double)b1 - 11 * (double)b0)/6;
			D = (double)b0;
			b = A*dk3 + B*dk2 + C*dk1 + D;

			if(r > 255.0) r = 255.0; else if(r < 0.0) r = 0.0;
			if(g > 255.0) g = 255.0; else if(g < 0.0) g = 0.0;
			if(b > 255.0) b = 255.0; else if(b < 0.0) b = 0.0;

			*dstptr++ = (byte)r;
			*dstptr++ = (byte)g;
			*dstptr++ = (byte)b;

			myevent();
		}

		/* Set next 3 line data */
		lineBuf[4] = lineBuf[0];
		lineBuf[0] = lineBuf[1]; lineBuf[1] = lineBuf[2];
		lineBuf[2] = lineBuf[3]; lineBuf[3] = lineBuf[4];
	}

	for(i=0; i<4; i++)
		XtFree((char *)(lineBuf[i]));
	XtFree((char *)(gim.subImageList->image));
	gim.subImageList->bits_per_pixel = 24;
	gim.subImageList->image  = dstdata;
	gim.subImageList->width  = xwidth;
	gim.subImageList->height = ywidth;
}

