---------------------------------------------------------------------
--  Framework.Plugs - Package body                                  --
--                                                                  --
--  This software  is (c) The European Organisation  for the Safety --
--  of Air  Navigation (EUROCONTROL) and Adalog  2004-2005. The Ada --
--  Controller  is  free software;  you can redistribute  it and/or --
--  modify  it under  terms of  the GNU  General Public  License as --
--  published by the Free Software Foundation; either version 2, or --
--  (at your  option) any later version.  This  unit is distributed --
--  in the hope  that it will be useful,  but WITHOUT ANY WARRANTY; --
--  without even the implied warranty of MERCHANTABILITY or FITNESS --
--  FOR A  PARTICULAR PURPOSE.  See the GNU  General Public License --
--  for more details.   You should have received a  copy of the GNU --
--  General Public License distributed  with this program; see file --
--  COPYING.   If not, write  to the  Free Software  Foundation, 59 --
--  Temple Place - Suite 330, Boston, MA 02111-1307, USA.           --
--                                                                  --
--  As  a special  exception, if  other files  instantiate generics --
--  from the units  of this program, or if you  link this unit with --
--  other files  to produce  an executable, this  unit does  not by --
--  itself cause the resulting executable  to be covered by the GNU --
--  General  Public  License.   This  exception  does  not  however --
--  invalidate any  other reasons why the executable  file might be --
--  covered by the GNU Public License.                              --
--                                                                  --
--  This  software is  distributed  in  the hope  that  it will  be --
--  useful,  but WITHOUT  ANY  WARRANTY; without  even the  implied --
--  warranty  of  MERCHANTABILITY   or  FITNESS  FOR  A  PARTICULAR --
--  PURPOSE.                                                        --
----------------------------------------------------------------------

-- ASIS
with
  Asis.Elements;

-- All general rules
with
  Rules.Abnormal_Function_Return,
  Rules.Allocators,
  Rules.Array_Declarations,
  Rules.Barrier_Expressions,
  Rules.Max_Call_Depth,
  Rules.Case_Statement,
  Rules.Control_Characters,
  Rules.Declarations,
  Rules.Default_Parameter,
  Rules.Directly_Accessed_Globals,
  Rules.Entities,
  Rules.Entity_Inside_Exception,
  Rules.Exception_Propagation,
  Rules.Expressions,
  Rules.Global_References,
  Rules.Header_Comments,
  Rules.If_For_Case,
  Rules.No_Safe_Initialization,
  Rules.Instantiations,
  Rules.Insufficient_Parameters,
  Rules.Local_Hiding,
  Rules.Local_Instantiation,
  Rules.Max_Blank_Lines,
  Rules.Max_Line_Length,
  Rules.Max_Nesting,
  Rules.Max_Parameters,
  Rules.Max_Statement_Nesting,
  Rules.Movable_Accept_Statements,
  Rules.Naming_Convention,
  Rules.Non_Static,
  Rules.Not_Elaboration_Calls,
  Rules.Other_Dependencies,
  Rules.Parameter_Aliasing,
  Rules.Potentially_Blocking_Operations,
  Rules.Pragmas,
  Rules.Reduceable_Scope,
  Rules.Representation_Clauses,
  Rules.Return_Type,
  Rules.Side_Effect_Parameters,
  Rules.Silent_Exceptions,
  Rules.Simplifiable_Expressions,
  Rules.Special_Comments,
  Rules.Statements,
  Rules.Style,
  Rules.Terminating_Tasks,
  Rules.Unnecessary_Use,
  Rules.Unsafe_Paired_Calls,
  Rules.Unsafe_Unchecked_Conversion,
  Rules.Usage,
  Rules.Use_Clauses,
  Rules.With_Clauses;
package body Framework.Plugs is

   ----------------
   -- Enter_Unit --
   ----------------

   -- This procedure is called before any processing of a library unit.
   -- Plug calls here for rules that need initialization for every unit.

   procedure Enter_Unit (Unit : in Asis.Compilation_Unit) is
   begin
      Rules.Max_Blank_Lines. Enter_Unit   (Unit);
      Rules.Declarations.    Process_Unit (Unit);
   end Enter_Unit;

   ---------------
   -- Exit_Unit --
   ---------------

   -- This procedure is called after all processing of a library unit.
   -- Plug calls here for rules that need finalization for every unit.

   procedure Exit_Unit (Unit : in Asis.Compilation_Unit) is
   begin
      Rules.With_Clauses. Process_Unit_Exit (Unit) ;
   end Exit_Unit;

   -----------------
   -- Enter_Scope --
   -----------------

   -- This procedure is called whenever we enter (Pre_Procedure) into a
   -- construct where elements can be declared. Since this happen in various
   -- places in the tree and is likely to be quite common use, it is easier to
   -- plug calls here rather than in every place that might require it.

   procedure Enter_Scope (Element : in Asis.Element) is
   begin
      Rules.Max_Nesting.Process_Scope_Enter (Element);
      Rules.Statements. Process_Scope_Enter (Element);
   end Enter_Scope;

   ----------------
   -- Exit_Scope --
   ----------------

   -- This procedure is called whenever we exit (Post_Procedure) from a
   -- construct where elements can be declared. Since this happen in various
   -- places in the tree and is likely to be quite common use, it is easier to
   -- plug calls here rather than in every place that might require it.

   procedure Exit_Scope (Element : in Asis.Element) is
   begin
      Rules.Max_Nesting.      Process_Scope_Exit (Element);
      Rules.Reduceable_Scope. Process_Scope_Exit (Element);
      Rules.Statements.       Process_Scope_Exit (Element);
      Rules.Unnecessary_Use.  Process_Scope_Exit (Element);
   end Exit_Scope;

   ---------------------
   -- True_Identifier --
   ---------------------

   --  Plug calls here to rules that need to process all occurrences
   --  of "true" identifiers, including operator symbols and
   --  enumeration literals, but excluding identifiers that are pragma
   --  names or attributes selectors

   procedure True_Identifier (Element : in Asis.Expression) is
   begin
      Rules.Directly_Accessed_Globals. Process_Identifier (Element);
      Rules.Entities.                  Process_Identifier (Element);
      Rules.Reduceable_Scope.          Process_Identifier (Element);
      Rules.Unnecessary_Use.           Process_Identifier (Element);
      Rules.Style.                     Process_Identifier (Element);
      Rules.Usage.                     Process_Identifier (Element);
      Rules.With_Clauses.              Process_Identifier (Element);
   end True_Identifier;

   -------------------
   -- Pre_Procedure --
   -------------------

   procedure Pre_Procedure (Element : in Asis.Element) is
      use Asis, Asis.Elements;
   begin
      case Element_Kind (Element) is
         when A_Clause =>
            Rules.Style. Process_Element (Element);

            case Clause_Kind (Element) is
               when A_Use_Package_Clause =>
                  Rules.Use_Clauses.Process_Use_Clause (Element);
               when A_Representation_Clause =>
                  Rules.Representation_Clauses.Process_Clause (Element);
               when A_With_Clause =>
                  Rules.Local_Hiding.       Process_With_Clause (Element);
                  Rules.Other_Dependencies. Process_With_Clause (Element);
                  Rules.With_Clauses.       Process_With_Clause (Element);
               when others =>
                  null;
            end case;

         when A_Defining_Name =>
            case Defining_Name_Kind (Element) is
               when A_Defining_Identifier =>
                  Rules.Local_Hiding.      Process_Defining_Name (Element);
                  Rules.Naming_Convention. Process_Defining_Name (Element);
                  Rules.Reduceable_Scope.  Process_Defining_Name (Element);
                  Rules.Style.             Process_Identifier    (Element);

               when A_Defining_Enumeration_Literal =>
                  Rules.Local_Hiding.      Process_Defining_Name (Element);
                  Rules.Naming_Convention. Process_Defining_Name (Element);
                  Rules.Style.             Process_Identifier    (Element);

               when others =>
                  null;
            end case;

         when A_Declaration =>
            Rules.Declarations.          Process_Declaration (Element);
            Rules.Exception_Propagation. Process_Declaration (Element);
            Rules.Style.                 Process_Element     (Element);

            case Declaration_Kind (Element) is
               when A_Variable_Declaration =>
                  Rules.Directly_Accessed_Globals. Process_Variable_Declaration (Element);
                  Rules.Usage.                     Process_Entity_Declaration   (Element);

               when A_Constant_Declaration
                 | A_Deferred_Constant_Declaration
                 | A_Number_Declaration
                 =>
                  Rules.Usage. Process_Entity_Declaration (Element);

               when A_Function_Body_Declaration =>
                  Rules.Abnormal_Function_Return. Process_Function_Body        (Element);
                  Rules.Exception_Propagation.    Process_SP_Declaration       (Element);
                  Rules.Global_References.        Process_Body                 (Element);
                  Rules.Max_Parameters.           Process_Declaration          (Element);
                  Rules.No_Safe_Initialization.   Process_Structure            (Element);
                  Rules.Return_Type.              Process_Function_Declaration (Element);
                  Rules.Statements.               Process_Function_Body        (Element);
                  Rules.Style.                    Process_Construct            (Element);
                  Rules.Style.                    Process_Declaration          (Element);

               when A_Procedure_Body_Declaration =>
                  Rules.Exception_Propagation.  Process_SP_Declaration (Element);
                  Rules.Global_References.      Process_Body           (Element);
                  Rules.Max_Parameters.         Process_Declaration    (Element);
                  Rules.No_Safe_Initialization. Process_Structure      (Element);
                  Rules.Style.                  Process_Construct      (Element);
                  Rules.Style.                  Process_Declaration    (Element);

               when An_Entry_Body_Declaration =>
                  Rules.Barrier_Expressions.    Process_Entry_Declaration (Element);
                  Rules.Global_References.      Process_Body              (Element);
                  Rules.No_Safe_Initialization. Process_Structure         (Element);
                  Rules.Style.                  Process_Construct         (Element);
                  Rules.Style.                  Process_Declaration       (Element);

               when A_Package_Declaration
                 | A_Task_Type_Declaration
                 | A_Protected_Type_Declaration
                 =>
                  Rules.Style. Process_Construct (Element);

               when A_Package_Body_Declaration =>
                  Rules.No_Safe_Initialization. Process_Structure (Element);
                  Rules.Style.                  Process_Construct (Element);

               when A_Generic_Package_Declaration =>
                  Rules.Style. Process_Construct   (Element);
                  Rules.Style. Process_Declaration (Element);

               when A_Protected_Body_Declaration =>
                 Rules.Style.                           Process_Construct      (Element);
                 Rules.Potentially_Blocking_Operations. Process_Protected_Body (Element);

               when A_Single_Task_Declaration
                 | A_Single_Protected_Declaration
                 =>
                  Rules.Style. Process_Construct          (Element);
                  Rules.Usage. Process_Entity_Declaration (Element);

               when A_Task_Body_Declaration
                 =>
                  Rules.Exception_Propagation.  Process_Task_Body (Element);
                  Rules.Global_References.      Process_Body      (Element);
                  Rules.No_Safe_Initialization. Process_Structure (Element);
                  Rules.Style.                  Process_Construct (Element);
                  Rules.Terminating_Tasks.      Process_Task_Body (Element);

               when An_Exception_Declaration =>
                  Rules.Usage. Process_Entity_Declaration (Element);

               when A_Package_Instantiation =>
                  Rules.Default_Parameter.      Process_Call_Or_Instantiation (Element);
                  Rules.Exception_Propagation.  Process_Instantiation         (Element);
                  Rules.Instantiations.         Process_Instantiation         (Element);
                  Rules.Local_Instantiation.    Process_Instantiation         (Element);
                  Rules.Non_Static.             Process_Instantiation         (Element);
                  Rules.Return_Type.            Process_Instantiation         (Element);
                  Rules.Side_Effect_Parameters. Process_Call_Or_Instantiation (Element);
                  Rules.Unnecessary_Use.        Process_Instantiation         (Element);
                  Rules.Usage.                  Process_Package_Instantiation (Element);

               when A_Formal_Package_Declaration =>
                  Rules.Default_Parameter.      Process_Call_Or_Instantiation (Element);
                  Rules.Side_Effect_Parameters. Process_Call_Or_Instantiation (Element);
                  Rules.Unnecessary_Use.        Process_Instantiation         (Element);

               when A_Procedure_Instantiation =>
                  Rules.Default_Parameter.      Process_Call_Or_Instantiation (Element);
                  Rules.Exception_Propagation.  Process_Instantiation         (Element);
                  Rules.Instantiations.         Process_Instantiation         (Element);
                  Rules.Local_Instantiation.    Process_Instantiation         (Element);
                  Rules.Max_Parameters.         Process_Declaration           (Element);
                  Rules.Non_Static.             Process_Instantiation         (Element);
                  Rules.Return_Type.            Process_Instantiation         (Element);
                  Rules.Side_Effect_Parameters. Process_Call_Or_Instantiation (Element);
                  Rules.Unnecessary_Use.        Process_Instantiation         (Element);

               when A_Function_Instantiation =>
                  Rules.Default_Parameter.           Process_Call_Or_Instantiation (Element);
                  Rules.Exception_Propagation.       Process_Instantiation         (Element);
                  Rules.Instantiations.              Process_Instantiation         (Element);
                  Rules.Local_Instantiation.         Process_Instantiation         (Element);
                  Rules.Max_Parameters.              Process_Declaration           (Element);
                  Rules.Non_Static.                  Process_Instantiation         (Element);
                  Rules.Return_Type.                 Process_Instantiation         (Element);
                  Rules.Side_Effect_Parameters.      Process_Call_Or_Instantiation (Element);
                  Rules.Unnecessary_Use.             Process_Instantiation         (Element);
                  Rules.Unsafe_Unchecked_Conversion. Process_Instantiation         (Element);

               when A_Procedure_Declaration
                 | A_Procedure_Body_Stub
                 | A_Generic_Procedure_Declaration
                 | An_Entry_Declaration
                 =>
                  Rules.Max_Parameters. Process_Declaration (Element);
                  Rules.Style.          Process_Declaration (Element);

               when A_Procedure_Renaming_Declaration =>
                  Rules.Style. Process_Declaration (Element);

               when A_Function_Renaming_Declaration =>
                  Rules.Return_Type. Process_Function_Declaration (Element);
                  Rules.Style.       Process_Declaration          (Element);

               when A_Formal_Procedure_Declaration   =>
                  Rules.Style. Process_Declaration (Element);

               when A_Function_Declaration
                  | A_Function_Body_Stub
                 | A_Generic_Function_Declaration
                 =>
                  Rules.Max_Parameters. Process_Declaration          (Element);
                  Rules.Return_Type.    Process_Function_Declaration (Element);
                  Rules.Style.          Process_Declaration          (Element);

               when A_Formal_Function_Declaration =>
                  Rules.Return_Type. Process_Function_Declaration (Element);
                  Rules.Style.       Process_Declaration          (Element);

               when others =>
                 null;
            end case;

         when A_Definition =>
            case Definition_Kind (Element) is
               when A_Type_Definition =>
                  case Type_Kind (Element) is
                     when A_Constrained_Array_Definition =>
                        Rules.Array_Declarations. Process_Array_Definition             (Element);
                        Rules.Non_Static.         Process_Constrained_Array_Definition (Element);
                     when An_Unconstrained_Array_Definition =>
                        Rules.Array_Declarations. Process_Array_Definition (Element);
                     when others =>
                        null;
                  end case;

               when A_Constraint =>
                  case Constraint_Kind (Element) is
                     when A_Discriminant_Constraint =>
                        Rules.Non_Static. Process_Discriminant_Constraint (Element);
                     when An_Index_Constraint =>
                        Rules.Array_Declarations. Process_Array_Definition (Element);
                        Rules.Non_Static.         Process_Index_Constraint (Element);
                     when others =>
                        null;
                  end case;

               when An_Others_Choice =>
                  Rules.Statements.Process_Others (Element);

               when A_Discrete_Range
                 | A_Discrete_Subtype_Definition
                 =>
                  Rules.Simplifiable_Expressions.Process_Range (Element);

               when others =>
                  null;
            end case;

         when An_Exception_Handler =>
            Rules.Entity_Inside_Exception. Process_Exception_Handler (Element);
            Rules.Silent_Exceptions.       Process_Exception_Handler (Element);

         when A_Statement =>
            Rules.Style.                 Process_Element   (Element);
            Rules.Max_Statement_Nesting. Process_Statement (Element);
            Rules.Statements.            Process_Statement (Element);

            case Statement_Kind (Element) is
               when A_Procedure_Call_Statement
                 | An_Entry_Call_Statement
                    =>
                  Rules.Max_Call_Depth.          Process_Call                  (Element);
                  Rules.Default_Parameter.       Process_Call_Or_Instantiation (Element);
                  Rules.Exception_Propagation.   Process_Call                  (Element);
                  Rules.Insufficient_Parameters. Process_Call                  (Element);
                  Rules.Not_Elaboration_Calls.   Process_Call                  (Element);
                  Rules.Parameter_Aliasing.      Process_Call                  (Element);
                  Rules.Side_Effect_Parameters.  Process_Call_Or_Instantiation (Element);
                  Rules.Unsafe_Paired_Calls.     Process_Call                  (Element);

               when An_If_Statement =>
                  Rules.If_For_Case. Process_If_Statement       (Element);
                  Rules.Style.       Process_If_Statement       (Element);
                  Rules.Style.       Process_Compound_Statement (Element);

               when A_Case_Statement =>
                  Rules.Case_Statement. Process_Case_Statement     (Element);
                  Rules.Style.          Process_Compound_Statement (Element);

               when An_Accept_Statement =>
                  Rules.Declarations.             Process_Statement          (Element);
                  Rules.Movable_Accept_Statements.Process_Accept_Statement   (Element);
                  Rules.No_Safe_Initialization.   Process_Structure          (Element);
                  Rules.Style.                    Process_Compound_Statement (Element);

               when A_Block_Statement =>
                  Rules.Declarations.           Process_Statement          (Element);
                  Rules.No_Safe_Initialization. Process_Structure          (Element);
                  Rules.Style.                  Process_Compound_Statement (Element);

               when A_Loop_Statement
                 | A_While_Loop_Statement
                 | A_For_Loop_Statement
                 =>
                  Rules.Statements. Pre_Process_Loop           (Element);
                  Rules.Style.      Process_Compound_Statement (Element);

               when A_Selective_Accept_Statement
                 | A_Timed_Entry_Call_Statement
                 | A_Conditional_Entry_Call_Statement
                 | An_Asynchronous_Select_Statement
                 =>
                  Rules.Style. Process_Compound_Statement (Element);

               when others =>
                  null;
            end case;

         when A_Path =>
            case Path_Kind (Element) is
               when A_Case_Path =>
                  Rules.Case_Statement. Process_Path (Element);
               when others =>
                  null;
            end case;


         when An_Expression =>
            Rules.Expressions. Process_Expression (Element);

            case Expression_Kind (Element) is
               when An_Attribute_Reference =>
                  Rules.Entities. Process_Identifier (Element);
                  Rules.Style.    Process_Attribute  (Element);

               when An_Allocation_From_Subtype
                 | An_Allocation_From_Qualified_Expression
                 =>
                  Rules.Allocators.Process_Allocator (Element);

                when A_Function_Call =>
                  Rules.Default_Parameter.        Process_Call_Or_Instantiation (Element);
                  Rules.Exception_Propagation.    Process_Call                  (Element);
                  Rules.Insufficient_Parameters.  Process_Call                  (Element);
                  Rules.Max_Call_Depth.           Process_Call                  (Element);
                  Rules.Not_Elaboration_Calls.    Process_Call                  (Element);
                  Rules.Side_Effect_Parameters.   Process_Call_Or_Instantiation (Element);
                  Rules.Simplifiable_Expressions. Process_Call                  (Element);
                  Rules.Statements.               Process_Function_Call         (Element);

               when A_Parenthesized_Expression =>
                  Rules.Simplifiable_Expressions. Process_Parenthesized (Element);

               when An_Integer_Literal
                  | A_Real_Literal
                  | A_Character_Literal
                  | A_String_Literal
                 =>
                  Rules.Style. Process_Literal (Element);

               when others =>
                  null;
            end case;

         when An_Association =>
            Rules.Style. Process_Association (Element);

         when A_Pragma =>
            Rules.Pragmas.Process_Pragma (Element);
            Rules.Style.  Process_Pragma (Element);

         when others =>
            null;
      end case;
   end Pre_Procedure;

   --------------------
   -- Post_Procedure --
   --------------------

   procedure Post_Procedure (Element : in Asis.Element) is
      use Asis, Asis.Elements;
   begin
      case Element_Kind (Element) is
         when A_Clause =>
            case Clause_Kind (Element) is
               when A_Use_Package_Clause =>
                  Rules.Unnecessary_Use. Process_Use_Clause (Element);
               when others =>
                  null;
            end case;

         when A_Declaration =>
            case Declaration_Kind (Element) is
               when A_Package_Body_Declaration =>
                  Rules.Directly_Accessed_Globals. Post_Process_Package_Body (Element);
               when A_Renaming_Declaration =>
                  Rules.Style. Process_Renaming (Element);
               when others =>
                  null;
            end case;

         when A_Statement =>
            case Statement_Kind (Element) is
               when A_Loop_Statement
                 | A_While_Loop_Statement
                 | A_For_Loop_Statement
                 =>
                  Rules.Statements. Post_Process_Loop (Element);
               when others =>
                  null;
            end case;

        when others =>
            null;
      end case;
   end Post_Procedure;

   ---------------------
   -- Text_Enter_Unit --
   ---------------------

   procedure Text_Enter_Unit (Unit : in Asis.Compilation_Unit) is
      pragma Unreferenced (Unit);
   begin
      Rules.Header_Comments. Enter_Unit;
   end Text_Enter_Unit;

   -------------------
   -- Text_Analysis --
   -------------------

   procedure Text_Analysis (Line : Asis.Program_Text; Loc : Location) is
   begin
      Rules.Control_Characters. Process_Line (Line, Loc);
      Rules.Header_Comments.    Process_Line (Line, Loc);
      Rules.Max_Blank_Lines.    Process_Line (Line, Loc);
      Rules.Max_Line_Length.    Process_Line (Line, Loc);
      Rules.Special_Comments.   Process_Line (Line, Loc);
   end Text_Analysis;

end Framework.Plugs;
