// ---------------------------------------------------------------------------
// - XmlRoot.cpp                                                             -
// - afnix:xml module - xml root node class implementation                   -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "XmlRoot.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default root node

  XmlRoot::XmlRoot (void) {}

  // return the class name

  String XmlRoot::repr (void) const {
    return "XmlRoot";
  }

  // get a clone of this node

  Object* XmlRoot::clone (void) const {
    rdlock ();
    try {
      // create a new node
      XmlRoot* result = new XmlRoot;
      // update the node info
      result->d_eflg = d_eflg;
      result->d_lnum = d_lnum;
      result->d_snam = d_snam;
      // here it is
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the declaration node - if any

  XmlDecl* XmlRoot::getdecl (void) const {
    rdlock ();
    try {
      // check if we have some nodes
      long len = lenchild ();
      // get the first node
      XmlNode* node = (len == 0) ? nilp : getchild (0);
      if (node == nilp) {
	unlock ();
	return nilp;
      }
      // normaly the first node is the declaration node
      XmlDecl* result = dynamic_cast <XmlDecl*> (node);
      if (result == nilp) {
	throw Exception ("xml-error", "first node must be a declaration node",
			 Object::repr (node));
      }
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // remove the declaration node if any

  void XmlRoot::deldecl (void) {
    wrlock ();
    try {
      // check if we have some nodes
      long len = lenchild ();
      // get the first node
      XmlNode* node = (len == 0) ? nilp : getchild (0);
      if (node == nilp) {
	unlock ();
	return;
      }
      // normaly the first node is the declaration node
      XmlDecl* result = dynamic_cast <XmlDecl*> (node);
      if (result == nilp) {
	unlock ();
	return;
      }
      // remove the first node
      delchild (0);
      unlock ();
      return;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a node to a buffer

  void XmlRoot::write (Buffer& buf) const {
    rdlock ();
    try {
      long len = lenchild ();
      for (long i = 0; i < len; i++) {
	XmlNode* node = getchild (i);
	if (node == nilp) continue;
	node->write (buf);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // write a node to an output stream

  void XmlRoot::write (Output& os) const {
    rdlock ();
    try {
      long len = lenchild ();
      for (long i = 0; i < len; i++) {
	XmlNode* node = getchild (i);
	if (node == nilp) continue;
	node->write (os);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 2;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GETDECL = zone.intern ("get-declaration");
  static const long QUARK_DELDECL = zone.intern ("del-declaration");

  // create a new object in a generic way

  Object* XmlRoot::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // create a default root node
    if (argc == 0) return new XmlRoot;
    throw Exception ("argument-error",
                     "too many argument with xml root node constructor");
  }

  // return true if the given quark is defined

  bool XmlRoot::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? XmlNode::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }

  // apply this object with a set of arguments and a quark
  
  Object* XmlRoot::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETDECL) {
	rdlock ();
	try {
	  XmlDecl* result = getdecl ();
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_DELDECL) {
	deldecl ();
	return nilp;
      }
    }
    // check the xml node object
    return XmlNode::apply (robj, nset, quark, argv);
  }
}
