/*  AO40TLMVIEW - decoder and viewer for AO-40 telemetry
 *
 *  Copyright (C) 2001-2002, Pieter-Tjerk de Boer -- pa3fwm@amsat.org
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of version 2 of the GNU General Public License as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <ncurses.h>
#include <math.h>
#include <strings.h>
#include <ctype.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <sys/ioctl.h>
#include <termios.h>

#define VERSION "1.03"

typedef unsigned char byte;

/*--------------------------------------------------------------------------------------------------------------------------*/

#define MKFN(NAME,FN) \
  double fn##NAME(int i) { double X=i; return FN; }
MKFN(RAW,  X)
MKFN(temp, 0.659 * X -69.7)
MKFN(100,   (X>101) ? pow(X/151.1985708,-4.999357462) : 46.4720 - 0.38452*X )
MKFN(101,   0.0815 * X - 1.253  )
MKFN(102,   0.0835 * X - 1.381  )
MKFN(103,   0.0503 * X - 0.3154  )
MKFN(104,   1.221 * X - 263.0537  )
MKFN(106,   0.2410 * X - 31.28  )
MKFN(107,   0.2035 * X - 2.85  )
MKFN(108,   0.197 * X -0.739  )
MKFN(109,   0.0412 * X -0.76  )
MKFN(10A,   0.1024 * X -0.653  )
MKFN(10B,   0.1548* X -1.484  )
MKFN(10E,   0.1522* X -1.06  )
MKFN(10F,   0.1318* X -0.923  )
MKFN(110,   0.0657* X - 0.712  )
MKFN(114,   2.3406* X - 197.1  )
MKFN(115,   0.1235* X -1.235  )
MKFN(116,   0.154* X -10.6  )
MKFN(119,   0.103*X -0.95  )
MKFN(11B,   -0.011*X*X + 3.66 *X - 284  )
MKFN(11D,   -0.004*X*X +1.25*X - 72  )
MKFN(11E,   0.254* X - 14.8  )
MKFN(11F,   0.457* X - 31.9  )
MKFN(120,   0.129* X - 7.9  )
MKFN(12B,   -0.413*X +103.8  )
MKFN(12E,   -31.501 + (0.3682*X) + (-0.001539 * X*X) + (0.00000361 * X*X*X)  )
MKFN(130,   acos(X / 255)*180/M_PI  )
MKFN(134,   -55.179 + (0.64187*X) + (-0.002447 * X*X) + (0.00000581 * X*X*X)  )
MKFN(171,   0.1014 * X -0.6212)
MKFN(175,   0.0125 * X -0.0875)
MKFN(17A,   0.0429 * X - 0.333)

MKFN(18C,   0.04*(X+256*(X<64)) + 17.76)
MKFN(18D,   0.10*(X+256*(X<128)) - 5.6)
MKFN(185,   10.337 - 0.0366*X)

int orb_hi, orb_lo;

double fn197(int i)
{
   int j;
   j=i+256*orb_hi;
   if (i>orb_lo) j-=256;
   return j;
}


double (*(tlmfns[]))(int) = {
   fn100, fn101, fn102, fn103, fn104, fnRAW, fn106, fn107, fn108, fn109, fn10A, fn10B, fn10B, fn10B, fn10E, fn10F,
   fn110, fn10F, fn110, fnRAW, fn114, fn115, fn116, fnRAW, fnRAW, fn119, fnRAW, fn11B, fnRAW, fn11D, fn11E, fn11F,
   fn120, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fn12B, fn12B, fnRAW, fn12E, fn12E,
   fn130, fn130, fn130, fn130, fn134, fn134, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW,
   fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp,
   fntemp, fntemp, fntemp, fntemp, fntemp, fnRAW, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fnRAW, fntemp, fntemp,
   fnRAW, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp, fntemp,
   fnRAW, fn171, fn171, fn171, fn171, fn175, fn171, fn171, fn175, fnRAW, fn17A, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 

   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fn185, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fn18C, fn18D, fn18D, fn18D, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
   fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, fnRAW, 
};

/*--------------------------------------------------------------------------------------------------------------------------*/
/* generic code related to drawing the telemetry windows */

int at_border;         /* attributes for window borders (incl. title) */
int at_helpborder;     /* attributes for border around help window */
int at_subborder;      /* attributes for borders of subwindows */
int at_on;             /* attributes for things marked switched ON */
int at_off;            /* for things switched OFF */
int at_broken;         /* for post-incident broken sensors */
int at_val;            /* telemetry values */
int at_lab;            /* labels, channel names, etc. */
int at_wrongcrc;       /* for 'raw' window if CRC wrong */
int at_matrix[]={A_BOLD, A_BOLD, A_BOLD, A_BOLD, A_BOLD, A_BOLD, A_BOLD};   /* array of 7 colours for active connections in the matrix */

int printraw=0;        /* flag, for showing raw telemetry values instead of in engineering units */
int printchan=0;       /* flag, for showing telemetry channel numbers instead of in engineering units */

int decode_A=1;         /* flag: decode A blocks? */
int decode_E=1;         /* flag: decode E blocks? */
int decode_wrongCRC=0;  /* flag: decode blocks with incorrect CRC? */


WINDOW *mainpad=NULL;

/* The below structure describes the display of one "measurement", e.g. a voltage or a temperature.
   An array of these describes a whole subwindow of the display, and the functions printlabels()
   and printvals() actually fill the subwindow.
   This method is used for most of the telemetry; however, some telemetry information does not fit
   this standard format (e.g. ON/OFF settings), and is therefore handled by specific code in the
   write_w*() functions.
*/
typedef struct {
   char label[12];         /* name of the quantity to be measured, i.e., a label in the window */
   int xpos;               /* horizontal position of the value as printed according to the below format */
                           /* note: vertical position is given by the position in the array of Measurment structs */
   int chan;               /* index in the byte array of a received telemetry block */
   double (*fn)(int);      /* pointer to function that converts the byte to engineering units */
   char format[12];        /* format (printf-style) for printing the value */
   int how;                /* extra information about how to print the value, defined below */
   int sameline;           /* flag, !=0 means next 'Measure' must be printed on same line */
} Measure;

enum { 
   how_ok,         /* simply print the value */
   how_broken,     /* channel non-functional since the December-2000 incident: print with attributes at_broken */
   how_onoff1,     /* only print if the flag onoff1 is nonzero, print a '-' otherwise; used for sensors which are not always on */
   how_onoff2,     /* same as how_onoff1, but using the flag onoff2 */
   how_none,       /* don't print a value, so print only the label; used for cases that don't fit the standard format */
   how_end         /* indicates the last element in the array */
};

void printlabels(WINDOW *w, Measure m[])
{
   int y;
   wattrset(w,at_lab);
   y=1;
   while (m->how!=how_end) {
      mvwaddstr(w,y,0,m->label);
      if (!m->sameline) y++;
      m++;
   }
}

void printvals(WINDOW *w, Measure m[], byte b[], int onoff1, int onoff2)
{
   int y;
   y=1;
   while (m->how!=how_end) {
      char s[20];
      if (m->how!=how_none) {
         int l;
         if (m->how==how_broken) wattrset(w,at_broken); else wattrset(w,at_val);
         l=sprintf(s,m->format,m->fn(b[m->chan]));
         if (printraw) sprintf(s,"%*i",l,b[m->chan]);
         else if (printchan) { wattrset(w,at_lab); sprintf(s,"%*X",l,m->chan); }
         else if ((m->how==how_onoff1 && !onoff1) || (m->how==how_onoff2 && !onoff2)) {
            int i;
            for (i=0;i<l;i++) s[i]= (i==(1+l)/2)?'-':' ';
         }
         mvwaddstr(w,y,m->xpos,s);
      }
      if (!m->sameline) y++;
      m++;
   }
   wattrset(w,at_val);
}



WINDOW *createwin(int height,int width,int y,int x,char *title)
{
   WINDOW *w;
   int i;

   w=subpad(mainpad,height,width,y,x);
   wattrset(w,at_border);
   box(w,ACS_VLINE,ACS_HLINE);
   i=(width-2-strlen(title))/2;
   mvwaddch(w,0,i,ACS_RTEE);
   waddstr(w,title);
   waddch(w,ACS_LTEE);
   return w;
}

WINDOW *createsubwin(WINDOW *parent,int height,int width,int y,int x,int hasrightborder,char *title)
{
   WINDOW *w;
   int i;
   w=subpad(parent,height,width+hasrightborder,y,x);
   wattrset(w,at_subborder);
   if (hasrightborder) {
      mvwhline(w,0,0,ACS_HLINE,width);
      mvwaddch(w,0,width,ACS_URCORNER);
      mvwvline(w,1,width,ACS_VLINE,height-1);
   } else {
      mvwhline(w,0,0,ACS_HLINE,width);
   }
   i=(width-strlen(title))/2;
   mvwaddstr(w,0,i,title);
   return w;
}

WINDOW *createmsubwin(WINDOW *parent,int height,int width,int y,int x,int hasrightborder,char *title,Measure m[])
{
   WINDOW *w;
   w=createsubwin(parent,height,width,y,x,hasrightborder,title);
   printlabels(w,m);
   return w;
}



void mvwaprintf(WINDOW *w,int y,int x,int attr1,int attr2,char *format, ...)
{
   char s[100];
   va_list ap;
   char *p;

   va_start(ap,format);
   wmove(w,y,x);
   vsprintf(s,format,ap);
   p=s;
   while (*p!=0) {
      if (*p!='@') waddch(w,*p);
      else switch (p[1]) {
         case '1': wattrset(w,attr1); p++; break;
         case '2': wattrset(w,attr2); p++; break;
      }
      p++;
   }
}

void wclearat(WINDOW *w,int a)
/* clear window using attribute a */
/* we seem to need this to get a black background on an xterm which by itself has a white background */
{
   int i;
   int x,y;
   getmaxyx(w,y,x);
   i=(x+1)*(y+1);
   wattrset(w,a);
   move(0,0);
   while (i>128) {
      waddstr(w,"                                                                                                                                ");
      i-=128;
   }
   addstr(128-i+"                                                                                                                                ");
}


int ofsx=0;
int ofsy=0;
int xsize=118;
int ysize=37;
int vxsize=118;
int vysize=37;
int layout=0;      /* 0: "wide" layout (118x37); 1: "tall" layout (80x60) */

int hexvisible=0;
WINDOW *hexpad=NULL;
int hexpadxsize,hexpadysize;

WINDOW *helppad=NULL;
int helpvisible=0;
int helpheight,helpwidth;

void do_prefresh(void)
{
   pnoutrefresh(mainpad,ofsy,ofsx,0,0,vysize-1,vxsize-1);
   if (hexvisible) {
      /* sigh, what a mess to calculate the right prefresh() params to make the hex
         window move together with the rest when pressing TAB; is there really no
         way to implement the hex window as a sub-pad of mainpad...? */
      int yw,xw;
      int ys,xs;
      int ym,xm;
      xw=0; yw=0;
      xs=1; ys=1;
      xs-=ofsx;
      if (xs<0) { xw-=xs; xm+=xs; xs=0; }
      ys-=ofsy;
      if (ys<0) { yw-=ys; ym+=ys; ys=0; }
      xm=vxsize; ym=vysize; 
      if (xm-xs+xw>hexpadxsize) xm=hexpadxsize-xw+xs;
      if (ym-ys+yw>hexpadysize) ym=hexpadysize-yw+ys;
      pnoutrefresh(hexpad,yw,xw,ys,xs,ym-1,xm-1);
   }
   if (helpvisible) {
      int x0,y0,x1,y1;
      x0=(vxsize-helpwidth-2)/2; if (x0<0) x0=0;
      x1=x0+1+helpwidth; if (x1>=COLS) x1=COLS-1;
      y0=(vysize-helpheight-2)/2; if (y0<0) y0=0;
      y1=y0+1+helpheight; if (y1>=LINES) y1=LINES-1;

      prefresh(helppad,0,0, y0,x0,y1,x1);
   }
   doupdate();
}

void setview(int view)
{
   if (view==1) {   /* TAB pressed */
      if (layout==0) {
         if (ofsx==0) ofsx=xsize-COLS;
         else ofsx=0;
      } else {
         if (ofsy==0) {
            if (ysize<=2*LINES) ofsy=ysize-LINES;
            else ofsy=18;
         } else if (ofsy==18 && ysize>2*LINES) {
            ofsy=ysize-LINES;
         }
         else ofsy=0;
      }
   } else if (view==2) {    /* move filename into view */
      if (layout==0) {
         if (xsize>COLS) ofsx=xsize-COLS;
      } else {
         ofsy=0;
      }
   }
}

/*--------------------------------------------------------------------------------------------------------------------------*/
/* code for each of the telemetry windows */

WINDOW *wraw,*wmatrix,*wtemp,*wnav,*wpower,*wstatus;
WINDOW *wtemptx,*wtemprx,*wtempmodules,*wtempframe,*wtemppipes,*wtempbatreg,*wtemptanks,*wtempsolar,*wtempbatt;
WINDOW *wnavmagnet,*wnavarcjet,*wnav400N,*wnaves,*wnavspin,*wnav3ax,*wnavwheels,*wnavaru;
WINDOW *wpowervolts,*wpoweramps,*wpowerbcr,*wpowertx,*wpowerstat;
WINDOW *wstatusinner;
WINDOW *wprogstat;


Measure mtemptx[] = {
   { "V",    5, 0x15e, fntemp, "%5.1f", how_ok },
   { "U",    5, 0x158, fntemp, "%5.1f", how_ok },
   { "U-PA", 5, 0x15f, fntemp, "%5.1f", how_ok },
   { "S1",   5, 0x15b, fntemp, "%5.1f", how_ok },
   { "S2",   5, 0x15c, fntemp, "%5.1f", how_ok },
   { "X",    5, 0x12b, fn12B,  "%5.1f", how_onoff1 },
   { "TWT",  5, 0x12c, fn12B,  "%5.1f", how_onoff1 },
   { "",     0, 0,     NULL,   "",      how_end },
};

Measure mtemprx[] = {
   { "U/V",  5, 0x159, fntemp, "%5.1f", how_ok },
   { "L1",   5, 0x15a, fntemp, "%5.1f", how_ok },
   { "L2",   5, 0x152, fntemp, "%5.1f", how_ok },
   { "S1/H", 5, 0x157, fntemp, "%5.1f", how_ok },
   { "S2/C", 5, 0x156, fntemp, "%5.1f", how_ok },
   { "",     0, 0,     NULL,   "",      how_end },
};

Measure mtempmodules[] = {
   { "IHU",  5, 0x161, fntemp, "%5.1f", how_ok },
   { "SEU",  5, 0x140, fntemp, "%5.1f", how_ok },
   { "EPU",  5, 0x141, fntemp, "%5.1f", how_ok },
   { "",     0, 0,     NULL,   "",      how_end },
};

Measure mtempframe[] = {
   { "Top",    6, 0x162, fntemp, "%5.1f", how_broken },
   { "Bottom", 6, 0x163, fntemp, "%5.1f", how_broken },
   { "Back",   6, 0x164, fntemp, "%5.1f", how_broken },
   { "Side2",  6, 0x16d, fntemp, "%5.1f", how_ok },
   { "Side4",  6, 0x165, fntemp, "%5.1f", how_ok },
   { "S ant",  6, 0x16e, fntemp, "%5.1f", how_ok },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mtemppipes[] = {
   { "1+X-Y",  6, 0x169, fntemp, "%5.1f", how_ok },
   { "2+X+Y",  6, 0x168, fntemp, "%5.1f", how_ok },
   { "2+X-Y",  6, 0x153, fntemp, "%5.1f", how_ok },
   { "2-X",    6, 0x154, fntemp, "%5.1f", how_broken },
   { "3+X",    6, 0x16a, fntemp, "%5.1f", how_ok },
   { "3-X",    6, 0x167, fntemp, "%5.1f", how_ok },
   { "4+X+Y",  6, 0x166, fntemp, "%5.1f", how_ok },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mtempbatreg[] = {
   { "BCR1",   6, 0x142, fntemp, "%5.1f", how_ok },
   { "BCR2",   6, 0x144, fntemp, "%5.1f", how_ok },
   { "BCR3",   6, 0x143, fntemp, "%5.1f", how_ok },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mtemptanks[] = {
   { "MMH-3",  6, 0x145, fntemp, "%5.1f", how_ok },
   { "N2O4+",  6, 0x16c, fntemp, "%5.1f", how_ok },
   { "N2O4-",  6, 0x16b, fntemp, "%5.1f", how_broken },
   { "NH3-2",  6, 0x148, fntemp, "%5.1f", how_ok },
   { "He",     6, 0x16f, fntemp, "%5.1f", how_broken },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mtempsolar[] = {
   { "Pnl1",   6, 0x14c, fntemp, "%5.1f", how_ok },
   { "Pnl2",   6, 0x14d, fntemp, "%5.1f", how_ok },
   { "Pnl3",   6, 0x14e, fntemp, "%5.1f", how_ok },
   { "Pnl4",   6, 0x14f, fntemp, "%5.1f", how_ok },
   { "Pnl5",   6, 0x150, fntemp, "%5.1f", how_ok },
   { "Pnl6",   6, 0x151, fntemp, "%5.1f", how_ok },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mtempbatt[] = {
   { "Main2",   6, 0x149, fntemp, "%5.1f", how_ok },
   { "Main4",   6, 0x14a, fntemp, "%5.1f", how_broken },
   { "Main6",   6, 0x14b, fntemp, "%5.1f", how_broken },
   { "Aux1",    6, 0x147, fntemp, "%5.1f", how_ok },
   { "Aux5",    6, 0x146, fntemp, "%5.1f", how_ok },
   { "",        0, 0,     NULL,   "",      how_end },
};


void write_wtemp(byte *b)
{
   printvals(wtemptx,mtemptx,b,(b[0x1c8]&0x20),0);
   printvals(wtemprx,mtemprx,b,0,0);
   printvals(wtempmodules,mtempmodules,b,0,0);
   printvals(wtempframe,mtempframe,b,0,0);
   printvals(wtemppipes,mtemppipes,b,0,0);
   printvals(wtempbatreg,mtempbatreg,b,0,0);
   printvals(wtemptanks,mtemptanks,b,0,0);
   printvals(wtempsolar,mtempsolar,b,0,0);
   printvals(wtempbatt,mtempbatt,b,0,0);
}


void write_wraw(byte *b,int crcok,int len)
{
   int i,j;
   WINDOW *w=wraw;
   if (!crcok) wattrset(wraw,at_wrongcrc);
   else wattrset(wraw,at_val);
   for (j=0;j<8;j++) {
      wmove(w,j+1,1);
      for (i=0;i<64;i++) {
         byte c;
         int inv;
         if (--len>=0) {
            c=*b++;
            inv = (c & 0x80);
            c &= 0x7f;
            if (c<32 || c==0x7f) c='.';
            if (inv) wattron(w,A_REVERSE);
            waddch(w,c);
            if (inv) wattroff(w,A_REVERSE);
         } else {
            waddch(w,' ');
         }
      }
   }
}


void write_wmatrix(byte *b)
{
   int i,j;
   WINDOW *w=wmatrix;
   int rxctrl=b[0x1ca];
   int txctrl=b[0x1c8]+(b[0x1c9]<<8);
   int antctrl=b[0x189];
   int beaconctrl=b[0x194];
   int x;
   char s[10];

   /* --------- "sources" (receivers, beacons, etc.) ------------------------------------- */

   wattrset(w,at_lab);
   mvwaddstr(w,1,1,"RX:   AGC:");

#define agcprnt(y,chan,fn,on)  \
   if (printchan) { wattrset(w,at_lab); mvwprintw(w,y,6,"  %3X",chan); }  \
   else if (printraw) { wattrset(w,at_val); mvwprintw(w,y,6,"  %3i",b[chan]); }  \
   else if (!(on)) { wattrset(w,at_val); mvwaddstr(w,y,6,"     "); }  \
   else { wattrset(w,at_val);  (fn==NULL) ? mvwprintw(w,y,6,"%5i",b[chan]) : mvwprintw(w,y,6,"%5.1f",((double (*)(int))(fn))(b[chan]));      }

#define rxprnt(y,name,on, chan,fn)  \
   wattrset(w,on?at_on:at_lab); mvwaddstr(w,y,1,name); agcprnt(y,chan,fn,on);

#define rx2prnt(y,name1,on1,name2,on2, chan,fn)  \
   wattrset(w,(on1)?at_on:at_lab); mvwaddstr(w,y,1,name1);  \
   wattrset(w,at_lab); waddch(w,'/');                     \
   if (on2) wattrset(w,at_on); waddstr(w,name2);          \
   agcprnt(y,chan,fn,(on1)|(on2));


   rx2prnt(2, "21",rxctrl&0x01, "24",rxctrl&0x02, 0x11c, NULL);
   rxprnt(3,(antctrl&1)?"V o  ":"V    ", rxctrl&0x04  ,0x11e,fn11E);
   rxprnt(4,(antctrl&2)?"U o  ":"U    ",!(rxctrl&0x04),0x11f,fn11F);
   switch (antctrl&0x0c) {
      case 0x00: strcpy(s,"L1bth"); break;
      case 0x04: strcpy(s,"L1 o "); break;
      case 0x08: strcpy(s,"L1   "); break;
      case 0x0c: strcpy(s,"L1?? "); break;
   }
   rxprnt(5,s,1,0x120,fn120);
   s[1]='2';
   rxprnt(6,s,rxctrl&0x20,0x116,fn116);
   rxprnt(7,"S1   ",rxctrl&0x40,0x11d,fn11D);
   rx2prnt(8, "S2",rxctrl&0x08, "C ",rxctrl&0x80, 0x11b,fn11B);


   if (beaconctrl&0x01) mvwaddstr(w,9,1,"GB        ");
   else {
      wattrset(w,at_on);
      mvwaddstr(w,9,1,"GB");
      waddch(w,(beaconctrl&0x02)?'+':' '); /* GB FSK */
      switch (beaconctrl&0x30) {
         case 0x00: waddstr(w,"       "); break;
         case 0x10: waddstr(w," rang. "); break;
         case 0x20: waddstr(w," =EB   "); break;
         case 0x30: waddstr(w," ???   "); break;
      }
      wattrset(w,at_lab);
   }

   wattrset(w,at_on);
   mvwaddstr(w,10,1,"MB        ");

   wattrset(w,(beaconctrl&0x08)?at_on:at_lab);
   mvwaddstr(w,11,1,"EB        ");

   wattrset(w,(b[0x188]&0x02)?at_on:at_lab);
   mvwaddstr(w,12,1,"Rudak-1   ");
   mvwaddstr(w,13,1,"Rudak-2   ");
   wattrset(w,at_lab);

   mvwaddstr(w,14,1,"Leila-1   ");
   mvwaddstr(w,15,1,"Leila-2   ");

   /* --------- matrix ------------------------------------- */

   {
      static int bin2row[14]=         /* maps groups of bits in telemetry to rows on matrix display */
         { 1, 10, 0, 2, 12, 5, 11, 13, 9, 7, 4, 8, 6, 3 };
      static int bin2mat[14][6]={     /* maps bitpositions in telemetry to indices in mat[][] array */
         { 1, 2, 4, 3, 5, 6},
         { 1, 2, 4, 3, 5, 6},
         { 0, 2, 4, 3, 5, 6},
         { 0, 2, 4, 3, 5, 6},
         { 0, 1, 4, 3, 5, 6},
         { 0, 1, 4, 3, 5, 6},
         { 0, 1, 2, 3, 5, 6},
         { 0, 1, 2, 3, 5, 6},
         { 0, 1, 2, 4, 5, 6},
         { 0, 1, 2, 4, 5, 6},
         { 0, 1, 2, 4, 3, 6},
         { 0, 1, 2, 4, 3, 6},
         { 0, 1, 2, 4, 3, 5},
         { 0, 1, 2, 4, 3, 5}
      };
      int bb;
      int colact[7];
      int rowcolour[14],colcolour[7];
      int nextcolour=0;
      int mat[14][7];
#define Xofs 12
#define Xstep 5

      /* scan the telemetry block and fill the array mat[][] */
      for (j=0;j<14;j++) {
         int row = bin2row[j];
         rowcolour[j]=-1;
         for (i=0;i<7;i++) mat[row][i]=0;
         if (j&0x01) bb=(b[0x1f1+j]>>6)|(b[0x1f2+j]<<2);
         else bb=b[0x1f2+j];
         for (i=0;i<6;i++) {
            if (bb&0x01) mat[row][bin2mat[j][i]]=2;
            else mat[row][bin2mat[j][i]]=1;
            bb>>=1;
         }
      }

      for (i=0;i<=6;i++) {
         colact[i]=0;
         colcolour[i]=-1;
      }

#define amaton(c) wattrset(w,at_matrix[c])
#define amatoff wattrset(w,0)
#define at_col(i) at_matrix[colcolour[i]]
#define at_row(i) at_matrix[rowcolour[i]]
      /* draw the entire matrix, row by row */
      wattrset(w,at_lab);
      for (j=0;j<14;j++) {
         int rowact=0;
         for (i=6;i>=0;i--) {
            int c;
            int k;
            switch (mat[j][i]) {
               default:  /* no relay present */
                  if (colact[i]) wattrset(w,at_col(i));
                  c=':';
                  break;
               case 1:   /* relay open */
                  c=ACS_VLINE; 
                  if (colact[i]) wattrset(w,at_col(i));
                  else if (rowact) {
                     c=ACS_HLINE; 
                     wattrset(w,at_row(j));
                  }
                  break;
               case 2:   /* relay closed */
                  c=ACS_BLOCK;
                  if (rowcolour[j]==-1) {
                     int rpt;
                     /* assign new colour, and check which other rows and columns should get the same colour */
                     rowcolour[j]=nextcolour;
                     do {
                        int k;
                        rpt=0;
                        for (k=0;k<14;k++) {
                           int l;
                           if (rowcolour[k]!=nextcolour) continue;
                           for (l=0;l<7;l++)
                              if (mat[k][l]==2 && colcolour[l]==-1) {
                                 int m;
                                 colcolour[l]=nextcolour;
                                 for (m=0;m<14;m++)
                                    if (mat[m][l]==2 && rowcolour[m]==-1) {
                                       rowcolour[m]=nextcolour;
                                       if (m<k) rpt=1;
                                    }
                              }
                        }
                     } while (rpt);
                     nextcolour++;
                  }
                  wattrset(w,at_col(i));
                  colact[i]=1; 
                  rowact=1; 
                  break;
            }
            mvwaddch(w,2+j,Xofs+Xstep*i,c);
            if (rowact) wattrset(w,at_row(j));
            else wattrset(w,at_lab);
            for (k=1;k<((i>0)?Xstep:2);k++) mvwaddch(w,2+j,Xofs+Xstep*i-k,ACS_HLINE);
            wattrset(w,at_lab);
         }
      }
      for (i=0;i<=6;i++) {
         if (colact[i]) wattrset(w,at_col(i));
         else wattrset(w,at_lab);
         mvwaddch(w,2+j,Xofs+Xstep*i,ACS_VLINE);
      }
   }

   /* --------- "destinations" (transmitters, leila) ------------------------------------- */


#define txvalprntfmt(y,x,chan,on,fmt,fmtchan)  \
   if (printchan) { wattrset(w,at_lab); mvwprintw(w,y,x,fmtchan,chan); } \
   else if ((on)|printraw) { wattrset(w,at_val); mvwprintw(w,y,x,fmt,b[chan]); }

#define txvalprnt(y,x,chan,on) txvalprntfmt(y,x,chan,on,"%3i","%3X")

   wattrset(w,at_lab);
   mvwaddstr(w,17,Xofs-5, "TX:                                  ");
   mvwaddstr(w,18,Xofs-6,"PWR:                                  ");
   mvwaddstr(w,19,Xofs-6,"ALC:                                  ");

   x=Xofs;
   wattrset(w,(txctrl&0x0200)?at_on:at_lab); mvwaddstr(w,17,x-0,(antctrl&1)?"V ":"Vo");
   txvalprnt(18,x-1,0x123,(txctrl&0x0200));
   txvalprnt(19,x-1,0x124,(txctrl&0x0200));

   x+=Xstep;
   {
      int on=(txctrl&0x0200);
      int lo=(txctrl&0x0004);
      if (on) {
         wattrset(w,at_on);
         if (lo) mvwaddstr(w,17,x-2,"Ulo");
         else mvwaddstr(w,17,x-2,"Uhi");
         waddch(w,(antctrl&2)?' ':'o');
      } else {
         wattrset(w,at_lab);
         mvwaddstr(w,17,x-2,(antctrl&2)?"  U ":" Uo ");
      }
      txvalprnt(18,x-2,0x125,on);
      txvalprnt(19,x-2,0x126,on&&(!lo));
   }

   x+=Xstep;
   wattrset(w,(txctrl&0x0800)?at_on:at_lab); mvwaddstr(w,17,x-1,"S1");
   txvalprnt(18,x-2,0x122,(txctrl&0x0800));

   x+=Xstep;
   wattrset(w,(txctrl&0x0001)?at_on:at_lab); mvwaddstr(w,17,x-2,"S2");
   wattrset(w,(txctrl&0x4001)?at_on:at_lab); waddch(w,'/');
   wattrset(w,(txctrl&0x4000)?at_on:at_lab); waddch(w,'K');
   txvalprnt(18,x-3,0x138,(txctrl&0x0001));
   txvalprnt(19,x-3,0x139,(txctrl&0x0001));
   txvalprntfmt(18,x+1,0x11a,(txctrl&0x4000),"%-3i","%-3X");

   x+=Xstep;
   if (txctrl&0x0020) {
      wattrset(w,at_on);
      if (txctrl&0x2000) mvwaddstr(w,17,x-2,"Xtwt");
      else mvwaddstr(w,17,x-1,"Xlo");
   } else {
      wattrset(w,at_lab);
      mvwaddstr(w,17,x-2,"  X ");
   }
   txvalprnt(18,x-1,0x117,(txctrl&0x0020));
   txvalprntfmt(19,x-1,0x184,(txctrl&0x0020),"#%02x","%3X");

   x+=Xstep;
   mvwaddstr(w,17,x-2,"Lei1");
   mvwprintw(w,18,x-2,"(%02X)",b[0x1d0]);
   mvwaddch(w,18,x-3,ACS_VLINE);
   mvwaddch(w,19,x-3,ACS_VLINE);
   mvwprintw(w,18,x-2," %2i ",b[0x1d0]&0x0f);
   mvwprintw(w,19,x-2,"%c%c%c%c",
      (b[0x1d0]&0x80) ? 'N' : ' ',   /* notch */
      (b[0x1d0]&0x40) ? 'J' : ' ',   /* jam */
      (b[0x1d0]&0x20) ? 'I' : ' ',   /* IHU control */
      (b[0x1d0]&0x10) ? 'S' : ' ');  /* scan */
   x+=Xstep;
   mvwaddstr(w,17,x-2,"Lei2");
   mvwprintw(w,18,x-2,"(%02X)",b[0x1d1]);
   mvwaddch(w,18,x-3,ACS_VLINE);
   mvwaddch(w,19,x-3,ACS_VLINE);
   mvwprintw(w,18,x-2," %2i ",b[0x1d1]&0x0f);
   mvwprintw(w,19,x-2,"%c%c%c%c",
      (b[0x1d0]&0x80) ? 'N' : ' ',   /* notch */
      (b[0x1d0]&0x40) ? 'J' : ' ',   /* jam */
      (b[0x1d0]&0x20) ? 'I' : ' ',   /* IHU control */
      (b[0x1d0]&0x10) ? 'S' : ' ');  /* scan */
}



Measure mnavmagnet[] = {
   { "",       7, 0,     NULL,   "",      how_none },
   { "M-Soll", 7, 0x1a1, fnRAW,  " %3.0f",  how_ok },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mnaves[] = {
   { "Sensi.:", 8, 0x18a, fnRAW,  "%3.0f",  how_ok },
   { "Pntng:",  6, 0x1c6, NULL,   "",       how_none },
   { "Edge:",   5, 0x1c6, NULL,   "",       how_none },
   { "Lockout:",8, 0x19a, fnRAW,  "%3.0f",  how_ok },
   { "ES1 top:",    6, 0, NULL,   "",       how_none },
   { " Cnt",   5, 0x19b,  fnRAW,  "%3.0f",  how_ok },
   { " MA:",   7, 0x196,  fnRAW,  " %3.0f", how_ok },
   { " Orb:",  6, 0x197,  fn197,  "%5.0f",  how_ok },
   { "ES2 bottom:", 6, 0, NULL,   "",       how_none },
   { " Cnt",   5, 0x19d,  fnRAW,  "%3.0f",  how_ok },
   { " MA:",   7, 0x198,  fnRAW,  " %3.0f", how_ok },
   { " Orb:",  6, 0x199,  fn197,  "%5.0f",  how_ok },
   { "",        0, 0,     NULL,   "",       how_end },
};

Measure mnavspin[] = {
   { "SS1cnt", 6, 0x190, fnRAW,  " %3.0f",  how_ok },
   { "SS2cnt", 6, 0x191, fnRAW,  " %3.0f",  how_ok },
   { "Angle",  6, 0x191, NULL,   "",        how_none },
   { "Thr.",   6, 0x1c6, NULL,   "",        how_none },
   { "Flags",  6, 0x192, fnRAW,  " %3.0f",  how_ok },
   { "Spin",   6, 0x100, fn100,  "%4.1f",   how_ok },
   { " Raw#",  6, 0x193, fnRAW,  " %3.0f",  how_ok },
   { "",       0, 0,     NULL,   "",        how_end },
};

Measure mnav3ax[] = {
   { "Y/25:",  5, 0x12f, fn12E,  "%5.1f", how_onoff1 },
   { "Y/45:",  5, 0x135, fn134,  "%5.1f", how_onoff2 },
   { "Z/25:",  5, 0x12e, fn12E,  "%5.1f", how_onoff1 },
   { "Z/45:",  5, 0x134, fn134,  "%5.1f", how_onoff2 },
   { "Up-X:",  5, 0x130, fn130,  "%5.1f", how_ok },
   { "Up-Y:",  5, 0x131, fn130,  "%5.1f", how_ok },
   { "Dn-X:",  5, 0x132, fn130,  "%5.1f", how_ok },
   { "Dn-Y:",  5, 0x133, fn130,  "%5.1f", how_ok },
   { "U/D Z:", 6, 0x12d, NULL,   "",      how_none },
   { "",       0, 0,     NULL,   "",      how_end },
};

Measure mnav400N[] = {
   { "LIUpwr:", 8, 0x19f, NULL,   "",       how_none },
   { "Valve",   6, 0x113, NULL,   "",       how_none },
   { "Hi pr:",  6, 0x114, fn114,  "%6.1f",  how_onoff1 },
   { "Lo pr:",  6, 0x115, fn115,  "%6.1f",  how_onoff1 },
   { "",        0, 0,     NULL,   "",       how_end },
};

Measure mnavarcjet[] = {
   { "Cnf",     4, 0x181, NULL,  "",        how_none },
   { "EPUpwr:", 7, 0x183, NULL,  "",        how_none },
   { "Flow:",   8, 0x186, fnRAW, " %3.0f",  how_ok },
   { "Mtr flow",8, 0x105, fnRAW, " %3.0f",  how_onoff1 },
   { "Cur.Set", 7, 0x185, fn185, "%5.1f",   how_ok },
   { "Tankpr:", 7, 0x102, fn102, "%5.1f",   how_onoff1 },
   { "Mtr.pr:", 7, 0x101, fn101, "%5.1f",   how_onoff1 },
   { "",        0, 0x103, fn103, "%5.2fA",  how_onoff1, 1 },
   { "",        6, 0x104, fn104, "%5.0fV",  how_onoff1 },
   { "",        0, 0,     NULL,  "",        how_end },
};

Measure mnavwheels[] = {
   { " on sync rpm", 0, 0, NULL, "",    how_none },
   { "1:",  2, 0,     NULL,   "",       how_none },
   { "2:",  2, 0,     NULL,   "",       how_none },
   { "3:",  2, 0,     NULL,   "",       how_none },
   { "",    0, 0,     NULL,   "",       how_end },
};

void write_wnav(byte *b)
{
   int i;

   mvwaddstr(wnav,1,7, (b[0x19f]&1)?"spin":"3-ax");

   printvals(wnavmagnet,mnavmagnet,b,0,0);
   mvwprintw(wnavmagnet,1,0, "%8.8s%3.3s",
      "Off     On      Off/und.Undespun" + 8*(b[0x1a0]&3),
      "                                --- --+ -+- -++ +-- +-+ ++- +++ "+4*(b[0x1a2]&0xf) );

   wattrset(wnavaru,at_val);
   mvwprintw(wnavaru,1,0,"%5s %5s", b[0x13a]<=15?"stowd":"rel'd", b[0x13b]<=15?"stowd":"rel'd");

   orb_lo=b[0x1a6];
   orb_hi=b[0x1a7];
   printvals(wnaves,mnaves,b,0,0);
   mvwprintw(wnaves, 2,6," %4.4s", (b[0x1c6]&0x03)*4+"noneside topboth" );
   mvwprintw(wnaves, 3,5,"   %3s",(b[0x1c6]&0x04)?"pos":"neg");
   mvwprintw(wnaves, 6,8,"%s",b[0x19c]?"upd":"   ");
   mvwprintw(wnaves,10,8,"%s",b[0x19e]?"upd":"   ");

   printvals(wnavspin,mnavspin,b,0,0);
   if (b[0x190]==0 || b[0x190]==255) {
      int a;
      a=b[0x191];
      if (a>=128) a=a-256;
      /* the below formula is from G3RUH's article "Sensorship, a question of attitudes"; apparently, AO-40's sun sensor has the same inclination as AO-10's */
      mvwprintw(wnavspin,3,5, "%5.1f", atan(tan(M_PI/3)*sin(M_PI*a/256))*180/M_PI);
   } else {
      mvwaddstr(wnavspin,3,5, "  -- ");
   }
   mvwprintw(wnavspin,4,6, "%3.1fv", (b[0x1c6]>>4)*0.3 );

   printvals(wnav3ax,mnav3ax,b,(b[0x136]>245),(b[0x137]>245));
   mvwprintw(wnav3ax,9,6, " %3s", (b[0x12d]>128)?"Up":"Dwn");

   printvals(wnav400N,mnav400N,b,(b[0x1c6]&0x08),0);
   mvwprintw(wnav400N,1,8," %3s",b[0x182]==0xaa?" ON":"OFF");
   mvwprintw(wnav400N,2,6,"%6s",b[0x113]<=89?"closed":"open");

   printvals(wnavarcjet,mnavarcjet,b,(b[0x1c6]&0x08),0);
   mvwprintw(wnavarcjet,1,3," %i%i%i%i%i%i%i%i",
      b[0x181]&0x80?1:0,
      b[0x181]&0x40?1:0,
      b[0x181]&0x20?1:0,
      b[0x181]&0x10?1:0,
      b[0x181]&0x08?1:0,
      b[0x181]&0x04?1:0,
      b[0x181]&0x02?1:0,
      b[0x181]&0x01?1:0);
   mvwprintw(wnavarcjet,2,7,"  %3s",b[0x183]==0xaa?" ON":"OFF");

   for (i=0;i<=2;i++) {
      double rpm;
      char *fmt;
      unsigned int r;
      int attr=at_val;
      r = b[0x1c0+2*i] + 256*b[0x1c1+2*i];
      rpm = 960./19. * 2.4e6 * (1./(r+2) - 1./0x6000);
      if (printraw) { rpm=r; fmt="%c %c%7.0f"; }
      else if (printchan) { attr=at_lab; fmt= (i==0)?"%c %c  1C0/1": (i==1)?"%c %c  1C2/3" : "%c %c  1C4/5"; }
      else if (rpm<9e4 && rpm>-9e3) fmt="%c %c%7.1f";
      else fmt="%c%c%8.1e";
      wattrset(wnavwheels,attr);
      mvwprintw(wnavwheels,i+2,2,fmt,
         (b[0x187]&(0x01<<i)) ? '*' : '-',
         (b[0x1da]&(0x04<<i)) ? '*' : '-',
         rpm );
   }
}

Measure mpowervolts[] = {
   { "Main bat", 8, 0x10b, fn10B, " %4.1f", how_ok },
   { "Aux.bat ", 8, 0x10c, fn10B, " %4.1f", how_ok },
   { "Bat.ofs",  7, 0x18c, fn18C, " %5.2f", how_ok },
   { "28V bus",  8, 0x10d, fn10B, " %4.1f", how_ok },
   { "",         0, 0,     NULL,  "",       how_end },
};

Measure mpoweramps[] = {
   { "Tot.bat",  7, 0x106, fn106, "%5.1f", how_ok },
   { "Main bus", 8, 0x108, fn108, "%4.1f", how_ok },
   { "28v-S",    6, 0x109, fn109, " %5.2f", how_ok },
   { "BCR",      6, 0x10A, fn10A, " %5.1f", how_ok },
   { "SEU",      6, 0x170, fnRAW, " %5.0f", how_broken },
   { "EPU",      6, 0x107, fn107, " %5.1f", how_ok },
   { "",         0, 0,     NULL,  "",       how_end },
};

Measure mpowerbcr[] = {
   { "V-in",    5, 0x10e,  fn10E,  "  %4.1f",     how_ok,  1 },
   { "",       11, 0x111,  fn10F,  "  %4.1f",     how_ok,  1 },
   { "",       17, 0x10f,  fn10F,  "  %4.1f",     how_ok,  0 },
   { "I-SA",    5, 0x171,  fn171,  "  %4.1f",     how_broken,  1 },
   { "",       11, 0x176,  fn171,  "  %4.1f",     how_broken,  1 },
   { "",       17, 0x174,  fn171,  "  %4.1f",     how_broken,  0 },
   { "I-SA",    5, 0x172,  fn171,  "  %4.1f",     how_broken,  1 },
   { "",       11, 0x177,  fn171,  "  %4.1f",     how_broken,  1 },
   { "",       17, 0x173,  fn171,  "  %4.1f",     how_broken,  0 },
   { "V-ofs",   5, 0x18d,  fn18D,  "  %4.1f",     how_ok,  1 },
   { "",       11, 0x18e,  fn18D,  "  %4.1f",     how_ok,  1 },
   { "",       17, 0x18f,  fn18D,  "  %4.1f",     how_ok,  0 },
   { "I-10vC", 11, 0x178,  fn175,  "  %4.1f",     how_broken,  1 },
   { "",       17, 0x175,  fn175,  "  %4.1f",     how_broken,  0 },
   { "U-10vC", 11, 0x112,  fn110,  "  %4.1f",     how_ok,  1 },
   { "",       17, 0x110,  fn110,  "  %4.1f",     how_ok,  0 },
   { "",        0, 0,     NULL,  "",       how_end },
};

Measure mpowertx[] = {
   { "S 28V",   7, 0x17a, fn17A, "%5.1f",   how_ok },
   { "S 10V",   7, 0x17b, fnRAW, "  %3.0f", how_ok },
   { "Xhelix",  7, 0x119, fn119, "%4.1fm",  how_ok },
   { "K",       7, 0x179, fnRAW, "  %3.0f", how_ok },
   { "",        0, 0,     NULL,  "",        how_end },
};

void write_wpower(byte *b)
{
   printvals(wpowervolts,mpowervolts,b,0,0);
   printvals(wpoweramps,mpoweramps,b,0,0);
   printvals(wpowertx,mpowertx,b,0,0);
   printvals(wpowerbcr,mpowerbcr,b,0,0);

#define prntonoff(w,y,x,o,s) if (o) wattrset(w,at_on); else wattrset(w,at_off); mvwaddstr(w,y,x,s)
   prntonoff(wpowerstat,1,0,b[0x18b]&1,"Main bat");
   prntonoff(wpowerstat,2,0,b[0x18b]&2,"Aux. bat");
   prntonoff(wpowerstat,3,0,b[0x18b]&4,"Charger");
   prntonoff(wpowerstat,4,0,b[0x18b]&8,"Heater");
}




void write_wstatus(byte *b)
{
   char s[12*8];

   mvwaprintf(wstatusinner,0,0,at_lab,at_val,"@1Day @2%-5i  @1UTC @2%02i:%02i:%02i.%02i  @1Orbit @2%-5i @1MA@2 %-3i  @1frac@2 %-5i          ",
      b[0x1ad]*256+b[0x1ac],b[0x1ab],b[0x1aa],b[0x1a9],b[0x1a8],
      b[0x1a7]*256+b[0x1a6],b[0x1a5],b[0x1a4]*256+b[0x1a3]);

   mvwaprintf(wstatusinner,1,0,at_lab,at_val,"@1Mem.err.: @2%1i",b[0x1d9]>>5);
   if (b[0x1ed]==0) wattrset(wstatusinner,at_off);
   else wattrset(wstatusinner,at_val);
   mvwprintw(wstatusinner,1,13,"Emergency:                                         ");
   s[0]=0;
   if (b[0x1ed]&0x01) strcat(s," LowBat");
   if (b[0x1ed]&0x02) strcat(s," VeryLowBat");
   if (b[0x1ed]&0x04) strcat(s," CommandLoss");
   if (b[0x1ed]&0x08) strcat(s," HighTemp");
   if (b[0x1ed]&0x10) strcat(s," SunAngle");
   if (b[0x1ed]&0x20) strcat(s," ?20?");
   if (b[0x1ed]&0x40) strcat(s," ?40?");
   if (b[0x1ed]&0x80) strcat(s," ?80?");
   if (strlen(s)>=64-23) strcpy(s+64-23-3,"...");
   mvwaddstr(wstatusinner,1,23,s);

   wattrset(wstatusinner,at_lab);
   mvwprintw(wstatusinner,2,0,"Equipment: ");
   prntonoff(wstatusinner,2,11,b[0x1c9]&0x01,"IHU-2");
   prntonoff(wstatusinner,2,17,b[0x188]&0x01,"ARU");
   prntonoff(wstatusinner,2,21,b[0x1c6]&0x08,"Instr");
   prntonoff(wstatusinner,2,27,b[0x1d9]&0x01,"LIU");
   prntonoff(wstatusinner,2,31,b[0x1d9]&0x04,"EPU");
   prntonoff(wstatusinner,2,35,b[0x1d9]&0x02,"Armed");

   wattrset(wstatusinner,at_lab);
   mvwprintw(wstatusinner,3,0,"Equipment: ");
   prntonoff(wstatusinner,3,13,b[0x188]&0x02,"Rudak");
   prntonoff(wstatusinner,3,19,b[0x188]&0x04,"GPS");
   prntonoff(wstatusinner,3,23,b[0x188]&0x08,"MonRX");
   prntonoff(wstatusinner,3,29,b[0x188]&0x10,"CamA");
   prntonoff(wstatusinner,3,34,b[0x188]&0x20,"CamB");
   prntonoff(wstatusinner,3,39,b[0x188]&0x80,"Cedex");
   prntonoff(wstatusinner,3,45,b[0x188]&0x40,"-ctl");
   prntonoff(wstatusinner,3,50,b[0x1a2]&0x10,"Laser");
}



void openwindows(void)
{
   wclearat(stdscr,at_lab);
   refresh();

   if (LINES<37) layout=1;
   else if (COLS>=118) layout=0;
   else if (LINES>=60) layout=1;
   else if (37*(118-COLS)>80*(60-LINES)) layout=1;
   else layout=0;
   switch (layout) {
      case 0:
         xsize = 118;
         ysize = 37;
         mainpad = newpad(ysize,xsize);
         wclearat(mainpad,at_lab);
         wraw=createwin(10,66,0,0,"RAW");
         wmatrix=createwin(21,45,16,0,"RX/TX/MATRIX");
         wtemp=createwin(21,36,16,45,"TEMPERATURES");
         wnav=createwin(21,37,16,81,"NAVIGATION");
         wpower=createwin(14,38,0,66,"POWER");
         wstatus=createwin(6,66,10,0,"STATUS");
         wprogstat=subpad(mainpad,2,118-66,14,66);
         break;
      case 1:
         if (COLS>=81) xsize = 81;
         else xsize=80;
         ysize = 60;
         mainpad = newpad(ysize,xsize);
         wclearat(mainpad,at_lab);
         wraw=createwin(10,66,0,0,"RAW");
         wstatus=createwin(6,66,10,0,"STATUS");
         wprogstat=subpad(mainpad,2,66,16,0);
         wmatrix=createwin(21,45,18,0,"TX/RX/MATRIX");
         wtemp=createwin(21,36,18,44+(xsize>=81),"TEMPERATURES");
         wnav=createwin(21,37,39,0,"NAVIGATION");
         wpower=createwin(14,38,39,37,"POWER");
         break;
   }

   wstatusinner= subpad(wstatus,4,64,1,1);

   wtemptx=      createmsubwin(wtemp,8,10,1,1,1,"TX",mtemptx);
   wtemprx=      createmsubwin(wtemp,6,10,9,1,1,"RX",mtemprx);
   wtempmodules= createmsubwin(wtemp,4,10,15,1,1,"Modules",mtempmodules);
   wtempframe=   createmsubwin(wtemp,7,11,1,12,1,"Frame",mtempframe);
   wtemppipes=   createmsubwin(wtemp,8,11,8,12,1,"Heatpipes",mtemppipes);
   wtempbatreg=  createmsubwin(wtemp,4,11,16,12,1,"Bat.Reg.",mtempbatreg);
   wtemptanks=   createmsubwin(wtemp,6,11,1,24,0,"Tanks",mtemptanks);
   wtempsolar=   createmsubwin(wtemp,7,11,7,24,0,"Solar",mtempsolar);
   wtempbatt=    createmsubwin(wtemp,6,11,14,24,0,"Batteries",mtempbatt);

   wnavmagnet=   createmsubwin(wnav,3,11,2,1,1,"Magnet",mnavmagnet);
   wnavspin=     createmsubwin(wnav,8,10,2,13,1,"Spin SunS.",mnavspin);
   wnavaru=      createsubwin (wnav,2,11,5,1,1,"Panels");
   wnaves=       createmsubwin(wnav,13,11,7,1,1,"Earth Sens.",mnaves);
   wnav3ax=      createmsubwin(wnav,10,10,10,13,1,"3-Ax SunS.",mnav3ax);
   wnav400N=     createmsubwin(wnav,5,12,1,24,0,"400 N",mnav400N);
   wnavarcjet=   createmsubwin(wnav,9,12,6,24,0,"ArcJet",mnavarcjet);
   wnavwheels=   createmsubwin(wnav,5,12,15,24,0,"Wheels",mnavwheels);
   wattrset(wnav,at_lab);
   mvwaddstr(wnav, 1,1, "Mode:");

   wpowerstat=   createsubwin (wpower,5,9,1,1,1,"Status");
   wpowervolts=  createmsubwin(wpower,5,13,1,11, 1,"Volts",  mpowervolts);
   wpoweramps=   createmsubwin(wpower,7,12,1,25,0,"Amps",   mpoweramps);
   wpowerbcr=    createmsubwin(wpower,7,23,6,1,1,"",        mpowerbcr);
   wpowertx=     createmsubwin(wpower,5,12,8,25,0,"TX Amps",mpowertx);
   wattrset(wpowerbcr,at_subborder);
   mvwaddch(wpowerbcr,0,9,'1');
   mvwaddch(wpowerbcr,0,15,'2');
   mvwaddch(wpowerbcr,0,21,'3');
   mvwaddstr(wpowerbcr,0,3,"BCR");
   wattrset(wpowerbcr,A_BOLD);

   vxsize=xsize; if (COLS<vxsize) vxsize=COLS;
   vysize=ysize; if (LINES<vysize) vysize=LINES;

   do_prefresh();
}

void closewindows(void)
{
   delwin(wtemptx);
   delwin(wtemprx);
   delwin(wtempmodules);
   delwin(wtempframe);
   delwin(wtemppipes);
   delwin(wtempbatreg);
   delwin(wtemptanks);
   delwin(wtempsolar);
   delwin(wtempbatt);
   delwin(wnavmagnet);
   delwin(wnavarcjet);
   delwin(wnav400N);
   delwin(wnaves);
   delwin(wnavspin);
   delwin(wnav3ax);
   delwin(wnavwheels);
   delwin(wnavaru);
   delwin(wpowervolts);
   delwin(wpoweramps);
   delwin(wpowerbcr);
   delwin(wpowertx);
   delwin(wpowerstat);

   delwin(wraw);
   delwin(wmatrix);
   delwin(wtemp);
   delwin(wnav);
   delwin(wpower);
   delwin(wstatus);
   delwin(wprogstat);

   delwin(mainpad);
}

/*--------------------------------------------------------------------------------------------------------------------------*/
/* code for dealing with files and TCP/UDP socket */

byte tcpbuf[522];
int tcplen=0;
char tcphostname[256]="garc9.gsfc.nasa.gov";
int tcpport=1024;
int tcpsocket=-1;

char udphostname[256]="localhost";
struct sockaddr_in udpsin;
int udpport=2222;
int udpsocket=-1;

byte *partialblock=NULL;  /* pointer to a partial block being received now */
int partiallen;           /* length of the partial block */


void tcpopen(void)
{
   struct hostent *hp;
   struct sockaddr_in sin;
   int so;

   hp = gethostbyname(tcphostname);
   if (!hp) return;

   bzero((char*)&sin,sizeof(sin));
   sin.sin_family=AF_INET;
   bcopy(hp->h_addr, (char*)&sin.sin_addr, hp->h_length);
   sin.sin_port=htons(tcpport);

   so = socket(PF_INET, SOCK_STREAM, 0);
   if (so<0) return;
   if (connect(so, (struct sockaddr*)&sin, sizeof(sin))<0) return;
   tcplen=0;

   fcntl(so,F_SETFL,O_NONBLOCK);  /* make the socket non-blocking; this should not be necessary, since we use select(), but otherwise sometimes recv() did block */

   tcpsocket=so;
}

byte *tcpread(void)
{
   int l;
   static byte syncvector[]={0x2f,0x8f,0x6e,0x4d,0x28,0x86,0x75,0x60};
   byte *p;

   /* read to fill 522 byte buffer */
   l=recv(tcpsocket,tcpbuf+tcplen,522-tcplen,0);
   if (l==-1 && errno==EAGAIN) return NULL;    /* catch the case where recv() would otherwise block; should not happen since we use select(), but it does... */
   if (l<=0) { close(tcpsocket); tcpsocket=-1; return NULL; }
   tcplen+=l;
   if (tcplen<9) return NULL;

   if (memcmp(tcpbuf,syncvector,8)==0) {
      /* syncvector found: return block or partial block */
      if (tcplen==522) {
         tcplen=0;
         partialblock=NULL;
         partiallen=0;
         return tcpbuf+8;
      } else {
         partialblock=tcpbuf+8;
         partiallen=tcplen-8;
         return NULL;
      }
   }

   /* syncvector not found: search for possible next occurrence of syncvector */
   p=memchr(tcpbuf+1,syncvector[0],tcplen-1);

   if (p==NULL) tcplen=0;
   else {
      memmove(tcpbuf,p,tcplen-(p-tcpbuf));
      tcplen-=(p-tcpbuf);
   }
   return NULL;
}



void udpopen(void)
{
   struct hostent *hp;
   struct sockaddr_in sin;
   int so;

   hp = gethostbyname(udphostname);
   if (!hp) return;
   bzero((char*)&udpsin,sizeof(udpsin));
   udpsin.sin_family=AF_INET;
   bcopy(hp->h_addr, (char*)&udpsin.sin_addr, hp->h_length);

   bzero((char*)&sin,sizeof(sin));
   sin.sin_family=AF_INET;
   sin.sin_addr.s_addr = htonl(INADDR_ANY);
   sin.sin_port=htons(udpport);

   so = socket(PF_INET, SOCK_DGRAM, 0);
   if (so<0) return;
   if (bind(so, (struct sockaddr*)&sin, sizeof(sin))<0) {
      perror("bind");
      return;
   }
   
   udpsocket=so;
}

byte *udpread(void)
{
   int l;
   byte s[8196];
   byte *p;
   int len;
   struct sockaddr_in sin;
   int addrlen=sizeof(sin);

   l=recvfrom(udpsocket,s,8196,0,(struct sockaddr*)&sin,&addrlen);
   if (l<=0) { close(udpsocket); udpsocket=-1; return NULL; }
   s[l]=0;

   if (sin.sin_addr.s_addr ^ udpsin.sin_addr.s_addr) return NULL;

   p=s;
   len=-1;
   do {
      if (strncmp(p,"Length:",7)==0) len=atoi(p+7);
      p=strchr(p,'\n');
      if (!p) return NULL;
      p++;
   } while (*p!='\r' && *p!='\n');
   if (p[0]=='\r' && p[1]=='\n') p++;
   p++;

   if (len!=4144 || p+len/8>s+l) return NULL;

   return p+4;   /* +4 to skip the sync vector */
}



int checkcrc(byte *b)
{
   long int crc;
   int i;
   crc=0xffff;
   for (i=0;i<512+2;i++) {
      int y;
      int j;
      y=*b++;
      for (j=0;j<8;j++) {
         crc<<=1;
         if ((y^(crc>>9))&0x80) crc ^= 0x1021;
         y<<=1;
      }
   }
   return ((crc&0xffff)==0);
}



FILE *ftlm=NULL;     /* the file from which telemetry blocks are read */
char tlmfname[256];  /* name of that file */
int numblocks=0;     /* this is the number of blocks in the *ftlm file */
FILE *flog=NULL;     /* the file to which telemetry blocks received via TCP or UDP are written; normally the same as ftlm */
int curblock=-1;     /* indicates which block is being shown now.  0...numblocks-1 refer to blocks from ftlm; numblocks refers to partial block; -1 is no block (typically because ftlm is empty) */

char atcallsign[16]="";   /* empty; or callsign of user, preceeded by '@'; included in names of log files (for the benefit of the telemetry archive) */
int logfileday=0;         /* if nonzero: day of month included in log file name */
int logfiledelay=3600;    /* maximum time (in seconds) between two consecutively received blocks that must still be written into the same logfile, even if we changed UTC day in the meantime */
time_t timelastlogwrite=0;  /* time of last write to log file, for use with logfiledelay */



void closetlm(void)
{
   if (ftlm) { fclose(ftlm); ftlm=NULL; }
   if (flog) { fclose(flog); flog=NULL; }
   if (tcpsocket) { close(tcpsocket); tcpsocket=-1; }
   if (udpsocket) { close(udpsocket); udpsocket=-1; }
   partialblock=0; partiallen=0;
   numblocks=0;
   curblock=-1;
   logfileday=0;
}

void opentlmfile(char *name)
{
   struct stat st;
   if (stat(name,&st)<0) return;
   ftlm = fopen(name,"rb");
   if (ftlm==NULL) return;
   strncpy(tlmfname,name,255); tlmfname[255]=0;
   numblocks=st.st_size/512;
   if (numblocks==0) curblock=-1;
   else curblock=0;
}

void openlogfile(void)
{
   time_t tt;
   struct tm *lt;
   char name[32];

   time(&tt);
   lt=gmtime(&tt);
   sprintf(name,"T%02i%02i%02i%s.RAW",lt->tm_year%100,lt->tm_mon+1,lt->tm_mday,atcallsign);
   logfileday = lt->tm_mday;

   flog=fopen(name,"a");
   if (!flog) return;
   fflush(flog);
   opentlmfile(name);
   curblock=numblocks-1;
}

void checklogfileday(void)
{
   /* Check whether we've moved to a new day; 
      if so, reopen the log file, so its name contains the proper date for the archive.
      However, check logfiledelay first
   */
   time_t tt;
   struct tm *lt;

   time(&tt);
   if (tt < timelastlogwrite+logfiledelay) return;

   lt=gmtime(&tt);
   if (lt->tm_mday==logfileday) return;
   closetlm();
   openlogfile();
}


byte *loadblock(int number)
{
   static byte b[512];
   fseek(ftlm,number*512,SEEK_SET);
   if (fread(b,512,1,ftlm)==1) return b;
   return NULL;
}


/*--------------------------------------------------------------------------------------------------------------------------*/
/* code related to drawing graphs */


int graph_gnuplot(int n,float *d0,float *d1,float *d2)
{
   FILE *gppipe;
   int i;
   static int gnuplotpresent=-1;

   if (!gnuplotpresent) return 0;
   if (gnuplotpresent==-1) {
      if (getenv("DISPLAY")==NULL) {
         gnuplotpresent=0;
         return 0;
      }
      gnuplotpresent = (system("gnuplot < /dev/null")==0);
      if (!gnuplotpresent) {
         wclearat(stdscr,at_lab); touchwin(stdscr); refresh();  /* to erase any possible error messages from the test */
         return 0;
      }
   }

   gppipe=popen("gnuplot > /dev/null 2> /dev/null","w");
   if (!d2) {
      fprintf(gppipe,"set nokey ; plot '-' with linespoints\n");
      for (i=0;i<n;i++) fprintf(gppipe,"%f %f\n",d0[i],d1[i]);
      fprintf(gppipe,"e\n");
   } else {
      fprintf(gppipe,"set nokey ; set y2tics ; set ytics nomirror ; plot '-' with linespoints, '' axis x1y2 with linespoints \n");
      for (i=0;i<n;i++) fprintf(gppipe,"%f %f\n",d0[i],d1[i]);
      fprintf(gppipe,"e\n");
      for (i=0;i<n;i++) fprintf(gppipe,"%f %f\n",d0[i],d2[i]);
      fprintf(gppipe,"e\n");
   }
   fflush(gppipe);
   return 1;
}


int at_graph1, at_graph2, at_graphcross;  /* attributes for the two curves in a graph, and for points where the graphs cross */
int asciigraph_visible=0;
WINDOW *w_asciigraph=NULL;

void graph_ascii(int n,float *d0,float *d1,float *d2)
{
   double minx,miny1,miny2=0,maxx,maxy1,maxy2=0;
   int i;
   WINDOW *w;
   int height=LINES-2, width=COLS-2;
   int lmargin,rmargin;

   /* figure out the range of the data */
   minx=maxx=d0[0];
   miny1=maxy1=d1[1];
   if (d2) {
      miny2=maxy2=d2[1];
   }
   for (i=0;i<n;i++) {
      if (d0[i]<minx) minx=d0[i];
      if (d0[i]>maxx) maxx=d0[i];
      if (d1[i]<miny1) miny1=d1[i];
      if (d1[i]>maxy1) maxy1=d1[i];
      if (d2) {
         if (d2[i]<miny2) miny2=d2[i];
         if (d2[i]>maxy2) maxy2=d2[i];
      }
   }
   if (minx==maxx) { minx*=0.99999; maxx*=1.00001; }
   if (miny1==maxy1) { miny1*=0.99999; maxy1*=1.00001; }
   if (d2 && miny2==maxy2) { miny2*=0.99999; maxy2*=1.00001; }

   /* open a "pad" for the graphs */
   w_asciigraph=newpad(height+2,width+2);
   wattrset(w_asciigraph,at_helpborder);
   box(w_asciigraph,ACS_VLINE,ACS_HLINE);
   w=subpad(w_asciigraph,height,width,1,1);
   wclearat(w,at_val);
   lmargin= 6;
   rmargin= d2 ? 6 : 0;

   /* draw the "axes"; actually, only the axes' end points */
   if (d2) wattrset(w,at_graph1);
   mvwprintw(w,0,0,"%5.1f",maxy1);
   mvwprintw(w,height-2,0,"%5.1f",miny1);
   wattrset(w,at_val);
   mvwprintw(w,height-1,lmargin,"%-9.3f",minx);
   mvwprintw(w,height-1,width-rmargin-9,"%9.3f",maxx);
   if (d2) {
      wattrset(w,at_graph2);
      mvwprintw(w,0,width-rmargin+1,"%-5.1f",maxy2);
      mvwprintw(w,height-2,width-rmargin+1,"%-5.1f",miny2);
      wattrset(w,at_val);
   }

   /* plot the curve */
   for (i=0;i<n;i++) {
      int x,y;
      char c,ch;
      int attr;

      x=lmargin+0.5+(d0[i]-minx)/(maxx-minx)*(width-lmargin-rmargin-1);
      y=0.5+2*(maxy1-d1[i])/(maxy1-miny1)*(height-1.5);
      if (y&1) c=','; else c='\'';
      y/=2;
      wmove(w,y,x);
      attr=winch(w);
      ch=attr&A_CHARTEXT;
      attr&=A_ATTRIBUTES|A_COLOR;
      if (ch!=' ' && ch!=c) c='|';
      if (d2) {
         if (ch==' ') attr=at_graph1;
         else if (attr!=at_graph1) attr=at_graphcross;
         wattrset(w,attr);
      }
      waddch(w,c);

      if (!d2) continue;

      y=0.5+2*(maxy2-d2[i])/(maxy2-miny2)*(height-1.5);
      if (y&1) c=','; else c='\'';
      y/=2;
      wmove(w,y,x);
      attr=winch(w);
      ch=attr&A_CHARTEXT;
      attr&=A_ATTRIBUTES|A_COLOR;
      if (ch!=' ' && ch!=c) c='|';
      if (ch==' ') attr=at_graph2;
      else if (attr!=at_graph2) attr=at_graphcross;
      wattrset(w,attr);
      waddch(w,c);
   }

   prefresh(w_asciigraph,0,0, 
      (LINES-height-2)/2, (COLS-width-2)/2,
      (LINES-height)/2+height, (COLS-width)/2+width );
   asciigraph_visible=1;
}

void del_asciigraph(void)
{
   delwin(w_asciigraph);
   wclearat(stdscr,at_val);
   asciigraph_visible=0;
   refresh();
   do_prefresh();
}


void do_graph(int chan1,int chan2)
{
   int i,n=0;
   float *d0,*d1,*d2;
   d0=malloc(numblocks*sizeof(float));
   d1=malloc(numblocks*sizeof(float));
   if (chan2>0) d2=malloc(numblocks*sizeof(float));
   else d2=NULL;
   for (i=0;i<numblocks;i++) {
      byte *b;
      b=loadblock(i);
      if (b && (b[0]=='A' || (b[0]==('A'|0x80) && decode_wrongCRC))) {
         int orbit,ma,Z;
         orbit = b[0x1a6]+256*b[0x1a7];
         ma = b[0x1a5];
         Z = 256*b[0x1a4]+b[0x1a3];
         if(Z>16100) Z=16100;
         d0[n] = orbit + ma/256. + (16100-Z)/256./16100.;    /* monotonic as long as the orbital period is < 24 hours */
         d1[n]=(tlmfns[chan1-0x100])(b[chan1]);
         if (d2) d2[n]=(tlmfns[chan2-0x100])(b[chan2]);
         n++;
      }
   }
   if (n>0) {
      if (!graph_gnuplot(n,d0,d1,d2)) graph_ascii(n,d0,d1,d2);
   }
   free(d0);
   free(d1);
   if (d2) free(d2);
}


int do_graph_wod(byte *b)
{
   float d0[384];
   float d1[384];
   int samples,chan,orbit,ma,orbit0,ma0;
   int n;
   float x,x0;

   if (  strncmp(b+2," Whole Orbit Data",17)!=0
      || sscanf(b+27,"Samples: %i",&samples)!=1
      || sscanf(b+41,"Captured Channel: #%4x",&chan)!=1
      || sscanf(b+7*64+22,"#%2x%2x",&orbit,&ma)!=2
      || sscanf(b+8*64-11,"#%2x%2x",&orbit0,&ma0)!=2
      )
      return 0;

   /* Unfortunately, the high-order bits of the orbit number are not contained
      in the WOD format, but it would be nice to have the correct orbit number on
      the horizontal axis. So let's make a guess based on the orbit number in any
      recent A or E blocks we may have seen.
      (Actually, it would be better to only consider orbit numbers from recent
      A blocks, not E blocks since those may be very old. But that would be
      messier to implement, so let's first see whether the present solution is
      practically acceptable.)
   */
   orbit += (orb_hi<<8);    
   orbit0 += (orb_hi<<8);
   if (orbit0 > 1+(orb_hi<<8)+orb_lo) { orbit0-=256; orbit-=256; }
   if (orbit0 < orbit) orbit-=256;

   x = orbit + ma/256.;
   x0 = orbit0 + (ma0+0.5)/256.;
   n = 0;
   while (x<x0) {
      d0[n]=x;
      d1[n]= (tlmfns[chan-0x100])(b[n+64]);
      x+=samples/256.;
      n++;
   }
   if (n<=0) return 0;
   if (!graph_gnuplot(n,d0,d1,NULL)) graph_ascii(n,d0,d1,NULL);
   return 1;
}


/*--------------------------------------------------------------------------------------------------------------------------*/


void preparehelp(void)
{
   helpheight=25;
   helpwidth=69;

   if (helppad==NULL) {
      WINDOW *w;
      helppad=newpad(helpheight+2,helpwidth+2);
      wattrset(helppad,at_helpborder);
      box(helppad,ACS_VLINE,ACS_HLINE);
      w=subpad(helppad,helpheight,helpwidth,1,1);
   
      wmove(w,0,0);
      wattrset(w,A_BOLD);
      waddstr(w,"AO40TLMVIEW " VERSION "  -  a viewer for AMSAT-OSCAR 40 telemetry\n");
      wattrset(w,0);
      waddstr(w,"                                     (c) 2001-2002, pa3fwm@amsat.org");
      wattrset(w,0);

      mvwaddstr(w,2,0,"Keystroke commands:\n");
      waddstr(w," q  - quit                                r - display raw values    \n");
      waddstr(w," a  - toggle decoding of A blocks         R - display channel number\n");
      waddstr(w," e  - toggle decoding of E blocks         x - hexadecimal display   \n");
      waddstr(w," c  - toggle decoding of blocks with incorrect CRC                  \n");
      waddstr(w," g  - make graph                          G - graph Whole Orbit Data\n");
      waddstr(w," f  - read telemetry blocks from file\n");
      waddstr(w," t u - receive telemetry through TCP resp. UDP (note: host and port\n"
                "      as specified on commandline; TCP defaults to Goddard server)\n");
      waddstr(w," left, up     - show previous telemetry block\n");
      waddstr(w," right, down  - show next telemetry block\n");
      waddstr(w," <  >         - show previous/next telemetry block of same category\n");
      waddstr(w," page-up      - go back 16 blocks\n");
      waddstr(w," page-down    - go forward 16 blocks\n");
      waddstr(w," home         - go to first block\n");
      waddstr(w," end          - go to last (most recent) block\n");
      waddstr(w," TAB          - toggle view (on small screens)\n");
      waddstr(w,"\n");
      waddstr(w,"For more information, see:\n");
      waddstr(w," http://www.amsat.org/amsat/sats/ao40/               - the satellite\n");
      waddstr(w," http://www.amsat.org/amsat/sats/ao40/ao40-tlm.html  - the telemetry\n");
      waddstr(w," http://www.cs.utwente.nl/~ptdeboer/ham/ao40/        - this software\n");
      waddstr(w,"and of course the man page.\n");

   }
}


void showhex(byte *b,int l)
{
   int i;
   int height=33, width=17*3+5;
   static WINDOW *w;

   hexpadysize=height+2;
   hexpadxsize=width+2;

   if (hexpad==NULL) {
      hexpad=newpad(hexpadysize,hexpadxsize);
      wattrset(hexpad,at_helpborder);
      box(hexpad,ACS_VLINE,ACS_HLINE);
      w=subpad(hexpad,height,width,1,1);
      wclearat(w,at_lab);
      mvwaddstr(w,0,5," 0  1  2  3   4  5  6  7   8  9  A  B   C  D  E  F");
      for (i=0;i<32;i++)
         mvwprintw(w,i+1,0,"%03X",i*0x10);
   }

   wattrset(w,at_val);
   i=0;
   while (i<512) {
      mvwprintw(w,1+i/16,5,"%02X %02X %02X %02X  %02X %02X %02X %02X  %02X %02X %02X %02X  %02X %02X %02X %02X",
         b[i], b[i+1], b[i+2], b[i+3], b[i+4], b[i+5], b[i+6], b[i+7],
         b[i+8], b[i+9], b[i+10], b[i+11], b[i+12], b[i+13], b[i+14], b[i+15] );
      if (i+16>l) break;
      i+=16;
   }
   if (i<512) {
      i=l;
      while (i%16) {
         mvwaddstr(w,1+i/16,5+3*(i%16)+(i%16)/4,"  ");
         i++;
      }
      while (i<512) {
         mvwaddstr(w,1+i/16,5,"                                                  ");
         i+=16;
      }
   }
}



char *inputstring(int yy,int xx)
{
   static char s[256];
   int x,y;
   setview(2);
   do_prefresh();
   getbegyx(wprogstat,y,x);
   attrset(at_val);
   untouchwin(stdscr);     /* note: mvwgetnstr() doesn't seem to work on a pad (like wprogstat), so we invoke it on stdscr */
   echo(); curs_set(1);
   mvwgetnstr(stdscr,y+yy-ofsy,x+xx-ofsx,s,255);
   noecho(); curs_set(0);
   return s;
}


void mainloop(void)
{
   int c=' ';
   int redraw=1;
   int redrawdec=0;
   fd_set readfds;
   unsigned char decbuf[512];  /* a copy of the last block (A- or E-type) we decoded; we need this for the 'r' and 'R' commands in case the current block is not A- or E-type  */
   int decbufvalid=0;
   struct timeval timeout;

   memset(decbuf,0,512);

   do {
      int i;

      if (redraw) {
         if (curblock>=0 && curblock<numblocks) {
            byte *b;
            b=loadblock(curblock);
            if (b) {
               int crcok=((b[0]&0x80)==0);
               write_wraw(b,crcok,512);
               if ( (crcok || decode_wrongCRC) 
                    && ( ((b[0]&0x7f)=='A' && decode_A) || ((b[0]&0x7f)=='E' && decode_E) ) 
                  ) {
                  memcpy(decbuf,b,512);
                  decbufvalid=1;
                  redrawdec=1;
               }
               if (hexvisible) showhex(b,512);
            }
         } else if (curblock==numblocks && partialblock!=NULL) {
            write_wraw(partialblock,1,partiallen);
            if (hexvisible) showhex(partialblock,partiallen);
         }
         redraw=0;
      }
      if (redrawdec && decbufvalid) {
         write_wtemp(decbuf);
         write_wmatrix(decbuf);
         write_wnav(decbuf);
         write_wpower(decbuf);
         write_wstatus(decbuf);
         redrawdec=0;
      }

      wattrset(wprogstat,at_lab);
      mvwaddstr(wprogstat,0,1,"                                                     ");
      if (tcpsocket>=0) {
         mvwaprintf(wprogstat,0,1,at_lab,at_val,"@1Source: @2TCP/IP %s:%i",tcphostname,tcpport);
      } else if (udpsocket>=0) {
         mvwaprintf(wprogstat,0,1,at_lab,at_val,"@1Source: @2UDP %s:%i",udphostname,udpport);
      } else if (ftlm!=NULL) {
         mvwaprintf(wprogstat,0,1,at_lab,at_val,"@1Source: @2file %-40s",tlmfname);
      } else {
         mvwaprintf(wprogstat,0,1,at_lab,at_val,"@1Source: @2none");
      }
      mvwaprintf(wprogstat,1,1,at_lab,at_val,"@1Block: @2%4i/%-4i",curblock+1,numblocks+(partialblock!=NULL));
      wattrset(wprogstat,at_lab);
      mvwaprintf(wprogstat,1,20,at_off,at_on,"Decode: %s A %s E %s wrongCRC",
         decode_A ? "@2" : "@1",
         decode_E ? "@2" : "@1",
         decode_wrongCRC ? "@2" : "@1" );
      wattrset(wprogstat,at_val);
      waddstr(wprogstat, printraw?" RAW " : printchan?" CHAN" :"     ");
      if (!asciigraph_visible) do_prefresh();

      FD_ZERO(&readfds);
      FD_SET(0,&readfds);
      if (tcpsocket>=0) FD_SET(tcpsocket,&readfds);
      if (udpsocket>=0) FD_SET(udpsocket,&readfds);
      i=1;
      if (tcpsocket>=i) i=tcpsocket+1;
      if (udpsocket>=i) i=udpsocket+1;
      timeout.tv_sec = 60;
      timeout.tv_usec = 0;
      select( i, &readfds, NULL, NULL, &timeout );
      if (FD_ISSET(0,&readfds)) {
         c=wgetch(wprogstat);
         if (asciigraph_visible) {
            del_asciigraph();
            if (c!=KEY_RESIZE) c=0;
         }
         switch(c) {
            case 'g':
               if (!ftlm || curblock<0) break;
               {
                  char *s;
                  int ch1,ch2;
                  int orig_printchan=printchan;
                  int orig_printraw=printraw;
                  mvwaddstr(wprogstat,0,1,"Graph of which channel(s) (hex) ?                   ");
                  mvwaddstr(wprogstat,1,1,"                                                    ");
                  printchan=1; printraw=0;
                  write_wtemp(decbuf); write_wmatrix(decbuf); write_wnav(decbuf); write_wpower(decbuf); write_wstatus(decbuf);
                  s=inputstring(0,35);
                  printchan=orig_printchan; printraw=orig_printraw;
                  write_wtemp(decbuf); write_wmatrix(decbuf); write_wnav(decbuf); write_wpower(decbuf); write_wstatus(decbuf);
                  while (*s==' ' || *s=='0' || *s=='x' || *s=='X') s++;
                  ch1=strtol(s,&s,16);
                  while (*s==' ' || *s==',' || *s==';' || *s=='0' || *s=='x' || *s=='X') s++;
                  ch2=strtol(s,&s,16);
                  if (ch1>=0x100 && ch1<=0x1ff && (ch2==0 || (ch2>=0x100 && ch2<=0x1ff))) do_graph(ch1,ch2);
               }
               break;
            case 'G':
               if (curblock<0) break;
               if (curblock==numblocks) break;
               {
                  byte *b;
                  b=loadblock(curblock);
                  if (b) do_graph_wod(b);
               }
               break;
            case 'f': 
               { 
                  char *newfname;
                  mvwaprintf(wprogstat,0,1,at_lab,at_val,"@1Source: @2file                                        ");
                  newfname=inputstring(0,14);
                  noecho(); curs_set(0);
                  closetlm();
                  opentlmfile(newfname);
                  redraw=1;
                  break;
               }
            case 't': 
               if (tcpsocket>=0) { closetlm(); break; }
               closetlm();
               tcpopen();
               if (tcpsocket>=0) {
                  openlogfile();
                  redraw=1;
               }
               break;
            case 'u': 
               if (udpsocket>=0) { closetlm(); break; }
               closetlm();
               udpopen();
               if (udpsocket>=0) {
                  openlogfile();
                  redraw=1;
               }
               break;
            case 'a':
               decode_A = !decode_A;
               if (decode_A) redraw=1;
               break;
            case 'e':
               decode_E = !decode_E;
               if (decode_E) redraw=1;
               break;
            case 'c':
               decode_wrongCRC = !decode_wrongCRC;
               if (decode_wrongCRC) redraw=1;
               break;
            case 'r':
               if (printchan) printchan=0;
               if (printraw) printraw=0;
               else printraw=1;
               redrawdec=1;
               break;
            case 'R':
               if (printraw) printraw=0;
               if (printchan) printchan=0;
               else printchan=1;
               redrawdec=1;
               break;
            case '?':
            case 'h':
               if (!helpvisible) {
                  preparehelp();
                  helpvisible=2;
               }
               break;
            case 'x':
               if (hexvisible) {
                  hexvisible=0;
                  do_prefresh();
               } else if (ftlm && curblock>=0) {
                  byte *b;
                  int l;
                  if (curblock<numblocks) { b=loadblock(curblock); l=512; }
                  else { b=partialblock; l=partiallen; }
                  if (!b) break;
                  showhex(b,l);
                  hexvisible=1;
               }
               break;
            case '\t':
               setview(1);
               break;
            case KEY_DOWN: case KEY_RIGHT:
               if (!ftlm) break;
               curblock++; 
               redraw=1;
               break;
            case KEY_UP: case KEY_LEFT:
               if (!ftlm) break;
               curblock--;
               redraw=1;
               break;
            case KEY_NPAGE:
               if (!ftlm) break;
               curblock+=16;
               redraw=1;
               break;
            case KEY_PPAGE:
               if (!ftlm) break;
               curblock-=16;
               redraw=1;
               break;
            case KEY_END:
               if (!ftlm) break;
               curblock=numblocks;
               redraw=1;
               break;
            case KEY_HOME:
               if (!ftlm) break;
               curblock=0;
               redraw=1;
               break;
            case KEY_SRIGHT:
            case '>':
               if (!ftlm) break;
               if (curblock==-1) break;
               {
                  byte *b;
                  int type;
                  int n;
                  if (curblock==numblocks) break;
                  b=loadblock(curblock);
                  if (!b) break;
                  type=(b[0]&0x7f);
                  n=curblock;
                  while (++n<=numblocks) {
                     int i;
                     if (n<numblocks) b=loadblock(n);
                     else b=partialblock;
                     if (!b) break;
                     i=(b[0]&0x7f);
                     if (i=='A' || i=='E') {
                        if (type==i) { curblock=n; redraw=1; break; }
                     } else {
                        if (type!='A' && type!='E') { curblock=n; redraw=1; break; }
                     }
                  }
               }
               break;
            case KEY_SLEFT:
            case '<':
               if (!ftlm) break;
               if (curblock==-1) break;
               {
                  byte *b;
                  int type;
                  int n;
                  if (curblock==numblocks) b=partialblock;
                  else b=loadblock(curblock);
                  if (!b) break;
                  type=(b[0]&0x7f);
                  n=curblock;
                  while (--n>=0) {
                     int i;
                     b=loadblock(n);
                     if (!b) break;
                     i=(b[0]&0x7f);
                     if (i=='A' || i=='E') {
                        if (type==i) { curblock=n; redraw=1; break; }
                     } else {
                        if (type!='A' && type!='E') { curblock=n; redraw=1; break; }
                     }
                  }
               }
               break;
            case KEY_RESIZE:
               closewindows();
               openwindows();
               ofsx=ofsy=0;
               keypad(wprogstat,TRUE);
               redraw=1;
               redrawdec=1;
               break;
         }
         helpvisible>>=1;
         if (curblock>=numblocks) {
            if (partialblock) curblock=numblocks;
            else curblock=numblocks-1;
         }
         if (curblock<0) curblock=0;
         if (numblocks==0 && !partialblock) { curblock=-1; redraw=0; }
      }
      if (logfileday) checklogfileday();
      if (tcpsocket>=0 && FD_ISSET(tcpsocket,&readfds)) {
         byte *bp;
         byte *prevpartialblock=partialblock;
         bp=tcpread();
         if (bp) {
            int crcok=checkcrc(bp);
            if (!crcok) bp[0]|=0x80;
            if (flog) { fwrite(bp,512,1,flog); fflush(flog); time(&timelastlogwrite); }
            if (curblock==numblocks-1 && !prevpartialblock) curblock++;
            if (curblock==numblocks) redraw=1;
            numblocks++;
         } else if (partialblock) {
            if (curblock==numblocks-1 && !prevpartialblock) curblock++;
            if (curblock==numblocks) redraw=1;
         }
      }
      if (udpsocket>=0 && FD_ISSET(udpsocket,&readfds)) {
         byte *bp;
         bp=udpread();
         if (bp) {
            int crcok=checkcrc(bp);
            if (!crcok) bp[0]|=0x80;
            if (flog) { fwrite(bp,512,1,flog); fflush(flog); time(&timelastlogwrite); }
            if (curblock==numblocks-1) curblock++;
            if (curblock==numblocks) redraw=1;
            numblocks++;
         }
      }
   } while (c!='q');
}


void usage(void)
{
   puts("AO40TLMVIEW  -  decode and view telemetry from the Amsat-Oscar 40 satellite");
   puts("Command line arguments:");
   puts(" -f filename   - read archived telemetry from file; the file should contain");
   puts("                 raw telemetry blocks (512 bytes each), without CRC");
   puts(" -t hostname:port  - read (live) telemetry via TCP/IP from the indicated source");
   puts("                     (using P3T's protocol)");
   puts(" -u hostname:port  - read (live) telemetry via UDP/IP, sent by the indicated host");
   puts("                     to the indicated port on this machine (using KA9Q's STP)");
   puts("Type '?' after starting the program for an overview of keystroke commands.");
   exit(1);
}


int sethostandport(int tcp,char *p)
/* returns 0 on success */
{
   p=strtok(p,":");
   if (!p) return 1;
   if (strlen(p)>255) return 1;
   if (tcp) strcpy(tcphostname,p); else strcpy(udphostname,p);
   p=strtok(NULL,"");
   if (!p) return 0;
   if (tcp) tcpport=atoi(p); else udpport=atoi(p);
   return 0;
}

void parse_rcfile(char *filename)
{
   char s[80],*p,*q;
   FILE *f;
   f=fopen(filename,"r");
   if (!f) return;
   while (fgets(s,80,f)) {
      p=strtok(s," \t\n\r");
      if (!p) continue;
      q=p; while (*q) *q++=tolower(*q);
      q=strtok(NULL," \t\n\r");
      if (strncmp(p,"call",4)==0) {
         if (!q) { puts("Missing callsign"); exit(1); }
         strncpy(atcallsign+1,q,sizeof(atcallsign)-1);
         atcallsign[0]='@';
         atcallsign[sizeof(atcallsign)-1]=0;
      } else if (strncmp(p,"tcp",3)==0) {
         if (!q || sethostandport(1,q)) { puts("Error setting TCP host/port"); exit(1); }
      } else if (strncmp(p,"udp",3)==0) {
         if (!q || sethostandport(0,q)) { puts("Error setting UDP host/port"); exit(1); }
      } else if (strncmp(p,"logfiledelay",3)==0) {
         if (!q) { puts("Missing delay value"); exit(1); }
         logfiledelay=atoi(q);
      }
   }
   fclose(f);
}

void parse_commandline(int argc, char **argv)
{
   int i;

   do {
      i=getopt(argc,argv,"f:t:u:");
      switch (i) {
         case 'f':
            if (!optarg) usage();
            opentlmfile(optarg);
            break;
         case 't':
            if (sethostandport(1,optarg)) usage();
            tcpopen();
            if (tcpsocket>=0) openlogfile();
            break;
         case 'u':
            if (sethostandport(0,optarg)) usage();
            udpopen();
            if (udpsocket>=0) openlogfile();
            break;
         case -1:
            break;
         default:
            usage();
            break;
      }
   } while (i!=-1);
}


void sigwinchhandler(int i)
{
   struct winsize siz;
   if (ioctl(1,TIOCGWINSZ,&siz)!=0) return;
   resizeterm(siz.ws_row,siz.ws_col);
}

void sigconthandler(int i)
{
   cbreak(); noecho(); nonl();
   curs_set(0);
   keypad(wprogstat,TRUE);
}


int main(int argc,char **argv)
{
   char *s;

   s=getenv("HOME");
   if (s) {
      int l;
      l=strlen(s);
      s=malloc(l+16);
      strcpy(s,getenv("HOME"));
      strcat(s,"/.ao40tlmviewrc");
      parse_rcfile(s);
   }
   parse_rcfile(".ao40tlmviewrc");
   parse_commandline(argc,argv);

   signal(SIGWINCH, sigwinchhandler);
   signal(SIGCONT, sigconthandler);

   initscr();

   at_border=0;
   at_subborder=0;
   at_on=A_REVERSE;
   at_off=A_DIM;
   at_broken=A_DIM;
   at_val=A_BOLD;
   at_lab=0;
   at_wrongcrc=0;
   at_graph1=at_graph2=at_graphcross=A_BOLD;
   if (has_colors()) {
      start_color();
      switch (COLOR_PAIRS) {
         default:
            init_pair(8, COLOR_WHITE, COLOR_BLACK);
            /* since some terminals (e.g. xterm) use black-on-white by default, and there seems to
               be no way to detect that this is the case (and adapt the colours to match), force
               all normal text to be white-on-black on any terminal with sufficient colours */
            at_on|=COLOR_PAIR(8);
            at_off|=COLOR_PAIR(8);
            at_broken|=COLOR_PAIR(8);
            at_val|=COLOR_PAIR(8);
            at_lab|=COLOR_PAIR(8);
            at_graphcross|=COLOR_PAIR(8);
         case 8:
            init_pair(7, COLOR_BLUE, COLOR_BLACK);
            at_matrix[5]=A_BOLD|COLOR_PAIR(7);
            if (!(termattrs() & A_DIM)) {
               /* some terminals, notably xterm, don't have dim; dark blue comes closest */
               at_off=COLOR_PAIR(7);     
               at_broken=COLOR_PAIR(7);
            }
         case 7:
            init_pair(6, COLOR_RED, COLOR_BLACK);
            at_matrix[4]=A_BOLD|COLOR_PAIR(6);
         case 6:
            init_pair(5, COLOR_MAGENTA, COLOR_BLACK);
            at_matrix[3]=A_BOLD|COLOR_PAIR(5);
         case 5:
            init_pair(4, COLOR_WHITE, COLOR_RED);
            at_wrongcrc=COLOR_PAIR(4);
         case 4:
            init_pair(3, COLOR_CYAN, COLOR_BLACK);
            at_border=COLOR_PAIR(3);
            at_matrix[2]=A_BOLD|COLOR_PAIR(3);
            at_graph1=A_BOLD|COLOR_PAIR(3);
         case 3:
            init_pair(2, COLOR_GREEN, COLOR_BLACK);
            at_subborder=COLOR_PAIR(2);
            at_matrix[1]=A_BOLD|COLOR_PAIR(2);
            at_graph2=A_BOLD|COLOR_PAIR(2);
         case 2:
            init_pair(1, COLOR_YELLOW, COLOR_BLACK);
            at_helpborder=A_BOLD|COLOR_PAIR(1);
            at_matrix[0]=A_BOLD|COLOR_PAIR(1);
         case 1:
         case 0:
            ;
      }
   }
   cbreak(); noecho(); nonl();
   curs_set(0);

   openwindows();
   keypad(wprogstat,TRUE);

   mainloop();

   endwin();
   return 0;
}

