
#define LOCAL_DEBUG
#include "debug.h"

#include "job.h"
#include <stdexcept>
#include <limits>
using namespace MYSTD;

#include "con.h"
#include "acfg.h"
#include "fileitem.h"
#include "dlcon.h"
#include "maintenance.h"

#include <sys/sendfile.h>
#include <errno.h>

// optimization on Linux; no-op for platforms that don't know it
#ifndef MSG_MORE
#define MSG_MORE 0
#endif

void DispatchAndRunMaintTask(const MYSTD::string &, int);


ssize_t sendfile_generic(int out_fd, int in_fd, off_t *offset, size_t count)
{
	char buf[4096];
	ssize_t cnt=0;
	
	if(!offset)
	{
		errno=EFAULT;
		return -1;
	}
	while(count>0)
	{
		int readcount=read(in_fd, buf, count>sizeof(buf)?sizeof(buf):count);
		if(readcount<=0)
			return readcount;
		
		*offset+=readcount;
		cnt+=readcount;
		count-=readcount;
		
		int nPos(0);
		while(nPos<readcount)
		{
			int r=write(out_fd, buf+nPos, readcount-nPos);
			if(r==0) continue; // not nice but needs to deliver it
			if(r<0)
				return r;
			nPos+=r;
		}
	}
	return cnt;
}

ssize_t linux_sendfile_with_fallback(int out_fd, int in_fd, off_t *offset, size_t count)
{
	ssize_t r=sendfile(out_fd, in_fd, offset, count);
	if(r==(ssize_t)-1 && (errno==ENOSYS || errno==EINVAL))
		return sendfile_generic(out_fd, in_fd, offset, count);
	else
		return r;
}


#ifndef __linux__
#define sendfile_linux(a,b,c,d) sendfile_generic(a,b,c,d)
#else
#define sendfile_linux(a,b,c,d) linux_sendfile_with_fallback(a,b,c,d)
#endif


job::job(const header &h, con *pParent) :
	m_filefd(-1),
	m_pParentCon(pParent),
	m_bReturnBody(true),
	m_bChunkMode(false),
	m_bLastJobCloseCon(true),
	m_bNonFatal(false),
	m_state(STATE_FRESH),
	m_ReqHeader(h),
	m_nSendPos(0),
	m_nRangeLast(std::numeric_limits<off_t>::max()),
	m_nCachedMaxSize(-1),
	m_nChunkRemainingBytes(0)
{
	ldbg("job creating, " << m_ReqHeader.frontLine << " and this: " << this);
	m_nAllDataCount=0;
}

job::~job() {
		if(m_filefd>=0)
			close(m_filefd);
		
		if(m_nAllDataCount>0 && ! m_sFileLoc.empty())
		{
			aclog::transfer(false, m_nAllDataCount, 
					m_pParentCon->m_sClientHost.c_str(), m_sFileLoc.c_str());
			
			uint64_t nIncommingCount(0);
			if(m_pItem)
				nIncommingCount=m_pItem->GetTransferCount();
			if(nIncommingCount>0)
				aclog::transfer(true, nIncommingCount, 
						m_pParentCon->m_sClientHost.c_str(), m_sFileLoc.c_str());
		}
		
		if(m_pItem)
			m_pItem->DelUser();
		
		ldbg("job destroyed");
	}

void replaceChars(string &s, const char *szBadChars, char goodChar)
{
	for(string::iterator p=s.begin();p!=s.end();p++)
		for(const char *b=szBadChars;*b;b++)
			if(*b==*p)
			{
				*p=goodChar;
				break;
			}
}

void job::PrepareDownload() {

    ldbg("job prepare");
    
    string sTmp; // for raw uri and other tasks
    tHttpUrl hi; // parsed URL
        
    acfg::tHostiVec * pBackends(NULL); // appropriate backends
    const string * psVname(NULL); // virtual name for storage pool, if available
    
    FiStatus nInitialStatus; // local file status, bit mask
    
    bool bForceFreshnessChecks; // force update of the file, i.e. on dynamic index files?

    if(!m_sErrorMsgBuffer.empty())
       	return ;
    
    switch(m_ReqHeader.type)
    {
    case(header::GET): m_bReturnBody=true; break;
    case(header::HEAD): m_bReturnBody=false; break;
    default: goto report_invpath;
    }
    
    sTmp=m_ReqHeader.get("Connection");
    if(0!=strcasecmp(sTmp.c_str(), "keep-alive") ||
    		( sTmp.empty() && 0==m_ReqHeader.frontLine.
    				compare(m_ReqHeader.frontLine.
    						length()-8, 8, "HTTP/1.1") ) )
    	m_bLastJobCloseCon = false;
    
    if(!m_ReqHeader.getUri(sTmp))
    	goto report_invpath; // invalid uri
    
    try
	{
    	// filesystem browsing attempt?
		if(stmiss != sTmp.find("..")) goto report_notallowed;

		if (0==sTmp.compare(0, 12, "apt-cacher?/"))
		sTmp.erase(0, 12);
		if (0==sTmp.compare(0, 11, "apt-cacher/"))
		sTmp.erase(11);
		
		if(!hi.SetHttpUrl(sTmp))	goto report_info;

		if(!hi.sPort.empty() && hi.sPort!="80")
			goto report_invport;

		// make a shortcut
		string & sPath=hi.sPath;

		// kill multiple slashes
		for(tStrPos pos; stmiss != (pos = sPath.find("//")); )
			sPath.erase(pos, 1);

		ldbg("input uri: "<<hi.ToString());

		tStrPos nRepNameLen=acfg::reportpage.size();
		if(nRepNameLen>0 && 0==hi.sHost.compare(0, nRepNameLen, acfg::reportpage))
		{
			m_sMaintCmd=hi.sHost;
			return;
		}
		if(!hi.sPath.empty() && hi.sPath[hi.sPath.size()-1]=='/' )
			goto report_info;
		
		m_type = m_pParentCon->rex.getFiletype(sPath.c_str());

		if ( m_type == rechecks::FILE_INVALID ) goto report_notallowed;

		// resolve to an internal location
		psVname = acfg::GetRepNameAndPathResidual(hi, sTmp);
		
		if(psVname)
			m_sFileLoc=*psVname+sPathSep+sTmp;
		else
		{
			m_sFileLoc=hi.sHost+hi.sPath;
			if(acfg::stupidfs)
			{
				// fix weird chars that we don't like in the filesystem
				replaceChars(hi.sPath, ENEMIESOFDOSFS, '_');
				replaceChars(hi.sPort, ENEMIESOFDOSFS, '_');
#ifdef WIN32
				replaceChars(hi.sPath, "/", '\\');
#endif
			}
		}
	}
    catch(out_of_range) // better safe...
    {
    	goto report_invpath;
    }
    
        
    bForceFreshnessChecks = ( ! acfg::offlinemode && m_type==rechecks::FILE_INDEX);
    m_pItem=fileitem::GetFileItem(m_sFileLoc, bForceFreshnessChecks);
    
    if(!m_pItem)
    	goto report_overload;
    
    off_t junk;
    nInitialStatus=m_pItem->GetStatus(junk, m_sErrorMsgBuffer);
    ldbg("Got initial file status: " << nInitialStatus);
    
    if(acfg::offlinemode) { // make sure there will be no problems later in SendData or prepare a user message
    	if(nInitialStatus&FIST_COMPLETE)
    		return; // perfect
    	if(m_sErrorMsgBuffer.empty())
    		goto report_offlineconf;
    	return; // either way, we are done here
    }
    dbgline;
    if(nInitialStatus & ( FIST_COMPLETE | FIST_DLRUNNING | FIST_ERROR) ) 
    	return ; // is downloadable or reportable as error
    dbgline;
    if( ! (nInitialStatus&FIST_DLRUNNING)) 
    {
    	dbgline;
    	if(!m_pParentCon->SetupDownloader())
    		goto report_overload;
    	
    	dbgline;
    	try
		{
			if(psVname && NULL != (pBackends=acfg::GetBackendVec(psVname)))
			{
				ldbg("Backends found, using them with " + sTmp);
				m_pParentCon->m_pDlClient->AddJob(bForceFreshnessChecks, m_pItem, pBackends, sTmp);
			}
			else
			{
			    if(acfg::forcemanaged)
			    	goto report_notallowed;
			    
			    m_pParentCon->m_pDlClient->AddJob(bForceFreshnessChecks, m_pItem, hi);
			}
			
		}
		catch(std::bad_alloc)
		{
			goto report_overload;
		};
	}
	return;
    
report_overload:
    m_sErrorMsgBuffer="503 Server overload, try later"; 
    return ;

report_notallowed:
    m_sErrorMsgBuffer="403 Forbidden file type or location";
    m_bNonFatal=true;
    return ;

report_offlineconf:
	m_sErrorMsgBuffer="503 Unable to download in offline mode";
	m_bNonFatal=true;
	return;

report_invpath:
    m_sErrorMsgBuffer="403 Invalid path specification"; 
    return ;

report_invport:
    m_sErrorMsgBuffer="403 Invalid or prohibited port"; 
    return ;

    report_info:
    m_sMaintCmd="/";
    return;
}

#define THROW_ERROR(x) { m_sErrorMsgBuffer=x; goto THROWN_ERROR; }
eJobResult job::SendData(int confd)
{
	if(!m_sMaintCmd.empty())
	{
		if(!m_bReturnBody)
		{
			m_bNonFatal=true;
			m_sErrorMsgBuffer="403 Use GET to see task pages";
		}
		else
		{
			DispatchAndRunMaintTask(m_sMaintCmd, confd);
			return R_DONE; // just stop and close connection
		}
	}
	
	ldbg("job::SendData");

	off_t nNewSize(0);
	FiStatus s(0);

	if (confd<0|| m_state==STATE_FAILMODE)
		return R_FAILURE; // shouldn't be here

	if (!m_sErrorMsgBuffer.empty()) 
	{  // should just report it
		m_state=STATE_FAILMODE;
	}
	else if(m_nCachedMaxSize>=0)
	{  // short circuit the fitem check -- sendfile was interrupted, just continue there 
		nNewSize=m_nCachedMaxSize;
	}
	else if(m_pItem)
	{

		/* Doesn't work anymore with file size check CACHING
		if(m_nSendPos>m_nRangeLast)
			return m_bLastJobCloseCon ? R_FAILURE : R_DONE;
		*/
		
		s=m_pItem->TrackStatusChanges(m_nSendPos, nNewSize, m_sErrorMsgBuffer);
		
		ldbg("status: "<<s);
		if (s&FIST_ERROR) {
			m_state=STATE_FAILMODE;
			m_bNonFatal=s&FIST_NONFATAL;
		}
		/*
		if(nNewSize>m_nRangeLast)
			nNewSize=m_nRangeLast+1;
		*/
	}
	else if(m_state != STATE_SEND_BUFFER)
	{
		ASSERT(!"no FileItem assigned and no sensible way to continue");
		return R_FAILURE;
	}

	while (true) // left by returning
	{
		try // for bad_alloc
		{
			switch(m_state)
			{
				case(STATE_FRESH):
				{
					ldbg("STATE_FRESH");
					header h;
					m_pItem->GetHeader(h);
					h.del("Content-Range");
					
					if(h.type == header::INVALID)
						THROW_ERROR("500 Rotten Data");
					
					//h.del("Content-Length"); // TESTCASE: Force chunked mode
					
					string con=m_ReqHeader.get("Connection");
					h.set("Connection", con.empty() ? "Keep-Alive" : con); // repeat close or keep-alive

					string sContLen=h.get("Content-Length");
					if(sContLen.empty())
					{
						if(m_ReqHeader.frontLine.find("HTTP/1.1") != 0)
							THROW_ERROR("505 HTTP version not supported for this file"); // you don't accept this, go away
						m_bChunkMode=true;
						h.set("Transfer-Encoding", "chunked");
					}
					else
					{
						// having Content-Length and the file is complete, can try some optimizations

						string ifmosince=m_ReqHeader.get("If-Modified-Since");
						string sReqRange=m_ReqHeader.get("Range");
						string sIfRange;
						// http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.27
						if(!sReqRange.empty()) 
							sIfRange=m_ReqHeader.get("If-Range");
						if(!sIfRange.empty())
							ifmosince=sIfRange;
						// If-Range this can be a date or an entity. We do not support entities, but date.
						// if the string won't match a date, it will have no effect
						
						ldbg("can send 304? ifmosince: " << ifmosince << " and last-modified" << h.get("Last-Modified"));
						
						bool bSendRange(false);
						
						if( !ifmosince.empty() )
						{
// TODO: implement header::getRef to work directly on the strings via reference, like with the map
							//////////////////////////////////////
							// Dirty workaround, local modification time is unknown (most likely an imported file)
							// but APT requested the time check... ignore it for static files
							if(h.get("Last-Modified").empty() && m_type==rechecks::FILE_PKG)
								h.set("Last-Modified", ifmosince);
								
							if(h.get("Last-Modified") == ifmosince )
							{
								if(!sReqRange.empty()) // still needs to send data
								{
									// extra check to avoid timeouts since the server may need time to get there
									if(s&FIST_COMPLETE) 
										bSendRange=true;
								}
								else
								{
									m_bNonFatal=true;
									THROW_ERROR("304 Not Modified");
								}
							}
						}
						else {
							if(!sReqRange.empty())
							{
								// extra check to avoid timeouts since the server may need time to get there
								if(s&FIST_COMPLETE)
									bSendRange=true;
							}
						}

						/*
						 * Range: bytes=453291-
						 * ...
						 * Content-Length: 7271829
						 * Content-Range: bytes 453291-7725119/7725120
						 */
						if(bSendRange)
						{
							int r=sscanf(sReqRange.c_str(), "bytes=%ld-%ld", &m_nSendPos, &m_nRangeLast);
							if(r>0)
							{
								long nSize=atol(sContLen.c_str());
								if(r==1)
									m_nRangeLast=nSize-1;
								
								char buf[100];
								sprintf(buf, "%ld", m_nRangeLast-m_nSendPos+1);
								h.set("Content-Length", buf);
								sprintf(buf, "bytes %ld-%ld/%s", m_nSendPos, m_nRangeLast, sContLen.c_str());
								h.set("Content-Range", buf);
								h.frontLine="HTTP/1.1 206 Partial Response";
							}
						}
					}

					h.fix();
					m_sPrependBuf=h.as_string(true);
					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_HEADER_SENT;
					continue;
				}
				case(STATE_HEADER_SENT):
				{
					ldbg("STATE_HEADER_SENT");
					if( ! m_bReturnBody) return R_DONE;

					m_filefd=m_pItem->GetFileFd();
					if(m_filefd<0)
					THROW_ERROR("503 IO error");

					m_state=m_bChunkMode ? STATE_SEND_CHUNK_HEADER : STATE_SEND_PLAIN_DATA;
					continue;
				}
				case(STATE_SEND_PLAIN_DATA):
				{
					ldbg("STATE_SEND_PLAIN_DATA untill " << nNewSize);
					
					if ( s & FIST_EOF )
						return m_bLastJobCloseCon ? R_FAILURE : R_DONE;
					
					if(nNewSize>m_nRangeLast)
						nNewSize=m_nRangeLast+1;
							
					//ldbg("~sendfile: on "<< m_nSendPos << " up to : " << nNewSize-m_nSendPos );
					int n=sendfile_linux(confd, m_filefd, &m_nSendPos, nNewSize-m_nSendPos);
					//ldbg("~sendfile: new m_nSendPos: " << m_nSendPos);

					 // sendfile was not done? Store that size to skip fitem polling next time, or reset it
					m_nCachedMaxSize = (nNewSize>m_nSendPos) ? nNewSize : -1;
					
					if(m_nSendPos>m_nRangeLast)
						return m_bLastJobCloseCon ? R_FAILURE : R_DONE;
					
					if(n<0)
						THROW_ERROR("400 Client error");
					m_nAllDataCount+=n;
					
					return R_AGAIN;
				}
				case(STATE_SEND_CHUNK_HEADER):
				{

					m_nChunkRemainingBytes=nNewSize-m_nSendPos;

					ldbg("STATE_SEND_CHUNK_HEADER for " << m_nChunkRemainingBytes);
					// if not on EOF then the chunk must have remaining size (otherwise the state would have been changed)
					ASSERT( 0==(s&FIST_EOF) || 0==m_nChunkRemainingBytes);

					char buf[23];
					snprintf(buf, 23, "%x\r\n", m_nChunkRemainingBytes);
					m_sPrependBuf=buf;

					if(m_nChunkRemainingBytes==0)
						m_sPrependBuf+="\r\n";

					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_SEND_CHUNK_DATA;
					continue;
				}
				case(STATE_SEND_CHUNK_DATA):
				{
					ldbg("STATE_SEND_CHUNK_DATA");

					if(m_nChunkRemainingBytes==0)
						return m_bLastJobCloseCon ? R_FAILURE : R_DONE; // yeah...

					int n=sendfile_linux(confd, m_filefd, &m_nSendPos, m_nChunkRemainingBytes);
					if(n<0)
						THROW_ERROR("400 Client error");
					m_nChunkRemainingBytes-=n;
					m_nAllDataCount+=n;
					if(m_nChunkRemainingBytes<=0)
					{ // append final newline
						m_sPrependBuf="\r\n";
						m_state=STATE_SEND_BUFFER;
						m_backstate=STATE_SEND_CHUNK_HEADER;
						continue;
					}
					return R_AGAIN;
				}
				case(STATE_SEND_BUFFER):
				{
					// special DELICATE state. Something fails here only when the connection is dead
					// and this state will be used by the error handler to send feedback.
					// ... or we send the report page
					try
					{
						ldbg("prebuf sende: "<< m_sPrependBuf);
						int r=send(confd, m_sPrependBuf.data(), m_sPrependBuf.length(), MSG_MORE);
						if (r<0)
						{
							if (errno==EAGAIN)
							{
								return R_AGAIN;
							}
							return R_FAILURE;
						}
						m_nAllDataCount+=r;
						m_sPrependBuf.erase(0, r);
						if(m_sPrependBuf.empty())
						{
							m_state=m_backstate;
							continue;
						}
					}
					catch(...)
					{
						return R_FAILURE;
					}
					return R_AGAIN;
				}
				
				case (STATE_FAILMODE):
					goto THROWN_ERROR;
				
				case (STATE_ERRORCONT):
					return m_bLastJobCloseCon ? R_FAILURE : R_DONE;
				
				case(STATE_NOWAYOUT):
				default:
					return R_FAILURE;
					
			}
			
			continue;
			
			
	THROWN_ERROR:
			if(m_nAllDataCount==0) // don't send errors when data header is underway
			{
				header h;
				h.frontLine="HTTP/1.1 ";
				h.frontLine+=m_sErrorMsgBuffer;
				h.set("Connection", (m_bLastJobCloseCon||!m_bNonFatal) ? "close" : "Keep-Alive");
				h.fix();
				h.del("Content-Type");
				h.del("Transfer-Encoding");
				m_sPrependBuf=h.as_string(true);
				m_backstate = m_bNonFatal ? STATE_ERRORCONT : STATE_NOWAYOUT;
				m_state=STATE_SEND_BUFFER;

				continue;
			}
			return R_FAILURE;
		}
		catch(bad_alloc) {
			// TODO: report memory failure?
			return R_FAILURE;
		}
		ASSERT(!"UNREACHED");
	}
}
