# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman, Enno Cramer
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.006;
use strict;
use warnings;

package ArchWay::Widget::Inventory::Entry;

use Glib qw(TRUE FALSE);
use Gtk2;

use base qw(Gtk2::Frame);

use Arch::Inventory qw(:category :type);
use POSIX qw(strftime);

use constant FIELDS => qw(path size type mode category mtime id);
use constant LABELS => ('Full Path', 'Size', 'Type', 'Mode', 'Category', 'Last Modified', 'Inventory ID');
use constant SIZES  => (1, 1, 1, 1, 1, 1, 3);

my %TYPES = (
	FILE()      => 'File',
	DIRECTORY() => 'Directory',
	SYMLINK()   => 'Symlink',
);

my %CATEGORIES = (
	TREE()         => 'Arch Tree Root',
	SOURCE()       => 'Source',
	PRECIOUS()     => 'Precious',
	BACKUP()       => 'Backup',
	JUNK()         => 'Junk',
	UNRECOGNIZED() => 'Unrecognized',
);

sub new ($$) {
	my $class     = shift;
	my $inventory = shift;

	my $self = bless Gtk2::Frame->new(' File Details '), $class;

	$self->_init();
	$self->show($inventory) if ($inventory);

	return $self;
}

sub _init ($) {
	my $self = shift;

	$self->set_border_width(3);

	my $table = Gtk2::Table->new(4, 4);
	$table->set_row_spacings(2);
	$table->set_col_spacings(10);
	$table->set_border_width(3);

	my $x = 0;
	my $y = 0;

	for (my $i = 0; $i < scalar(FIELDS); ++$i) {
		my $lbl = Gtk2::Label->new();
		$lbl->set_markup('<b>'.(LABELS)[$i].':</b>');
		$lbl->set_alignment(0.0, 0.0);
		$table->attach($lbl, $x, $x+1, $y, $y+1, [ 'fill' ], [ 'fill' ], 0, 0);

		++$x;

		my $val = Gtk2::Label->new();
		$val->set_alignment(0.0, 0.0);
		$val->set_selectable(TRUE);
		$table->attach($val, $x, $x+(SIZES)[$i], $y, $y+1, [ 'expand', 'fill' ], [ 'fill' ], 0, 0);

		$x += (SIZES)[$i];
		
		if ($x >= 4) {
			++$y;
			$x = 0;
		}

		$self->{(FIELDS)[$i]} = $val;
	}

	$self->add($table);
}

sub show ($$) {
	my $self      = shift;
	my $inventory = shift;

	if (defined $inventory) {
		$self->{type}->set_text($TYPES{$inventory->{type}});

		$self->{path}->set_text(
			$inventory->{path} . (
				exists $inventory->{symlink}
				? ' -> ' . $inventory->{symlink}
				: ""
			)
		);

		$self->{category}->set_text($CATEGORIES{$inventory->{category}});

		if (defined $inventory->{id}) {
			$self->{id}->set_text($inventory->{id});
		} else {
			$self->{id}->set_markup('<i>none</i>');
		}

		if (exists $inventory->{stat}) {
			$self->{mode}->set_text(mode_string($inventory->{stat}->[2]));
			$self->{size}->set_text($inventory->{stat}->[7]);
			$self->{mtime}->set_text(
				strftime(
					'%c',
					localtime($inventory->{stat}->[9])
				)
			);
		} else {
			foreach my $name (qw(mode size mtime)) {
				$self->{$name}->set_markup('<i>unknown</i>');
			}
		}
	} else {
		foreach my $field (FIELDS) {
			$self->{$field}->set_text('');
		}
	}
}

sub mode_string ($) {
	my $mode = shift;

	my @modes = qw(--- --x -w- -wx r-- r-x rw- rwx);

	return join(' ', map { $modes[($mode >> 3*$_) & 7] } (2, 1, 0));
}

1;

__END__
