/* $Id: ArkModelState.h,v 1.15 2003/03/20 17:23:25 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_MODELSTATE_H
#define ARK_MODELSTATE_H

#include <Ark/ArkModel.h>

namespace Ark
{

   class ARK_DLL_API ModelState
   {
      public:
	 enum PlayMode { ONCE, LOOP };
	 
      private:
	 ModelPtr m_Model;
	 
      public:
	 Vector3 m_Position;
	 Quat m_Rotation;
	 Matrix44 m_Matrix;

	 /// Skin
	 SkinPtr m_Skin;

	 /// Bone matrices.
	 Matrix44 *m_BoneMatrices;

	 /// Animation stuff
	 PlayMode m_Mode;	 
	 SequencePtr m_Sequence;
	 String m_SequenceName;
	 bool m_SequenceChanged;
	 scalar m_Time;

	 /// Current position of controlled bones
	 scalar m_Controller[4];
	 
	 /// Mouth position
	 scalar m_Mouth;
	 bool  m_Changed;
	 
      public:
	 /// Create a new model state using 'model'.
	 ModelState (Model *model);
	 
	 /// Create an empty model state
	 ModelState ();

	 /// Destroy the data associated with this model state.
	 ~ModelState();

	 /**
	  * Set the skin of the model.
	  */
	 void SetSkin (Skin *skin);

	 /// Set the model, with a default skin.
	 bool SetModel (Model *model);

	 /**
	  * Set the model with the given name and skin. The 'name' should be
	  * in the form skin@model where 'skin' is the name of the skin file
	  * and 'model' the name of the model.
	  */
	 bool SetModel (Cache *cache, const String &name);

	 /// Return the current model.
	 Model *GetModel()
	    {return &*m_Model;} 

	 /// Clear the state. Set controllers and animation to default, etc..
	 void Clear ();
	 
	 /// Controllers
	 float SetController (int iController, float flValue);
	 float SetMouth (float flValue);
	 
	 /// Update the model state
	 void Update (scalar dt);

	 /// Compute the state matrix using the current position/rotation
	 void ComputeMatrix ();
	 
	 /**
	  * Play an animation sequence.
	  *  \param sequence is the name of the sequence
	  *  \pm is the PlayMode (either ONCE either LOOP).
	  */
	 bool Play (PlayMode pm, const String &sequence);
	 
	 /// Get the bounding box of the model in its current state.
	 BBox ExtractBbox () const;
	 
	 /// Serialization (FIXME use operators << and >> to serialize)
	 bool Read (Stream &file);
	 bool Write (WriteStream &file);

	 /**
	  * Render the model with the given position, animation frame, as 
	  * specified in this modelstate.
	  */
	 void Render (Renderer &renderer)
	 {
	    m_Model->Render (renderer, *this);
	 }
      };

}

#endif
