/* $Id: ArkVBuffer.cpp,v 1.13 2003/03/12 23:45:31 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
 #include <config.h>
#endif

#include <Ark/ArkVBuffer.h>

namespace Ark
{
   
   /**
    * This creates an empty vertex buffer. You have to call the
    * SetFormat() and Reserve() functions before you're really able to
    * access vertex data.
    */
   VertexBuffer::VertexBuffer ()
   {}

   /**
    * Deletes the vertex buffer.
    */
   VertexBuffer::~VertexBuffer()
   {}

   /**
    * Sets the format of the vertex buffer. This function must be
    * called before any access function since it computes the size
    * each vertex will take in memory.
    * Be aware that if you call this function after any access or
    * reserve function, all data will be lost.
    *
    * \param format Mask of or'ed VB_HAS_xx values specifying which data
    *        will be included for each vertex in the buffer.
    * \param desired_vertex_size If the computed size is lower than this
    *        value then the vertex buffer is padded with zero's in order
    *        to reach the desired size.
    */
    void
    VertexBuffer::SetFormat(int format, unsigned int desired_size)
    {
       m_VertexSize = 0;
       m_Format = 0;
       m_UVOffset = 0;
       
       if (format & VB_HAS_UV0)
       {
	  m_IndexUV0 = m_VertexSize;
	  m_Format |= VB_HAS_UV0;
	  m_VertexSize += sizeof (Vector2);
       }
       
       if (format & VB_HAS_UV1)
       {
	  m_IndexUV1 = m_VertexSize;
	  m_Format |= VB_HAS_UV1;
	  m_VertexSize += sizeof (Vector2);
       }
       
       if (format & VB_HAS_COLOR)
       {
	  m_IndexColor4 = m_VertexSize;
	  m_Format |= VB_HAS_COLOR;
	  m_VertexSize += sizeof (RGBA);
       }
       
       if (format & VB_HAS_NORMAL)
       {
	  m_IndexNormal = m_VertexSize;
	  m_Format |= VB_HAS_NORMAL;
	  m_VertexSize += sizeof (Vector3);
       }
       
       if (format & VB_HAS_COORD)
       {
	  m_IndexCoord = m_VertexSize;
	  m_Format |= VB_HAS_COORD;
	  m_VertexSize +=  sizeof (Vector3);
       }
       
       if (m_VertexSize < desired_size)
	  m_VertexSize = desired_size;
 
       Resize(0);
    }

   /**
    * Reserve \c n vertices for the vertex buffer. This should be called
    * before calling any access function.
    */
   void
   VertexBuffer::Resize(unsigned int n)
   {
      m_Buffer.resize (n * m_VertexSize);
      m_Size = n;
   }

   /**
    * Remaps the vertices with the given order, ie indice n will be
    * mapped to newIndices[n].
    */
   void
   VertexBuffer::Remap (std::vector<size_t> newIndices)
   {
      assert (!"FIXME: doesn't work (VertexBuffer::Remap function)");
      assert (newIndices.size() == Size());

      VertexBuffer oldvb;
      oldvb.SetFormat (Format());
      oldvb.m_Size = m_Size;
      oldvb.m_Buffer = m_Buffer;

      for (size_t i = 0; i < newIndices.size(); i++)
      {
	 if (m_Format & VB_HAS_COORD)
	    Coord(newIndices[i]) = oldvb.Coord(i); 
	 if (m_Format & VB_HAS_NORMAL)
	    Normal(newIndices[i]) = oldvb.Normal(i);
	 if (m_Format & VB_HAS_COLOR)
	    Color4(newIndices[i]) = oldvb.Color4(i);
	 if (m_Format & VB_HAS_UV0)
	    UV0(newIndices[i]) = oldvb.UV0(i);
	 if (m_Format & VB_HAS_UV1)
	    UV1(newIndices[i]) = oldvb.UV1(i);
      }
   }


#define DEFINE_ACCESS(type, name)                                       \
   type& VertexBuffer::name (unsigned int n) {                          \
      assert ((0 <= n) && (n < Size()));                                \
      return * ( (type*) &m_Buffer[n * m_VertexSize + m_Index##name]);  \
   }                                                                    \
   const type& VertexBuffer::name (unsigned int n) const {              \
      assert ((0 <= n) && (n < Size()));                                \
      return *((const type*)&m_Buffer[n*m_VertexSize + m_Index##name]); \
   }                                                                    \
  type* VertexBuffer::name##P () const {                                \
     return (type *) &m_Buffer[m_Index##name];                          \
  }

   DEFINE_ACCESS(Vector2, UV0);
   DEFINE_ACCESS(Vector2, UV1);
   DEFINE_ACCESS(Vector3, Coord);
   DEFINE_ACCESS(Vector3, Normal);
   DEFINE_ACCESS(RGBA, Color4);
}
