/*****************************************************************************/
/*	asmounter - AfterStep All-Purpose Mounter			     */
/*	Version 0.3 						             */
/*	By David Taylor							     */		
/*	dtaylor@cit.nepean.uws.edu.au					     */
/*	http://www.cit.nepean.uws.edu.au/~dtaylor			     */
/*      								     */
/*	This program is based on asmodem, asmail, and wmmount.		     */
/*	It is intended to be an all-purpose mounter for use in		     */
/*	the AfterStep Window Manager.					     */
/*      								     */
/*****************************************************************************/


#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#ifdef HAVE_SYS_VFS_H
  #include <sys/vfs.h>
#else
  #include <sys/statvfs.h>
#endif
#include <X11/Xlib.h>
#include <X11/xpm.h>
#include <X11/extensions/shape.h>


#include "XPM/tile.xpm"
#include "XPM/mask.xpm"
#include "XPM/hdd_mounted.xpm"
#include "XPM/hdd_unmounted.xpm"
#include "XPM/fd_mounted.xpm"
#include "XPM/fd_unmounted.xpm"
#include "XPM/cd_mounted.xpm"
#include "XPM/cd_unmounted.xpm"
#include "XPM/zip_mounted.xpm"
#include "XPM/zip_unmounted.xpm"


#define NAME		"asmounter"
#define CLASS		"ASMounter"
#define BACKCLR		"#282828"
#define SYS_RC_FILE	"/usr/lib/asmounter/sys.asmounter"
#define USER_RC_FILE	".asmounter"
#define MOUNTS		"/proc/mounts"
#define MAX_TICS	50
#define MAX_STR_SIZE	256
#define MAX_MPOINTS	20
#define MAX_ARGS	10
#define TRUE		1
#define FALSE		0


typedef unsigned int bool;


typedef struct _XpmIcon {
	Pixmap pixmap;
	Pixmap mask;
	XpmAttributes attributes;
} XpmIcon;


XpmIcon Current, 
        CDmounted, 
	CDunmounted, 
	FDmounted, 
	FDunmounted, 
	HDDmounted,
	HDDunmounted,
	ZIPmounted,
	ZIPunmounted;


Pixmap disp, back, mask, tile;


typedef struct MPInfo{
    char name[MAX_STR_SIZE];
    char mpoint[MAX_STR_SIZE];
    char type[8];
    bool shwcap;
} MPInfo;


MPInfo *mpoint[MAX_MPOINTS];
#ifdef HAVE_SYS_VFS_H
  struct statfs sfs;
#else
  struct statvfs sfs;
#endif
bool 	cmounted;
int	mpoints = 0;
int 	curmnt  = 0;
int 	ccap    = 0;


void	Help(void);
void	Version(void);
void	ParseCmdLine(int argc, char *argv[]);
void	ParseRCFile(void);
void	GenerateRCFile(void);
void	GetMPType (MPInfo *);
void	GetMPLocation (MPInfo *);
void	GetMPName (MPInfo *);
void	GetMPShowCap (MPInfo *);
void	WriteMPtoRC (FILE *, MPInfo *);
void	ExecCommand(char *command);
void	ExecRelatedCmd(void);
void	MountOrUmount(void);
void	Cleanup(void);
void	MainLoop(void);
void	InitFonts(void);
void	InitXWin(int argc, char *argv[]);
void	GetXPM(void);
void	RedrawWindow(XpmIcon *);
int	IsMounted (char *);
Pixel	GetColor(char *);


Display	*Disp;	 
Window	Root;
int	screen;
Window	Win[2];
GC	WinGC;
Atom	deleteWin;
bool	finished  = FALSE;
bool	withdrawn = FALSE;
bool	shape = FALSE;
bool	cycle = FALSE;
bool	debug = FALSE;
char	*geometry = NULL;
char	*display  = NULL;
char	mountcmd[MAX_STR_SIZE]	= "mount %s &";
char	umountcmd[MAX_STR_SIZE]	= "umount %s &";
char	execmd[MAX_STR_SIZE]	= "xterm -e mc %s";
char	txtfont[MAX_STR_SIZE]	= "-*-helvetica-bold-r-*-*-10-*-*-*-*-*-*-*";
char	txtcolor[MAX_STR_SIZE]	= "gray90";
char	capfont[MAX_STR_SIZE]	= "-*-helvetica-medium-r-*-*-10-*-*-*-*-*-*-*";
char	capcolor[MAX_STR_SIZE]	= "gray90";
char	bckcol[MAX_STR_SIZE]	= BACKCLR;
int	txtx1, txty1, txtx2, txty2 = 40;
int 	activeWin = 0;
Font	 gcf1, gcf2;
unsigned long gcc1, gcc2;



int main(int argc,char *argv[])
{       	
	ParseCmdLine(argc, argv);      
	ParseRCFile();
	if (mpoints) {
		InitXWin(argc, argv);
		InitFonts();
		GetXPM();
		MainLoop();
		Cleanup();
	}
	return 0;
}



void Help()
{       
	fprintf(stderr,"asmounter - Version 0.3\n");
	fprintf(stderr,"Copyright (C) 1999 - David Taylor\n"); 
	fprintf(stderr,"<dtaylor@cit.nepean.uws.edu.au>\n");
	fprintf(stderr,"usage:  asmounter [-options ...] \n");
	fprintf(stderr,"options:\n");
	fprintf(stderr,"        -w | -withdrawn       withdrawn mode (WindowMaker users)\n");
	fprintf(stderr,"        -s | -shaped          transparent edge mode (AfterStep users)\n");
	fprintf(stderr,"        -D | -debug           debug mode (extra info for bug reporting)\n");
	fprintf(stderr,"        -d | -display name    specify the display name\n");
	fprintf(stderr,"        -C | -create          create a new rc file\n");
	fprintf(stderr,"	-c | -cycle           cycle mountpoints whilst idle\n");
	fprintf(stderr,"        -v | -version         display the version number\n");
        fprintf(stderr,"        -h | -help            display this help screen\n");
	fprintf(stderr,"\n");       
	exit(EXIT_SUCCESS);
}



void Version(void)
{
	fprintf(stderr,"asmounter - Version 0.3\n");
	fprintf(stderr,"\n");       
	exit(EXIT_SUCCESS);
}



void CreateWin(Window *win)
{
	XClassHint classHint;
	*win = XCreateSimpleWindow(Disp, Root, 10, 10, 64, 64, 0, 0, 0);
	classHint.res_name = NAME;
	classHint.res_class = CLASS;
	XSetClassHint(Disp, *win, &classHint);
}



void InitXWin(int argc, char *argv[])
{
	XWMHints hints;
	XSizeHints shints;
	XGCValues gcv;
	unsigned long gcm;
	XpmAttributes pixatt;
	XpmColorSymbol ledcols[1]={{"back_color", NULL, 0}};
	unsigned int dummy = 0;
	unsigned int wx = 0, wy = 0;

	/* Open display */
	if (NULL == (Disp = XOpenDisplay(display))) { 
		fprintf(stderr,"asmounter: can't open display %s\n", XDisplayName(display));
		exit (EXIT_FAILURE); 
	} 

	deleteWin = XInternAtom(Disp, "WM_DELETE_WINDOW", False);
        screen = DefaultScreen(Disp);
	Root = RootWindow(Disp, screen);

	CreateWin(&Win[0]);
	CreateWin(&Win[1]);

	hints.window_group = Win[0];
	hints.icon_window  = Win[1];
	shints.min_width  = 64;
	shints.min_height = 64;
	shints.max_width  = 64;
	shints.max_height = 64;
	shints.x = 0;
	shints.y = 0;
	shints.flags = PMinSize | PMaxSize | PPosition;            

	XWMGeometry(Disp, screen, geometry, NULL, 1, &shints, &shints.x, 
	            &shints.y, &shints.width, &shints.height, &dummy);

	if (withdrawn) {
		shints.flags |= USPosition;
		hints.initial_state = WithdrawnState;
		hints.flags = WindowGroupHint | StateHint | IconWindowHint;
		activeWin=0;
	} else {
	        hints.initial_state = NormalState;
	        hints.flags = WindowGroupHint | StateHint;
	        activeWin = 0;
	}

	XSetWMHints(Disp, Win[0], &hints);
	XSetWMNormalHints(Disp, Win[0], &shints);
	XSetCommand(Disp, Win[0], argv, argc);
	XStoreName(Disp, Win[0], NAME);
	XSetIconName(Disp, Win[0], NAME);
	XSetWMProtocols(Disp, Win[activeWin], &deleteWin, 1);

	gcm = GCForeground|GCBackground|GCGraphicsExposures;
	gcv.graphics_exposures = False;
	WinGC = XCreateGC(Disp, Root, gcm, &gcv);

	ledcols[0].pixel=GetColor(bckcol);
	pixatt.numsymbols=1;
	pixatt.colorsymbols=ledcols;
	pixatt.exactColors=FALSE;
	pixatt.closeness=40000;
	pixatt.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness;
	XpmCreatePixmapFromData(Disp, Root, mask_xpm, &back, &mask, &pixatt);
	XpmCreatePixmapFromData(Disp, Root, tile_xpm, &tile, NULL, &pixatt);
	disp = XCreatePixmap(Disp, Root, 64, 64, DefaultDepth(Disp,DefaultScreen(Disp)));

	/* Install mask or copy background tile */
	if( withdrawn || shape)
		XShapeCombineMask(Disp, Win[activeWin], ShapeBounding, 0, 0, mask, ShapeSet);
	else
		XCopyArea(Disp, tile, disp, WinGC, 0, 0, 64, 64, 0, 0);

	/* Copy background */
	XSetClipMask(Disp, WinGC, mask);
	XCopyArea(Disp, back, disp, WinGC, 0, 0, 64, 64, 0, 0);
	XSetClipMask(Disp, WinGC, None);
		
	XSelectInput(Disp, Win[activeWin], ExposureMask | ButtonPressMask | ButtonReleaseMask);
	XMapWindow(Disp, Win[0]);
	if (geometry) {
		if (sscanf(geometry, "+%d+%d", &wx, &wy) != 2) {
			fprintf(stderr, "asmounter: Bad geometry string\n");
			exit(1);
		}
		XMoveWindow(Disp, Win[0], wx, wy);
	}
}



void InitFonts(void)
{
	XFontStruct *xfs;

	if (debug) {
		fprintf(stderr, "asmounter: Initialising Fonts...\n");
		fprintf(stderr, "           %s\n", txtfont);
		fprintf(stderr, "           %s\n", txtcolor);
		fprintf(stderr, "           %s\n", capfont);
		fprintf(stderr, "           %s\n", capcolor);
	}

	xfs = XLoadQueryFont(Disp, txtfont);
 	if (xfs == 0) {
		fprintf(stderr, "asmounter: Unable to load text font\n");
		exit(EXIT_FAILURE);                                      
	}
 	gcf1 = xfs->fid;
	gcc1 = GetColor(txtcolor);
	if (gcc1 == 0) {
		fprintf(stderr, "asmounter: Unable to load text colour\n");
		exit(EXIT_FAILURE);
	}
	txtx1 = 6-xfs->min_bounds.lbearing;
	txty1 = 5+xfs->ascent;

	xfs = XLoadQueryFont(Disp, capfont);
	if (xfs == 0) {
		fprintf(stderr, "Unable to load cap font\n");
		exit(EXIT_FAILURE);
	}
	gcf2 = xfs->fid;
	gcc2 = GetColor(capcolor);
	if (gcc2 == 0) {
		fprintf(stderr, "Unable to load cap color\n");
		exit(EXIT_FAILURE);
	}
	txtx2 = 6-xfs->min_bounds.lbearing;
	txty2 = txty2+xfs->ascent;
	
	if (debug) fprintf(stderr, "           ... Done\n");

}



void Cleanup(void)
{
	if (debug) fprintf(stderr, "asmounter: Cleaning Up... ");

	while (mpoints>0){
		free(mpoint[mpoints-1]);
		mpoints--;
	}

	XFreeGC(Disp, WinGC);
	XUnloadFont(Disp, gcf1);
	XUnloadFont(Disp, gcf2);
	XFreePixmap(Disp, disp);
	XFreePixmap(Disp, mask);
	XFreePixmap(Disp, tile);
	XFreePixmap(Disp, back);
	XDestroyWindow(Disp, Win[0]);
	XDestroyWindow(Disp, Win[1]);
	XCloseDisplay(Disp);				
	
	if (debug) fprintf(stderr, "Done\n");
}
											


void MainLoop()
{
	XEvent	Event;
	int	Idle = 0;
	
	/* Main loop */
	while(!finished)
	{
		/* Check events */
		while (XPending(Disp))
		{
			XNextEvent(Disp,&Event);
			switch(Event.type)
			{		     
				case Expose:
					if(Event.xexpose.count == 0)
						RedrawWindow(&Current); 
					break;		     
				case ButtonPress:
					Idle = 0;
					if (Event.xbutton.button == 1)
						/* Cycle mount points */
						curmnt = (curmnt+1) % mpoints;
					if (Event.xbutton.button == 2)
						/* Execute command */
						ExecRelatedCmd();
					if (Event.xbutton.button == 3)
						/* Toggle fs mounting */
						MountOrUmount();
					break;			 
				case ClientMessage:
					finished = (Event.xclient.data.l[0] == deleteWin);
			}
		}
		if ( (Idle++ > MAX_TICS) && cycle ) {
			Idle = 0;
			curmnt = (curmnt+1) % mpoints;
		}
		cmounted = IsMounted(mpoint[curmnt]->mpoint);
		if (cmounted) {
			if (0 == strcmp(mpoint[curmnt]->type, "cd"))
				Current = CDmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "fd"))
				Current = FDmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "hdd"))
				Current = HDDmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "zip"))
				Current = ZIPmounted;
			if (mpoint[curmnt]->shwcap) {
				statfs(mpoint[curmnt]->mpoint,&sfs);
				ccap = (int)((sfs.f_blocks-sfs.f_bfree)*100.0/(sfs.f_blocks-sfs.f_bfree+sfs.f_bavail)+.5);
			}
		} else {
			if (0 == strcmp(mpoint[curmnt]->type, "cd"))
				Current = CDunmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "fd"))
				Current = FDunmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "hdd"))
				Current = HDDunmounted;
			else if (0 == strcmp(mpoint[curmnt]->type, "zip"))
				Current = ZIPunmounted;
		}
		RedrawWindow(&Current);
		XFlush(Disp);
		usleep(50000L);
	}
	
}



void GetXPM(void)
{
	XWindowAttributes Attributes;
	int ret;
	
	if (debug) fprintf(stderr, "asmounter: Loading Pixmaps... ");

	XGetWindowAttributes(Disp,Root,&Attributes);		

	HDDmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, hdd_mounted_xpm, &HDDmounted.pixmap, 
				      &HDDmounted.mask, &HDDmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	HDDunmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, hdd_unmounted_xpm, &HDDunmounted.pixmap, 
				      &HDDunmounted.mask, &HDDunmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	FDmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, fd_mounted_xpm, &FDmounted.pixmap, 
				      &FDmounted.mask, &FDmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	FDunmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, fd_unmounted_xpm, &FDunmounted.pixmap, 
				      &FDunmounted.mask, &FDunmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	CDmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, cd_mounted_xpm, &CDmounted.pixmap, 
				      &CDmounted.mask, &CDmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	CDunmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, cd_unmounted_xpm, &CDunmounted.pixmap, 
				      &CDunmounted.mask, &CDunmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	ZIPmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, zip_mounted_xpm, &ZIPmounted.pixmap, 
				      &ZIPmounted.mask, &ZIPmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	ZIPunmounted.attributes.valuemask |= (XpmReturnPixels|XpmReturnExtensions);
	ret = XpmCreatePixmapFromData(Disp, Root, zip_unmounted_xpm, &ZIPunmounted.pixmap, 
				      &ZIPunmounted.mask, &ZIPunmounted.attributes);
	if(ret != XpmSuccess)
	{
		fprintf(stderr, "asmounter: not enough free color cells\n");
		exit(EXIT_FAILURE);
	}

	
	if (debug) fprintf(stderr, "Done\n");

	Current = HDDmounted;

}



void RedrawWindow(XpmIcon *Icon)
{
	XEvent xev;

	XCopyArea(Disp, disp, Win[activeWin], WinGC, 0, 0, 64, 64, 0, 0);
	while(XCheckTypedEvent(Disp, Expose, &xev));	

	XCopyArea(Disp,Icon->pixmap,disp,WinGC,
		  0,0,Icon->attributes.width, Icon->attributes.height,0,0);

	XSetFont(Disp, WinGC, gcf1);
	XSetForeground(Disp, WinGC, gcc1);
	XDrawString(Disp, disp, WinGC, txtx1, txty1, (char *)mpoint[curmnt]->name, strlen(mpoint[curmnt]->name)); 

	if(cmounted && mpoint[curmnt]->shwcap){
		char capstr[20] = "\0";
		sprintf(capstr,"%d%%",ccap);
		XSetFont(Disp, WinGC, gcf2);
		XSetForeground(Disp, WinGC, gcc2);
		XDrawString(Disp, disp, WinGC, txtx2, txty2, (char *)capstr, strlen(capstr));
	}
}



Pixel GetColor(char *ColorName)
{
	XColor Color;
	XWindowAttributes Attributes;
	
	XGetWindowAttributes(Disp,Root,&Attributes);
	Color.pixel = 0;
	
	if (!XParseColor (Disp, Attributes.colormap, ColorName, &Color)) 
		fprintf(stderr,"asmounter: can't parse %s\n", ColorName);
	else if(!XAllocColor (Disp, Attributes.colormap, &Color)) 
		fprintf(stderr,"asmounter: can't allocate %s\n", ColorName);       
	
	return Color.pixel;
}



void ParseCmdLine(int argc, char *argv[])
{
	int i;       
	
	for(i = 1; i < argc; i++) {
		if ( (0 == strcmp(argv[i],"-h")) || (0 == strcmp(argv[i],"-help")))
			Help();
		if ( (0 == strcmp(argv[i],"-v")) || (0 == strcmp(argv[i],"-version")))
			Version();
		if ( (0 == strcmp(argv[i],"-C")) || (0 == strcmp(argv[i],"-create")))
			GenerateRCFile();
		if ( (0 == strcmp(argv[i],"-w")) || (0 == strcmp(argv[i],"-withdrawn")))
			withdrawn = TRUE;
		if ( (0 == strcmp(argv[i],"-s")) || (0 == strcmp(argv[i],"-shaped")) )
			shape = TRUE;
		if ( (0 == strcmp(argv[i],"-D")) || (0 == strcmp(argv[i],"-debug")) )
			debug = TRUE;
		if ( (0 == strcmp(argv[i],"-c")) || (0 == strcmp(argv[i],"-cycle")) )
			cycle = TRUE;
		if ( (0 == strcmp(argv[i],"-g")) || (0 == strcmp(argv[i],"-geometry")) ) {
			if (i < argc-1) geometry = argv[++i];
			continue;
		}
		if ( (0 == strcmp(argv[i],"-d")) || (0 == strcmp(argv[i],"-display")) ) {
			if (i < argc-1) display = argv[++i];
			continue;
		}
	}
}



void ParseRCFile (void)
/* Searches for local or system-wide resource file */
/* and, if found, loads the settings from this.    */
{
	FILE *rcfile;
	char rcfname[MAX_STR_SIZE];
	char buf[MAX_STR_SIZE];
	XpmAttributes pixatt;
	
	if (debug) fprintf(stderr, "asmounter: Parsing Resource File... ");

	sprintf(rcfname,"%s/%s",getenv("HOME"),USER_RC_FILE);
	if (NULL == (rcfile = fopen(rcfname, "r"))) {
		fprintf(stderr, "asmounter: Unable to open file %s\n", rcfname);
		if (NULL == (rcfile = fopen(SYS_RC_FILE, "r"))) {
			fprintf(stderr, "asmounter: Unable to open file %s\n", SYS_RC_FILE);
			return;
		}
	}
	
	pixatt.exactColors = FALSE;
	pixatt.closeness = 40000;
	pixatt.valuemask = XpmExactColors | XpmCloseness;	
	while (NULL != fgets(buf, MAX_STR_SIZE, rcfile)) {
		buf[MAX_STR_SIZE-1] = '\0';
		if (strstr(buf,"mountcmd=")==(char *)buf)
			sprintf(mountcmd,"%s",buf+strlen("mountcmd="));
		if (strstr(buf,"umountcmd=")==(char *)buf)
			sprintf(umountcmd,"%s",buf+strlen("umountcmd="));
		if (strstr(buf,"execmd=")==(char *)buf)
			sprintf(execmd,"%s",buf+strlen("execmd="));
         	if (strstr(buf,"mountname ")==(char *)buf){
			mpoint[mpoints]=(MPInfo *)malloc(sizeof(MPInfo));
			sprintf(mpoint[mpoints]->name,"%s",buf+strlen("mountname "));
			fscanf(rcfile," mountpoint %s",mpoint[mpoints]->mpoint);
			fscanf(rcfile," type %s",mpoint[mpoints]->type);
			fscanf(rcfile," showcapacity %s",buf);
			mpoint[mpoints]->shwcap=(strcasecmp(buf,"true")==0);
			mpoints++;
		}
	}

	if (debug) fprintf(stderr, "Done\n");

}



void GenerateRCFile (void)
/* The purpose of this procedure is to allow the */
/* user to generate a generic resource file for  */
/* asmounter.  I'm trying to be user-friendly... */
{
	FILE *rcfile;
	MPInfo mp;
	char rcfname[MAX_STR_SIZE];
	char cont = 'y';

	printf("WARNING: This will remove your existing RC file!\n");
	printf("Continue (y/n)? ");
	scanf("%c%*c", &cont);
	
	if (cont != 'y') return;
	
	sprintf(rcfname,"%s/%s",getenv("HOME"),USER_RC_FILE);
	if (NULL == (rcfile = fopen(rcfname, "w"))) {
		fprintf(stderr, "asmounter: Unable to open file %s\n", rcfname);
		return;
	}
	
	fprintf(rcfile, "# asmounter - Copyright (C) 1999 David Taylor\n");
	fprintf(rcfile, "# \n");
	fprintf(rcfile, "# This file was generated using the -c option\n");
	fprintf(rcfile, "# of asmounter.\n\n\n");

	do {
		GetMPType(&mp);
		GetMPLocation(&mp);
		GetMPName(&mp);
		GetMPShowCap(&mp);
		WriteMPtoRC(rcfile, &mp);
		printf("\nAnother mount point (y/n)? ");
		scanf("%c%*c", &cont);
	} while (cont == 'y');

	fprintf(rcfile, "# EOF\n");

	fclose(rcfile);
	
}



void WriteMPtoRC (FILE *file, MPInfo *mp)
{
	fprintf(file, "# Mount Point #\n");
	fprintf(file, "mountname %s\n", mp->name);
	fprintf(file, "\tmountpoint %s\n", mp->mpoint);
	fprintf(file, "\ttype %s\n", mp->type);
	if (mp->shwcap)
		fprintf(file, "\tshowcapacity TRUE\n");
	else
		fprintf(file, "\tshowcapacity FALSE\n");
	fprintf(file, "\n\n");
}



void GetMPType (MPInfo *mp)
{
	int dev;
	
	printf("\n\n");
	printf("Mount Point\n");
	printf("-----------\n\n");
	printf("Types of Devices: \n");
	printf("1. Hard Disk Drive\n");
	printf("2. Floppy Disk Drive\n");
	printf("3. CD-ROM Drive\n");
	printf("4. ZIP Drive\n");
	printf("Selection [1-4]: ");

	scanf("%d%*c", &dev);

	switch (dev) {
		case 1  :  strcpy(mp->type,"hdd");
		           break;
		case 2  :  strcpy(mp->type,"fd");
		           break;
		case 3  :  strcpy(mp->type,"cd");
		           break;
		case 4  :  strcpy(mp->type,"zip");
		           break;
		default :  strcpy(mp->type,"hdd");
	}
}



void GetMPLocation (MPInfo *mp)
{
	printf("Mount Point location (eg /mnt/zip): ");
	fgets(mp->mpoint, MAX_STR_SIZE, stdin);
}



void GetMPName (MPInfo *mp)
{
	printf("Mount Point name (eg ZIP): ");
	fgets(mp->name, MAX_STR_SIZE, stdin);
}



void GetMPShowCap (MPInfo *mp)
{
	char showcap = 'y';
	printf("Show Capacity (y/n)? ");
	scanf("%c%*c", &showcap);
	mp->shwcap = (showcap == 'y');
}



void ExecCommand (char *command)
/* ExecCommand is used to execute the mount/unmount */
/* command - or any command, for that matter.       */
{
	char *argv[MAX_ARGS]={"sh","-c",command,0};
	int pid=fork();
	int status;
	if (pid==0) {
		execv("/bin/sh", argv);
		exit(127);
	}
	do {
		if (waitpid(pid,&status,0)!=-1)
			return;
	} while(1);
}



int IsMounted (char *mntpoint)
/* IsMounted is used to determine whether or not */
/* a mount point is already mounted.             */
{
	FILE *fd;
	char buf[MAX_STR_SIZE] = "\0";
	bool found = FALSE;
	
	/* Open MOUNTS file to search */
	if (NULL == (fd = fopen(MOUNTS, "r"))) {
		fprintf(stderr, "asmounter: cannot open %s\n", MOUNTS);
		fprintf(stderr, "/proc filesystem must be enabled in the kernel\n");
		return(-1);
	}

	/* Search for mountpoint ... */
	while (NULL != fgets(buf, MAX_STR_SIZE, fd)) {
		found = (NULL != strstr(buf,mntpoint));
		if (found) break;
	}
     
	fclose(fd);
	return(found);
}



void MountOrUmount(void)
/* This code toggles the mount status.  If    */
/* the mount point is already mounted it will */
/* be unmounted, and vice-versa.              */
{
	char cmd[MAX_STR_SIZE];
	
	if (cmounted)
		sprintf(cmd,umountcmd,mpoint[curmnt]->mpoint);
	else
		sprintf(cmd,mountcmd,mpoint[curmnt]->mpoint);
	
	if (debug)
		fprintf(stderr, "asmounter: Executing \"%s\"\n", cmd);

	ExecCommand(cmd);
}



void ExecRelatedCmd(void)
/* Executes the user defined command ("execmd=") */
/* for the current mount point.                  */
{
	char cmd[MAX_STR_SIZE];
	
	if (cmounted)
		sprintf(cmd,execmd,mpoint[curmnt]->mpoint);
	
	if (debug)
		fprintf(stderr, "asmounter: Executing \"%s\"\n", cmd);

	ExecCommand(cmd);
}



