// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __CTemplateInstance_h__
#define __CTemplateInstance_h__

// template instance info

#include "Puma/DeducedArgument.h"
#include "Puma/CTemplateInfo.h"
#include "Puma/CTranslationUnit.h"

namespace Puma {


class CT_SimpleName;

class CTemplateInstance {
protected:
  CTemplateInfo           *_TemplateInfo;
  CT_SimpleName           *_PointOfInstantiation;
  CScopeInfo              *_PointOfInstantiationScope;
  Array<DeducedArgument*>  _DeducedArgs;
  Array<DeducedArgument*>  _InstantiationArgs;
  CTranslationUnit        *_InstanceSrc;
  bool                     _Pseudo;
  bool                     _Delete;

public:
  CTemplateInstance ();
  ~CTemplateInstance ();

  CTemplateInfo *Template () const;        // the template parameter info
  void Template (CTemplateInfo *);

  CT_SimpleName *PointOfInstantiation () const;
  CScopeInfo *PointOfInstantiationScope () const;
  void PointOfInstantiation (CT_SimpleName *, CScopeInfo *);
  
  void TranslationUnit (CTranslationUnit *);
  CTranslationUnit *TranslationUnit () const;
  
  void canDelete ();

  void isPseudoInstance (bool);
  bool isPseudoInstance () const;
  
  void addDeducedArg (DeducedArgument *);
  void addInstantiationArg (DeducedArgument *);
  DeducedArgument *DeducedArg (unsigned) const;
  DeducedArgument *InstantiationArg (unsigned) const;
  unsigned DeducedArgs () const;
  unsigned InstantiationArgs () const;
  void clearDeducedArgs ();
};

inline CTemplateInstance::CTemplateInstance () :
  _TemplateInfo ((CTemplateInfo*)0),
  _PointOfInstantiation ((CT_SimpleName*)0),
  _InstanceSrc (0),
  _Pseudo (false),
  _Delete (false)
 {}
inline CTemplateInstance::~CTemplateInstance ()
 { if (_TemplateInfo && _TemplateInfo->isSpecialization ())
     for (unsigned i = 0; i < InstantiationArgs (); i++) delete InstantiationArg (i); 
   for (unsigned i = 0; i < DeducedArgs (); i++) delete DeducedArg (i); 
   if (_Delete && _InstanceSrc) { 
     Unit *u = _InstanceSrc->unit (); 
     delete _InstanceSrc; 
     if (u) delete u;
   } 
 }

inline void CTemplateInstance::TranslationUnit (CTranslationUnit *tu)
 { _InstanceSrc = tu; }
inline CTranslationUnit *CTemplateInstance::TranslationUnit () const
 { return _InstanceSrc; } 

inline void CTemplateInstance::canDelete () 
 { _Delete = true; }

inline void CTemplateInstance::isPseudoInstance (bool v)
 { _Pseudo = v; }
inline bool CTemplateInstance::isPseudoInstance () const
 { return _Pseudo; }
  
inline void CTemplateInstance::addDeducedArg (DeducedArgument *a)
 { _DeducedArgs.append (a); }
inline DeducedArgument *CTemplateInstance::DeducedArg (unsigned i) const
 { return _DeducedArgs.lookup (i); }
inline unsigned CTemplateInstance::DeducedArgs () const
 { return _DeducedArgs.length (); }
inline void CTemplateInstance::addInstantiationArg (DeducedArgument *a)
 { _InstantiationArgs.append (a); }
inline DeducedArgument *CTemplateInstance::InstantiationArg (unsigned i) const
 { return _InstantiationArgs.lookup (i); }
inline unsigned CTemplateInstance::InstantiationArgs () const
 { return _InstantiationArgs.length (); }
inline void CTemplateInstance::clearDeducedArgs () 
 { _DeducedArgs.reset (); _InstantiationArgs.reset (); }

inline CTemplateInfo *CTemplateInstance::Template () const
 { return _TemplateInfo; }
inline void CTemplateInstance::Template (CTemplateInfo *info) 
 { _TemplateInfo = info; }

inline CT_SimpleName *CTemplateInstance::PointOfInstantiation () const
 { return _PointOfInstantiation; }
inline CScopeInfo *CTemplateInstance::PointOfInstantiationScope () const
 { return _PointOfInstantiationScope; }
inline void CTemplateInstance::PointOfInstantiation (CT_SimpleName *poi,
  CScopeInfo *pois)
 { _PointOfInstantiation = poi;
   _PointOfInstantiationScope = pois;
 }


} // namespace Puma

#endif /* __CTemplateInstance_h__ */
