#include "includes.h"

/*
** This file contains the definitions about which counters have to be 
** read from /proc
**                                                      
** New counters can be added to the tables in this file as follows:
**       Define if the counter-contents is static (e.g. a kernel-param)
**       and only has to be read once, or if the counter dynamically  
**       changes (to be read per sample). In case of a static counter
**       add its specification to the 'oncedef' table and for dynamic
**       counter, add its specification to the 'sampdef' table.      
**                                                                
** ---------------------------------------------------------------------
** Author:      Gerlof Langeveld - AT Computing, Nijmegen, Holland
** E-mail:      gerlof@ATComputing.nl
** Date:        Januar  1995
** LINUX-port:  Februar 1999
**
** $Log: fetchdef.c,v $
** Revision 1.22  2004/07/08 12:52:03  gerlof
** Adapted for kernel version 2.6.
** Support of new disk-names.
**
** Revision 1.21  2002/09/03 12:06:16  gerlof
** Corrected number of open files and maximum number of open inodes.
** 
**
** Revision 1.20  2001/03/16 09:37:18  root
** Reimplement FTP/HTTP counters.
**
** Revision 1.19  2001/03/14 14:04:53  gerlof
** Changed numbers of disks 2.2
**
** Revision 1.18  2001/03/14 11:08:01  gerlof
** TTY statistics added.
**
** Revision 1.17  2001/03/12 14:57:48  gerlof
** IPv6 counters.
**
** Revision 1.16  2001/03/09 15:08:55  gerlof
** Support IP version 6 counters (/proc/net/snmp6).
**
** Revision 1.15  2001/02/26 15:34:53  gerlof
** Gather load-averages from /proc/loadavg.
**
** Revision 1.14  2001/02/26 15:28:14  gerlof
** Use counters of type 'double' i.s.o. long for HTTP and FTP.
**
** Revision 1.13  2000/11/07 09:23:07  gerlof
** Support for modified per-partition statistics.
**
** Revision 1.12  2000/08/09 11:27:32  gerlof
** Changes needed for per-partition counters (larger disk-names).
**
** Revision 1.11  2000/06/29 05:52:06  gerlof
** Bug: the file opened for NFS statistics was never closed, so after 1024
** samples, atsadc ran out of file-descriptors.
**
** Revision 1.10  1999/10/18 12:58:53  gerlof
** Separate counters for FTP-input and -output.
**
** Revision 1.9  1999/09/16 09:27:07  gerlof
** Minor code-restyling.
**
** Revision 1.8  1999/09/01 07:12:09  gerlof
** Ported to Alpha.
**
** Revision 1.7  1999/08/31 09:34:29  gerlof
** Implement support for WEB-servers.
** Porting to version 2.0.
**
** Revision 1.6  1999/08/27 14:05:32  gerlof
** Added NFS-statistics.
**
** Revision 1.5  1999/08/26 07:47:19  gerlof
** Code-cleanup, removed all counters obtained from /dev/kmem (obtained via
** /proc now) and gathered new counters (a.o. socket-stats).
**
** Revision 1.4  1999/05/18 08:31:19  gerlof
** Back-port from version 2.2 to 2.0.
**
** Revision 1.3  1999/05/11 09:07:33  gerlof
** Add support for disk-partition statistics and kernel-tables.
**
** Revision 1.2  1999/05/10 10:13:19  gerlof
** Gather statistics about kernel-table occupation (parinfo).
**
** Revision 1.1  1999/05/05 11:38:33  gerlof
** Initial revision
**
** This program is free software; you can redistribute it and/or modify it
** under the terms of the GNU General Public License as published by the
** Free Software Foundation; either version 2, or (at your option) any
** later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
** See the GNU General Public License for more details.
*/

static const char rcsid[] = "$Id: fetchdef.c,v 1.22 2004/07/08 12:52:03 gerlof Exp $";

static int	isrealdisk(char *, char *, int);

#define	EQ	0

/*************************************************************/
/* specific routines which are defined above                 */
/*************************************************************/
#define	MAXCNT	64

/*
** Gather general system statistics from /proc and store in binary form.
*/
void
genget(struct fetchdef *fdef, struct osrel *osrel, struct genstat *gs)
{
	register int	nr, i;
	FILE 		*fp;
	char		linebuf[8192], nam[64];
	count_t		cnts[MAXCNT];

	memset(gs, 0, sizeof(struct genstat));

	/*
	** gather various general statistics from the file /proc/stat and
	** store them in binary form in a proprietary structure.
	*/
	if ( (fp = fopen("/proc/stat", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
		     nr = sscanf(linebuf,
			"%s   %lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld %lld "
			"%lld %lld %lld %lld %lld %lld\n",
				nam,
				&cnts[0],  &cnts[1],  &cnts[2],  &cnts[3],
				&cnts[4],  &cnts[5],  &cnts[6],  &cnts[7],
				&cnts[8],  &cnts[9],  &cnts[10], &cnts[11],
				&cnts[12], &cnts[13], &cnts[14], &cnts[15],
				&cnts[16], &cnts[17], &cnts[18], &cnts[19],
				&cnts[20], &cnts[21], &cnts[22], &cnts[23],
				&cnts[24], &cnts[25], &cnts[26], &cnts[27],
				&cnts[28], &cnts[29], &cnts[30], &cnts[31],
				&cnts[32], &cnts[33], &cnts[34], &cnts[35],
				&cnts[36], &cnts[37], &cnts[38], &cnts[39],
				&cnts[40], &cnts[41], &cnts[42], &cnts[43],
				&cnts[44], &cnts[45], &cnts[46], &cnts[47],
				&cnts[48], &cnts[49], &cnts[50], &cnts[51],
				&cnts[52], &cnts[53], &cnts[54], &cnts[55],
				&cnts[56], &cnts[57], &cnts[58], &cnts[59],
				&cnts[60], &cnts[61], &cnts[62], &cnts[63]);

			if (nr < 1)		/* wrong format --> skip */
				continue;

			if ( strcmp("cpu", nam) == EQ)
			{
				gs->cpu_user	= cnts[0];
				gs->cpu_nice	= cnts[1];
				gs->cpu_system	= cnts[2];
				gs->cpu_idle	= cnts[3];

				if (osrel->rel >= 2 && osrel->vers >= 6)
				{
					gs->cpu_wait	= cnts[4];
					gs->cpu_irq	= cnts[5];
					gs->cpu_sirq	= cnts[6];
				}

				continue;
			}

			if ( nam[0] == 'c' && nam[1] == 'p' && nam[2] == 'u' )
			{
				int cpunr = atoi( &nam[3] );

				if (cpunr < MAXCPU && cpunr >= 0)
				{
					gs->per_cpu_user[cpunr]	   = cnts[0];
					gs->per_cpu_nice[cpunr]	   = cnts[1];
					gs->per_cpu_system[cpunr]  = cnts[2];
					gs->per_cpu_idle[cpunr]	   = cnts[3];

					if (osrel->rel  >= 2 &&
					    osrel->vers >= 6)
					{
					    gs->per_cpu_wait[cpunr] = cnts[4];
					    gs->per_cpu_irq [cpunr] = cnts[5];
					    gs->per_cpu_sirq[cpunr] = cnts[6];
					}
				}

				continue;
			}

			if ( strcmp("ctxt", nam) == EQ)
			{
				gs->context_swtch	= cnts[0];

				continue;
			}

			if ( strcmp("intr", nam) == EQ)
			{
				int max = (nr-2) < MAXIRQ ? (nr-2) : MAXIRQ;

				/* first counter is the sum of all counters */
				for (i=1; i < max; i++)
					gs->irqs[i-1] = cnts[i];

				continue;
			}

			if ( strcmp("page", nam) == EQ)
			{
				gs->pgpgin	= cnts[0];
				gs->pgpgout	= cnts[1];
				continue;
			}

			if ( strcmp("swap", nam) == EQ)
			{
				gs->pswpin	= cnts[0];
				gs->pswpout	= cnts[1];
				continue;
			}

			/*
			** handle disk-statistics
			** only for kernel-version 2.4
			*/
			if ( memcmp("disk_io", nam, 7) == EQ)
			{
				char 	*p;
				count_t	major, minor, tot, rio, rblk, wio, wblk;
				int	dkix;

				/*
				** redo the parsing due to format:
				**  (major,minor):(tot,rio,rbk,wio,wbk) ....
				**
				** search for stats of first disk
				*/
				if ( (p = strchr(linebuf, ' ')) == NULL)
					continue;

				while (*++p == ' ');

				dkix = 0;

				while ((nr=sscanf(p,
					   "(%lld,%lld):(%lld,%lld,%lld,"
					   "%lld,%lld) ",
						&major, &minor, &tot,
						&rio, &rblk, &wio, &wblk)) ==7)
				{ 
					gs->dk_drive_maj [dkix]	= major;
					gs->dk_drive_min [dkix]	= minor;
					gs->dk_drive_tot [dkix]	= tot;
					gs->dk_drive_rio [dkix]	= rio;
					gs->dk_drive_wio [dkix]	= wio;
					gs->dk_drive_rblk[dkix]	= rblk;
					gs->dk_drive_wblk[dkix]	= wblk;

					dkix++;

					if (dkix >= MAXDISK)
						break;

					if ( (p = strchr(p, ' ')) == NULL)
						break;

					while (*++p == ' ');
				}

				continue;
			}

			/*
			** handle disk-statistics kernel-version < 2.4
			*/
			if ( strcmp("disk", nam) == EQ)
			{
				int max = (nr-1) < MAXDISK ? (nr-1) : MAXDISK;
	
				for (i=0; i < max; i++)
				{
					gs->dk_drive_min[i]	= i+1;
					gs->dk_drive_tot[i]	= cnts[i];
				}

				continue;
			}

			if ( strcmp("disk_rio", nam) == EQ)
			{
				int max = (nr-1) < MAXDISK ? (nr-1) : MAXDISK;

				for (i=0; i < max; i++)
					gs->dk_drive_rio[i]	= cnts[i];

				continue;
			}

			if ( strcmp("disk_wio", nam) == EQ)
			{
				int max = (nr-1) < MAXDISK ? (nr-1) : MAXDISK;

				for (i=0; i < max; i++)
					gs->dk_drive_wio[i]	= cnts[i];

				continue;
			}

			if ( strcmp("disk_rblk", nam) == EQ)
			{
				int max = (nr-1) < MAXDISK ? (nr-1) : MAXDISK;

				for (i=0; i < max; i++)
					gs->dk_drive_rblk[i]	= cnts[i];

				continue;
			}

			if ( strcmp("disk_wblk", nam) == EQ)
			{
				int max = (nr-1) < MAXDISK ? (nr-1) : MAXDISK;

				for (i=0; i < max; i++)
					gs->dk_drive_wblk[i]	= cnts[i];

				continue;
			}
		}

		fclose(fp);
	}

	/*
	** gather counters of per-cpu interrupt-statistics
	*/
	if ( (fp = fopen("/proc/interrupts", "r")) != NULL)
	{
		/* skip header-line */
		(void)  fgets(linebuf, sizeof(linebuf), fp);

		/* read counter-lines */
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
		    	int irq, max;

		    	nr = sscanf(linebuf,
			            "%d:  %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld\n",
				&irq,
				&cnts[0],  &cnts[1],  &cnts[2],  &cnts[3],
				&cnts[4],  &cnts[5],  &cnts[6],  &cnts[7],
				&cnts[8],  &cnts[9],  &cnts[10], &cnts[11],
				&cnts[12], &cnts[13], &cnts[14], &cnts[15],
				&cnts[16], &cnts[17], &cnts[18], &cnts[19],
				&cnts[20], &cnts[21], &cnts[22], &cnts[23],
				&cnts[24], &cnts[25], &cnts[26], &cnts[27],
				&cnts[28], &cnts[29], &cnts[30], &cnts[31],
				&cnts[32], &cnts[33], &cnts[34], &cnts[35],
				&cnts[36], &cnts[37], &cnts[38], &cnts[39],
				&cnts[40], &cnts[41], &cnts[42], &cnts[43],
				&cnts[44], &cnts[45], &cnts[46], &cnts[47],
				&cnts[48], &cnts[49], &cnts[50], &cnts[51],
				&cnts[52], &cnts[53], &cnts[54], &cnts[55],
				&cnts[56], &cnts[57], &cnts[58], &cnts[59],
				&cnts[60], &cnts[61], &cnts[62], &cnts[63]);

			if (nr < 2)
				continue;

			max = (nr-1) < MAXCPU ? (nr-1) : MAXCPU;

			for (i=0; i < max; i++)
				gs->per_irqs[i][irq] = cnts[i];
		}

		fclose(fp);
	}

	/*
	** gather load-averages
	*/
	if ( (fp = fopen("/proc/loadavg", "r")) != NULL)
	{
		if ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
		    	nr = sscanf(linebuf, "%f %f %f %lld/%lld",
				&(gs->loadavg1),
				&(gs->loadavg5),
				&(gs->loadavg15),
				&(gs->nrrun),
				&(gs->nrproc));
		}

		fclose(fp);
	}

	/*
	** gather page-/swap-counters kernel version >= 2.6
	*/
	if ( (fp = fopen("/proc/vmstat", "r")) != NULL)
	{
		int	nrfields = 4;
		count_t	cnt;

		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL &&
								nrfields > 0)
		{
			nr = sscanf(linebuf, "%s %lld", nam, &cnt);

			if (nr < 2)		/* invalid line --> skip */
				continue;

			if ( strcmp("pgpgin", nam) == EQ)
			{
                                gs->pgpgin = cnt;
				nrfields--;
			}

			if ( strcmp("pgpgout", nam) == EQ)
			{
                                gs->pgpgout = cnt;
				nrfields--;
			}

			if ( strcmp("pswpin", nam) == EQ)
			{
                                gs->pswpin = cnt;
				nrfields--;
			}

			if ( strcmp("pswpout", nam) == EQ)
			{
                                gs->pswpout = cnt;
				nrfields--;
			}
		}

		fclose(fp);
	}
}

/*
** Gather memory-related statistics from /proc and store in binary form.
*/
void
memget(struct fetchdef *fdef, struct osrel *osrel, struct memstat *mi)
{
	register int	nr, nrfields=7;
	FILE 		*fp;
	char		linebuf[1024];
	char		itsname[32];
	count_t		cnt;

	memset(mi, 0, sizeof(struct memstat));

	if ( (fp = fopen("/proc/meminfo", "r")) == NULL)
		return;

	while ( fgets(linebuf, sizeof(linebuf), fp) != NULL && nrfields > 0)
	{
		nr = sscanf(linebuf, "%s %lld", itsname, &cnt);

		if (nr < 2)		/* invalid line --> skip */
			continue;

		if ( strcmp("MemTotal:", itsname) == EQ)
		{
			mi->memtot = cnt;
			nrfields--;
		}

		if ( strcmp("MemFree:", itsname) == EQ)
		{
			mi->memfree = cnt;
			nrfields--;
		}

		if ( strcmp("Buffers:", itsname) == EQ)
		{
			mi->membuf = cnt;
			nrfields--;
		}

		if ( strcmp("Cached:", itsname) == EQ)
		{
			mi->memcache = cnt;
			nrfields--;
		}

		if ( strcmp("Slab:", itsname) == EQ)
		{
			mi->memslab = cnt;
			nrfields--;
		}

		if ( strcmp("SwapTotal:", itsname) == EQ)
		{
			mi->swptot = cnt;
			nrfields--;
		}

		if ( strcmp("SwapFree:", itsname) == EQ)
		{
			mi->swpfree = cnt;
			nrfields--;
		}
	}

	fclose(fp);
}


/*
** Gather network-interface statistics from /proc and store in binary form.
*/
int		getifstats(struct osrel *, struct ifstat *, int);

void
ifinit(struct fetchdef *fdef, struct osrel *osrel)
{
	int		numnet=0;
	struct ifstat 	ifs[MAXNET];

	numnet = getifstats(osrel, ifs, MAXNET);

	/*
	** current number of instances defined;
	** however, this number might be changed dynamically, so
	** reserve some spare entries (round up to unit of 8);
	** unused entries will be filled with binary zeroes during
	** each sample
	*/
	fdef->cdef.kinst = ((numnet>>3) + 1) << 3;

	if (fdef->cdef.kinst > MAXNET)
		fdef->cdef.kinst = MAXNET;
}

void
ifget(struct fetchdef *fdef, struct osrel *osrel, struct ifstat *ifsbuf)
{
	memset(ifsbuf, 0, sizeof(struct ifstat) * fdef->cdef.kinst);

	(void) getifstats(osrel, ifsbuf, fdef->cdef.kinst);
}

int
getifstats(struct osrel *osrel, struct ifstat *ifsp, int maxif)
{
	register int	i=0, nr;
	FILE 		*fp;
	char		linebuf[1024], *cp;

	if ( (fp = fopen("/proc/net/dev", "r")) == NULL)
		return(0);

	while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
	{
		if (i >= maxif)
			break;


		if ( (cp = strchr(linebuf, ':')) != NULL)
			*cp = ' ';	/* substitute ':' by space */

		if (osrel->rel == 2 && osrel->vers < 2)
		{
			memset(ifsp, 0, sizeof(struct ifstat));

	                nr = sscanf(linebuf,
				    "%s   %lld %lld %lld %lld %lld %lld %lld "
				    "%lld %lld %lld %lld\n",
                                          ifsp->name,
                                        &(ifsp->rpack),
                                        &(ifsp->rerrs),
                                        &(ifsp->rdrop),
                                        &(ifsp->rfifo),
                                        &(ifsp->rframe),
                                        &(ifsp->spack),
                                        &(ifsp->serrs),
                                        &(ifsp->sdrop),
                                        &(ifsp->sfifo),
                                        &(ifsp->scollis),
                                        &(ifsp->scarrier));

			if (nr < 12)	/* skip header & lines without stats */
			{
				ifsp->name[0] = 0;
				continue;
			}
		}
		else
		{
			nr = sscanf(linebuf,
			 	    "%s   %lld %lld %lld %lld %lld %lld %lld "
			            "%lld %lld %lld %lld %lld %lld %lld %lld "
			            "%lld\n",
					  ifsp->name,
					&(ifsp->rbyte),
					&(ifsp->rpack),
					&(ifsp->rerrs),
					&(ifsp->rdrop),
					&(ifsp->rfifo),
					&(ifsp->rframe),
					&(ifsp->rcompr),
					&(ifsp->rmultic),
					&(ifsp->sbyte),
					&(ifsp->spack),
					&(ifsp->serrs),
					&(ifsp->sdrop),
					&(ifsp->sfifo),
					&(ifsp->scollis),
					&(ifsp->scarrier),
					&(ifsp->scompr));

			if (nr < 16)	/* skip header & lines without stats */
			{
				ifsp->name[0] = 0;
				continue;
			}
		}

		i++;
		ifsp++;
	}

	fclose(fp);

	return i;
}

/*
** Gather ip-icmp-tcp-udp statistics from /proc and store in binary form
** in the snmp mib-structure where the counters are derived from.
*/
struct ipv6_mib		ipv6_tmp;
struct icmpv6_mib	icmpv6_tmp;
struct udp_mib		udpv6_tmp;

struct v6tab {
	char 	*nam;
	count_t *val;
} v6tab[] = {
    {"Ip6InReceives",		&ipv6_tmp.Ip6InReceives,                     },
    {"Ip6InHdrErrors",		&ipv6_tmp.Ip6InHdrErrors,                    },
    {"Ip6InTooBigErrors",	&ipv6_tmp.Ip6InTooBigErrors,                 },
    {"Ip6InNoRoutes",		&ipv6_tmp.Ip6InNoRoutes,                     },
    {"Ip6InAddrErrors",		&ipv6_tmp.Ip6InAddrErrors,                   },
    {"Ip6InUnknownProtos",	&ipv6_tmp.Ip6InUnknownProtos,                },
    {"Ip6InTruncatedPkts",	&ipv6_tmp.Ip6InTruncatedPkts,                },
    {"Ip6InDiscards",		&ipv6_tmp.Ip6InDiscards,                     },
    {"Ip6InDelivers",		&ipv6_tmp.Ip6InDelivers,                     },
    {"Ip6OutForwDatagrams",	&ipv6_tmp.Ip6OutForwDatagrams,               },
    {"Ip6OutRequests",		&ipv6_tmp.Ip6OutRequests,                    },
    {"Ip6OutDiscards",		&ipv6_tmp.Ip6OutDiscards,                    },
    {"Ip6OutNoRoutes",		&ipv6_tmp.Ip6OutNoRoutes,                    },
    {"Ip6ReasmTimeout",		&ipv6_tmp.Ip6ReasmTimeout,                   },
    {"Ip6ReasmReqds",		&ipv6_tmp.Ip6ReasmReqds,                     },
    {"Ip6ReasmOKs",		&ipv6_tmp.Ip6ReasmOKs,                       },
    {"Ip6ReasmFails",		&ipv6_tmp.Ip6ReasmFails,                     },
    {"Ip6FragOKs",		&ipv6_tmp.Ip6FragOKs,                        },
    {"Ip6FragFails",		&ipv6_tmp.Ip6FragFails,                      },
    {"Ip6FragCreates",		&ipv6_tmp.Ip6FragCreates,                    },
    {"Ip6InMcastPkts",		&ipv6_tmp.Ip6InMcastPkts,                    },
    {"Ip6OutMcastPkts",		&ipv6_tmp.Ip6OutMcastPkts,                   },
    
    {"Icmp6InMsgs",		&icmpv6_tmp.Icmp6InMsgs,                     },
    {"Icmp6InErrors",		&icmpv6_tmp.Icmp6InErrors,                   },
    {"Icmp6InDestUnreachs",	&icmpv6_tmp.Icmp6InDestUnreachs,             },
    {"Icmp6InPktTooBigs",	&icmpv6_tmp.Icmp6InPktTooBigs,               },
    {"Icmp6InTimeExcds",	&icmpv6_tmp.Icmp6InTimeExcds,                },
    {"Icmp6InParmProblems",	&icmpv6_tmp.Icmp6InParmProblems,             },
    {"Icmp6InEchos",		&icmpv6_tmp.Icmp6InEchos,                    },
    {"Icmp6InEchoReplies",	&icmpv6_tmp.Icmp6InEchoReplies,              },
    {"Icmp6InGroupMembQueries",	&icmpv6_tmp.Icmp6InGroupMembQueries,         },
    {"Icmp6InGroupMembResponses",
				&icmpv6_tmp.Icmp6InGroupMembResponses,       },
    {"Icmp6InGroupMembReductions",
				&icmpv6_tmp.Icmp6InGroupMembReductions,      },
    {"Icmp6InRouterSolicits",	&icmpv6_tmp.Icmp6InRouterSolicits,           },
    {"Icmp6InRouterAdvertisements",
				&icmpv6_tmp.Icmp6InRouterAdvertisements,     },
    {"Icmp6InNeighborSolicits",	&icmpv6_tmp.Icmp6InNeighborSolicits,         },
    {"Icmp6InNeighborAdvertisements",
				&icmpv6_tmp.Icmp6InNeighborAdvertisements,   },
    {"Icmp6InRedirects",	&icmpv6_tmp.Icmp6InRedirects,                },
    {"Icmp6OutMsgs",		&icmpv6_tmp.Icmp6OutMsgs,                    },
    {"Icmp6OutDestUnreachs",	&icmpv6_tmp.Icmp6OutDestUnreachs,            },
    {"Icmp6OutPktTooBigs",	&icmpv6_tmp.Icmp6OutPktTooBigs,              },
    {"Icmp6OutTimeExcds",	&icmpv6_tmp.Icmp6OutTimeExcds,               },
    {"Icmp6OutParmProblems",	&icmpv6_tmp.Icmp6OutParmProblems,            },
    {"Icmp6OutEchoReplies",	&icmpv6_tmp.Icmp6OutEchoReplies,             },
    {"Icmp6OutRouterSolicits",	&icmpv6_tmp.Icmp6OutRouterSolicits,          },
    {"Icmp6OutNeighborSolicits",&icmpv6_tmp.Icmp6OutNeighborSolicits,        },
    {"Icmp6OutNeighborAdvertisements",
				&icmpv6_tmp.Icmp6OutNeighborAdvertisements,  },
    {"Icmp6OutRedirects",	&icmpv6_tmp.Icmp6OutRedirects,               },
    {"Icmp6OutGroupMembResponses",
				&icmpv6_tmp.Icmp6OutGroupMembResponses,      },
    {"Icmp6OutGroupMembReductions",
				&icmpv6_tmp.Icmp6OutGroupMembReductions,     },

    {"Udp6InDatagrams",		&udpv6_tmp.UdpInDatagrams,                   },
    {"Udp6NoPorts",		&udpv6_tmp.UdpNoPorts,                       },
    {"Udp6InErrors",		&udpv6_tmp.UdpInErrors,                      },
    {"Udp6OutDatagrams",	&udpv6_tmp.UdpOutDatagrams,                  },
};

int	v6tab_entries = sizeof(v6tab)/sizeof(struct v6tab);

void
netget(struct fetchdef *fdef, struct osrel *osrel, struct netstat *ns)
{
	register int	nr, cur=0;
	FILE 		*fp;
	char		linebuf[1024];
	char		itsname[32];
	count_t		cnts[40];

	memset(ns, 0, sizeof(struct netstat));

	/*
	** Obtain de IP version 4 statistics
	*/
	if ( (fp = fopen("/proc/net/snmp", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
		    nr = sscanf(linebuf,
			        "%s   %lld %lld %lld %lld %lld %lld %lld %lld "
			        "%lld %lld %lld %lld %lld %lld %lld %lld %lld "
			        "%lld %lld %lld %lld %lld %lld %lld %lld %lld "
			        "%lld %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld\n",
				itsname,
				&cnts[0],  &cnts[1],  &cnts[2],  &cnts[3],
				&cnts[4],  &cnts[5],  &cnts[6],  &cnts[7],
				&cnts[8],  &cnts[9], &cnts[10],  &cnts[11],
				&cnts[12], &cnts[13], &cnts[14], &cnts[15],
				&cnts[16], &cnts[17], &cnts[18], &cnts[19],
				&cnts[20], &cnts[21], &cnts[22], &cnts[23],
				&cnts[24], &cnts[25], &cnts[26], &cnts[27],
				&cnts[28], &cnts[29], &cnts[30], &cnts[31],
				&cnts[32], &cnts[33], &cnts[34], &cnts[35],
				&cnts[36], &cnts[37], &cnts[38], &cnts[39]);

			if (nr < 2)		/* headerline ? --> skip */
				continue;

			if ( strcmp("Ip:", itsname) == EQ)
			{
				memcpy( &ns->ip, cnts, sizeof(struct ip_mib) );
				continue;
			}
	
			if ( strcmp("Icmp:", itsname) == EQ)
			{
				memcpy( &ns->icmp, cnts, sizeof(struct icmp_mib) );
				continue;
			}
	
			if ( strcmp("Tcp:", itsname) == EQ)
			{
				memcpy( &ns->tcp, cnts, sizeof(struct tcp_mib) );
				continue;
			}
	
			if ( strcmp("Udp:", itsname) == EQ)
			{
				memcpy( &ns->udp, cnts, sizeof(struct udp_mib) );
				continue;
			}
		}
	
		fclose(fp);
	}

	/*
	** Obtain de IP version 6 statistics
	** This file has one name-value pair per line.
	*/
	if ( (fp = fopen("/proc/net/snmp6", "r")) != NULL)
	{
		cur = 0;

		memset(&ipv6_tmp,   0, sizeof ipv6_tmp);
		memset(&icmpv6_tmp, 0, sizeof icmpv6_tmp);
		memset(&udpv6_tmp,  0, sizeof udpv6_tmp);

		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
		   	nr = sscanf(linebuf, "%s %lld", itsname, &cnts[0]);

		   	if (strcmp(v6tab[cur].nam, itsname) == EQ)
		   	{
		   		*(v6tab[cur].val) = cnts[0];
		   	}
		   	else
		   	{
		   		for (cur=0; cur < v6tab_entries; cur++)
					if (strcmp(v6tab[cur].nam, itsname)
									 == EQ)
						break;

				if (cur < v6tab_entries) /* found ? */
		   			*(v6tab[cur].val) = cnts[0];
			}

			if (++cur >= v6tab_entries)
				cur = 0;
		}

		memcpy(&ns->ip6,   &ipv6_tmp,   sizeof ipv6_tmp);
		memcpy(&ns->icmp6, &icmpv6_tmp, sizeof icmpv6_tmp);
		memcpy(&ns->udp6,  &udpv6_tmp,  sizeof udpv6_tmp);

		fclose(fp);
	}
}

/*
** Gather nfs (& rpc) statistics from /proc and store in binary form
** in the proprietary nfsstat structure.
*/
void
nfsget(struct fetchdef *fdef, struct osrel *osrel, struct nfsstat *ns)
{
	register int	nr;
	FILE 		*fp;
	char		linebuf[1024];
	char		itsname[32];
	count_t		cnts[32];
	void		nfsfill(char *, count_t [], int,
					struct nfsstat *, int);

	memset(ns, 0, sizeof(struct nfsstat));

	/*
	** Get NFS-client statistics
	*/
	if ( (fp = fopen("/proc/net/rpc/nfs", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			nr = sscanf(linebuf,
				"%s   %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld\n",
				   itsname,
				   &cnts[0],  &cnts[1],  &cnts[2],  &cnts[3],
				   &cnts[4],  &cnts[5],  &cnts[6],  &cnts[7],
				   &cnts[8],  &cnts[9], &cnts[10],  &cnts[11],
				   &cnts[12], &cnts[13], &cnts[14], &cnts[15],
				   &cnts[16], &cnts[17], &cnts[18], &cnts[19],
				   &cnts[20], &cnts[21], &cnts[22], &cnts[23],
				   &cnts[24], &cnts[25], &cnts[26], &cnts[27],
				   &cnts[28], &cnts[29], &cnts[30], &cnts[31]);

			if (nr < 2)
				continue;

			nfsfill(itsname, cnts, nr-1, ns, 1);
		}

		fclose(fp);
	}


	/*
	** Get NFS-server statistics
	*/
	if ( (fp = fopen("/proc/net/rpc/nfsd", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			nr = sscanf(linebuf,
				"%s   %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld %lld %lld %lld "
				"%lld %lld %lld %lld %lld %lld\n",
				   itsname,
				   &cnts[0],  &cnts[1],  &cnts[2],  &cnts[3],
				   &cnts[4],  &cnts[5],  &cnts[6],  &cnts[7],
				   &cnts[8],  &cnts[9], &cnts[10],  &cnts[11],
				   &cnts[12], &cnts[13], &cnts[14], &cnts[15],
				   &cnts[16], &cnts[17], &cnts[18], &cnts[19],
				   &cnts[20], &cnts[21], &cnts[22], &cnts[23],
				   &cnts[24], &cnts[25], &cnts[26], &cnts[27],
				   &cnts[28], &cnts[29], &cnts[30], &cnts[31]);

			if (nr < 2)
				continue;

			nfsfill(itsname, cnts, nr-1, ns, 0);
		}

		fclose(fp);
	}
}

void
nfsfill(char *nam, count_t cnts[], int nc,
			struct nfsstat *ns, int isclient)
{
	if ( strcmp("net", nam) == EQ)
	{
		if (isclient)
		{
			ns->cl_netcnt		= cnts[0];
			ns->cl_netudpcnt	= cnts[1];
			ns->cl_nettcpcnt	= cnts[2];
			ns->cl_nettcpconn	= cnts[3];
		}
		else
		{
			ns->sv_netcnt		= cnts[0];
			ns->sv_netudpcnt	= cnts[1];
			ns->sv_nettcpcnt	= cnts[2];
			ns->sv_nettcpconn	= cnts[3];
		}

		return;
	}

	if ( strcmp("rpc", nam) == EQ)
	{
		if (isclient)
		{
			ns->cl_rpccnt		= cnts[0];
			ns->cl_rpcretrans	= cnts[1];
			ns->cl_rpcauthrefresh	= cnts[2];
		}
		else
		{
			ns->sv_rpccnt		= cnts[0];
			ns->sv_rpcbadtot	= cnts[1];
			ns->sv_rpcbadfmt	= cnts[2];
			ns->sv_rpcbadauth	= cnts[3];
			ns->sv_rpcbadclnt	= cnts[4];
		}

		return;
	}

	if ( nam[0] == 'p' && nam[1] == 'r' &&
	     nam[2] == 'o' && nam[3] == 'c'   )
	{
		int version = atoi(nam+4);

		switch (version)	{
		   case 2:
			if (isclient)
				memcpy(&ns->cl_proc2, &cnts[1],
						sizeof(ns->cl_proc2) );
			else
				memcpy(&ns->sv_proc2, &cnts[1],
						sizeof(ns->sv_proc2) );
			break;

		   case 3:
			if (isclient)
				memcpy(&ns->cl_proc3, &cnts[1],
						sizeof(ns->cl_proc3) );
			else
				memcpy(&ns->sv_proc3, &cnts[1],
						sizeof(ns->sv_proc3) );
			break;
		}

		return;
	}

	if ( strcmp("rc", nam) == EQ)
	{
		if (isclient)
			return;

		ns->sv_rchits		= cnts[0];
		ns->sv_rcmisses		= cnts[1];
		ns->sv_rcnocache	= cnts[2];

		if (nc == 9)	/* kernel 2.2 */
		{
			ns->sv_fhstale		= cnts[7];
			ns->sv_fhlookup		= cnts[6];
			ns->sv_fhanon		= cnts[5];
		}
		return;
	}

	if ( strcmp("fh", nam) == EQ)	/* kernel 2.4 */
	{
		if (isclient)
			return;

		ns->sv_fhstale		= cnts[0];
		ns->sv_fhlookup		= cnts[1];
		ns->sv_fhanon		= cnts[2];
		ns->sv_fhnocachedir	= cnts[3];
		ns->sv_fhnocachenondir	= cnts[4];

		return;
	}

	if ( strcmp("io", nam) == EQ)	/* kernel 2.4 */
	{
		if (isclient)
			return;

		ns->sv_ioread		= cnts[0];
		ns->sv_iowrite		= cnts[1];

		return;
	}

	if ( strcmp("th", nam) == EQ)	/* kernel 2.4 */
	{
		if (isclient)
			return;

		ns->sv_thcount		= cnts[0];
		ns->sv_thlastcnt	= cnts[1];

		return;
	}

	if ( strcmp("ra", nam) == EQ)	/* kernel 2.4 */
	{
		if (isclient)
			return;

		ns->sv_rasize		= cnts[0];

		return;
	}

}

/*
** Gather statistics about the utilization of parameters from /proc and
** store in binary form.
*/
void
tabget(struct fetchdef *fdef, struct osrel *osrel, struct tabstat *ps)
{
	register int	baselen;
	FILE 		*fp;
	char		linebuf[1024], firstword[80], basepath[132];
	count_t		cnt;

	memset(ps, 0, sizeof(struct tabstat));

	/*
	** obtain info about number of processes started
	*/
	if ( (fp = fopen("/proc/stat", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			sscanf(linebuf, "%s %lld", firstword, &cnt);
	
			if ( strcmp(firstword, "processes") == EQ)
			{
				ps->totforks = cnt;
				break;
			}
		}
		fclose(fp);
	}

	/*
	** obtain info about number of record-locks
	*/
	cnt = 0;

	if ( (fp = fopen("/proc/locks", "r")) != NULL)
	{
		/* just count number of lines */
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
			cnt++;

		fclose(fp);
	}

	ps->curlocks = cnt;

	/*
	** obtain info about occupation of disk-quota
	*/
	if ( (fp = fopen("/proc/sys/fs/dquot-max", "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->maxdquot));
		fclose(fp);
	}

	if ( (fp = fopen("/proc/sys/fs/dquot-nr", "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->curdquot));
		fclose(fp);
	}

	/*
	** obtain info about occupation of superblocks
	*/
	if ( (fp = fopen("/proc/sys/fs/super-max", "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->maxsuper));
		fclose(fp);
	}

	if ( (fp = fopen("/proc/sys/fs/super-nr", "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->cursuper));
		fclose(fp);
	}

	/*
	** Next file-names under /proc are dependent on the release/version
	** of this system; they might be stored under /proc/sys/kernel or
	** under /proc/sys/fs. General preparation:
	*/
	if (osrel->rel == 2 && osrel->vers < 2)
		strcpy(basepath, "/proc/sys/kernel/");
	else
		strcpy(basepath, "/proc/sys/fs/");

	baselen = strlen(basepath);

	/*
	** obtain info about occupation of open files
	*/
	strcpy(&basepath[baselen], "file-max");

	if ( (fp = fopen(basepath, "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->maxfiles));
		fclose(fp);
	}

	strcpy(&basepath[baselen], "file-nr");

	if ( (fp = fopen(basepath, "r")) != NULL)
	{
		count_t	alloctot, allocfree;

		fscanf(fp, "%lld %lld", &alloctot, &allocfree);
		fclose(fp);

		ps->curfiles = alloctot - allocfree;
	}

	/*
	** obtain info about occupation of inodes
	*/
	strcpy(&basepath[baselen], "inode-max");

	if ( (fp = fopen(basepath, "r")) != NULL)
	{
		fscanf(fp, "%lld", &(ps->maxinode));
		fclose(fp);
	}

	strcpy(&basepath[baselen], "inode-nr");

	if ( (fp = fopen(basepath, "r")) != NULL)
	{
		fscanf(fp, "%lld %lld", &(ps->curinode), &(ps->maxinode));
		fclose(fp);
	}
}

/*
** Gather statistics about the utilization of disk-partitions from /proc and
** store in binary form.
*/
#define	SCANDK	"%255s %lld %lld %lld %lld %lld %lld %lld %lld %lld %lld %lld"
#define	SCANDS	"%hu %hu "
#define	SCANPT	"%hu %hu %lld "

static char usediskstats;

void
dkinit(struct fetchdef *fdef, struct osrel *osrel)
{
	int		numpart=0, nr=0;
	FILE		*fp;
	struct dkstat	dkstat;
	char		buf[256], diskname[256];

	/*
	** define number of partitions and define if per-partition stats
	** are supported on this system
	*/
	if ( (fp = fopen("/proc/partitions", "r")) != NULL)
	{
		while ( fgets(buf, sizeof(buf), fp) )
		{
			nr = sscanf(buf, SCANPT SCANDK,
                       		&dkstat.major,
				&dkstat.minor,
                       		&dkstat.totsize,
                       		 diskname,

                       		&dkstat.readblocks,
                       		&dkstat.readmerges,
                       		&dkstat.readsectors,
                       		&dkstat.read_msecs,

                       		&dkstat.writblocks,
                       		&dkstat.writmerges,
                       		&dkstat.writsectors,
                       		&dkstat.writ_msecs,

                       		&dkstat.reqpending,
                       		&dkstat.rdwr_msecs,
                       		&dkstat.avq);

			if (nr >= 8 &&
				isrealdisk(diskname, dkstat.name, MAXDKNAME) )
				numpart++;
		}

		fclose(fp);
	}

	if (numpart > 0)	/* partition-stats found */
	{
		fdef->cdef.kinst = numpart;
		return;
	}

	/*
	** no per-partition statistics found; try /proc/diskstats (>= 2.6)
	*/
	if ( (fp = fopen("/proc/diskstats", "r")) != NULL)
	{
		while ( fgets(buf, sizeof(buf), fp) )
		{
			nr = sscanf(buf, SCANDS SCANDK,
                       		&dkstat.major,
				&dkstat.minor,
                       		 diskname,

                       		&dkstat.readblocks,
                       		&dkstat.readmerges,
                       		&dkstat.readsectors,
                       		&dkstat.read_msecs,

                       		&dkstat.writblocks,
                       		&dkstat.writmerges,
                       		&dkstat.writsectors,
                       		&dkstat.writ_msecs,

                       		&dkstat.reqpending,
                       		&dkstat.rdwr_msecs,
                       		&dkstat.avq);

			if (nr >= 7 &&
				isrealdisk(diskname, dkstat.name, MAXDKNAME) )
				numpart++;
		}

		if (numpart > 0)
			usediskstats++;

		fclose(fp);
	}

	fdef->cdef.kinst = numpart;
}

void
dkget(struct fetchdef *fdef, struct osrel *osrel, struct dkstat *ds)
{
	FILE	*fp;
	int	numpart=0, nr=0;
	char	buf[1024], diskname[256];

	if (fdef->cdef.kinst == 0)
		return;

	memset(ds, 0, sizeof(struct dkstat) * fdef->cdef.kinst);

	if (usediskstats)
	{
		fp = fopen("/proc/diskstats",  "r");

		while (fgets(buf, sizeof(buf), fp) && numpart< fdef->cdef.kinst)
		{
			nr = sscanf(buf, SCANDS SCANDK,
                      		&(ds->major),
				&(ds->minor),
                       		  diskname,

                       		&(ds->readblocks),
                       		&(ds->readmerges),
                       		&(ds->readsectors),
                       		&(ds->read_msecs),

                       		&(ds->writblocks),
                       		&(ds->writmerges),
                       		&(ds->writsectors),
                       		&(ds->writ_msecs),

                       		&(ds->reqpending),
                       		&(ds->rdwr_msecs),
                       		&(ds->avq));

			if (nr < 7 ||
				!isrealdisk(diskname, ds->name, MAXDKNAME) )
				continue;

			if (nr == 7) 	/* partition i.s.o. full disk ? */
			{
				ds->writblocks  = ds->readsectors;
				ds->writsectors	= ds->read_msecs;
				ds->readsectors	= ds->readmerges;
				ds->readmerges	= 0;
				ds->read_msecs	= 0;
				ds->writmerges	= 0;
				ds->writ_msecs	= 0;
				ds->reqpending	= 0;
				ds->rdwr_msecs	= 0;
				ds->avq		= 0;
			}

			numpart++;
			ds++;
		}
	}
	else
	{
		fp = fopen("/proc/partitions", "r");

		while (fgets(buf, sizeof(buf), fp) && numpart< fdef->cdef.kinst)
		{
			nr = sscanf(buf, SCANPT SCANDK,
                      		&(ds->major),
				&(ds->minor),
                       		&(ds->totsize),
                       		  diskname,
                       		&(ds->readblocks),
                       		&(ds->readmerges),
                       		&(ds->readsectors),
                       		&(ds->read_msecs),

                       		&(ds->writblocks),
                       		&(ds->writmerges),
                       		&(ds->writsectors),
                       		&(ds->writ_msecs),

                       		&(ds->reqpending),
                       		&(ds->rdwr_msecs),
                       		&(ds->avq));

			if (nr >= 8 && isrealdisk(diskname, ds->name,MAXDKNAME))
			{
				numpart++;
				ds++;
			}
		}
	}

	fclose(fp);
}

/*
** set of subroutines to determine which disks should be monitored
** and to translate long name strings into short name strings
*/
static void
nullmodname(char *curname, char *newname, int maxlen)
{
	strncpy(newname, curname, maxlen-1);
	*(newname+maxlen-1) = 0;
}

static void
abbrevname1(char *curname, char *newname, int maxlen)
{
	char	cutype[128];
	int	hostnum, busnum, targetnum, lunnum, partnum, nr;

	nr = sscanf(curname, "%[^/]/host%d/bus%d/target%d/lun%d/part%d",
			cutype, &hostnum, &busnum, &targetnum,
			        &lunnum,  &partnum);

	if (nr == 5)	/* full disk */
		snprintf(newname, maxlen, "%c-h%db%dt%d", 
			cutype[0], hostnum, busnum, targetnum);
	else		/* partition */
		snprintf(newname, maxlen, "%c-h%db%dt%d.%d", 
			cutype[0], hostnum, busnum, targetnum, partnum);
}

/*
** table contains the names (in regexp format) of valid disks
** to be supported, together a function to modify the name-strings
** (i.e. to abbreviate long strings);
** this table is used in the function isrealdisk()
*/
static struct {
	char 	*regexp;
	regex_t	compreg;
	void	(*modname)(char *, char *, int);
} validdisk[] = {
	{	"^sd[a-z][a-z]*",			{0}, 	nullmodname, },
	{	"^hd[a-z]",				{0},	nullmodname, },
	{	"^rd/c[0-9][0-9]*d[0-9][0-9]*",		{0},	nullmodname, },
	{	"^cciss/c[0-9][0-9]*d[0-9][0-9]*",	{0},	nullmodname, },
	{ 	"/host.*/bus.*/target.*/lun",		{0},	abbrevname1, },
};

static int
isrealdisk(char *curname, char *newname, int maxlen)
{
	static int	firstcall = 1;
	register int	i;

	if (firstcall)		/* compile the regular expressions */
	{
		for (i=0; i < sizeof validdisk/sizeof validdisk[0]; i++)
			regcomp(&validdisk[i].compreg, validdisk[i].regexp,
								REG_NOSUB);
		firstcall = 0;
	}

	/*
	** try to recognize one of the compiled regular expressions
	*/
	for (i=0; i < sizeof validdisk/sizeof validdisk[0]; i++)
	{
		if (regexec(&validdisk[i].compreg, curname, 0, NULL, 0) == 0)
		{
			/*
			** name-string recognized; modify name-string
			*/
			(*validdisk[i].modname)(curname, newname, maxlen);

			return 1;
		}
	}

	return 0;
}


/*
** Gather statistics about the utilization of sockets from /proc and
** store in binary form.
*/
void
sockget(struct fetchdef *fdef, struct osrel *osrel, struct sockstat *ss)
{
	FILE 	*fp;
	char	linebuf[1024];
	char	itsname[32], notused[64];
	count_t	nowuse, maxuse;

	memset(ss, 0, sizeof(struct sockstat));

	if ( (fp = fopen("/proc/net/sockstat", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			maxuse = 0;	/* 2.4 kernel: no support max usage */

			if ( sscanf(linebuf, "%s  %s %lld %s %lld\n",
				itsname, notused, &nowuse, notused, &maxuse)<3)
					continue;

			if ( strcmp("TCP:", itsname) == EQ)
			{
				ss->tcpnow = nowuse;
				ss->tcpmax = maxuse;
				continue;
			}

			if ( strcmp("UDP:", itsname) == EQ)
			{
				ss->udpnow = nowuse;
				ss->udpmax = maxuse;
				continue;
			}
		}

		fclose(fp);
	}

	if ( (fp = fopen("/proc/net/sockstat6", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			maxuse = 0;	/* 2.4 kernel: no support max usage */

			if ( sscanf(linebuf, "%s  %s %lld %s %lld\n",
				itsname, notused, &nowuse, notused, &maxuse)<3)
					continue;

			if ( strcmp("TCP6:", itsname) == EQ)
			{
				ss->tcp6now = nowuse;
				ss->tcp6max = maxuse;
				continue;
			}

			if ( strcmp("UDP6:", itsname) == EQ)
			{
				ss->udp6now = nowuse;
				ss->udp6max = maxuse;
				continue;
			}
		}

		fclose(fp);
	}
}


/*
** Gather statistics about the utilization of serial interfaces and
** store in binary form.
*/
void
ttyget(struct fetchdef *fdef, struct osrel *osrel, struct serialstat *ts)
{
	register char	*p;
	FILE 		*fp;
	char		linebuf[1024];
	char		nam[9][16], par[9][16];
	int		ttynum, nr, i, t=0, colons;

	memset(ts, 0, sizeof(struct serialstat) * fdef->cdef.kinst);

	if ( (fp = fopen("/proc/tty/driver/serial", "r")) != NULL)
	{
		while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		{
			/*
			** replace colons by spaces to ease parsing
			** and to determine if this line contains counters
			*/
			for (p=linebuf, colons=0; *p; p++)
			{
				if (*p != ':')
					continue;

				*p = ' ';
				colons++;
			}

			if (colons < 5)		/* unused serial line */
				continue;

			/*
			** counter available: parse line
			*/
			nr = sscanf(linebuf, "%d %s %s %s %s %s %s %s %s"
			                     "   %s %s %s %s %s %s %s %s"
			                     "   %s %s\n",
						&ttynum, 
						nam[0], par[0], nam[1], par[1],
						nam[2], par[2], nam[3], par[3],
						nam[4], par[4], nam[5], par[5],
						nam[6], par[6], nam[7], par[7],
						nam[8], par[8]);

			nr = (nr - 1) / 2;	/* # name-value pairs */

			for (i=1; i < nr; i++)
			{
				if ( strcmp("port", nam[i]) == EQ)
				{
					ts->port = strtol(par[i], NULL, 16);
					continue;
				}

				if ( strcmp("tx", nam[i]) == EQ)
				{
					ts->tx = atoi(par[i]);
					continue;
				}

				if ( strcmp("rx", nam[i]) == EQ)
				{
					ts->rx = atoi(par[i]);
					continue;
				}

				if ( strcmp("fe", nam[i]) == EQ)
				{
					ts->fe = atoi(par[i]);
					continue;
				}

				if ( strcmp("brk", nam[i]) == EQ)
				{
					ts->br = atoi(par[i]);
					continue;
				}

				if ( strcmp("pe", nam[i]) == EQ)
				{
					ts->pe = atoi(par[i]);
					continue;
				}

				if ( strcmp("oe", nam[i]) == EQ)
				{
					ts->oe = atoi(par[i]);
					continue;
				}
			}

			if (ts->port == 0)
				continue;

			if (++t >= fdef->cdef.kinst)
				break;

			ts++;
		}

		fclose(fp);
	}
}


#ifdef	WEBSUPPORT
/*
** Gather statistics about the number of FTP-transfers
*/
void
ftpinit(struct fetchdef *fdef, struct osrel *osrel)
{
	FILE 		*fp;
	char		linebuf[1024];

	fdef->cdef.kinst = 0;

	if ( (fp = fopen("/var/log/atsar/ftpstat", "r")) == NULL)
		return;

	while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		fdef->cdef.kinst++;

	fclose(fp);
}

void
ftpget(struct fetchdef *fdef, struct osrel *osrel, struct ftpstat *fs)
{
	FILE 		*fp;
	char		linebuf[1024], filename[256], symname[256];
	int			i;
	count_t		notused;

	memset(fs, 0, sizeof(struct ftpstat) * fdef->cdef.kinst);

	if ( (fp = fopen("/var/log/atsar/ftpstat", "r")) == NULL)
		return;

	for (i=0; i < fdef->cdef.kinst; i++, fs++)
	{
		if  ( fgets(linebuf, sizeof(linebuf), fp) == NULL)
			break;

		sscanf(linebuf, "%lf %lf %lf %lf %lf %lf %lld %lld %lld %s %s\n",
			&fs->ocumxfer, &fs->ocumseconds, &fs->ocumkbytes,
			&fs->icumxfer, &fs->icumseconds, &fs->icumkbytes,
			&notused, &notused, &notused, filename, symname);

		strncpy(fs->logname, symname, MAXLOGNAME-1);
		fs->logname[MAXLOGNAME-1] = '\0';
	}
	fclose(fp);
}

/*
** Gather HTTP-statistics
*/
void
httpinit(struct fetchdef *fdef, struct osrel *osrel)
{
	FILE 		*fp;
	char		linebuf[1024];

	fdef->cdef.kinst = 0;

	if ( (fp = fopen("/var/log/atsar/httpstat", "r")) == NULL)
		return;

	while ( fgets(linebuf, sizeof(linebuf), fp) != NULL)
		fdef->cdef.kinst++;

	fclose(fp);
}

void
httpget(struct fetchdef *fdef, struct osrel *osrel, struct httpstat *hs)
{
	register int	i;
	FILE 		*fp;
	char		linebuf[1024], filename[256], symname[256];
	count_t		notused;

	memset(hs, 0, sizeof(struct httpstat) * fdef->cdef.kinst);

	if ( (fp = fopen("/var/log/atsar/httpstat", "r")) == NULL)
		return;

	for (i=0; i < fdef->cdef.kinst; i++, hs++)
	{
		if  ( fgets(linebuf, sizeof(linebuf), fp) == NULL)
			break;

		sscanf(linebuf, "%lf %lf %lf %lf %lf %lld %lld %lld %s %s\n",
			&hs->cumhead, &hs->cumgets, &hs->cumputs,
			&hs->cumpost, &hs->cumdels,
			&notused, &notused, &notused, filename, symname);

		strncpy(hs->logname, symname, MAXLOGNAME-1);
		hs->logname[MAXLOGNAME-1] = '\0';
	}

	fclose(fp);
}
#endif


/*********************************************************************/
/* Counter fetch definition tables.                                  */
/*                                                                   */
/* The layout of these tables is as follows:                         */
/*     Column 1:                                                     */
/*        Internal name of counter-set, to be filled by atsadc and   */
/*        to be printed by atsar.                                    */
/*                                                                   */
/*     Column 2:                                                     */
/*        Size (number of bytes) per instance of this counter or     */
/*        counter-structure.                                         */
/*                                                                   */
/*     Column 3:                                                     */
/*        Number of instances of this counter or counter-structure   */
/*        (e.g. if this is a table of counters).                     */
/*                                                                   */
/*     Column 4:                                                     */
/*        Entry-point (optional) of a function which may initialise  */
/*        this table-entry before its contents is taken into account.*/
/*        If e.g. the number of instances (column 3) can only be     */
/*        determined at runtime, a special function can be written   */
/*        to determine this value.                                   */
/*                                                                   */
/*        This function will be called with two arguments, i.e. a    */
/*        pointer to the relevant fetchdef[] table entry to be       */
/*        filled or modified, and info about the OS-release.         */
/*                                                                   */
/*        The return-value is not relevant.                          */
/*                                                                   */
/*     Column 5:                                                     */
/*        Entry-point (mandatory!) of a function which can be called */
/*        to read the statistical counters during every sample.      */
/*                                                                   */
/*        This function will be called with three arguments, i.e.    */
/*        a pointer to the relevant fetchdef[] table entry to be     */
/*        used for information, info about the OS-release and a      */
/*        pointer to the area in which the counter-data should be    */
/*        stored.                                                    */
/*                                                                   */
/*        The return-value is not relevant.                          */
/*********************************************************************/

	/*******************************************************/
	/* counters which are different on every system, but   */
	/* do not change on-the-fly (only read once).          */
	/*******************************************************/
struct fetchdef oncedef[] =
{

};

int	oncecnt = sizeof(oncedef)/sizeof(struct fetchdef);

	/*******************************************************/
	/* counters which hold the real statistics which       */
	/* constantly increase (read every <t> seconds).       */
	/*******************************************************/
struct fetchdef sampdef[] =
{
	{ {"genstat",   sizeof(struct genstat),	    1}, 0,       genget,  },
	{ {"memstat",   sizeof(struct memstat),	    1}, 0,       memget,  },
	{ {"ifstat",    sizeof(struct ifstat),	    1}, ifinit,  ifget,   },
	{ {"netstat",   sizeof(struct netstat),	    1}, 0,       netget,  },
	{ {"nfsstat",   sizeof(struct nfsstat),	    1}, 0,       nfsget,  },
	{ {"tabstat",   sizeof(struct tabstat),	    1}, 0,       tabget,  },
	{ {"partstat",  sizeof(struct dkstat),	    1}, dkinit,  dkget,   },
	{ {"sockstat",  sizeof(struct sockstat),    1}, 0,       sockget, },
	{ {"ttystat",   sizeof(struct serialstat), 32}, 0,       ttyget,  },
 
#ifdef	WEBSUPPORT
	{ {"ftpstat",   sizeof(struct ftpstat),	    1}, ftpinit,  ftpget, },
	{ {"httpstat",  sizeof(struct httpstat),    1}, httpinit, httpget,},
#endif
};

int	sampcnt = sizeof(sampdef)/sizeof(struct fetchdef);
