/***************************************************************************
 *   Copyright (C) 2006 by Bjorn Hansen                                    *
 *   holomorph@users.sourceforge.net                                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "../../include/menu/playersetup.h"
#include "../../include/menu/menu.h"
#include "../../include/configmanager.h"
#include "../../include/menu/sampleprobewidget.h"
#include "../../include/menu/ailistmodel.h"
#include "../../include/input.h"
#include <sstream>

using namespace Balder;

PlayerSetupMenu::PlayerSetupMenu(gcn::Container* top, SubMenu* parent):SubMenu(top,parent)
{
    page->setDimension(gcn::Rectangle(0, 0, 600, 400));
    page->setPosition(100,200);

    backButton = new gcn::Button("Back");
    backButton->setEventId("back");
    backButton->addActionListener(this);

    currentPlayer = 1;
    std::stringstream ss;
    ss << "Player " << currentPlayer;
    currentPlayerLabel = new gcn::Label(ss.str());

    nextPlayerButton = new gcn::Button(" > ");
    nextPlayerButton->setEventId("nextPlayer");
    nextPlayerButton->addActionListener(this);

    prevPlayerButton = new gcn::Button(" < ");
    prevPlayerButton->setEventId("prevPlayer");
    prevPlayerButton->addActionListener(this);

    setRotateRightButton = new KeyBindButton("Rotate Right:", this);
    setRotateRightButton->setEventId("bindRight");
    setRotateRightButton->addActionListener(this);
    rotateRightLabel = new gcn::Label(ConfigManager::GetPlayerControl(currentPlayer, "right"));
    setRotateRightButton->setAlignment(0);

    setRotateLeftButton = new KeyBindButton("Rotate Left:", this);
    setRotateLeftButton->setEventId("bindLeft");
    setRotateLeftButton->addActionListener(this);
    rotateLeftLabel = new gcn::Label(ConfigManager::GetPlayerControl(currentPlayer, "left"));
    setRotateLeftButton->setAlignment(0);

    setStickPushButton = new KeyBindButton("Stick/Push Off:", this);
    setStickPushButton->setEventId("bindStick");
    setStickPushButton->addActionListener(this);
    stickPushLabel = new gcn::Label(ConfigManager::GetPlayerControl(currentPlayer, "stick"));
    setStickPushButton->setAlignment(0);

    setFireButton = new KeyBindButton("Fire:", this);
    setFireButton->setEventId("bindFire");
    setFireButton->addActionListener(this);
    fireLabel = new gcn::Label(ConfigManager::GetPlayerControl(currentPlayer, "fire"));
    setFireButton->setAlignment(0);

    setRotateRightButton->setWidth(setStickPushButton->getWidth());
    setRotateLeftButton->setWidth(setStickPushButton->getWidth());
    setFireButton->setWidth(setStickPushButton->getWidth());

    // Loads colors from config file to init color sliders.
    probe_color playerColor = ConfigManager::GetPlayerColor(currentPlayer, 255);
    sampleProbe = new SampleProbeWidget();
    sampleProbe->setProbeColor(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cBlue = Uint8(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cGreen = Uint8(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cRed = Uint8(playerColor);

    redImage= new gcn::Image("menu/redspectrum.png");
    redIcon = new gcn::Icon(redImage);
    redSlider = new gcn::Slider(0, 255);
    redSlider->setWidth(redIcon->getWidth());
    redSlider->setHeight(15);
    redSlider->setValue(double(cRed));
    redSlider->setEventId("colorChange");
    redSlider->addActionListener(this);

    greenImage= new gcn::Image("menu/greenspectrum.png");
    greenIcon = new gcn::Icon(greenImage);
    greenSlider = new gcn::Slider(0, 255);
    greenSlider->setWidth(redIcon->getWidth());
    greenSlider->setHeight(15);
    greenSlider->setValue(double(cGreen));
    greenSlider->setEventId("colorChange");
    greenSlider->addActionListener(this);

    blueImage= new gcn::Image("menu/bluespectrum.png");
    blueIcon = new gcn::Icon(blueImage);
    blueSlider = new gcn::Slider(0, 255);
    blueSlider->setWidth(redIcon->getWidth());
    blueSlider->setHeight(15);
    blueSlider->setValue(double(cBlue));
    blueSlider->setEventId("colorChange");
    blueSlider->addActionListener(this);

    // AI Script list box.
    aiLabel = new gcn::Label("AI Script: ");
    ailistModel = new AIListModel();
    aiScrollArea = new gcn::ScrollArea();
    aiListBox = new gcn::ListBox();
    aiListBox->setWidth(redIcon->getWidth() - 12);
    aiListBox->setEventId("changeAIScript");
    aiListBox->addActionListener(this);
    aiDropDown = new gcn::DropDown(ailistModel, aiScrollArea, aiListBox);
    aiDropDown->setWidth(redIcon->getWidth());
    aiDropDown->setSelected(ailistModel->FindScriptIndex(ConfigManager::GetControlType(currentPlayer)));

    page->add(backButton, 0, 300);
    page->add(prevPlayerButton, 0,0);
    page->add(currentPlayerLabel, prevPlayerButton->getWidth(), 0);
    page->add(nextPlayerButton, prevPlayerButton->getWidth() + currentPlayerLabel->getWidth(), 0 );
    page->add(setRotateRightButton, 0, 60);
    page->add(rotateRightLabel, setRotateRightButton->getWidth(), 60);
    page->add(setRotateLeftButton, 0, 120);
    page->add(rotateLeftLabel, setRotateLeftButton->getWidth(), 120);
    page->add(setStickPushButton, 0, 180);
    page->add(stickPushLabel, setStickPushButton->getWidth(), 180);
    page->add(setFireButton, 0, 240);
    page->add(fireLabel, setFireButton->getWidth(), 240);

    // Color selection controls for the selected player.
    colorTop = 60; // This sets the top most position for the color selection interface.
    colorLeft = 400; // This sets the left most position for the color selection interface.
    page->add(sampleProbe, colorLeft +(redIcon->getWidth() - sampleProbe->getWidth())/2, colorTop);
    page->add(redIcon, colorLeft, colorTop + sampleProbe->getHeight() + 10);
    page->add(redSlider, colorLeft, colorTop + sampleProbe->getHeight() + 10 + redIcon->getHeight());
    page->add(greenIcon, colorLeft, colorTop + sampleProbe->getHeight() + 10 + redIcon->getHeight() + redSlider->getHeight() + 10);
    page->add(greenSlider, colorLeft, colorTop + sampleProbe->getHeight() + 10 + redIcon->getHeight() + redSlider->getHeight() + 10 + greenIcon->getHeight());
    page->add(blueIcon, colorLeft, colorTop + sampleProbe->getHeight() + 10 + redIcon->getHeight() + redSlider->getHeight() + 10 + greenIcon->getHeight() + greenSlider->getHeight() + 10);
    page->add(blueSlider, colorLeft, colorTop + sampleProbe->getHeight() + 10 + redIcon->getHeight() + redSlider->getHeight() + 10 + greenIcon->getHeight() + greenSlider->getHeight() + 10 + blueIcon->getHeight());

    // AI Script Listbox
    page->add(aiDropDown, colorLeft, nextPlayerButton->getY());
    page->add(aiLabel, aiDropDown->getX() - aiLabel->getWidth(), aiDropDown->getY());
    // lastly call refreshKeys to make sure everything displays correctly
    RefreshKeys();
}

PlayerSetupMenu::~PlayerSetupMenu()
{
}

void PlayerSetupMenu::action(const std::string& action)
{
    // Player options menu events.**********************************************
    if (action == "nextPlayer")
    {
        if (currentPlayer < ConfigManager::GetNumPlayers())
        {
            ++currentPlayer;
            RefreshKeys();
        }
    }
    else if (action == "prevPlayer")
    {
        if (currentPlayer > 1 )
        {
            --currentPlayer;
            RefreshKeys();
        }
    }
    else if (action == "bindRight")
    {
        rotateRightLabel->setCaption(" ");
        actionToBind = "right";
        bindNextKey = true;
    }
    else if (action == "bindLeft")
    {
        rotateLeftLabel->setCaption(" ");
        actionToBind = "left";
        bindNextKey = true;
    }
    else if (action == "bindStick")
    {
        stickPushLabel->setCaption(" ");
        actionToBind = "stick";
        bindNextKey = true;
    }
    else if (action == "bindFire")
    {
        fireLabel->setCaption(" ");
        actionToBind = "fire";
        bindNextKey = true;
    }
    else if (action == "changeAIScript")
    {
        ConfigManager::SetControlType(currentPlayer, ailistModel->getElementAt(aiListBox->getSelected()).c_str());
        RefreshKeys();
    }
    else if (action == "colorChange")
    {
        Uint32 rgbColor = Uint8(redSlider->getValue());
        rgbColor = rgbColor << 8;
        rgbColor += Uint8(greenSlider->getValue());
        rgbColor = rgbColor << 8;
        rgbColor += Uint8(blueSlider->getValue());
        sampleProbe->setProbeColor((rgbColor << 8) + 0xFF);
        ConfigManager::SetPlayerColor(currentPlayer, rgbColor);
    }
    else if (action == "back")
    {
        hide();
        parent->show();
    }
}

void PlayerSetupMenu::RefreshKeys()
{
    int numPlayers = ConfigManager::GetNumPlayers();
    if (currentPlayer > numPlayers) {currentPlayer = numPlayers;}
    std::stringstream ss;
    ss << "Player " << currentPlayer;
    currentPlayerLabel->setCaption(ss.str());
    // Refreshes the AI script selector for the players menu
    aiDropDown->setSelected(ailistModel->FindScriptIndex(ConfigManager::GetControlType(currentPlayer)));
    if (aiDropDown->getSelected()) { // in otherwords != 0, which is human
        // hide all the key binding stuff
        setRotateRightButton->setVisible(false);
        rotateRightLabel->setVisible(false);
        setRotateLeftButton->setVisible(false);
        rotateLeftLabel->setVisible(false);
        setStickPushButton->setVisible(false);
        stickPushLabel->setVisible(false);
        setFireButton->setVisible(false);
        fireLabel->setVisible(false);
    }
    else { // it's a human player, so we want to show keybinging stuff
        setRotateRightButton->setVisible(true);
        rotateRightLabel->setVisible(true);
        setRotateLeftButton->setVisible(true);
        rotateLeftLabel->setVisible(true);
        setStickPushButton->setVisible(true);
        stickPushLabel->setVisible(true);
        setFireButton->setVisible(true);
        fireLabel->setVisible(true);
        rotateRightLabel->setCaption(ConfigManager::GetPlayerControl(currentPlayer, "right"));
        rotateRightLabel->adjustSize();
        rotateLeftLabel->setCaption(ConfigManager::GetPlayerControl(currentPlayer, "left"));
        rotateLeftLabel->adjustSize();
        stickPushLabel->setCaption(ConfigManager::GetPlayerControl(currentPlayer, "stick"));
        stickPushLabel->adjustSize();
        fireLabel->setCaption(ConfigManager::GetPlayerControl(currentPlayer, "fire"));
        fireLabel->adjustSize();
    }

    // Refreshes the color sliders.
    probe_color playerColor = ConfigManager::GetPlayerColor(currentPlayer, 255);
    sampleProbe->setProbeColor(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cBlue = Uint8(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cGreen = Uint8(playerColor);
    playerColor = playerColor >> 8;
    Uint8 cRed = Uint8(playerColor);
    redSlider->setValue(double(cRed));
    greenSlider->setValue(double(cGreen));
    blueSlider->setValue(double(cBlue));
}

void PlayerSetupMenu::BindKey(const gcn::Key& key)
{
    int value = key.getValue();
    if (value == gcn::Key::ESCAPE)
    {
        bindNextKey = false;
        RefreshKeys();
    }
    else if (bindNextKey && Menu::lastKeyPressed)
    {
        ConfigManager::SetPlayerControl(currentPlayer, actionToBind, Input::GetKeyString(Menu::lastKeyPressed));
        bindNextKey = false;
        RefreshKeys();
    }
}
