/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "GzipCompressor.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include <zlib.h>
#include <memory>
#include <cassert>

using namespace std;

char const GzipCompressor::m_sHeader[] = {
	0x1f, 0x8b,             // format id
	0x08,                   // compression: deflate
	0x00,                   // flags
	0x00, 0x00, 0x00, 0x00, // mtime: unknown
	0x00,                   // extra flags
	0xff                    // OS: unknown
};


GzipCompressor::GzipCompressor()
:	m_state(ST_WRITING_HEADER),
	m_isEOF(false),
	m_inputLength(0),
	m_inputCRC32(0),
	m_headerBytesWritten(0),
	m_footerBytesWritten(0),
	m_deflateCompressor(DeflateCompressor::RAW)
{
}

GzipCompressor::~GzipCompressor()
{
}

void
GzipCompressor::reset()
{
	m_deflateCompressor.reset();
	m_state = ST_WRITING_HEADER;
	m_isEOF = false;
	m_inputLength = 0;
	m_inputCRC32 = 0;
	m_headerBytesWritten = 0;
	m_footerBytesWritten = 0;
	m_footer.clear();
}

bool
GzipCompressor::isError() const
{
	return (m_state == ST_ERROR);
}

void
GzipCompressor::consume(SplittableBuffer& data, bool eof)
{
	if (m_state == ST_ERROR) {
		data.clear();
		return;
	}
	if (m_isEOF) {
		assert(data.empty() && eof);
		return;
	}
	
	m_inputLength += data.size();
	updateInputCRC32(data);
	
	m_deflateCompressor.consume(data, eof);
	m_isEOF = eof;
	if (m_deflateCompressor.isError()) {
		m_state = ST_ERROR;
	}
}

size_t
GzipCompressor::retrieveOutput(SplittableBuffer& buf, size_t limit)
{
	size_t output_produced = 0;
	
	switch (m_state) {
		case ST_WRITING_HEADER: {
			size_t todo = sizeof(m_sHeader) - m_headerBytesWritten;
			size_t len = std::min<size_t>(todo, limit);
			BString empty;
			buf.append(BString(
				empty, m_sHeader + m_headerBytesWritten,
				m_sHeader + m_headerBytesWritten + len
			));
			m_headerBytesWritten += len;
			output_produced += len;
			limit -= len;
			if (len != todo) {
				break;
			}
			m_state = ST_COMPRESSING;
			// fall through
		}
		case ST_COMPRESSING: {
			size_t written = 0;
			do {
				written = m_deflateCompressor.retrieveOutput(buf, limit);
				output_produced += written;
				limit -= written;
				if (m_deflateCompressor.isError()) {
					m_state = ST_ERROR;
					break;
				}
			} while (m_isEOF && written != 0 && limit != 0);
			
			if (m_isEOF && written == 0 && limit != 0) {
				prepareFooter();
				m_state = ST_WRITING_FOOTER;
			} else {
				break;
			}
			// fall through
		}
		case ST_WRITING_FOOTER: {
			size_t todo = m_footer.size() - m_footerBytesWritten;
			size_t len = std::min<size_t>(todo, limit);
			buf.append(BString(
				m_footer, m_footer.begin() + m_footerBytesWritten,
				m_footer.begin() + m_footerBytesWritten + len
			));
			m_footerBytesWritten += len;
			output_produced += len;
			limit -= len;
			if (len != todo) {
				break;
			}
			m_state = ST_FINISHED;
			break;
		}
		default: break;
	}
	
	return output_produced;
}

void
GzipCompressor::updateInputCRC32(SplittableBuffer const& data)
{
	SplittableBuffer::SpanIterator it(data.spanBegin());
	SplittableBuffer::SpanIterator const end(data.spanEnd());
	for (; it != end; ++it) {
		m_inputCRC32 = crc32(m_inputCRC32, (Bytef*)it->begin(), it->size());
	}
}

void
GzipCompressor::prepareFooter()
{
	auto_ptr<DataChunk> chunk(DataChunk::create(8));
	uint8_t* p = (uint8_t*)chunk->getDataAddr();
	p[0] = m_inputCRC32;
	p[1] = m_inputCRC32>>8;
	p[2] = m_inputCRC32>>16;
	p[3] = m_inputCRC32>>24;
	p[4] = m_inputLength;
	p[5] = m_inputLength>>8;
	p[6] = m_inputLength>>16;
	p[7] = m_inputLength>>24;
	m_footer = BString(chunk);
}
