/*
    BFilter - a web proxy which removes banner ads
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <iostream>
#include <fstream>
#include <sstream>
#include "lexgen.h"
#include "NfaChar.h"
#include "NfaCharClass.h"
#include "NfaNegatedCharClass.h"
#include "NfaConcatenation.h"
#include "NfaClosure.h"
#include "NfaPositiveClosure.h"
#include "NfaOption.h"
#include "NfaString.h"
#include "NfaInsensitiveString.h"
#include "NfaUnion.h"
#include "NfaEpsilon.h"

#include "HeaderLexerDefinitions.h"

int main(int argc, char** argv)
{
	if (argc < 8) {
		std::cerr << "Usage: makelexer <OutputClass> <out.h> <out.cpp> "
			"<Definitions> <definitions.h> <Subclass> <subclass.h>" << std::endl;
		return 1;
	}
	const char* out_class = argv[1];
	const char* out_header = argv[2];
	const char* out_impl = argv[3];
	const char* def_class = argv[4];
	const char* def_header = argv[5];
	const char* subclass = argv[6];
	const char* subclass_header = argv[7];
	
	std::ofstream header_strm(out_header);
	if (header_strm.fail()) {
		std::cerr << "Failed opening " << out_header << " for writing" << std::endl;
	}
	
	std::ofstream impl_strm(out_impl);
	if (impl_strm.fail()) {
		std::cerr << "Failed opening " << out_impl << " for writing" << std::endl;
	}
	
	LexGen lexgen(out_class, subclass);
	lexgen.trackStreamPosition(true);
	
	std::string space_chars(" \t\r\n");
	NfaEpsilon epsilon;
	NfaCharClass space(space_chars);
	NfaNegatedCharClass tchar("=,;\"\f");
	tchar.addChars(space_chars);
	NfaNegatedCharClass any;
	NfaChar comma(',');
	NfaChar equal('=');
	NfaChar quote('"');
	NfaChar backslash('\\');
	NfaChar semicolon(';');
	
	typedef HeaderLexerDefinitions Defs;
	
	
	lexgen.addRule(Defs::INITIAL, NfaPositiveClosure(space),
		"/* skip spaces */"
	);
	lexgen.addRule(Defs::INITIAL, NfaPositiveClosure(tchar),
		"BEGIN(AFTER_ELEMENT_NAME);\n\t"
		"obj->processElementName(tok_begin, tok_end);"
	);
	lexgen.addRule(Defs::INITIAL, any,
		"obj->processError(obj->tokenPosition(tok_begin).streamPos);\n\t"
		"obj->abort();"
	);
	
	
	lexgen.addRule(Defs::AFTER_ELEMENT_NAME, NfaPositiveClosure(space),
		"/* skip spaces */"
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_NAME, equal,
		"BEGIN(ELEMENT_VALUE);"
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_NAME, semicolon,
		"BEGIN(INITIAL);\n\t"
		"obj->processNullElementValue();\n\t"
		"obj->setProcessingParam(true);"
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_NAME, NfaUnion(comma, epsilon),
		"BEGIN(INITIAL);\n\t"
		"obj->processNullElementValue();\n\t"
		"obj->setProcessingParam(false);"
		
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_NAME, any,
		"obj->processError(obj->tokenPosition(tok_begin).streamPos);\n\t"
		"obj->abort();"
	);
	
	
	lexgen.addRule(Defs::ELEMENT_VALUE, NfaPositiveClosure(space),
		"/* skip spaces */"
	);
	lexgen.addRule(Defs::ELEMENT_VALUE, quote,
		"BEGIN(QUOTED_ELEMENT_VALUE);"
	);
	lexgen.addRule(Defs::ELEMENT_VALUE, epsilon,
		"BEGIN(UNQUOTED_ELEMENT_VALUE);"
	);
	
	
	lexgen.addRule(Defs::UNQUOTED_ELEMENT_VALUE, NfaClosure(tchar),
		"BEGIN(AFTER_ELEMENT_VALUE);\n\t"
		"obj->processElementValue(tok_begin, tok_end);"
	);
	
	
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE, NfaClosure(NfaNegatedCharClass("\"\\")), quote,
		"BEGIN(AFTER_ELEMENT_VALUE);\n\t"
		"obj->processElementValue(tok_begin, tok_end);\n\t"
		"CONSUME_TRAILER();"
	);
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE, epsilon,
		"BEGIN(QUOTED_ELEMENT_VALUE2);"
	);
	
	
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE2, NfaPositiveClosure(NfaNegatedCharClass("\"\\")),
		"obj->appendElementValue(tok_begin, tok_end);"
	);
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE2, backslash, any,
		"obj->appendElementValueQuoted(*tok_end);\n\t"
		"CONSUME_TRAILER();"
	);
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE2, quote,
		"BEGIN(AFTER_ELEMENT_VALUE);\n\t"
		"obj->appendElementValueDone();"
	);
	lexgen.addRule(Defs::QUOTED_ELEMENT_VALUE2, epsilon,
		"obj->processError(obj->tokenPosition(tok_begin).streamPos);\n\t"
		"obj->abort();"
	);
	
	
	lexgen.addRule(Defs::AFTER_ELEMENT_VALUE, semicolon,
		"BEGIN(INITIAL);\n\t"
		"obj->setProcessingParam(true);"
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_VALUE, comma,
		"BEGIN(INITIAL);\n\t"
		"obj->setProcessingParam(false);"
	);
	lexgen.addRule(Defs::AFTER_ELEMENT_VALUE, any,
		"obj->processError(obj->tokenPosition(tok_begin).streamPos);\n\t"
		"obj->abort();"
	);
	
	
	lexgen.writeLexer(header_strm, impl_strm, def_class, def_header, subclass_header);
	return 0;
}
