/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "SelectReactor.h"
#include "EventHandler.h"
#include "ReactorHelpers.h"
#include "SynchFactory.h"
#include "MtGuard.h"
#include "AtomicOps.h"
#include "TimeDelta.h"
#include "TimeStamp.h"
#include "MonotonicTimer.h"
#include <ace/config-lite.h>
#include <ace/Lock.h>
#include <ace/OS_NS_sys_select.h>
#include <cassert>
#include <cstdlib>
#include <cerrno>
#include <limits>
#include <algorithm>

#define EXCLUSIVE_WAITSET_ACCESS()    \
MtGuard<ACE_Lock> guard(*m_ptrMutex); \
if (m_isInsideDemux) wakeup();        \
MtGuard<ACE_Lock> demux_guard(*m_ptrDemuxMutex);

using namespace std;
using namespace ReactorHelpers;

/*
When registering a handler, we store the current dispatch wave
in Handler::flags. When dispatching events, we skip handlers with
flags == m_dispatchWave.
That's done to prevent a situation like this:
1. Multiple events have occured on a handle.
2. The first event is dispatched.
3. The event handler unregisters itself and closes the handle.
4. Another handle is created with the same value as the one closed.
5. The new handle is registered in the reactor.
6. The rest of the events ocurred on the old handle are dispatched
to the new handler, which is wrong.

We also properly handle the wraparound in m_dispatchWave.
The reason for not using HANDNER_NEW as PollReactor does, is that we
don't iterate over handlers before select(), so we have no chance to
update Handler::flags.
*/


static void checkCapacity(ACE_HANDLE handle, size_t num_handlers);


SelectReactor::SelectReactor(
	AbstractSynchFactory const& synch_factory, bool wakeup_on_signal)
:	m_ptrHandlers(new HandlerRepository),
	m_ptrTimers(new TimerQueue),
	m_ptrMutex(synch_factory.createMutex()),
	m_ptrDemuxMutex(synch_factory.createMutex()),
	m_wakeupPipe(),
	m_waitSet(),
	m_dispatchSet(),
	m_dispatchWave(0),
	m_ioIter(m_dispatchSet),
	m_wakeupOnSignal(wakeup_on_signal),
	m_isInsideDemux(false),
	m_isStopped(false),
	m_isStoppedAtomic(0)
{
	checkCapacity(m_wakeupPipe.getReadHandle(), 0);
	m_waitSet.read.set_bit(m_wakeupPipe.getReadHandle());
}

SelectReactor::~SelectReactor()
{
}

ReactorHandlerId
SelectReactor::findHandler(ACE_HANDLE handle) const
{
	return m_ptrHandlers->findByHandle(handle);
}

ReactorHandlerId
SelectReactor::registerHandler(
	ACE_HANDLE handle, EventHandlerPtr const& handler, IOEvents events)
{
	assert(handler);
	assert(handle != ACE_INVALID_HANDLE);
	
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	if (m_isInsideDemux) {
		wakeup();
	}
	
	checkCapacity(handle, m_ptrHandlers->size());
	
	Handler new_handler(handle, handler, events, m_dispatchWave, 0);
	ReactorHandlerId id = m_ptrHandlers->add(new_handler);
	// may throw a DuplicateHandleException
	
	setBits(m_waitSet, handle, events);
	
	return id;
}

void
SelectReactor::unregisterHandler(ReactorHandlerId const& id)
{
	assert(id);
	EXCLUSIVE_WAITSET_ACCESS();
	
	ACE_HANDLE handle = HandlerRepository::iterFromId(id)->handle;
	setBits(m_waitSet, handle, Reactor::NO_EVENTS);
	m_ptrHandlers->remove(id);
}

void
SelectReactor::unregisterAllHandlers()
{
	EXCLUSIVE_WAITSET_ACCESS();
	
	m_ptrHandlers->clear();
	m_ptrHandlers->first_inactive = m_ptrHandlers->end();
	m_waitSet.reset();
	m_dispatchSet.reset();
	m_ioIter.reset(0);
	m_waitSet.read.set_bit(m_wakeupPipe.getReadHandle());
}

void
SelectReactor::enableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);

	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	IOEvents const new_events = it->events | events;
	m_ptrHandlers->setEvents(it, new_events);
	setBits(m_waitSet, it->handle, new_events);
}

void
SelectReactor::disableEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	IOEvents const new_events = it->events & ~events;
	m_ptrHandlers->setEvents(it, new_events);
	setBits(m_waitSet, it->handle, new_events);
}

void
SelectReactor::setEvents(ReactorHandlerId const& id, IOEvents events)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);

	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	m_ptrHandlers->setEvents(it, events);
	setBits(m_waitSet, it->handle, events);
}

Reactor::IOEvents
SelectReactor::getEvents(ReactorHandlerId const& id) const
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	HandlerRepository::iterator const& it = HandlerRepository::iterFromId(id);
	return it->events;
}

ReactorTimerId
SelectReactor::registerTimer(EventHandlerPtr const& handler, TimeDelta const* timeout)
{
	assert(handler);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	TimeStamp was_earliest_time = tq->getEarliestTime();
	ReactorTimerId id = tq->add(handler, timeout);
	assert(id);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
	
	return id;
}

void
SelectReactor::rescheduleTimer(ReactorTimerId const& id, TimeDelta const* timeout)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	TimerQueue* tq = m_ptrTimers.get();
	TimeStamp was_earliest_time = tq->getEarliestTime();
	tq->reschedule(id, timeout);
	
	if (m_isInsideDemux && tq->getEarliestTime() < was_earliest_time) {
		wakeup();
	}
}

void
SelectReactor::unregisterTimer(ReactorTimerId const& id)
{
	assert(id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->erase(TimerQueue::iterFromId(id));
}

void
SelectReactor::unregisterAllTimers()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	m_ptrTimers->clear();
}

TimeDelta
SelectReactor::getRemainingTime(ReactorTimerId const& timer_id) const
{
	assert(timer_id);
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	return m_ptrTimers->getRemainingTime(timer_id);
}

Reactor::Status
SelectReactor::handleEvents()
{
	MtGuard<ACE_Lock> guard(*m_ptrMutex);
	
	if (m_isStopped) {
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	bool progress_made = continueIODispatching();
	progress_made |= m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	if (m_isStopped) {
		return STOPPED;
	}
	if (progress_made) {
		/*
		It's mandatory to return here, as otherwise a situation
		is possible where one of the events we've just dispatched
		does all of the pending work so there is nothing more to do,
		and we end up waiting for nothing.
		*/
		return SUCCESS;
	}

	{
		MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
		m_beforeSleepSignal.emit(); // may add / remove handlers
	}
	if (m_isStopped) {
		return STOPPED;
	}
	
	m_dispatchSet = m_waitSet;
	bool woken_up = false;
	bool interrupted = false;
	int res = 0;
	{
		DemultiplexerGuard guard(*m_ptrMutex, *m_ptrDemuxMutex, m_isInsideDemux);
		res = select(woken_up, interrupted);

		// Note: another thread may add/remove handlers while we are in
		// DemultiplexerGuard destructor.
	}
	if (res < 0) {
		assert(!woken_up);
		return interrupted ? INTERRUPTED : FAILURE;
	}
	
	nextDispatchWave();
	m_ioIter.reset(res);
	m_ptrTimers->updateDispatchThreshold();
	
	if (woken_up) {
		m_isStopped = AtomicOps::add(&m_isStoppedAtomic, 0);
		if (m_isStopped) {
			return STOPPED;
		}
	}
	
	if (res > 0) {
		continueIODispatching();
	}
	m_ptrTimers->continueDispatching(*m_ptrMutex, m_isStopped);
	
	return m_isStopped ? STOPPED : SUCCESS;
}

Reactor::Status
SelectReactor::runEventLoop()
{
	Status status = SUCCESS;
	do {
		status = handleEvents();
	} while (status == SUCCESS);
	return status;
}

void
SelectReactor::wakeup()
{
	m_wakeupPipe.activate();
}

void
SelectReactor::stop()
{
	if (AtomicOps::set(&m_isStoppedAtomic, 1) == 0) {
		wakeup();
	}
}

void
SelectReactor::restart()
{
	AtomicOps::set(&m_isStoppedAtomic, 0);
}

sigc::signal<void>&
SelectReactor::beforeSleepSignal()
{
	return m_beforeSleepSignal;
}

int
SelectReactor::select(bool& woken_up, bool& interrupted)
{
	HandleSet& hs = m_dispatchSet;
	int max = -1;
#if !defined(ACE_WIN32)
	max = hs.read.max_set();
	max = std::max(max, hs.write.max_set());
	max = std::max(max, hs.except.max_set());
#endif
	
	int res = 0;
	while (true) {
		struct timeval tv;
		struct timeval* tv_ptr = 0;
		TimeDelta timeout = getSelectTimeout();
		if (timeout != TimeDelta::max()) {
			tv = timeout.toTimeval();
			tv_ptr = &tv;
		}
		res = ::select(
			max + 1, hs.read.fdset(), hs.write.fdset(),
			hs.except.fdset(), tv_ptr
		);
		woken_up = res > 0 && hs.read.is_set(m_wakeupPipe.getReadHandle());
		if (woken_up) {
			m_wakeupPipe.deactivate();
		}
		interrupted = false;
		if (res < 0 && errno == EINTR) {
			if (m_wakeupOnSignal) {
				interrupted = true;
			} else {
				continue;
			}
		}
		break;
	}
	
#if !defined(ACE_WIN32)
	if (res >= 0) {
		hs.read.sync(hs.read.max_set() + 1);
		hs.write.sync(hs.write.max_set() + 1);
		hs.except.sync(hs.except.max_set() + 1);
	}
#endif
	
	return res;
}

TimeDelta
SelectReactor::getSelectTimeout() const
{
	TimeStamp const target_time = m_ptrTimers->getEarliestTime();
	if (target_time == TimeStamp::max()) {
		return TimeDelta::max();
	}
	
	TimeStamp const cur_time = MonotonicTimer::getTimestamp();
	if (cur_time + TimeDelta::fromUsec(2) >= target_time) {
		/*
		These 2 microseconds compensate the effects of
		TimerQueue::updateDispatchThreshold() and
		TimerQueue::ensureTimeAfterThreshold().
		*/
		return TimeDelta::zero();
	}
	
	return target_time - cur_time;
}

void
SelectReactor::nextDispatchWave()
{
	if (++m_dispatchWave == 0) {
		HandlerRepository::iterator it = m_ptrHandlers->begin();
		HandlerRepository::iterator const end = m_ptrHandlers->end();
		for (; it != end; ++it) {
			it->flags = 0;
		}
		++m_dispatchWave; // make it not equal to flags
	}
}

bool
SelectReactor::continueIODispatching()
{
	bool progress_made = false;
	while (!m_isStopped && m_ioIter.next()) {
		ACE_HANDLE handle = m_ioIter.handle;
		if (handle == m_wakeupPipe.getReadHandle()) {
			continue;
		}
		
		typedef HandlerRepository::HandleIdx HandleIdx;
		HandleIdx& idx = m_ptrHandlers->get<HandleTag>();
		HandleIdx::iterator it = idx.find(handle);
		if (it != idx.end() && it->flags != m_dispatchWave) {
			dispatchIO(*it);
			progress_made = true;
		}
	}
	return progress_made;
}

void
SelectReactor::dispatchIO(Handler const& handler)
{
	/*
	Unfortunately we can't dispatch all events in one go, because
	event handlers may erase the current handler, may advance m_ioIter,
	and may even change the dispatch set.
	*/
	
	ACE_HANDLE handle = handler.handle;
	EventHandlerBase* eh = handler.handler.get();
	switch (m_ioIter.phase) {
		case IODispatchIter::WRITE_PHASE:
		if (handler.events & WRITE) {
			dispatchIOEvent(handle, eh, &EventHandlerBase::handleWrite);
		}
		break;
		case IODispatchIter::EXCEPT_PHASE:
		if (handler.events & EXCEPT) {
			dispatchIOEvent(handle, eh, &EventHandlerBase::handleExcept);
		}
		break;
		case IODispatchIter::READ_PHASE:
		if (handler.events & READ) {
			dispatchIOEvent(handle, eh, &EventHandlerBase::handleRead);
		}
	};
}

void
SelectReactor::dispatchIOEvent(
	ACE_HANDLE handle, EventHandlerBase* eh, HandlerFuncPtr func)
{
	MtAntiGuard<ACE_Lock> anti_guard(*m_ptrMutex);
	(eh->*func)(handle);
}

void
SelectReactor::setBits(HandleSet& hs, ACE_HANDLE handle, IOEvents events)
{
	if (events & READ) {
		hs.read.set_bit(handle);
	} else {
		hs.read.clr_bit(handle);
	}
	
	if (events & WRITE) {
		hs.write.set_bit(handle);
	} else {
		hs.write.clr_bit(handle);
	}
	
	if (events & EXCEPT) {
		hs.except.set_bit(handle);
	} else {
		hs.except.clr_bit(handle);
	}
}


void checkCapacity(ACE_HANDLE handle, size_t num_handlers)
{
#if defined(ACE_WIN32)
	if (num_handlers >= FD_SETSIZE) {
#else
	if (handle >= FD_SETSIZE) {
#endif
		throw Reactor::CapacityException();
	}
}


/*==================== SelectReactor::IODispatchIter ====================*/

SelectReactor::IODispatchIter::IODispatchIter(HandleSet const& hs)
:	write_iter(hs.write),
	except_iter(hs.except),
	read_iter(hs.read),
	todo(0),
	handle(ACE_INVALID_HANDLE),
	phase(INITIAL_PHASE)
{
}

void
SelectReactor::IODispatchIter::reset(unsigned todo)
{
	write_iter.reset_state();
	except_iter.reset_state();
	read_iter.reset_state();
	this->todo = todo;
	handle = ACE_INVALID_HANDLE;
	phase = INITIAL_PHASE;
}

bool
SelectReactor::IODispatchIter::next()
{
	if (todo == 0) {
		return false;
	}
	
	// indexed by phase
	ACE_Handle_Set_Iterator* iterators[] = {
		&write_iter, &except_iter, &read_iter
	};

	for (int i = 0; i < 3; ++i) {
		nextPhase();
		handle = (*iterators[phase])();
		if (handle != ACE_INVALID_HANDLE) {
			--todo;
			return true;
		}
	}
	
	// No more active handles exist, although todo is greater than zero.
	// This can happen if a thread calls unregisterAllHandlers()
	// while another thread is inside select().
	// FYI: unregisterAllHandlers() will do its work when the thread
	// that was inside select() is inside DemultiplexerGuard destructor.

	todo = 0;
	return false;
}
